"""
Diálogo principal do plugin GeoCAR Vision
Interface gráfica para seleção de sensor, bandas e projeção
"""

from qgis.PyQt.QtWidgets import (
    QDialog,
    QVBoxLayout,
    QHBoxLayout,
    QLabel,
    QPushButton,
    QGroupBox,
    QMessageBox,
    QProgressBar,
    QFileDialog,
    QButtonGroup,
    QRadioButton,
    QScrollArea,
    QWidget
)
from qgis.PyQt.QtCore import Qt, QThread, pyqtSignal, QUrl, QSize
from qgis.PyQt.QtGui import QPixmap, QCursor, QDesktopServices, QIcon
from qgis.core import (
    QgsProject,
    QgsProcessingContext,
    QgsCoordinateReferenceSystem,
    QgsProcessingFeedback,
    QgsMapLayerProxyModel
)
from qgis.gui import QgsMapLayerComboBox, QgsProjectionSelectionWidget
import os

from ..processing_algorithm import LandsatFusionAlgorithm


class ProcessingThread(QThread):
    """Thread para executar o processamento sem bloquear a interface"""
    
    progress = pyqtSignal(int)
    finished = pyqtSignal(bool, str)
    
    def __init__(self, algorithm, parameters, context):
        super().__init__()
        self.algorithm = algorithm
        self.parameters = parameters
        self.context = context

    def run(self):
        """Executa o processamento"""
        try:
            feedback = QgsProcessingFeedback()
            
            # Usar o método run() do algoritmo em vez de processAlgorithm()
            results = self.algorithm.run(
                self.parameters,
                self.context,
                feedback
            )
            
            if results[0]:  # Se foi bem-sucedido
                self.finished.emit(True, 'Processamento concluído com sucesso!')
            else:
                self.finished.emit(False, f'Erro durante o processamento: {results[1]}')
        except Exception as e:
            self.finished.emit(False, f'Erro durante o processamento: {str(e)}')


class LandsatFusionDialog(QDialog):
    """Diálogo principal do plugin"""

    def __init__(self, iface):
        super().__init__()
        self.iface = iface
        self.processing_thread = None
        self.setup_ui()
        self.setWindowTitle('')
        self.setMinimumSize(650, 800)
        self.resize(650, 800)

    def setup_ui(self):
        """Configura a interface gráfica"""
        main_layout = QVBoxLayout()
        main_layout.setSpacing(10)
        main_layout.setContentsMargins(15, 15, 15, 15)

        # ==================== CABEÇALHO COM LOGO ====================
        header_layout = QVBoxLayout()
        header_layout.setSpacing(5)
        
        # Logo ForgeGeo
        logo_label = QLabel()
        logo_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)),
            'resources',
            'geocar_vision_logo.png'
        )
        if os.path.exists(logo_path):
            pixmap = QPixmap(logo_path)
            # Ajustar tamanho para exibir logo completo
            scaled_pixmap = pixmap.scaled(140, 140, Qt.KeepAspectRatio, Qt.SmoothTransformation)
            logo_label.setPixmap(scaled_pixmap)
            logo_label.setAlignment(Qt.AlignCenter)
        
        # Título (removido - apenas logo)
        
        header_layout.addWidget(logo_label)
        main_layout.addLayout(header_layout)

        # ==================== SELEÇÃO DE SENSOR ====================
        sensor_group = QGroupBox('Seleção de Sensor')
        sensor_layout = QHBoxLayout()
        
        self.sensor_button_group = QButtonGroup()
        self.landsat8_radio = QRadioButton('Landsat 8')
        self.landsat5_radio = QRadioButton('Landsat 5')
        self.cbers4a_radio = QRadioButton('CBERS-4A')
        self.sentinel2_radio = QRadioButton('Sentinel-2')
        
        self.landsat8_radio.setChecked(True)
        
        self.sensor_button_group.addButton(self.landsat8_radio, 0)
        self.sensor_button_group.addButton(self.landsat5_radio, 1)
        self.sensor_button_group.addButton(self.cbers4a_radio, 2)
        self.sensor_button_group.addButton(self.sentinel2_radio, 3)
        
        sensor_layout.addWidget(self.landsat8_radio)
        sensor_layout.addWidget(self.landsat5_radio)
        sensor_layout.addWidget(self.cbers4a_radio)
        sensor_layout.addWidget(self.sentinel2_radio)
        sensor_group.setLayout(sensor_layout)
        main_layout.addWidget(sensor_group)
        
        # Conectar mudança de sensor
        self.sensor_button_group.buttonClicked.connect(self.on_sensor_changed)

        # ==================== BANDAS LANDSAT 8 ====================
        self.l8_bands_group = QGroupBox('Bandas Landsat 8')
        l8_bands_layout = QVBoxLayout()
        l8_bands_layout.setSpacing(8)

        # Banda B4
        l8_b4_layout = QHBoxLayout()
        l8_b4_label = QLabel('Banda B4 (Red):')
        l8_b4_label.setMinimumWidth(180)
        self.l8_b4_combo = QgsMapLayerComboBox()
        self.l8_b4_combo.setFilters(1)
        self.l8_b4_combo.setMinimumHeight(30)
        l8_b4_layout.addWidget(l8_b4_label)
        l8_b4_layout.addWidget(self.l8_b4_combo, 1)
        l8_bands_layout.addLayout(l8_b4_layout)

        # Banda B5
        l8_b5_layout = QHBoxLayout()
        l8_b5_label = QLabel('Banda B5 (Near Infrared):')
        l8_b5_label.setMinimumWidth(180)
        self.l8_b5_combo = QgsMapLayerComboBox()
        self.l8_b5_combo.setFilters(1)
        self.l8_b5_combo.setMinimumHeight(30)
        l8_b5_layout.addWidget(l8_b5_label)
        l8_b5_layout.addWidget(self.l8_b5_combo, 1)
        l8_bands_layout.addLayout(l8_b5_layout)

        # Banda B6
        l8_b6_layout = QHBoxLayout()
        l8_b6_label = QLabel('Banda B6 (SWIR 1):')
        l8_b6_label.setMinimumWidth(180)
        self.l8_b6_combo = QgsMapLayerComboBox()
        self.l8_b6_combo.setFilters(1)
        self.l8_b6_combo.setMinimumHeight(30)
        l8_b6_layout.addWidget(l8_b6_label)
        l8_b6_layout.addWidget(self.l8_b6_combo, 1)
        l8_bands_layout.addLayout(l8_b6_layout)

        # Banda B8
        l8_b8_layout = QHBoxLayout()
        l8_b8_label = QLabel('Banda B8 (Panchromatic):')
        l8_b8_label.setMinimumWidth(180)
        self.l8_b8_combo = QgsMapLayerComboBox()
        self.l8_b8_combo.setFilters(1)
        self.l8_b8_combo.setMinimumHeight(30)
        l8_b8_layout.addWidget(l8_b8_label)
        l8_b8_layout.addWidget(self.l8_b8_combo, 1)
        l8_bands_layout.addLayout(l8_b8_layout)

        self.l8_bands_group.setLayout(l8_bands_layout)
        main_layout.addWidget(self.l8_bands_group)

        # ==================== BANDAS LANDSAT 5 ====================
        self.l5_bands_group = QGroupBox('Bandas Landsat 5')
        l5_bands_layout = QVBoxLayout()
        l5_bands_layout.setSpacing(8)

        # Banda B3
        l5_b3_layout = QHBoxLayout()
        l5_b3_label = QLabel('Banda B3 (Red):')
        l5_b3_label.setMinimumWidth(180)
        self.l5_b3_combo = QgsMapLayerComboBox()
        self.l5_b3_combo.setFilters(1)
        self.l5_b3_combo.setMinimumHeight(30)
        l5_b3_layout.addWidget(l5_b3_label)
        l5_b3_layout.addWidget(self.l5_b3_combo, 1)
        l5_bands_layout.addLayout(l5_b3_layout)

        # Banda B4
        l5_b4_layout = QHBoxLayout()
        l5_b4_label = QLabel('Banda B4 (Near Infrared):')
        l5_b4_label.setMinimumWidth(180)
        self.l5_b4_combo = QgsMapLayerComboBox()
        self.l5_b4_combo.setFilters(1)
        self.l5_b4_combo.setMinimumHeight(30)
        l5_b4_layout.addWidget(l5_b4_label)
        l5_b4_layout.addWidget(self.l5_b4_combo, 1)
        l5_bands_layout.addLayout(l5_b4_layout)

        # Banda B5
        l5_b5_layout = QHBoxLayout()
        l5_b5_label = QLabel('Banda B5 (SWIR):')
        l5_b5_label.setMinimumWidth(180)
        self.l5_b5_combo = QgsMapLayerComboBox()
        self.l5_b5_combo.setFilters(1)
        self.l5_b5_combo.setMinimumHeight(30)
        l5_b5_layout.addWidget(l5_b5_label)
        l5_b5_layout.addWidget(self.l5_b5_combo, 1)
        l5_bands_layout.addLayout(l5_b5_layout)

        self.l5_bands_group.setLayout(l5_bands_layout)
        self.l5_bands_group.setVisible(False)
        main_layout.addWidget(self.l5_bands_group)

        # ==================== BANDAS CBERS-4A ====================
        self.cb4a_bands_group = QGroupBox('Bandas CBERS-4A')
        cb4a_bands_layout = QVBoxLayout()
        cb4a_bands_layout.setSpacing(8)

        # Banda B3
        cb4a_b3_layout = QHBoxLayout()
        cb4a_b3_label = QLabel('Banda B3 (Red):')
        cb4a_b3_label.setMinimumWidth(180)
        self.cb4a_b3_combo = QgsMapLayerComboBox()
        self.cb4a_b3_combo.setFilters(1)
        self.cb4a_b3_combo.setMinimumHeight(30)
        cb4a_b3_layout.addWidget(cb4a_b3_label)
        cb4a_b3_layout.addWidget(self.cb4a_b3_combo, 1)
        cb4a_bands_layout.addLayout(cb4a_b3_layout)

        # Banda B4
        cb4a_b4_layout = QHBoxLayout()
        cb4a_b4_label = QLabel('Banda B4 (Near Infrared):')
        cb4a_b4_label.setMinimumWidth(180)
        self.cb4a_b4_combo = QgsMapLayerComboBox()
        self.cb4a_b4_combo.setFilters(1)
        self.cb4a_b4_combo.setMinimumHeight(30)
        cb4a_b4_layout.addWidget(cb4a_b4_label)
        cb4a_b4_layout.addWidget(self.cb4a_b4_combo, 1)
        cb4a_bands_layout.addLayout(cb4a_b4_layout)

        # Banda B5
        cb4a_b5_layout = QHBoxLayout()
        cb4a_b5_label = QLabel('Banda B5 (SWIR):')
        cb4a_b5_label.setMinimumWidth(180)
        self.cb4a_b5_combo = QgsMapLayerComboBox()
        self.cb4a_b5_combo.setFilters(1)
        self.cb4a_b5_combo.setMinimumHeight(30)
        cb4a_b5_layout.addWidget(cb4a_b5_label)
        cb4a_b5_layout.addWidget(self.cb4a_b5_combo, 1)
        cb4a_bands_layout.addLayout(cb4a_b5_layout)

        self.cb4a_bands_group.setLayout(cb4a_bands_layout)
        self.cb4a_bands_group.setVisible(False)
        main_layout.addWidget(self.cb4a_bands_group)

        # ==================== BANDAS SENTINEL-2 ====================
        self.s2_bands_group = QGroupBox('Bandas Sentinel-2')
        s2_bands_layout = QVBoxLayout()
        s2_bands_layout.setSpacing(8)

        # Banda B4
        s2_b4_layout = QHBoxLayout()
        s2_b4_label = QLabel('Banda B4 (Red):')
        s2_b4_label.setMinimumWidth(180)
        self.s2_b4_combo = QgsMapLayerComboBox()
        self.s2_b4_combo.setFilters(1)
        self.s2_b4_combo.setMinimumHeight(30)
        s2_b4_layout.addWidget(s2_b4_label)
        s2_b4_layout.addWidget(self.s2_b4_combo, 1)
        s2_bands_layout.addLayout(s2_b4_layout)

        # Banda B8
        s2_b8_layout = QHBoxLayout()
        s2_b8_label = QLabel('Banda B8 (Near Infrared):')
        s2_b8_label.setMinimumWidth(180)
        self.s2_b8_combo = QgsMapLayerComboBox()
        self.s2_b8_combo.setFilters(1)
        self.s2_b8_combo.setMinimumHeight(30)
        s2_b8_layout.addWidget(s2_b8_label)
        s2_b8_layout.addWidget(self.s2_b8_combo, 1)
        s2_bands_layout.addLayout(s2_b8_layout)

        # Banda B11
        s2_b11_layout = QHBoxLayout()
        s2_b11_label = QLabel('Banda B11 (SWIR):')
        s2_b11_label.setMinimumWidth(180)
        self.s2_b11_combo = QgsMapLayerComboBox()
        self.s2_b11_combo.setFilters(1)
        self.s2_b11_combo.setMinimumHeight(30)
        s2_b11_layout.addWidget(s2_b11_label)
        s2_b11_layout.addWidget(self.s2_b11_combo, 1)
        s2_bands_layout.addLayout(s2_b11_layout)

        self.s2_bands_group.setLayout(s2_bands_layout)
        self.s2_bands_group.setVisible(False)
        main_layout.addWidget(self.s2_bands_group)

        # ==================== PROJEÇÃO ====================
        projection_group = QGroupBox('Sistema de Coordenadas de Saída')
        projection_layout = QVBoxLayout()
        
        self.crs_widget = QgsProjectionSelectionWidget()
        self.crs_widget.setCrs(QgsCoordinateReferenceSystem('EPSG:4674'))
        projection_layout.addWidget(self.crs_widget)
        
        projection_group.setLayout(projection_layout)
        main_layout.addWidget(projection_group)

        # ==================== ARQUIVO DE SAÍDA ====================
        output_group = QGroupBox('Arquivo de Saída')
        output_layout = QHBoxLayout()
        
        self.output_path_label = QLabel('Nenhum arquivo selecionado')
        output_layout.addWidget(self.output_path_label, 1)
        
        self.browse_button = QPushButton('Procurar...')
        self.browse_button.clicked.connect(self.select_output_path)
        self.browse_button.setMinimumHeight(30)
        output_layout.addWidget(self.browse_button)
        
        output_group.setLayout(output_layout)
        main_layout.addWidget(output_group)

        # ==================== BARRA DE PROGRESSO ====================
        self.progress_bar = QProgressBar()
        self.progress_bar.setVisible(False)
        self.progress_bar.setMinimumHeight(25)
        main_layout.addWidget(self.progress_bar)

        # ==================== BOTÕES DE AÇÃO ====================
        button_layout = QHBoxLayout()
        
        self.process_button = QPushButton('Processar')
        self.process_button.clicked.connect(self.process)
        self.process_button.setMinimumHeight(35)
        
        self.close_button = QPushButton('Fechar')
        self.close_button.clicked.connect(self.close)
        self.close_button.setMinimumHeight(35)
        
        button_layout.addWidget(self.process_button)
        button_layout.addWidget(self.close_button)
        
        main_layout.addLayout(button_layout)

        # ==================== RODAPÉ COM BOTÕES ====================
        footer_layout = QHBoxLayout()
        footer_layout.setSpacing(15)
        footer_layout.setContentsMargins(0, 10, 0, 0)
        
        # Botão Instagram elegante
        instagram_btn = QPushButton()
        instagram_icon_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)),
            'resources',
            'instagram_icon.png'
        )
        if os.path.exists(instagram_icon_path):
            instagram_btn.setIcon(QIcon(instagram_icon_path))
            instagram_btn.setIconSize(QSize(28, 28))
        instagram_btn.setFixedSize(50, 50)
        instagram_btn.setStyleSheet("""
            QPushButton {
                background-color: #f09433;
                border: none;
                border-radius: 10px;
                padding: 8px;
            }
            QPushButton:hover {
                background-color: #e07d2f;
            }
            QPushButton:pressed {
                background-color: #d06d25;
            }
        """)
        instagram_btn.setToolTip('Visite nosso Instagram: @forgegeo')
        instagram_btn.setCursor(QCursor(Qt.PointingHandCursor))
        instagram_btn.clicked.connect(self.open_instagram)
        
        # Botão Website elegante
        web_btn = QPushButton()
        web_icon_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)),
            'resources',
            'web_icon.png'
        )
        if os.path.exists(web_icon_path):
            web_btn.setIcon(QIcon(web_icon_path))
            web_btn.setIconSize(QSize(28, 28))
        web_btn.setFixedSize(50, 50)
        web_btn.setStyleSheet("""
            QPushButton {
                background-color: #1f6feb;
                border: none;
                border-radius: 10px;
                padding: 8px;
            }
            QPushButton:hover {
                background-color: #1a5dd4;
            }
            QPushButton:pressed {
                background-color: #1550c4;
            }
        """)
        web_btn.setToolTip('Visite nosso site: forgegeo.com.br')
        web_btn.setCursor(QCursor(Qt.PointingHandCursor))
        web_btn.clicked.connect(self.open_website)
        
        footer_layout.addStretch()
        footer_layout.addWidget(instagram_btn)
        footer_layout.addWidget(web_btn)
        footer_layout.addStretch()
        
        main_layout.addLayout(footer_layout)

        self.setLayout(main_layout)
        self.output_path = None

    def on_sensor_changed(self, button):
        """Callback quando o sensor é alterado"""
        sensor_id = self.sensor_button_group.id(button)
        
        # Ocultar todos os grupos de bandas
        self.l8_bands_group.setVisible(False)
        self.l5_bands_group.setVisible(False)
        self.cb4a_bands_group.setVisible(False)
        self.s2_bands_group.setVisible(False)
        
        # Mostrar o grupo correspondente
        if sensor_id == 0:  # Landsat 8
            self.l8_bands_group.setVisible(True)
            self.filter_layers_by_sensor('Landsat 8')
        elif sensor_id == 1:  # Landsat 5
            self.l5_bands_group.setVisible(True)
            self.filter_layers_by_sensor('Landsat 5')
        elif sensor_id == 2:  # CBERS-4A
            self.cb4a_bands_group.setVisible(True)
            self.filter_layers_by_sensor('CBERS-4A')
        elif sensor_id == 3:  # Sentinel-2
            self.s2_bands_group.setVisible(True)
            self.filter_layers_by_sensor('Sentinel-2')
    
    def filter_layers_by_sensor(self, sensor_name):
        """Filtra as camadas disponíveis conforme o sensor selecionado"""
        from qgis.core import QgsProject
        
        # Obter todas as camadas raster do projeto
        layers = QgsProject.instance().mapLayers().values()
        
        # Palavras-chave para cada sensor
        keywords = {
            'Landsat 8': ['B4', 'B5', 'B6', 'B8', 'L8', 'Landsat 8'],
            'Landsat 5': ['B3', 'B4', 'B5', 'L5', 'Landsat 5', 'TM'],
            'CBERS-4A': ['B3', 'B4', 'B5', 'CBERS', 'CB4A'],
            'Sentinel-2': ['B4', 'B8', 'B11', 'S2', 'Sentinel', 'MSI']
        }
        
        sensor_keywords = keywords.get(sensor_name, [])
        
        # Atualizar combos de banda com filtro
        if sensor_name == 'Landsat 8':
            self.update_layer_combo(self.l8_b4_combo, sensor_keywords)
            self.update_layer_combo(self.l8_b5_combo, sensor_keywords)
            self.update_layer_combo(self.l8_b6_combo, sensor_keywords)
            self.update_layer_combo(self.l8_b8_combo, sensor_keywords)
        elif sensor_name == 'Landsat 5':
            self.update_layer_combo(self.l5_b3_combo, sensor_keywords)
            self.update_layer_combo(self.l5_b4_combo, sensor_keywords)
            self.update_layer_combo(self.l5_b5_combo, sensor_keywords)
        elif sensor_name == 'CBERS-4A':
            self.update_layer_combo(self.cb4a_b3_combo, sensor_keywords)
            self.update_layer_combo(self.cb4a_b4_combo, sensor_keywords)
            self.update_layer_combo(self.cb4a_b5_combo, sensor_keywords)
        elif sensor_name == 'Sentinel-2':
            self.update_layer_combo(self.s2_b4_combo, sensor_keywords)
            self.update_layer_combo(self.s2_b8_combo, sensor_keywords)
            self.update_layer_combo(self.s2_b11_combo, sensor_keywords)
    
    def update_layer_combo(self, combo, keywords):
        """Atualiza o combo de camadas com filtro de palavras-chave"""
        from qgis.core import QgsProject, QgsRasterLayer
        
        # Limpar o combo
        combo.setFilters(QgsMapLayerProxyModel.RasterLayer)
        
        # Obter todas as camadas raster
        layers = QgsProject.instance().mapLayers().values()
        raster_layers = [l for l in layers if isinstance(l, QgsRasterLayer)]
        
        # Filtrar por palavras-chave
        filtered_layers = []
        for layer in raster_layers:
            layer_name = layer.name().upper()
            for keyword in keywords:
                if keyword.upper() in layer_name:
                    filtered_layers.append(layer)
                    break

    def select_output_path(self):
        """Abre diálogo para selecionar caminho de saída"""
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            'Salvar raster processado',
            '',
            'GeoTIFF (*.tif);;All Files (*.*)'
        )
        
        if file_path:
            if not file_path.lower().endswith('.tif'):
                file_path += '.tif'
            
            self.output_path = file_path
            self.output_path_label.setText(os.path.basename(file_path))

    def open_instagram(self):
        """Abre o Instagram da ForgeGeo"""
        QDesktopServices.openUrl(QUrl('https://www.instagram.com/forgegeo/'))

    def open_website(self):
        """Abre o website da ForgeGeo"""
        QDesktopServices.openUrl(QUrl('https://forgegeo.com.br'))

    def validate_inputs(self):
        """Valida os dados de entrada"""
        sensor_id = self.sensor_button_group.checkedId()
        
        if sensor_id == 0:  # Landsat 8
            if self.l8_b4_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B4 do Landsat 8')
                return False
            if self.l8_b5_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B5 do Landsat 8')
                return False
            if self.l8_b6_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B6 do Landsat 8')
                return False
            if self.l8_b8_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B8 do Landsat 8')
                return False
        
        elif sensor_id == 1:  # Landsat 5
            if self.l5_b3_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B3 do Landsat 5')
                return False
            if self.l5_b4_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B4 do Landsat 5')
                return False
            if self.l5_b5_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B5 do Landsat 5')
                return False
        
        elif sensor_id == 2:  # CBERS-4A
            if self.cb4a_b3_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B3 do CBERS-4A')
                return False
            if self.cb4a_b4_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B4 do CBERS-4A')
                return False
            if self.cb4a_b5_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B5 do CBERS-4A')
                return False
        
        elif sensor_id == 3:  # Sentinel-2
            if self.s2_b4_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B4 do Sentinel-2')
                return False
            if self.s2_b8_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B8 do Sentinel-2')
                return False
            if self.s2_b11_combo.currentLayer() is None:
                QMessageBox.warning(self, 'Aviso', 'Selecione a Banda B11 do Sentinel-2')
                return False
        
        if self.output_path is None:
            QMessageBox.warning(self, 'Aviso', 'Selecione um caminho de saída')
            return False
        
        return True

    def process(self):
        """Inicia o processamento"""
        if not self.validate_inputs():
            return

        sensor_type = self.sensor_button_group.checkedId()
        
        parameters = {
            'SENSOR_TYPE': sensor_type,
            'OUTPUT_CRS': self.crs_widget.crs(),
            'OUTPUT_RASTER': str(self.output_path) if self.output_path else '',
            # Inicializar todos os parametros como None
            'INPUT_L8_B4': None,
            'INPUT_L8_B5': None,
            'INPUT_L8_B6': None,
            'INPUT_L8_B8': None,
            'INPUT_L5_B3': None,
            'INPUT_L5_B4': None,
            'INPUT_L5_B5': None,
            'INPUT_CB4A_B3': None,
            'INPUT_CB4A_B4': None,
            'INPUT_CB4A_B5': None,
            'INPUT_S2_B4': None,
            'INPUT_S2_B8': None,
            'INPUT_S2_B11': None,
        }
        
        if sensor_type == 0:  # Landsat 8
            parameters['INPUT_L8_B4'] = self.l8_b4_combo.currentLayer()
            parameters['INPUT_L8_B5'] = self.l8_b5_combo.currentLayer()
            parameters['INPUT_L8_B6'] = self.l8_b6_combo.currentLayer()
            parameters['INPUT_L8_B8'] = self.l8_b8_combo.currentLayer()
        
        elif sensor_type == 1:  # Landsat 5
            parameters['INPUT_L5_B3'] = self.l5_b3_combo.currentLayer()
            parameters['INPUT_L5_B4'] = self.l5_b4_combo.currentLayer()
            parameters['INPUT_L5_B5'] = self.l5_b5_combo.currentLayer()
        
        elif sensor_type == 2:  # CBERS-4A
            parameters['INPUT_CB4A_B3'] = self.cb4a_b3_combo.currentLayer()
            parameters['INPUT_CB4A_B4'] = self.cb4a_b4_combo.currentLayer()
            parameters['INPUT_CB4A_B5'] = self.cb4a_b5_combo.currentLayer()
        
        elif sensor_type == 3:  # Sentinel-2
            parameters['INPUT_S2_B4'] = self.s2_b4_combo.currentLayer()
            parameters['INPUT_S2_B8'] = self.s2_b8_combo.currentLayer()
            parameters['INPUT_S2_B11'] = self.s2_b11_combo.currentLayer()

        self.process_button.setEnabled(False)
        self.progress_bar.setVisible(True)
        self.progress_bar.setValue(0)

        context = QgsProcessingContext()
        context.setProject(QgsProject.instance())

        algorithm = LandsatFusionAlgorithm()
        self.processing_thread = ProcessingThread(algorithm, parameters, context)
        self.processing_thread.progress.connect(self.progress_bar.setValue)
        self.processing_thread.finished.connect(self.on_processing_finished)
        self.processing_thread.start()

    def on_processing_finished(self, success, message):
        """Callback quando o processamento termina"""
        self.progress_bar.setVisible(False)
        self.process_button.setEnabled(True)

        if success:
            QMessageBox.information(self, 'Sucesso', message)
            
            layer = self.iface.addRasterLayer(
                self.output_path,
                os.path.splitext(os.path.basename(self.output_path))[0]
            )
            
            if layer:
                self.iface.mapCanvas().zoomToFullExtent()
        else:
            QMessageBox.critical(self, 'Erro', message)
