// Post-load helpers for the MapLibre HTML generated by qmap_maplibre.py
// This script expects the map instance to be passed to window.qmap_init_after_load
// or will be called by the HTML generator as window.qmap_init_after_load(map).

(function() {
  // Define a single post-load initializer that will be invoked with the map instance
  window.qmap_init_after_load = function(map) {
    try {
      // Add helper: addWfsLayer (mirrors previous inline implementation)
      // 付帯ヘルパー: スタイルから幾何タイプを推定
      // 幾何推定ヘルパー（フォールバック簡易版）
      function inferGeomFromData(features) {
        if (!features || !features.length) return null;
        for (var i=0;i<features.length;i++) {
          var g = features[i] && features[i].geometry;
          if (!g || !g.type) continue;
          var t = g.type.toLowerCase();
          if (t.indexOf('point')>=0) return 'Point';
          if (t.indexOf('line')>=0) return 'LineString';
          if (t.indexOf('polygon')>=0) return 'Polygon';
        }
        return null;
      }

      function addWfsLayer(wfsUrl, sourceId, layerId, labelId, layerTitle, labelTitle) {
        // Guard: avoid adding the same WFS source multiple times across
        // re-initializations (map.setStyle triggers re-run of this script).
        // Use a global Set on window so it survives style reloads in the
        // same page context.
        try {
          if (!window.__qmap_seen_wfs_sources) window.__qmap_seen_wfs_sources = new Set();
          if (window.__qmap_seen_wfs_sources.has(sourceId)) {
            console.log('[addWfsLayer] source already registered, skipping:', sourceId);
            return;
          }
          // mark immediately to avoid races from concurrent calls
          window.__qmap_seen_wfs_sources.add(sourceId);
        } catch (e) { console.warn('failed to guard repeated WFS registration', e); }
        // 1) まず /maplibre-style?typename=sourceId を取得して QGIS 由来スタイルを注入する試行
        // 2) 失敗した場合のみ WFS GeoJSON を直接取得して簡易フォールバックスタイルを付与
        // 3) ポリゴン「ブラシなし」は QGIS スタイル注入成功時は fill レイヤ未生成となり輪郭線のみ (alpha=0 判定)
        // 4) 成功/失敗双方でラベルレイヤは必ず追加

        const STYLE_ENDPOINT = '/maplibre-style?typename=' + encodeURIComponent(sourceId);
        const fetchTimeoutMs = 5000;
        let aborted = false;

        function withTimeout(promise, ms) {
          if (typeof AbortController === 'undefined') return promise; // 古いブラウザ
          const controller = new AbortController();
          const id = setTimeout(() => { aborted = true; controller.abort(); }, ms);
          return Promise.race([
            promise(controller.signal).finally(() => clearTimeout(id)),
            new Promise((_r, rej) => setTimeout(() => rej(new Error('timeout')), ms + 50))
          ]);
        }

        function injectQgisStyle(styleJson) {
          try {
            if (!styleJson || !styleJson.sources || !styleJson.sources[sourceId]) {
              console.warn('[style-inject] 対象ソースが style JSON に含まれないためスキップ:', sourceId);
              return false;
            }
            // ソース追加 / 更新
            if (!map.getSource(sourceId)) {
              map.addSource(sourceId, styleJson.sources[sourceId]);
              console.log('[style-inject] addSource:', sourceId, styleJson.sources[sourceId]);
            } else {
              // 既存ソースは GeoJSON の場合 setData で更新可能
              try {
                const existing = map.getSource(sourceId);
                if (existing && typeof existing.setData === 'function' && styleJson.sources[sourceId].data) {
                  existing.setData(styleJson.sources[sourceId].data);
                  console.log('[style-inject] update existing source data via setData');
                }
              } catch (e) {
                console.warn('[style-inject] ソース更新失敗 (非致命):', e);
              }
            }
            // レイヤ追加 (sourceId を参照するもののみ / 既存重複はスキップ)
            const styleLayers = styleJson.layers || [];
            let addedCount = 0;
            styleLayers.forEach(function(ly) {
              if (!ly || ly.source !== sourceId) return;
              if (map.getLayer(ly.id)) return;
              try {
                map.addLayer(ly);
                addedCount++;
              } catch (e) {
                console.warn('[style-inject] addLayer 失敗:', ly.id, e);
              }
            });
            console.log(`[style-inject] source=${sourceId} 追加レイヤ数=${addedCount}`);
            return addedCount > 0;
          } catch (e) {
            console.warn('[style-inject] 例外 (非致命):', e);
            return false;
          }
        }

        function ensureLabelLayer(geomTypeGuess) {
          if (map.getLayer(labelId)) return;
          const layout = { 'text-field': ['get', 'label'], 'text-size': 14 };
          if (geomTypeGuess === 'LineString') layout['symbol-placement'] = 'line';
          map.addLayer({
            id: labelId,
            type: 'symbol',
            source: sourceId,
            filter: ['has', 'label'],
            layout: layout,
            paint: { 'text-color': '#000', 'text-halo-color': '#fff', 'text-halo-width': 2 }
          });
          console.log('[label] 追加:', labelId);
        }

        function registerLayersInControl() {
          if (typeof wmtsLayers === 'undefined' || !Array.isArray(wmtsLayers)) return;
            try {
              const current = map.getStyle().layers || [];
              current.forEach(function(ly) {
                try {
                  if (ly.source === sourceId) {
                    // Avoid duplicates by id or title
                    if (!wmtsLayers.some(function(e) { return e && (e.id === ly.id || e.title === (ly.id.indexOf('label') >= 0 ? labelTitle : layerTitle)); })) {
                      const title = (ly.id.indexOf('label') >= 0) ? labelTitle : layerTitle;
                      wmtsLayers.push({ id: ly.id, title });
                      console.log('[layerControl] 登録:', ly.id);
                    }
                  }
                } catch (e) { console.warn('registerLayersInControl inner failure', e); }
              });
            } catch (e) { console.warn('[layerControl] 登録失敗', e); }
        }

        function determineGeomTypeFromLayers() {
          try {
            const types = new Set();
            (map.getStyle().layers || []).forEach(function(ly) { if (ly.source === sourceId) types.add(ly.type); });
            if (types.has('fill')) return 'Polygon';
            if (types.has('line')) return 'LineString';
            if (types.has('circle')) return 'Point';
          } catch (e) {}
          return null;
        }

        // フォールバック WFS フェッチ処理 (style 取得失敗時のみ呼ぶ)
        function proceedWfsFetch() {
        
  const controller = typeof AbortController !== 'undefined' ? new AbortController() : null;
  const signal = controller ? controller.signal : null;
  let timeoutId = null;
  if (controller) timeoutId = setTimeout(() => controller.abort(), 5000);

        fetch(wfsUrl, { signal })
          .then(function(resp) {
            if (controller && timeoutId) clearTimeout(timeoutId);
            if (!resp.ok) throw new Error('WFS fetch failed: ' + resp.status);
            return resp.json();
          })
          .then(function(data) {
            try {
              if (data && data.features && data.features.length > 0) {
                const _props0 = data.features[0].properties || null;
                console.log('WFS: first feature properties ->', _props0);
                if (_props0 && typeof _props0._qgis_style !== 'undefined') {
                  console.log('WFS: _qgis_style ->', _props0._qgis_style);
                }
              } else {
                console.log('WFS: no features returned');
              }

              if (!map.getSource(sourceId)) {
                map.addSource(sourceId, { type: 'geojson', data: data });
              } else {
                try {
                  var s = map.getSource(sourceId);
                  if (s && typeof s.setData === 'function') { s.setData(data); }
                } catch (e) { console.warn('Failed to update existing source data', e); }
              }

              var geomType = null;
              if (data && data.features && data.features.length > 0) {
                for (var fi = 0; fi < data.features.length; fi++) {
                  var g = data.features[fi] && data.features[fi].geometry;
      
                // スタイルから幾何タイプを推定（注入成功時のラベル配置に利用）
                function inferGeomFromStyleBySource(srcId) {
                  try {
                    var types = new Set();
                    var allLayers = map.getStyle().layers || [];
                    for (var i = 0; i < allLayers.length; i++) {
                      var lyr = allLayers[i];
                      if (lyr && lyr.source === srcId && lyr.type) types.add(String(lyr.type));
                    }
                    if (types.has('fill')) return 'Polygon';
                    if (types.has('line')) return 'LineString';
                    if (types.has('circle')) return 'Point';
                  } catch (e) {}
                  return null;
                }
                  if (!g || !g.type) continue;
                  var coords = g.coordinates;
                  if (coords && Array.isArray(coords) && coords.length === 0) continue;
                  geomType = String(g.type);
                  break;
                }
              }

              if (geomType) {
                geomType = geomType.charAt(0).toUpperCase() + geomType.slice(1);
                if (geomType.indexOf('Multi') === 0) { geomType = geomType.replace(/Multi/i, ''); }
              }

              // スタイルは /maplibre-style エンドポイントから既に適用されているため、
              // レイヤーが既に存在する場合は何もしない（スタイルを上書きしない）
              // レイヤーが存在しない場合のみ、フォールバックスタイルで追加
              var layerExists = false;
              try {
                // Check if any layers using this source already exist
                // (they would have been added by the style JSON)
                var allLayers = map.getStyle().layers || [];
                for (var i = 0; i < allLayers.length; i++) {
                  if (allLayers[i].source === sourceId) {
                    layerExists = true;
                    console.log('Layer already exists in style for source:', sourceId);
                    break;
                  }
                }
              } catch (e) {
                console.warn('Failed to check existing layers', e);
              }

              // Only add a MINIMAL fallback style if style injection failed.
              // Goal: 表示は確保するが「QGISスタイル」と誤解させないシンプル記号。
              // Polygon は線のみ、Point は小さなアウトライン付き円、Line は細線。
              if (!layerExists) {
                console.warn('No QGIS style injected for source ' + sourceId + '; using minimal fallback (data-only look)');
                if (!geomType || geomType === 'Point' || geomType === 'MultiPoint') {
                  if (!map.getLayer(layerId)) {
                    map.addLayer({
                      id: layerId,
                      type: 'circle',
                      source: sourceId,
                      paint: {
                        'circle-radius': 4,
                        'circle-color': '#ffffff',
                        'circle-stroke-color': '#555555',
                        'circle-stroke-width': 1
                      },
                      layout: { 'visibility': 'visible' }
                    });
                  }
                } else if (geomType === 'LineString') {
                  if (!map.getLayer(layerId)) {
                    map.addLayer({
                      id: layerId,
                      type: 'line',
                      source: sourceId,
                      paint: { 'line-color': '#555555', 'line-width': 1, 'line-opacity': 1.0 },
                      layout: { 'visibility': 'visible' }
                    });
                  }
                } else if (geomType === 'Polygon') {
                  // 線のみ（塗り無し）
                  if (!map.getLayer(layerId)) {
                    map.addLayer({
                      id: layerId,
                      type: 'line',
                      source: sourceId,
                      paint: { 'line-color': '#555555', 'line-width': 1, 'line-opacity': 1.0 },
                      layout: { 'visibility': 'visible' }
                    });
                  }
                } else {
                  // 不明 → point扱い
                  if (!map.getLayer(layerId)) {
                    map.addLayer({
                      id: layerId,
                      type: 'circle',
                      source: sourceId,
                      paint: { 'circle-radius': 4, 'circle-color': '#ffffff', 'circle-stroke-color': '#555555', 'circle-stroke-width': 1 },
                      layout: { 'visibility': 'visible' }
                    });
                  }
                }
              }

        // ラベルレイヤー追加
        ensureLabelLayer(geomType);

              // Expose added layers to the WMTS/vector layer control
              // スタイルから読み込まれたレイヤーも含めて、すべてのWFSレイヤーを登録
              if (typeof wmtsLayers !== 'undefined' && Array.isArray(wmtsLayers)) {
                // Check if this layer is already in wmtsLayers (from style JSON)
                var layerExists = wmtsLayers.some(function(l) { return l && (l.id === layerId || l.id === labelId); });
                if (!layerExists) {
                  // Only add if not already present
                  // Get actual layer IDs from the style (may differ from our expected IDs)
                  try {
                    var styleLayers = map.getStyle().layers || [];
                    var foundIds = [];
                    for (var i = 0; i < styleLayers.length; i++) {
                      if (styleLayers[i].source === sourceId) {
                        foundIds.push(styleLayers[i].id);
                      }
                    }
                    // Register all layers from this source
                    for (var j = 0; j < foundIds.length; j++) {
                      var fid = foundIds[j];
                      // Skip if already registered
                      if (wmtsLayers.some(function(l) { return l && l.id === fid; })) continue;
                      // Determine title based on layer type
                      var fTitle = layerTitle;
                      if (fid.indexOf('label') >= 0 || fid.indexOf('symbol') >= 0) {
                        fTitle = labelTitle;
                      }
                      wmtsLayers.push({ id: fid, title: fTitle });
                    }
                  } catch (e) {
                    console.warn('Failed to register style layers in control', e);
                    // Fallback: register expected IDs
                    wmtsLayers.push({ id: layerId, title: layerTitle });
                    wmtsLayers.push({ id: labelId, title: labelTitle });
                  }
                }
              }

            } catch (e) { console.warn('Failed to add WFS layer after successful fetch:', e); }
          })
          .catch(function(err) { console.warn('WFS fetch error (skipping WFS layer):', err); });
        }

        // ---- スタイル取得フェーズ ----
        withTimeout((sig) => fetch(STYLE_ENDPOINT, { signal: sig }))
          .then(resp => {
            if (!resp.ok) throw new Error('style HTTP ' + resp.status);
            return resp.json();
          })
          .then(styleJson => {
            console.log('[style-inject] 取得成功:', STYLE_ENDPOINT);
            const injected = injectQgisStyle(styleJson);
            let geomTypeGuess = determineGeomTypeFromLayers();
            if (!geomTypeGuess) {
              // style JSON から推定できない場合、後段 WFS データで再推定するため fallback fetch へ
              if (!injected) throw new Error('no layers injected');
            }
            ensureLabelLayer(geomTypeGuess);
            registerLayersInControl();
          })
          .catch(err => {
            console.warn('[style-inject] 失敗 -> フォールバックWFS適用:', err.message || err);
            proceedWfsFetch();
          });
      }

      // Expose addWfsLayer globally for other scripts if needed
      window.qmap_addWfsLayer = addWfsLayer;

      // Populate GetCapabilities UI and call addWfsLayer for each available type
      try {
        fetch('/wfs?SERVICE=WFS&REQUEST=GetCapabilities')
          .then(resp => resp.text())
          .then(xmlText => {
            const parser = new window.DOMParser();
            const xml = parser.parseFromString(xmlText, 'text/xml');
            const featureTypes = xml.getElementsByTagName('FeatureType');
            const control = document.getElementById('layerControl');
            
            // WFSレイヤー用のチェックボックスを作成する前に、既存のものをチェック
            var processedSources = new Set();
            
            for (let i = 0; i < featureTypes.length; i++) {
              const ft = featureTypes[i];
              const nameElem = ft.getElementsByTagName('Name')[0];
              const titleElem = ft.getElementsByTagName('Title')[0];
              if (!nameElem) continue;
              const id = nameElem.textContent;
              const safeId = encodeURIComponent(id);
              const expectedLayerId = id + '_layer';
              const expectedLabelId = id + '_label';
              const labelText = (titleElem ? titleElem.textContent : id);

              // 既にwmtsLayersに登録されているかチェック（重複防止）
              var alreadyInWmts = false;
              if (typeof wmtsLayers !== 'undefined' && Array.isArray(wmtsLayers)) {
                alreadyInWmts = wmtsLayers.some(function(l) {
                  return l && (l.id === id || l.id === expectedLayerId || l.title === labelText || l.title === `WFS: ${id}`);
                });
              }
              
              // UIに既に追加されているかチェック
              var existingCb = document.getElementById('cb_' + safeId);
              
              if (!existingCb && !alreadyInWmts) {
                const row = document.createElement('div');
                row.style.marginBottom = '4px';
                const cb = document.createElement('input'); 
                cb.type = 'checkbox'; 
                cb.id = 'cb_' + safeId; 
                cb.checked = true; 
                cb.style.marginRight = '6px';
                const label = document.createElement('label'); 
                label.htmlFor = cb.id; 
                label.innerHTML = labelText;
                row.appendChild(cb); 
                row.appendChild(label); 
                control.appendChild(row);

                cb.addEventListener('change', function() {
                  try {
                    const visibility = cb.checked ? 'visible' : 'none';
                    // スタイルJSONから読み込まれたレイヤーを含む、このソースを使用するすべてのレイヤーを制御
                    var controlledCount = 0;
                    try {
                      var allLayers = map.getStyle().layers || [];
                      for (var li = 0; li < allLayers.length; li++) {
                        var layer = allLayers[li];
                        if (layer.source === id) {
                          // このソースを使用するすべてのレイヤーの表示を切り替え
                          try {
                            map.setLayoutProperty(layer.id, 'visibility', visibility);
                            controlledCount++;
                            console.log('Set visibility for layer:', layer.id, 'to', visibility);
                          } catch (e) {
                            console.warn('Failed to set visibility for layer:', layer.id, e);
                          }
                        }
                      }
                    } catch (e) {
                      console.warn('Failed to enumerate layers for source:', id, e);
                    }
                    
                    // フォールバック: 期待されるレイヤーIDも試す
                    if (controlledCount === 0) {
                      if (map.getLayer(expectedLayerId)) { 
                        map.setLayoutProperty(expectedLayerId, 'visibility', visibility); 
                        controlledCount++;
                      }
                      if (map.getLayer(expectedLabelId)) { 
                        map.setLayoutProperty(expectedLabelId, 'visibility', visibility); 
                        controlledCount++;
                      }
                    }
                    
                    if (controlledCount === 0) {
                      console.warn('No layers found for source:', id);
                    }
                  } catch (e) { console.warn('Failed to toggle layer visibility', e); }
                });
                
                processedSources.add(id);
              }

              const already = (typeof wmtsLayers !== 'undefined') ? wmtsLayers.find(function(e) { return e && e.title === `WFS: ${id}`; }) : null;
              if (!already) {
                addWfsLayer(`/wfs?SERVICE=WFS&REQUEST=GetFeature&TYPENAMES=${encodeURIComponent(id)}&OUTPUTFORMAT=application/json&MAXFEATURES=1000`, id, expectedLayerId, expectedLabelId, `WFS: ${id}`, `WFS: ${id} (labels)`);
              }
            }
          })
          .catch(err => { console.warn('Failed to fetch WFS GetCapabilities:', err); });
      } catch (e) { console.warn('WFS GetCapabilities setup failed', e); }

      // Build the static layer control UI from wmtsLayers (may have been appended by addWfsLayer above)
      try {
        const control = document.getElementById('layerControl');
        if (typeof wmtsLayers !== 'undefined' && Array.isArray(wmtsLayers)) {
          wmtsLayers.forEach(function(layerInfo) {
            // 既にUIに追加されているかチェック（重複防止）
            var cbId = 'cb_' + encodeURIComponent(layerInfo.id);
            if (document.getElementById(cbId)) {
              console.log('Layer control already exists for:', layerInfo.id);
              return; // skip duplicates
            }
            
            const row = document.createElement('div'); 
            row.style.marginBottom = '4px';
            const cb = document.createElement('input'); 
            cb.type = 'checkbox'; 
            cb.id = cbId; 
            cb.checked = true; 
            cb.style.marginRight = '6px';
            const label = document.createElement('label'); 
            label.htmlFor = cb.id; 
            label.textContent = layerInfo.title || layerInfo.id;
            row.appendChild(cb); 
            row.appendChild(label); 
            control.appendChild(row);

            cb.addEventListener('change', function() {
              try {
                const visibility = cb.checked ? 'visible' : 'none';
                var controlled = false;
                
                // Try to control the specific layer by ID first
                if (map.getLayer(layerInfo.id)) { 
                  map.setLayoutProperty(layerInfo.id, 'visibility', visibility); 
                  controlled = true;
                  console.log('Set visibility for layer:', layerInfo.id, 'to', visibility);
                }
                
                // If layer not found, try to find it by source (for style JSON layers)
                if (!controlled) {
                  try {
                    var allLayers = map.getStyle().layers || [];
                    for (var i = 0; i < allLayers.length; i++) {
                      var layer = allLayers[i];
                      // Match by layer ID or source ID
                      if (layer.id === layerInfo.id || layer.source === layerInfo.id) {
                        map.setLayoutProperty(layer.id, 'visibility', visibility);
                        controlled = true;
                        console.log('Set visibility for layer via source:', layer.id, 'to', visibility);
                      }
                    }
                  } catch (e) {
                    console.warn('Failed to search layers by source', e);
                  }
                }
                
                if (!controlled) {
                  console.warn('Layer not found on map yet:', layerInfo.id);
                }
              } catch (e) { console.warn('Failed to toggle layer visibility', e); }
            });
          });
        }
      } catch (e) { console.warn('layer control setup failed', e); }

      // Pitch toggle setup
      try {
        const pitchBtn = document.getElementById('pitchToggle');
        let pitchLocked = true;
        const _enforcePitch = function() { try { if (map.getPitch && Math.abs(map.getPitch()) > 0.0001) { map.setPitch(0); } } catch (e) { } };
        function lockPitch() { try { map.setPitch(0); } catch(e) {} try { if (map.on) map.on('move', _enforcePitch); } catch(e) {} pitchLocked = true; pitchBtn.textContent = '斜め許可'; }
        function unlockPitch() { try { if (map.off) map.off('move', _enforcePitch); } catch(e) {} pitchLocked = false; pitchBtn.textContent = '斜め禁止'; }
        pitchBtn.addEventListener('click', function() { if (!pitchLocked) lockPitch(); else unlockPitch(); });
        try { lockPitch(); } catch(e) {}
      } catch (e) { console.warn('pitch toggle setup failed', e); }

      // --- Theme selector setup ---
      try {
        const themeSelect = document.getElementById('qmp-themes');
        if (themeSelect && typeof themes !== 'undefined' && Array.isArray(themes)) {
          // Add prompt option
          const promptOpt = document.createElement('option');
          promptOpt.value = '';
          promptOpt.textContent = 'Select theme';
          promptOpt.selected = true;
          themeSelect.appendChild(promptOpt);
          
          // Add theme options
          themes.forEach(function(themeName) {
            try {
              const opt = document.createElement('option');
              opt.value = themeName;
              opt.textContent = themeName;
              themeSelect.appendChild(opt);
            } catch (e) { console.warn('Failed to add theme option', e); }
          });
          
          // Theme change handler
          themeSelect.addEventListener('change', function() {
            try {
              const selectedTheme = this.value;
              if (!selectedTheme) return;
              
              console.log('Theme changed to:', selectedTheme);
              
              // Request new style with theme parameter
              const styleUrl = window.__QMAP_CONFIG__ && window.__QMAP_CONFIG__.style 
                ? window.__QMAP_CONFIG__.style 
                : '/maplibre-style';
              
              // Build new style URL with theme parameter
              const themeStyleUrl = styleUrl + (styleUrl.indexOf('?') >= 0 ? '&' : '?') + 'theme=' + encodeURIComponent(selectedTheme);
              
              console.log('Loading style with theme:', themeStyleUrl);
              
              // Reload style (this will trigger a map refresh)
              map.setStyle(themeStyleUrl);
              
              // Reset select to prompt after a delay
              setTimeout(function() {
                try { themeSelect.value = ''; } catch (e) {}
              }, 500);
            } catch (e) { console.warn('Theme change failed', e); }
          });
          
          console.log('Theme selector initialized with', themes.length, 'themes');
        } else if (themeSelect) {
          // No themes available - hide the selector
          themeSelect.style.display = 'none';
        }
      } catch (e) { console.warn('Theme selector setup failed', e); }

      // --- Bookmark selector setup ---
      try {
        const bookmarkSelect = document.getElementById('qmp-bookmarks');
        
        // Debug: check if bookmarks variable is available
        console.log('Bookmark setup - bookmarks variable:', typeof bookmarks !== 'undefined' ? bookmarks : 'UNDEFINED');
        console.log('Bookmark setup - initialX:', typeof initialX !== 'undefined' ? initialX : 'UNDEFINED');
        console.log('Bookmark setup - initialY:', typeof initialY !== 'undefined' ? initialY : 'UNDEFINED');
        console.log('Bookmark setup - initialZoom:', typeof initialZoom !== 'undefined' ? initialZoom : 'UNDEFINED');
        
        if (bookmarkSelect && typeof bookmarks !== 'undefined' && Array.isArray(bookmarks)) {
          console.log('Setting up bookmark selector with', bookmarks.length, 'bookmarks');
          
          // Add prompt option
          const promptOpt = document.createElement('option');
          promptOpt.value = '';
          promptOpt.textContent = 'Select bookmark';
          promptOpt.selected = true;
          bookmarkSelect.appendChild(promptOpt);
          
          // Add Home option (returns to initial position)
          const homeOpt = document.createElement('option');
          homeOpt.value = '__home';
          homeOpt.textContent = 'Home';
          bookmarkSelect.appendChild(homeOpt);
          
          // Add bookmark options
          bookmarks.forEach(function(bookmark, index) {
            try {
              const opt = document.createElement('option');
              opt.value = String(index);
              opt.textContent = bookmark.name || ('Bookmark ' + (index + 1));
              bookmarkSelect.appendChild(opt);
              console.log('Added bookmark option:', bookmark.name, 'at index', index, 'lon:', bookmark.lon, 'lat:', bookmark.lat);
            } catch (e) { console.warn('Failed to add bookmark option', e); }
          });
          
          // Bookmark change handler
          bookmarkSelect.addEventListener('change', function() {
            try {
              const selectedValue = this.value;
              console.log('Bookmark selected:', selectedValue);
              
              if (!selectedValue) return;
              
              if (selectedValue === '__home') {
                // Return to initial position
                console.log('Navigating to home:', initialX, initialY, initialZoom);
                if (typeof initialX !== 'undefined' && typeof initialY !== 'undefined') {
                  map.flyTo({
                    center: [initialX, initialY],
                    zoom: typeof initialZoom !== 'undefined' ? initialZoom : 12,
                    duration: 1000
                  });
                  console.log('Returned to home position');
                }
              } else {
                // Navigate to selected bookmark
                const index = parseInt(selectedValue, 10);
                console.log('Parsed bookmark index:', index);
                
                if (!isNaN(index) && index >= 0 && index < bookmarks.length) {
                  const bookmark = bookmarks[index];
                  console.log('Navigating to bookmark:', bookmark);
                  
                  // MapLibre expects [longitude, latitude] in WGS84
                  if (bookmark && typeof bookmark.lon !== 'undefined' && typeof bookmark.lat !== 'undefined') {
                    console.log('Flying to: [lon:', bookmark.lon, ', lat:', bookmark.lat, '] zoom:', bookmark.zoom || 14);
                    map.flyTo({
                      center: [bookmark.lon, bookmark.lat],
                      zoom: bookmark.zoom || 14,
                      duration: 1000
                    });
                    console.log('Navigated to bookmark:', bookmark.name);
                  } else {
                    console.warn('Bookmark missing lon/lat coordinates:', bookmark);
                  }
                } else {
                  console.warn('Invalid bookmark index:', index, 'bookmarks length:', bookmarks.length);
                }
              }
              
              // Reset select to prompt
              setTimeout(function() {
                try { bookmarkSelect.value = ''; } catch (e) {}
              }, 1500);
            } catch (e) { console.warn('Bookmark navigation failed', e); }
          });
          
          console.log('Bookmark selector initialized with', bookmarks.length, 'bookmarks');
        } else if (bookmarkSelect) {
          // No bookmarks available - hide the selector
          console.warn('Hiding bookmark selector - bookmarks not available or not an array');
          bookmarkSelect.style.display = 'none';
        }
      } catch (e) { console.warn('Bookmark selector setup failed', e); }

      // --- WMTS identity watch: on map move, re-check GetCapabilities and
      // reload WMTS tile sources when the identity hash changes.
      try {
        // Simple helper: fetch GetCapabilities and extract ?v=... from
        // ServiceMetadataURL or ResourceURL template. Returns identity string
        // or null.
        function getWmtsIdentity() {
          return fetch('/wmts?SERVICE=WMTS')
            .then(function(resp) { if (!resp.ok) throw new Error('GetCapabilities failed'); return resp.text(); })
            .then(function(xmlText) {
              try {
                var parser = new window.DOMParser();
                var xml = parser.parseFromString(xmlText, 'text/xml');
                // Try ResourceURL template first
                var resUrls = xml.getElementsByTagName('ResourceURL');
                for (var i = 0; i < resUrls.length; i++) {
                  var t = resUrls[i].getAttribute && resUrls[i].getAttribute('template');
                  if (t) {
                    var m = t.match(/[?&]v=([^&]+)/);
                    if (m) return decodeURIComponent(m[1]);
                  }
                }
                // Fallback: ServiceMetadataURL text content
                var sm = xml.getElementsByTagName('ServiceMetadataURL');
                if (sm && sm.length > 0) {
                  var txt = sm[0].textContent || '';
                  var m2 = txt.match(/[?&]v=([^&]+)/);
                  if (m2) return decodeURIComponent(m2[1]);
                }
              } catch (e) {
                console.warn('parse GetCapabilities failed', e);
              }
              return null;
            })
            .catch(function(err) { console.warn('getWmtsIdentity error', err); return null; });
        }

        // Replace or append ?v= in a tile template URL
        function replaceVParamInTile(tileUrl, newV) {
          try {
            if (!tileUrl) return tileUrl;
            if (/[?&]v=[^&]*/.test(tileUrl)) {
              return tileUrl.replace(/([?&]v=)[^&]*/, '$1' + encodeURIComponent(newV));
            } else {
              return tileUrl + (tileUrl.indexOf('?') >= 0 ? '&' : '?') + 'v=' + encodeURIComponent(newV);
            }
          } catch (e) { return tileUrl; }
        }

        // Update only WMTS tile sources (sources with `tiles` array or raster url)
        // to use the new identity value. This avoids calling map.setStyle(),
        // which re-initializes the entire page and can cause WFS layers to be
        // re-registered. Strategy: for each tile source, read its current
        // descriptor from the style, compute new tiles/urls with ?v=, remove
        // the source from the map and immediately re-add it with the same id
        // and new descriptor. Layers referencing the source remain and will
        // use the new source automatically.
        function applyNewWmtsIdentity(map, newV) {
          try {
            var style = map.getStyle();
            if (!style || !style.sources) return;
            var srcIds = Object.keys(style.sources || {});
            srcIds.forEach(function(sid) {
              try {
                var src = style.sources[sid];
                if (!src) return;
                var isTileSource = Array.isArray(src.tiles) && src.tiles.length > 0;
                var isRasterUrl = !!(src.url && (src.type === 'raster' || src.type === 'raster'));
                if (!isTileSource && !isRasterUrl) return; // skip non-tile sources

                // Build new descriptor by cloning and replacing tiles/url
                var newDesc = JSON.parse(JSON.stringify(src));
                if (isTileSource) {
                  newDesc.tiles = (src.tiles || []).map(function(t) { return replaceVParamInTile(t, newV); });
                }
                if (isRasterUrl && newDesc.url) {
                  newDesc.url = replaceVParamInTile(newDesc.url, newV);
                }

                // To safely replace a source, first collect and remove any
                // layers that reference it, then remove the source, add the
                // new source descriptor, and finally re-add the layers in the
                // original order. This avoids MapLibre errors about removing
                // sources that are still used by layers and prevents duplicate
                // source addition.
                try {
                  var fullStyle = map.getStyle() || { layers: [] };
                  var allLayers = fullStyle.layers || [];
                  var referencingLayers = [];
                  var firstIndex = null;
                  var lastIndex = null;
                  for (var li = 0; li < allLayers.length; li++) {
                    var L = allLayers[li];
                    if (L && L.source === sid) {
                      if (firstIndex === null) firstIndex = li;
                      lastIndex = li;
                      referencingLayers.push(JSON.parse(JSON.stringify(L)));
                    }
                  }

                  // Determine id of the layer that followed the last referencing
                  // layer so we can insert before it when restoring. If null,
                  // re-adding will place layers at the top.
                  var nextLayerId = (lastIndex !== null && allLayers[lastIndex + 1]) ? allLayers[lastIndex + 1].id : null;

                  // Remove referencing layers from the map (if present)
                  referencingLayers.forEach(function(ld) {
                    try { if (map.getLayer && map.getLayer(ld.id)) { map.removeLayer(ld.id); } } catch (e) { /* ignore */ }
                  });

                  // Now remove the source if present
                  try { if (map.getSource && map.getSource(sid)) { map.removeSource(sid); } } catch (e) { /* ignore */ }

                  // Add the new source descriptor
                  try {
                    if (!map.getSource || !map.getSource(sid)) {
                      map.addSource(sid, newDesc);
                    } else {
                      // If source persists for some reason, skip adding.
                    }
                    console.log('[WMTS] updated tile source', sid);
                  } catch (e) {
                    console.warn('[WMTS] failed to re-add source', sid, e);
                  }

                  // Re-add layers in the saved order, inserting before nextLayerId
                  for (var ri = 0; ri < referencingLayers.length; ri++) {
                    try {
                      var layerDef = referencingLayers[ri];
                      // Avoid re-adding if it already exists
                      if (map.getLayer && map.getLayer(layerDef.id)) continue;
                      if (nextLayerId && map.getLayer && map.getLayer(nextLayerId)) {
                        map.addLayer(layerDef, nextLayerId);
                      } else {
                        // no next layer -> add to top
                        map.addLayer(layerDef);
                      }
                    } catch (e) { console.warn('[WMTS] failed to re-add layer', layerDef && layerDef.id, e); }
                  }
                } catch (e) { console.warn('applyNewWmtsIdentity replacement failure', e); }
              } catch (e) { console.warn('applyNewWmtsIdentity inner failure', e); }
            });
          } catch (e) { console.warn('applyNewWmtsIdentity failed', e); }
        }

        // Debounce helper
        function debounce(fn, ms) {
          var t = null;
          return function() {
            var args = arguments;
            clearTimeout(t);
            t = setTimeout(function() { fn.apply(null, args); }, ms);
          };
        }

        // Check and reload if identity changed
        var __qmap_current_wmts_identity = null;
        getWmtsIdentity().then(function(v) { __qmap_current_wmts_identity = v; });

        function checkWmtsIdentityAndReload() {
          getWmtsIdentity().then(function(v) {
            try {
              if (!v) return;
              if (__qmap_current_wmts_identity !== v) {
                console.log('[WMTS] identity changed', __qmap_current_wmts_identity, '->', v);
                __qmap_current_wmts_identity = v;
                applyNewWmtsIdentity(map, v);
              }
            } catch (e) { console.warn('checkWmtsIdentityAndReload failed', e); }
          });
        }

        // Attach to moveend (user finished panning/zooming) with debounce
        try {
          map.on('moveend', debounce(checkWmtsIdentityAndReload, 800));
        } catch (e) {
          console.warn('Failed to attach moveend handler', e);
        }
      } catch (e) { console.warn('WMTS identity watch setup failed', e); }

    } catch (e) { console.warn('qmap_postload failed', e); }
  };
})();
