# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeoAgent
                                 A QGIS plugin
 Plugin for geospatial workflow
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-12-15
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Tek Kshetri
        email                : iamtekson@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import (
    QSettings,
    QTranslator,
    QCoreApplication,
    Qt,
    QThread,
    QMetaObject,
    QTimer,
    Q_ARG,
    QEventLoop,
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QSizePolicy, QProgressDialog
from qgis.PyQt.QtGui import QFont
from qgis.core import Qgis, QgsMessageLog, QgsApplication

# Import the code for the dialog
from .dialogs.geo_agent_dialog import GeoAgentDialog

# Import agent and LLM components
from .config.settings import (
    API_KEY_FILE,
    SUPPORTED_MODELS,
    DEFAULT_MODEL,
    DEBUG_MODE,
    QGIS_MESSAGE_DURATION,
    SHOW_DEBUG_LOGS,
)
from .logger.logger import UILogHandler
from .llm.worker import LLMWorker
from .prompts.system import GENERAL_SYSTEM_PROMPT
from .utils.canvas_refresh import (
    RefreshDispatcher,
    set_qgis_interface,
    set_refresh_callback,
)
from .utils.layer_operations import (
    LayerRemovalDispatcher,
    set_layer_removal_callback,
)
from .utils.project_loader import (
    ProjectLoadDispatcher,
    set_project_load_callback,
)
from .utils.markdown_converter import markdown_to_html
from typing import Optional
import importlib
import subprocess
import sys
import re

import os
import os.path
import traceback
import logging


class GeoAgent:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # Dispatcher for main-thread canvas refresh
        self._refresh_dispatcher = RefreshDispatcher(self.iface)

        # Dispatcher for main-thread project loading
        self._project_load_dispatcher = ProjectLoadDispatcher(self.iface)

        # Dispatcher for main-thread layer removal
        self._layer_removal_dispatcher = LayerRemovalDispatcher(self.iface)

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "GeoAgent_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&GeoAgent")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Agent and LLM components
        self.llm = None
        self.app = None
        self.current_model = DEFAULT_MODEL
        self.api_key = self._load_api_key()
        self.thread_id = "geo-agent"
        self._has_started_thread = False
        self._last_temperature = None
        self._error_log_path = os.path.join(self.plugin_dir, "geo_agent_error.log")
        self._worker_thread: Optional[QThread] = None
        self._is_processing = False

    def _log_error(self, context: str, exc: Exception):
        try:
            details = traceback.format_exc()
            # Write to plugin error log file
            with open(self._error_log_path, "a", encoding="utf-8") as f:
                f.write(f"[{context}] {str(exc)}\n")
                f.write(details + "\n\n")
            # Also send to QGIS log panel
            QgsMessageLog.logMessage(details, "GeoAgent", level=Qgis.Critical)
            # Mirror to UI logger if available
            try:
                if hasattr(self, "_ui_logger"):
                    self._ui_logger.error(f"[{context}] {exc}\n{details}")
            except Exception:
                pass
        except Exception:
            pass

    def _setup_ui_logging(self):
        """Wire all logging sources to the GeoAgent Logs tab."""
        ui_handler = None
        try:
            if hasattr(self, "dlg") and hasattr(self.dlg, "get_ui_log_handler"):
                ui_handler = self.dlg.get_ui_log_handler()
        except Exception:
            ui_handler = None

        if not ui_handler or not isinstance(ui_handler, UILogHandler):
            return

        self._ui_log_handler = ui_handler

        # Configure UI logger
        level = logging.DEBUG if SHOW_DEBUG_LOGS else logging.INFO
        ui_handler.setLevel(level)
        try:
            ui_handler.set_show_debug(SHOW_DEBUG_LOGS)
        except Exception as exc:
            # log and continue if the UI handler does not support debug toggling
            QgsMessageLog.logMessage(
                f"GeoAgent UI handler does not support debug toggle: {exc}",
                "GeoAgent",
                level=Qgis.Warning,
            )

        self._ui_logger = logging.getLogger("GeoAgent.UI")
        self._ui_logger.setLevel(level)
        self._ui_logger.propagate = False

        # Avoid duplicate handlers on reload
        existing = [h for h in self._ui_logger.handlers if h is ui_handler]
        if not existing:
            # Clear stale handlers to prevent duplicate console outputs
            self._ui_logger.handlers.clear()

            console_handler = logging.StreamHandler()
            console_handler.setLevel(level)
            console_handler.setFormatter(
                logging.Formatter(
                    "%(asctime)s [%(levelname)s] %(name)s: %(message)s",
                    datefmt="%Y-%m-%d %H:%M:%S",
                )
            )
            self._ui_logger.addHandler(console_handler)
            self._ui_logger.addHandler(ui_handler)

        # Plugin-level logger for general events; let it propagate to root
        self._plugin_logger = logging.getLogger("GeoAgent")
        self._plugin_logger.setLevel(level)
        self._plugin_logger.propagate = True
        # Do not attach ui_handler here to avoid duplication; root will handle it

        # Attach handler to root logger so any logging call is mirrored to UI once
        root_logger = logging.getLogger()
        root_logger.setLevel(level)
        if ui_handler not in root_logger.handlers:
            root_logger.addHandler(ui_handler)

        # Attach processing logger to the same UI handler
        try:
            from .logger.processing_logger import set_processing_ui_log_handler

            set_processing_ui_log_handler(ui_handler)
        except Exception:
            pass

        # Connect QGIS message log stream
        self._connect_qgis_message_log()

        try:
            self._plugin_logger.info("GeoAgent UI logging initialized")
        except Exception:
            pass

    def _connect_qgis_message_log(self):
        """Forward QgsMessageLog messages to the UI logger."""
        try:
            if getattr(self, "_message_log_connected", False):
                return
            msg_log = QgsApplication.messageLog()
            if msg_log is None:
                return
            msg_log.messageReceived.connect(self._on_qgis_message)
            self._message_log_connected = True
            self._qgis_msg_log = msg_log
        except Exception:
            pass

    def _on_qgis_message(self, message, tag, level):
        """Slot to mirror QGIS messages into the UI log."""
        try:
            level_map = {
                getattr(Qgis, "Info", 0): logging.INFO,
                getattr(Qgis, "Success", 0): logging.INFO,
                getattr(Qgis, "Warning", 0): logging.WARNING,
                getattr(Qgis, "Critical", 0): logging.ERROR,
                getattr(Qgis, "Fatal", 0): logging.CRITICAL,
            }
            log_level = level_map.get(level, logging.INFO)
            if hasattr(self, "_ui_logger"):
                self._ui_logger.log(log_level, f"[{tag}] {message}")
        except Exception:
            pass

    def _refresh_callback(self):
        """Thread-safe refresh callback invoked by tools. Queues a dispatcher slot on the main thread."""
        try:
            ok = QMetaObject.invokeMethod(
                self._refresh_dispatcher,
                "doRefresh",
                Qt.QueuedConnection,
            )
            if not ok:
                QgsMessageLog.logMessage(
                    "QMetaObject.invokeMethod('doRefresh') returned False; "
                    "falling back to QTimer.singleShot.",
                    "GeoAgent",
                    level=Qgis.Warning,
                )
                QTimer.singleShot(0, self._refresh_dispatcher.doRefresh)
        except Exception as e:
            self._log_error("refresh_callback", e)

    def _project_load_callback(self, path):
        """Thread-safe project load callback using signals."""
        try:

            def on_result_ready():
                loop.quit()

            # connect signal to quit when result is ready
            self._project_load_dispatcher.result_ready.connect(on_result_ready)

            # queue the load operation on the main thread
            QMetaObject.invokeMethod(
                self._project_load_dispatcher,
                "doLoadProject",
                Qt.QueuedConnection,
                Q_ARG(str, path),
            )
            # wait for the dispatcher to signal completion
            loop = QEventLoop()

            # max wait 30 seconds timeout
            QTimer.singleShot(30000, loop.quit)
            loop.exec_()

            # disconnect signal to avoid memory leaks
            self._project_load_dispatcher.result_ready.disconnect(on_result_ready)

            # return the result that was set by the dispatcher
            return self._project_load_dispatcher.result
        except Exception as e:
            error_msg = f"_project_load_callback error: {e}"
            QgsMessageLog.logMessage(error_msg, "GeoAgent", level=Qgis.Warning)
            return {"success": False, "error": error_msg}

    def _layer_removal_callback(self, layer_id, layer_name):
        """Thread-safe layer removal callback using signals."""
        try:

            def on_result_ready():
                loop.quit()

            # connect signal to quit when result is ready
            self._layer_removal_dispatcher.result_ready.connect(on_result_ready)

            # queue the removal operation on the main thread
            QMetaObject.invokeMethod(
                self._layer_removal_dispatcher,
                "doRemoveLayer",
                Qt.QueuedConnection,
                Q_ARG(str, layer_id),
                Q_ARG(str, layer_name),
            )
            # wait for the dispatcher to signal completion
            loop = QEventLoop()

            # max wait 10 seconds timeout
            QTimer.singleShot(10000, loop.quit)
            loop.exec_()

            # disconnect signal to avoid memory leaks
            self._layer_removal_dispatcher.result_ready.disconnect(on_result_ready)

            # return the result that was set by the dispatcher
            return self._layer_removal_dispatcher.result
        except Exception as e:
            error_msg = f"_layer_removal_callback error: {e}"
            QgsMessageLog.logMessage(error_msg, "GeoAgent", level=Qgis.Warning)
            return {"success": False, "error": error_msg}

    def _ensure_dependencies_installed(self):
        """Ensure required Python packages are installed via pyproject.toml."""
        pkg_to_import = {
            "langgraph": "langgraph",
            "langchain-core": "langchain_core",
            "langchain-community": "langchain_community",
            "langchain-openai": "langchain_openai",
            "langchain-google-genai": "langchain_google_genai",
            "langchain-ollama": "langchain_ollama",
            "requests": "requests",
            "markdown": "markdown",
        }

        # Read dependencies from pyproject.toml
        deps = []
        pyproject_path = os.path.join(self.plugin_dir, "pyproject.toml")
        try:
            with open(pyproject_path, "r", encoding="utf-8") as f:
                content = f.read()
            try:
                import tomllib  # Python 3.11+

                data = tomllib.loads(content)
                deps = data.get("project", {}).get("dependencies", []) or []
            except Exception:
                m = re.search(r"dependencies\s*=\s*\[(.*)\]", content, re.DOTALL)
                if m:
                    raw = m.group(1)
                    for line in raw.splitlines():
                        line = line.strip().strip(",")
                        if not line:
                            continue
                        # handle both single and double quoted strings
                        if (line.startswith('"') and line.endswith('"')) or (line.startswith("'") and line.endswith("'")):
                            deps.append(line[1:-1])
        except Exception:
            deps = []

        # Normalize package names (strip version specifiers)
        pkgs = []
        for d in deps:
            name = d.split(";")[0].split(" ")[0]
            name = name.split(">=")[0].split("==")[0]
            pkgs.append(name)

        # Determine missing based on import availability
        missing = []
        for pkg in pkgs:
            import_name = pkg_to_import.get(pkg)
            if not import_name:
                continue
            try:
                importlib.import_module(import_name)
            except Exception:
                missing.append(pkg)

        if not missing:
            return True

        # Ask user to confirm installation
        try:
            pkg_list = "\n".join(missing)
            reply = QMessageBox.question(
                self.iface.mainWindow(),
                "Install Dependencies",
                f"The following packages are required:\n\n{pkg_list}\n\nInstall now?",
                QMessageBox.Yes | QMessageBox.No,
            )
            if reply != QMessageBox.Yes:
                return False
        except Exception:
            pass

        # Inform user via message bar
        try:
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Installing missing dependencies: {' '.join(missing)}",
                level=Qgis.Info,
                duration=0,
            )
        except Exception:
            pass

        # Install missing packages
        try:
            py_exec = sys.executable
            lower = py_exec.lower()
            if lower.endswith("qgis-bin.exe") or lower.endswith("qgis-ltr-bin.exe"):
                py_exec = os.path.join(os.path.dirname(py_exec), "python.exe")
            elif lower.endswith("pythonw.exe"):
                py_exec = os.path.join(os.path.dirname(py_exec), "python.exe")

            # Progress dialog during installation
            progress = None
            try:
                progress = QProgressDialog(
                    "Installing dependencies...",
                    None,
                    0,
                    len(missing),
                    self.iface.mainWindow(),
                )
                progress.setWindowTitle("GeoAgent")
                progress.setWindowModality(Qt.ApplicationModal)
                progress.setAutoClose(True)
                progress.setAutoReset(True)
                progress.setMinimumDuration(0)
            except Exception:
                progress = None

            for idx, pkg in enumerate(missing, start=1):
                subprocess.run(
                    [py_exec, "-m", "pip", "install", "--upgrade", pkg], check=True
                )
                try:
                    if progress:
                        progress.setValue(idx)
                        QCoreApplication.processEvents()
                except Exception:
                    pass

            self.iface.messageBar().pushMessage(
                "GeoAgent",
                "Dependencies installed successfully.",
                level=Qgis.Success,
                duration=QGIS_MESSAGE_DURATION,
            )
            try:
                if progress:
                    progress.close()
            except Exception:
                pass
            return True
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Failed to install dependencies: {e}",
                level=Qgis.Critical,
                duration=QGIS_MESSAGE_DURATION,
            )

    def _get_message_classes(self):
        msgs_mod = importlib.import_module("langchain_core.messages")
        return msgs_mod.SystemMessage, msgs_mod.HumanMessage

    def _get_agents(self):
        """Import agent functions from agents module."""
        mod = importlib.import_module(".agents", package=__package__)
        return mod.build_unified_graph, mod.invoke_app, mod.invoke_app_async

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("GeoAgent", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(self.plugin_dir, "icons", "icon.png")
        self.add_action(
            icon_path,
            text=self.tr("geoAgent"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&GeoAgent"), action)
            self.iface.removeToolBarIcon(action)

        if hasattr(self, "dlg"):
            self.iface.removeDockWidget(self.dlg)

        # Disconnect QGIS message log signal on unload
        if getattr(self, "_message_log_connected", False):
            try:
                if hasattr(self, "_qgis_msg_log"):
                    self._qgis_msg_log.messageReceived.disconnect(self._on_qgis_message)
            except Exception:
                pass

    def run(self):
        """Run method that performs all the real work"""
        # Ensure dependencies before loading graph or message classes
        self._ensure_dependencies_installed()

        # Initialize QGIS interface for tools
        try:
            # Register QGIS interface for tools module
            set_qgis_interface(self.iface)

            # Register thread-safe refresh callback method
            set_refresh_callback(self._refresh_callback)

            # Register thread-safe project load callback method
            set_project_load_callback(self._project_load_callback)

            # Register thread-safe layer removal callback method
            set_layer_removal_callback(self._layer_removal_callback)
        except Exception as e:
            self._log_error("initialize_tools_interface", e)

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = GeoAgentDialog(self.iface.mainWindow())
            # Prefer bottom dock area and allow only bottom
            try:
                self.dlg.setAllowedAreas(Qt.BottomDockWidgetArea)
            except Exception:
                pass
            self.iface.addDockWidget(Qt.BottomDockWidgetArea, self.dlg)
            # Encourage larger content footprint in bottom area
            try:
                # Increase minimum height on dock and its main widget if accessible
                self.dlg.setMinimumHeight(300)
                if hasattr(self.dlg, "widget") and callable(
                    getattr(self.dlg, "widget")
                ):
                    w = self.dlg.widget()
                    if w is not None:
                        w.setMinimumHeight(300)
                        sp = w.sizePolicy()
                        sp.setVerticalPolicy(QSizePolicy.Expanding)
                        sp.setHorizontalPolicy(QSizePolicy.Expanding)
                        w.setSizePolicy(sp)
            except Exception:
                pass
            # Wire up UI actions
            if hasattr(self.dlg, "send_chat"):
                try:
                    self.dlg.send_chat.clicked.disconnect()
                except Exception:
                    pass
                self.dlg.send_chat.clicked.connect(self.send_message)
            if hasattr(self.dlg, "question") and hasattr(
                self.dlg.question, "returnPressed"
            ):
                try:
                    self.dlg.question.returnPressed.disconnect()
                except Exception:
                    pass
                self.dlg.question.returnPressed.connect(self.send_message)
            # Wire up export and clear buttons
            if hasattr(self.dlg, "export_ans"):
                try:
                    self.dlg.export_ans.clicked.disconnect()
                except Exception:
                    pass
                self.dlg.export_ans.clicked.connect(self.export_chat)
            if hasattr(self.dlg, "clear_ans"):
                try:
                    self.dlg.clear_ans.clicked.disconnect()
                except Exception:
                    pass
                self.dlg.clear_ans.clicked.connect(self.clear_chat)

            # Setup unified UI logging once the dialog exists
            try:
                self._setup_ui_logging()
            except Exception as e:
                QgsMessageLog.logMessage(
                    f"Failed to setup UI logging: {e}", "GeoAgent", level=Qgis.Warning
                )

        # show and focus the dock widget
        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()
        # Try to give it a reasonable initial height in bottom area
        try:
            # Ask for more vertical space in bottom dock area
            self.iface.mainWindow().resizeDocks([self.dlg], [300], Qt.Vertical)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Failed to resize: {e}",
                "GeoAgent",
                level=Qgis.Warning,
            )

        # Ensure UI logging stays connected on subsequent runs
        try:
            self._setup_ui_logging()
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Failed to ensure UI logging stays connected: {e}",
                "GeoAgent",
                level=Qgis.Warning,
            )

    def showMessage(self, title, msg, button, icon, fontsize=9):
        msgBox = QMessageBox()
        if icon == "Warning":
            msgBox.setIcon(QMessageBox.Warning)
        if icon == "Info":
            msgBox.setIcon(QMessageBox.Information)
        msgBox.setWindowTitle(title)
        msgBox.setText(msg)
        msgBox.setStandardButtons(QMessageBox.Ok)
        msgBox.setStyleSheet(
            "background-color: rgb(83, 83, 83);color: rgb(255, 255, 255);"
        )
        font = QFont()
        font.setPointSize(fontsize)
        msgBox.setFont(font)
        msgBox.setWindowFlags(Qt.Dialog | Qt.WindowStaysOnTopHint)
        buttonY = msgBox.button(QMessageBox.Ok)
        buttonY.setText(button)
        buttonY.setFont(font)
        msgBox.exec_()

    def send_message(self):
        """Send a message and get a response from the LLM."""
        # Validate input
        question = self.dlg.question.text().strip()
        if not question:
            return

        # Disable buttons during processing
        self.dlg.send_chat.setEnabled(False)
        self.dlg.question.setEnabled(False)

        try:
            # Get settings from UI
            temperature = self.dlg.temperature.value()
            max_tokens = self.dlg.max_tokens.value()
            model_name = self.dlg.model.currentText()

            # Display user question
            self._display_user_message(question)

            # Get current mode from UI
            current_mode = self.dlg.get_current_mode()

            # Initialize app if needed, model changed, or mode changed
            if (
                self.app is None
                or self.current_model != model_name
                or getattr(self, "_current_mode", None) != current_mode
            ):
                self._initialize_agent(
                    model_name,
                    temperature=temperature,
                    max_tokens=max_tokens,
                    mode=current_mode,
                )
                self._current_mode = current_mode

            # Rebuild app if temperature changed
            if (
                self._last_temperature is None
                or abs(self._last_temperature - temperature) > 1e-9
            ):
                # Recreate LLM and app to apply new temperature
                self._initialize_agent(
                    model_name,
                    temperature=temperature,
                    max_tokens=max_tokens,
                    mode=current_mode,
                )
                self._last_temperature = temperature
                self._current_mode = current_mode

            if self.app is None:
                raise RuntimeError("LLM app is not initialized")

            # Prepare messages
            SystemMessage, HumanMessage = self._get_message_classes()

            # Build messages for this turn; include system prompt only on the first turn
            if not self._has_started_thread:
                msgs = [
                    SystemMessage(content=GENERAL_SYSTEM_PROMPT),
                    HumanMessage(content=question),
                ]
                self._has_started_thread = True
            else:
                msgs = [HumanMessage(content=question)]
            _, _, invoke_app_async = self._get_agents()

            # Disable send button to prevent multiple submissions
            self.dlg.send_chat.setEnabled(False)
            self.dlg.send_chat.setText("Processing...")
            self.dlg.question.setEnabled(False)
            # self.dlg.send_chat.setText("Processing...")

            # Create and start worker thread for non-blocking inference
            self._worker_thread = LLMWorker(
                self.app, self.thread_id, msgs, invoke_app_async
            )
            self._worker_thread.result_ready.connect(self._on_invoke_result)
            self._worker_thread.error.connect(self._on_invoke_error)
            self._worker_thread.finished.connect(self._on_invoke_finished)
            self._worker_thread.start()

        except Exception as e:
            error_msg = f"Error: {str(e)}"
            self._log_error("send_message", e)
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                error_msg,
                level=Qgis.Critical,
                duration=QGIS_MESSAGE_DURATION,
            )
            # Show a popup with a hint to the log location
            try:
                if DEBUG_MODE:
                    self.showMessage(
                        "GeoAgent Error",
                        f"{error_msg}\n\nSee full log at:\n{self._error_log_path}",
                        "OK",
                        "Warning",
                    )
            except Exception:
                pass
            # Re-enable buttons on error
            self.dlg.send_chat.setEnabled(True)
            self.dlg.send_chat.setText("Send")
            self.dlg.question.setEnabled(True)

    def _on_invoke_result(self, last_msg):
        """Callback when worker thread completes successfully."""
        try:
            # Extract content and ensure it's a string
            if hasattr(last_msg, "content"):
                content = last_msg.content
                # Handle list content (e.g., from structured responses)
                if isinstance(content, list):
                    # Join list items or extract text from content blocks
                    response_text = " ".join(
                        (
                            item.get("text", str(item))
                            if isinstance(item, dict)
                            else str(item)
                        )
                        for item in content
                    )
                else:
                    response_text = str(content)
            else:
                response_text = str(last_msg)

            # Display response
            self._display_ai_response(response_text)
            # Clear input and scroll to bottom
            self.dlg.question.setText("")
            self._scroll_to_bottom()
        except Exception as e:
            self._log_error("_on_invoke_result", e)
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Error displaying response: {str(e)}",
                level=Qgis.Critical,
                duration=QGIS_MESSAGE_DURATION,
            )

    def _on_invoke_error(self, error_msg: str):
        """Callback when worker thread encounters an error."""
        self._log_error("LLM inference", Exception(error_msg))
        self.iface.messageBar().pushMessage(
            "GeoAgent",
            f"LLM Error: {error_msg}",
            level=Qgis.Critical,
            duration=QGIS_MESSAGE_DURATION,
        )
        try:
            if DEBUG_MODE:
                self.showMessage(
                    "GeoAgent Error",
                    f"LLM Error: {error_msg}\n\nSee full log at:\n{self._error_log_path}",
                    "OK",
                    "Warning",
                )
        except Exception:
            pass
        # Re-enable buttons on error
        self.dlg.send_chat.setEnabled(True)
        self.dlg.send_chat.setText("Send")
        self.dlg.question.setEnabled(True)

    def _on_invoke_finished(self):
        """Callback when worker thread finishes (success or error)."""
        # Re-enable buttons
        self.dlg.send_chat.setEnabled(True)
        self.dlg.send_chat.setText("Send")
        self.dlg.question.setEnabled(True)
        self._is_processing = False
        # Clean up thread reference
        if self._worker_thread:
            self._worker_thread.quit()
            self._worker_thread.wait()
            self._worker_thread = None

    def _initialize_agent(
        self,
        model_name: str,
        temperature: float = 0.7,
        max_tokens: int = None,
        mode: str = "general",
    ) -> None:
        """Initialize the LangChain chat model and LangGraph app.

        Args:
            model_name: The model to use
            temperature: Temperature parameter for LLM
            max_tokens: Maximum tokens for response
            mode: Either 'general' or 'processing'
        """
        try:
            from .llm.client import create_llm, ollama_model_exists, ollama_pull_model
        except Exception as e:
            self._log_error("_initialize_agent", e)
            raise RuntimeError(
                "Failed to import LLM client module. Ensure dependencies are installed."
            ) from e
        
        try:
            if model_name not in SUPPORTED_MODELS:
                raise ValueError(f"Unsupported model: {model_name}")

            model_config = SUPPORTED_MODELS[model_name]
            provider = model_config["type"]

            # Get API key if required
            api_key = None
            if model_config.get("requires_api_key"):
                api_key = self.dlg.custom_apikey.text().strip()
                if not api_key:
                    api_key = self.api_key
                if not api_key:
                    raise ValueError(f"API key required for {model_name}")

            # Create LLM based on provider
            client_kwargs = {"temperature": temperature, "max_tokens": max_tokens}
            if provider == "ollama":
                # Use UI-provided Ollama model name if specified; else default to llama3.2:3b
                try:
                    ollama_model_name = (
                        self.dlg.ollama_model_name.text().strip()
                        if hasattr(self.dlg, "ollama_model_name")
                        else ""
                    )
                except Exception:
                    ollama_model_name = ""

                # Optional base URL from UI; default to http://localhost:11434
                try:
                    ollama_base_url = (
                        self.dlg.ollama_base_url.text().strip()
                        if hasattr(self.dlg, "ollama_base_url")
                        else ""
                    )
                except Exception:
                    ollama_base_url = ""

                client_kwargs["model"] = (
                    ollama_model_name if ollama_model_name else "llama3.2:3b"
                )
                if ollama_base_url:
                    client_kwargs["base_url"] = ollama_base_url
            elif provider == "openai":
                # try to get the model from UI if available
                try:
                    openai_model_name = (
                        self.dlg.openai_model_name.text().strip()
                        if hasattr(self.dlg, "model_name")
                        else ""
                    )
                except Exception:
                    openai_model_name = ""

                client_kwargs["model"] = (
                    openai_model_name
                    if openai_model_name
                    else model_config.get("default_model", "gpt-4")
                )
            elif provider == "google":
                # try to get the model from UI if available
                try:
                    google_model_name = (
                        self.dlg.google_model_name.text().strip()
                        if hasattr(self.dlg, "model_name")
                        else ""
                    )
                except Exception:
                    google_model_name = ""
                client_kwargs["model"] = (
                    google_model_name
                    if google_model_name
                    else model_config.get("default_model", "gemini-3-flash-preview")
                )

            # Validate Ollama availability/model
            if provider == "ollama":
                base_url = client_kwargs.get("base_url", "http://localhost:11434")
                model_str = client_kwargs.get("model", "llama3.2:3b")
                if not ollama_model_exists(base_url, model_str):
                    reply = QMessageBox.question(
                        self.iface.mainWindow(),
                        "Ollama Model Not Found",
                        f"The model '{model_str}' is not installed.\n\n"
                        f"Would you like to pull it now? This may take a few minutes.",
                        QMessageBox.Yes | QMessageBox.No,
                    )
                    if reply == QMessageBox.Yes:
                        self.iface.messageBar().pushMessage(
                            "GeoAgent",
                            f"Pulling model '{model_str}'. This may take a few minutes...",
                            level=Qgis.Info,
                            duration=QGIS_MESSAGE_DURATION,
                        )
                        if not ollama_pull_model(base_url, model_str):
                            raise RuntimeError(
                                f"Failed to pull '{model_str}'. Check Ollama server."
                            )
                        self.iface.messageBar().pushMessage(
                            "GeoAgent",
                            f"Successfully pulled '{model_str}'. Initializing...",
                            level=Qgis.Success,
                            duration=QGIS_MESSAGE_DURATION,
                        )
                    else:
                        raise RuntimeError(
                            f"Model '{model_str}' not installed. To install manually, run: ollama pull {model_str}"
                        )

            # Create LLM and compile LangGraph app
            self.llm = create_llm(provider, api_key=api_key, **client_kwargs)
            build_unified_graph, _, _ = self._get_agents()

            # Use unified graph builder that routes based on mode
            self.app = build_unified_graph(self.llm, mode=mode)
            self.current_model = model_name
            # Reset thread on re-init
            self.thread_id = f"geo-agent:{model_name}:{mode}"
            self._has_started_thread = False

            mode_display = "Processing" if mode == "processing" else "General"
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Connected to {model_name} ({mode_display} mode)",
                level=Qgis.Success,
                duration=QGIS_MESSAGE_DURATION,
            )

        except Exception as e:
            self._log_error("_initialize_agent", e)
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Failed to initialize {model_name}: {str(e)}",
                level=Qgis.Critical,
                duration=QGIS_MESSAGE_DURATION,
            )
            try:
                self.showMessage(
                    "Initialization Error",
                    f"Failed to initialize {model_name}: {str(e)}\n\nSee full log at:\n{self._error_log_path}",
                    "OK",
                    "Warning",
                )
            except Exception:
                pass
            self.llm = None
            self.app = None
            raise

    def _display_user_message(self, message: str) -> None:
        """Display user message in the chat area."""
        self.dlg.llm_response.append("\n")
        self.dlg.llm_response.append("." * 40)
        self.dlg.llm_response.append(f"\n<b>User:</b> {message}")
        # Show a processing indicator immediately
        self.dlg.llm_response.append("\n<i>Agent is processing…</i>")
        try:
            self.dlg.llm_response.repaint()
        except Exception:
            pass
        self._scroll_to_bottom()

    def _display_ai_response(self, response: str) -> None:
        """Display AI response in the chat area."""
        # Ensure response is a string (safety check)
        if not isinstance(response, str):
            response = str(response)

        # Get cursor and remove the processing indicator line
        cursor = self.dlg.llm_response.textCursor()
        cursor.movePosition(cursor.End)
        # Move back to select the "Agent is processing..." line
        cursor.select(cursor.LineUnderCursor)
        cursor.removeSelectedText()
        # Remove the extra newline if present
        cursor.deletePreviousChar()

        # Markdown formatting to HTML for proper rendering
        formatted_response = markdown_to_html(response)
        
        # Append agent response with HTML formatting support
        self.dlg.llm_response.append("\n<b>Agent:</b> " + formatted_response)
        self.dlg.llm_response.append("\n" + "." * 40)

        # Re-enable buttons only after response is displayed
        self.dlg.send_chat.setEnabled(True)
        self.dlg.send_chat.setText("Send")
        self.dlg.question.setEnabled(True)

    def _scroll_to_bottom(self) -> None:
        """Scroll chat area to the bottom."""
        scroll_bar = self.dlg.llm_response.verticalScrollBar()
        scroll_bar.setValue(scroll_bar.maximum())

    def _load_api_key(self) -> str:
        """Load API key from file if it exists."""
        try:
            if os.path.exists(API_KEY_FILE):
                with open(API_KEY_FILE, "r") as f:
                    return f.read().strip()
        except Exception:
            pass
        return ""

    def _save_api_key(self, api_key: str) -> None:
        """Save API key to file."""
        try:
            os.makedirs(os.path.dirname(API_KEY_FILE), exist_ok=True)
            with open(API_KEY_FILE, "w") as f:
                f.write(api_key)
            self.api_key = api_key
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Failed to save API key: {str(e)}",
                level=Qgis.Warning,
                duration=QGIS_MESSAGE_DURATION,
            )

    def export_chat(self) -> None:
        """Export chat history to a text file."""
        try:
            from qgis.PyQt.QtWidgets import QFileDialog
            import datetime

            # Get chat content
            chat_text = self.dlg.llm_response.toPlainText()
            if not chat_text.strip():
                self.iface.messageBar().pushMessage(
                    "GeoAgent",
                    "Chat is empty. Nothing to export.",
                    level=Qgis.Info,
                    duration=QGIS_MESSAGE_DURATION,
                )
                return

            # Ask user where to save
            file_path, _ = QFileDialog.getSaveFileName(
                self.dlg,
                "Export Chat",
                f"geo_agent_chat_{datetime.datetime.now().strftime('%Y%m%d_%H%M%S')}.txt",
                "Text Files (*.txt);;All Files (*)",
            )

            if file_path:
                with open(file_path, "w", encoding="utf-8") as f:
                    f.write(chat_text)
                self.iface.messageBar().pushMessage(
                    "GeoAgent",
                    f"Chat exported to {file_path}",
                    level=Qgis.Success,
                    duration=QGIS_MESSAGE_DURATION,
                )
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Failed to export chat: {str(e)}",
                level=Qgis.Critical,
                duration=QGIS_MESSAGE_DURATION,
            )

    def clear_chat(self) -> None:
        """Clear the chat history."""
        try:
            self.dlg.llm_response.clear()
            # Start a fresh thread id (clears memory)
            import uuid

            self.thread_id = f"geo-agent:{uuid.uuid4().hex}"
            self._has_started_thread = False
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                "Chat cleared.",
                level=Qgis.Info,
                duration=QGIS_MESSAGE_DURATION,
            )
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "GeoAgent",
                f"Failed to clear chat: {str(e)}",
                level=Qgis.Critical,
                duration=QGIS_MESSAGE_DURATION,
            )
