from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtGui import QIcon
from qgis.core import (
    QgsProject,
    QgsCoordinateTransform,
    QgsCoordinateReferenceSystem
)
from PyQt5.QtCore import QStandardPaths
import os


class GELiveViewPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.enabled = False
        self.action = None

        # ✅ CORRECT DESKTOP PATH (Windows-safe)
        self.desktop = QStandardPaths.writableLocation(
            QStandardPaths.DesktopLocation
        )

        self.KML_PATH = os.path.join(
            self.desktop,
            "QGIS_GE_Live_View.kml"
        )

    def initGui(self):
        icon_path = os.path.join(
            os.path.dirname(__file__),
            "icon.png"
        )

        self.action = QAction(
            QIcon(icon_path),
            "GE Live Sync",
            self.iface.mainWindow()
        )

        self.action.setCheckable(True)
        self.action.triggered.connect(self.toggle_ge_sync)

        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&GE Live Sync", self.action)

        self.iface.messageBar().pushMessage(
            "GE Live Sync",
            f"KML will be created at: {self.KML_PATH}",
            level=0,
            duration=4
        )

    def unload(self):
        if self.action:
            self.iface.removeToolBarIcon(self.action)
            self.iface.removePluginMenu("&GE Live Sync", self.action)

        try:
            self.canvas.extentsChanged.disconnect(self.update_ge_view)
        except Exception:
            pass

    def toggle_ge_sync(self, checked):
        self.enabled = checked

        if checked:
            self.canvas.extentsChanged.connect(self.update_ge_view)
            self.update_ge_view()

            self.iface.messageBar().pushMessage(
                "GE Live Sync",
                "Live Sync ON",
                level=0,
                duration=2
            )
        else:
            try:
                self.canvas.extentsChanged.disconnect(self.update_ge_view)
            except Exception:
                pass

            self.iface.messageBar().pushMessage(
                "GE Live Sync",
                "Live Sync OFF",
                level=0,
                duration=2
            )

    def update_ge_view(self):
        if not self.enabled:
            return

        extent = self.canvas.extent()
        center = extent.center()

        src_crs = self.canvas.mapSettings().destinationCrs()
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")

        transform = QgsCoordinateTransform(
            src_crs,
            wgs84,
            QgsProject.instance()
        )

        center_wgs = transform.transform(center)

        lon = center_wgs.x()
        lat = center_wgs.y()

        scale = self.canvas.scale()
        altitude = max(10, scale / 2)

        kml = f"""<?xml version="1.0" encoding="UTF-8"?>
<kml xmlns="http://www.opengis.net/kml/2.2">
<Document>
  <LookAt>
    <longitude>{lon}</longitude>
    <latitude>{lat}</latitude>
    <range>{altitude}</range>
    <tilt>0</tilt>
    <heading>0</heading>
  </LookAt>
</Document>
</kml>
"""

        # ✅ Ensure file is always writable
        with open(self.KML_PATH, "w", encoding="utf-8") as f:
            f.write(kml)
