# Import basic libs
import json

# Import PyQt libs
from qgis.PyQt.QtCore import QObject, QUrl, pyqtSignal
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest

from gbif_extractor.__about__ import __plugin_name__, __version__


class MaxObs(QObject):
    finished_dl = pyqtSignal()
    """Launch a request to fetch only one obs in the extent and get the total number of
    observation in the extent as a property."""

    def __init__(
        self,
        network_manager=None,
        extent=None,
        url=None,
    ):
        super().__init__()
        self.network_manager = network_manager
        self.extent = extent
        self.url = url

        self.nb_obs = 0

        self.download()

    def download(self):
        url = "{url}?advanced=false&geometry={polygon}&limit={limit}".format(
            url=self.url,
            polygon=self.extent.asWktPolygon(),
            limit=1,
        )
        url = QUrl(url)
        request = QNetworkRequest(url)
        # Add an user-agent to the request so the website maintainer can check where
        # the requests are from.
        request.setRawHeader(
            b"User-Agent", bytes(__plugin_name__ + "/" + __version__, encoding="utf-8")
        )
        request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
        reply = self.network_manager.get(request)
        reply.finished.connect(lambda: self.handle_finished(reply))

    def handle_finished(self, reply):
        # Check if the request raises an error.
        if reply.error() != QNetworkReply.NoError:
            print(f"code: {reply.error()} message: {reply.errorString()}")
            if reply.error() == 403:
                print("Service down")
        else:
            # Decode data fetch from the get request and create a dictionnary.
            data_request = reply.readAll().data().decode()
            res = json.loads(data_request)
            # Get the observation number in the extent based on a specific property
            self.nb_obs = res["count"]
            self.finished_dl.emit()
