# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GazetteerCreator
                                 A QGIS plugin
 This plugin is for generate gazetteer pdf from layer vector data.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-24
        git sha              : $Format:%H$
        copyright            : (C) 2023 by BIG - UNGEGN
        email                : nna-ina@big.go.id
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog

from qgis.core import QgsProject, QgsMapLayerType, Qgis, QgsWkbTypes, QgsDataSourceUri, QgsVectorLayer

from shapely.geometry import LineString

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .gazetteer_creator_dialog import GazetteerCreatorDialog
import os.path
from datetime import date
import math
import platform
import subprocess
import sys

try:
    import geopandas as gpd
except ImportError:
    parent = os.path.dirname(sys.executable)
    if platform.system() == 'Windows':
        python = os.path.join(parent, 'python.exe')
    elif platform.system() == 'Darwin':
        bin_path = os.path.join(parent, 'bin')
        python = os.path.join(bin_path, 'python3')
    else:
        python = os.path.join(parent, 'python.exe')
    command_geopandas = [python, '-m', 'pip', 'install', 'geopandas']
    subprocess.check_call(command_geopandas)
try:
    import pandas as pd
except ImportError:
    parent = os.path.dirname(sys.executable)
    if platform.system() == 'Windows':
        python = os.path.join(parent, 'python.exe')
    elif platform.system() == 'Darwin':
        bin_path = os.path.join(parent, 'bin')
        python = os.path.join(bin_path, 'python3')
    else:
        python = os.path.join(parent, 'python.exe')
    command_pandas = [python, '-m', 'pip', 'install', 'pandas']
    subprocess.check_call(command_pandas)

try:
    from fpdf import FPDF
except ImportError:
    parent = os.path.dirname(sys.executable)
    if platform.system() == 'Windows':
        python = os.path.join(parent, 'python.exe')
    elif platform.system() == 'Darwin':
        bin_path = os.path.join(parent, 'bin')
        python = os.path.join(bin_path, 'python3')
    else:
        python = os.path.join(parent, 'python.exe')
    command_fpdf = [python, '-m', 'pip', 'install', 'fpdf']
    subprocess.check_call(command_fpdf)

class GazetteerCreator:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'GazetteerCreator_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Toponyms Management')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('GazetteerCreator', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/gazetteer_creator/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Gazetteer Creator'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Toponyms Management'),
                action)
            self.iface.removeToolBarIcon(action)

    def calculate_lat_long(self, gdf):
        if gdf.geometry.geom_type.isin(['Point', 'MultiPoint']).any():
            if gdf.geometry.geom_type.isin(['MultiPoint']).any():
                gdf['geometry'] = gdf.representative_point()
            if 'lat' not in gdf.columns:
                gdf['lat'] = None 
            if 'long' not in gdf.columns:
                gdf['long'] = None
            gdf['lat'] = gdf['lat'].fillna(gdf.geometry.y.apply("{:.5f}".format))
            gdf['long'] = gdf['long'].fillna(gdf.geometry.x.apply("{:.5f}".format))
            gdf['lat'] = gdf['lat'].astype(float).round(5)
            gdf['long'] = gdf['long'].astype(float).round(5)
            return gdf
        elif gdf.geometry.geom_type.isin(['LineString', 'MultiLineString']).any():
            if gdf.geometry.geom_type.isin(['MultiLineString']).any():
                gdf = gdf.explode(ignore_index=True)
            if 'lat' not in gdf.columns:
                gdf['lat'] = None 
            if 'long' not in gdf.columns:
                gdf['long'] = None
            for index, row in gdf.iterrows():
                first_coords = row['geometry'].coords[0]
                last_coords = row['geometry'].coords[-1]
                lat_first_coords = first_coords[1]
                long_first_coords = first_coords[0]
                if gdf.at[index, 'lat'] is None:
                    gdf.at[index, 'lat'] = "{:.5f}".format(lat_first_coords)
                if gdf.at[index, 'long'] is None:
                    gdf.at[index, 'long'] = "{:.5f}".format(long_first_coords)
            return gdf
        elif gdf.geometry.geom_type.isin(['Polygon', 'MultiPolygon']).any():
            if gdf.geometry.geom_type.isin(['MultiPolygon']).any():
                gdf = gdf.explode(ignore_index=True)
            if 'lat' not in gdf.columns:
                gdf['lat'] = None 
            if 'long' not in gdf.columns:
                gdf['long'] = None
            gdf.crs = "epsg:4326"
            centroid = gdf.centroid
            gdf['lat'] = gdf['lat'].fillna(centroid.y.apply("{:.5f}".format))
            gdf['long'] = gdf['long'].fillna(centroid.x.apply("{:.5f}".format))
            gdf['lat'] = gdf['lat'].astype(float).round(5)
            gdf['long'] = gdf['long'].astype(float).round(5)
            return gdf
        else:
            gdf['lat'] = gdf['long'] = None
            return gdf

    def get_data(self, point_filename, line_filename, area_filename):
        if point_filename[0] and self.dlg.checkBox.isChecked():
            gdf = gpd.read_file(point_filename[0])
            gdf = self.calculate_lat_long(gdf)
        else:
            gdf = gpd.GeoDataFrame()
            gdf['ftype'] = gdf['name'] = gdf['country'] = gdf['lat'] = gdf['long'] = None
            gdf.geometry = gpd.GeoSeries([])

        if line_filename[0] and self.dlg.checkBox_2.isChecked():
            line_gdf = gpd.read_file(line_filename[0])
            line_gdf = self.calculate_lat_long(line_gdf)
        else:
            line_gdf = gpd.GeoDataFrame()
            line_gdf['ftype'] = line_gdf['name'] = line_gdf['lat'] = line_gdf['long'] = None
            line_gdf.geometry = gpd.GeoSeries([])

        if area_filename[0] and self.dlg.checkBox_3.isChecked():
            area_gdf = gpd.read_file(area_filename[0])
            area_gdf = self.calculate_lat_long(area_gdf)
        else:
            area_gdf = gpd.GeoDataFrame()
            area_gdf['ftype'] = area_gdf['name'] = area_gdf['lat'] = area_gdf['long'] = None
            area_gdf.geometry = gpd.GeoSeries([])

        gdf = pd.concat([gdf, line_gdf, area_gdf], ignore_index=True, sort=False)
        gdf = gdf.sort_values('name')

        column_mapping = {
            'ftype': 'Feature Type',
            'name': 'Name',
            'lat': 'Latitude',
            'long': 'Longitude'
        } 

        gdf = gdf.rename(columns=column_mapping)

        return gdf[['Feature Type', 'Name', 'Latitude', 'Longitude']]

    def print_to_pdf2(self, data, gazetteer_title, gazetteer_version, logo, filename):
        pdf = FPDF('portrait', 'mm', 'A4')
        pdf.set_left_margin(20)
        pdf.set_top_margin(20)
        pdf.set_right_margin(20)
        pdf.set_auto_page_break(True, margin=20)

        # Define a custom class derived from FPDF
        class CustomPDF(FPDF):
            def __init__(self):
                super().__init__()
                self.first_page = True

            def header(self):
                if not self.page_no() - 1 == 0:
                    # Add the header content for subsequent pages
                    self.set_y(20)
                    self.set_x(20)
                    self.set_font('Arial', 'B', 12)
                    col_widths = {}  # Custom widths for specific column names
                    col_widths['Feature Type'] = 40
                    col_widths['Name'] = 70
                    col_widths['Latitude'] = 30
                    col_widths['Longitude'] = 30
                    for col in data.columns:
                        if col in col_widths:
                            col_width = col_widths[col]
                        else:
                            col_width = 45  # Default width for other columns
                        self.cell(col_width, 5, col, border='B', align='C')
                    self.ln()

            def footer(self):
                # Add the footer content
                if not self.page_no() - 1 == 0:
                    self.set_y(-20)
                    self.set_font('Arial', 'I', 8)
                    page_number = self.page_no() - 1
                    footer_text = f'Page {page_number}'
                    self.cell(0, 10, footer_text, 0, 0, 'R')

            def add_page(self, orientation=''):
                if self.first_page:
                    self.first_page = False
                    super().add_page(orientation)
                else:
                    super().add_page(orientation)

        # Create a new PDF object using the CustomPDF class
        pdf = CustomPDF()
        pdf.add_page()

        # Add logo
        pdf.set_font('Arial', 'B', 24)
        pdf.image(logo, x=50, y=80, w=100, h=100)
        pdf.cell(0, 60, gazetteer_title, ln=True, align='C')

        # Add column headers
        pdf.add_page()
        pdf.set_font('Arial', 'B', 11)
        col_widths = {}  # Custom widths for specific column names
        col_widths['Feature Type'] = 40
        col_widths['Name'] = 70
        col_widths['Latitude'] = 30
        col_widths['Longitude'] = 30
        # for col in data.columns:
        #     if col in col_widths:
        #         col_width = col_widths[col]
        #     else:
        #         col_width = 45  # Default width for other columns
        #     pdf.cell(col_width, 8, col, border='B', align='C')

        # Add table contents
        pdf.set_font('Arial', '', 10)
        for row in data.values:
            pdf.ln()
            pdf.set_x(20)
            for item, col in zip(row, data.columns):
                if col in col_widths:
                    col_width = col_widths[col]
                else:
                    col_width = 45  # Default width for other columns
                pdf.cell(col_width, 7, str(item), border=0)

        # Save PDF file
        if os.path.exists(filename):
            os.remove(filename)
        pdf.output(f'{filename}')

    # Function to print tables to PDF files
    def print_to_pdf(self, data, gazetteer_title, gazetteer_version, logo, filename):
        pdf = FPDF('portrait', 'mm', 'A4')
        pdf.set_left_margin(20)
        pdf.set_top_margin(20)
        pdf.set_right_margin(20)
        pdf.set_auto_page_break(True, margin=20)

        # Add cover page
        pdf.add_page()
        if logo:
            img_width = 100  # Set desired image width
            img_height = 100  # Set desired image height

            # Calculate image position to center it on the page
            x = (pdf.w - img_width) / 2
            y = (pdf.h - img_height) / 2

            pdf.image(logo, x=x, y=y - 40, w=img_width)

        pdf.set_font('Arial', 'B', 24)
        pdf.set_y(pdf.h - 90)  # Position of the title
        pdf.multi_cell(0, 10, gazetteer_title, 0, align='C')

        # Add content page
        pdf.add_page()

        # Setting Columns Width
        col_widths = {}  # Custom widths for specific column names
        col_widths['Feature Type'] = 45
        col_widths['Feature'] = 45
        col_widths['Ftype'] = 45
        col_widths['Name'] = 65
        col_widths['Latitude'] = 30
        col_widths['Lat'] = 30
        col_widths['Longitude'] = 30
        col_widths['Long'] = 30
        header_height = 8

        # Determine max_width for each column
        max_widths = {}
        for col in data.columns:
            max_widths[col] = max([pdf.get_string_width(str(item)) for item in data[col]])

        def column_header_line():
            # Draw header line
            header_line_y = pdf.t_margin + header_height
            pdf.set_line_width(0.5)
            pdf.line(pdf.l_margin, header_line_y, pdf.w - pdf.r_margin, header_line_y)

        def footer():
            # Set position of the footer
            pdf.set_y(pdf.h - 17)
            # Footer bagian kiri
            pdf.set_font('Arial', '', 6)
            pdf.cell(12, 5, str(date.today()), align='L', ln=False)

            # Footer bagian tengah
            pdf.cell(0, 5, gazetteer_version, align='C', ln=False)

            # Footer bagian kanan
            total_rows = len(data.columns)
            rows_per_page = int((pdf.h - pdf.b_margin - pdf.t_margin - header_height) / 7)
            total_pages = math.ceil((total_rows / rows_per_page) + 1)
            page_number = pdf.page_no() if pdf.page_no() != 0 else 1
            pdf.cell(0, 5, f'Page {page_number-1} of {total_pages-1}', align='R')

            # Draw footer line
            footer_line_y = pdf.h - 18
            pdf.set_line_width(0.5)
            pdf.line(pdf.l_margin, footer_line_y, pdf.w - pdf.r_margin, footer_line_y)

            pdf.set_y(pdf.h - 13)
            # Footer bagian kiri bawah (notes for Line data)
            pdf.set_font('Arial', '', 5)
            pdf.cell(12, 5, "*)The coordinates for the line feature represent the starting point.", align='L', ln=False)

        def column_header(): #repeat column headers
            pdf.set_font('Arial', 'B', 11)
            for col in data.columns:  # Iterate over selected columns
                if col in col_widths:
                    col_width = col_widths[col]
                else:
                    col_width = max_widths[col]  # Default width for other columns
                pdf.cell(col_width, header_height, col, border=0, ln=False)
            pdf.ln()

        # Add 1st page column headers
        column_header()

        # Set header and footer function as the footer callback
        column_header_line()
        pdf.footer = footer

        # Set initial y position for table contents
        table_y = pdf.t_margin + header_height

        # Add table contents
        for i, row in enumerate(data[data.columns].values):
            column_header_line()
            if table_y + 7 > pdf.h - pdf.b_margin:  # Check if the table contents exceed the page height
                pdf.add_page()  # Add new page
                column_header()  # Print column headers
                table_y = pdf.t_margin + header_height  # Reset y position for table contents

            pdf.set_xy(pdf.l_margin, table_y)  # Set starting position for table contents

            pdf.set_font('Arial', '', 8)
            for item, col in zip(row, data.columns):
                row_height = 6
                if col in col_widths:
                    col_width = col_widths[col]
                else:
                    col_width = max_widths[col]  # Default width for other columns

                # Draw dashed border
                pdf.set_draw_color(192)  # Set the drawing color to light gray
                pdf.set_line_width(0.1)  # Set the line width for dashed lines

                dash_length = 1
                gap_length = 2

                x_start = pdf.get_x()
                y_start = pdf.get_y() + row_height
                x_end = pdf.w - pdf.r_margin
                y_end = y_start

                while x_start < x_end:
                    pdf.line(x_start, y_start, x_start + dash_length, y_end)
                    x_start += dash_length + gap_length

                # Reset drawing settings
                pdf.set_draw_color(0)  # Reset the drawing color to black (default)
                pdf.set_line_width(0.1)  # Reset the line width

                pdf.cell(col_width, row_height, str(item), border=0)

            table_y += 7  # Increment y position for next row

        # Save PDF file
        pdf.output(f'{filename}')
    
    def start_checkbox(self):
        self.dlg.checkBox.setChecked(True)
        self.dlg.checkBox_2.setChecked(True)
        self.dlg.checkBox_3.setChecked(True)
    
    def start_combobox(self):
        self.dlg.comboBox.clear()
        self.dlg.comboBox_2.clear()
        self.dlg.comboBox_3.clear()

    def filter_checked(self):
        self.dlg.checkBox.stateChanged.connect(self.update_check_box)
        self.dlg.checkBox_2.stateChanged.connect(self.update_check_box_2)
        self.dlg.checkBox_3.stateChanged.connect(self.update_check_box_3)

    def update_check_box(self):
        if self.dlg.checkBox.isChecked():
            self.dlg.comboBox.setEnabled(True)
        else:
            self.dlg.comboBox.setEnabled(False)
    
    def update_check_box_2(self):
        if self.dlg.checkBox_2.isChecked():
            self.dlg.comboBox_2.setEnabled(True)
        else:
            self.dlg.comboBox_2.setEnabled(False)

    def update_check_box_3(self):
        if self.dlg.checkBox_3.isChecked():
            self.dlg.comboBox_3.setEnabled(True)
        else:
            self.dlg.comboBox_3.setEnabled(False)

    def get_layer_source(self, selected_layer):
        layer_list = QgsProject.instance().mapLayersByName(selected_layer)
        if layer_list:
            layer_name = layer_list[0]
            data_source = layer_name.dataProvider().dataSourceUri()
            provider = layer_name.dataProvider().name()
        else:
            layer = data_source = provider = None
        return [data_source, provider]

    def select_input_logo(self):
        filename, _filter = QFileDialog.getOpenFileName(
            self.dlg, "Select logo file ")
        self.dlg.lineEdit_3.setText(filename)

    def select_output_file(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output file ","", '*.pdf')
        self.dlg.lineEdit_4.setText(filename)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = GazetteerCreatorDialog()
            self.dlg.pushButton.clicked.connect(self.select_input_logo)
            self.dlg.pushButton_2.clicked.connect(self.select_output_file)

        layers = QgsProject.instance().mapLayers().values()
        # Clear the contents of the comboBox from previous runs and checklist checkbox and run filter
        self.start_checkbox()
        self.start_combobox()
        self.filter_checked()
        # Populate the comboBox with names of all the loaded layers
        vector_layers = [layer for layer in layers if layer.type() == QgsMapLayerType.VectorLayer]
        point_vector_layer = [layer.name() for layer in vector_layers if layer.geometryType() == QgsWkbTypes.PointGeometry]
        line_vector_layer = [layer.name() for layer in vector_layers if layer.geometryType() == QgsWkbTypes.LineGeometry or layer.geometryType() == QgsWkbTypes.MultiLineString]
        area_vector_layer = [layer.name() for layer in vector_layers if layer.geometryType() == QgsWkbTypes.PolygonGeometry or layer.geometryType() == QgsWkbTypes.MultiPolygon]

        self.dlg.comboBox.addItems(point_vector_layer)
        self.dlg.comboBox_2.addItems(line_vector_layer)
        self.dlg.comboBox_3.addItems(area_vector_layer)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            selected_point_layer_name = self.dlg.comboBox.currentText()
            point_data_source = self.get_layer_source(selected_point_layer_name)
            
            selected_line_layer_name = self.dlg.comboBox_2.currentText()
            line_data_source = self.get_layer_source(selected_line_layer_name)

            selected_area_layer_name = self.dlg.comboBox_3.currentText()
            area_data_source = self.get_layer_source(selected_area_layer_name)

            title = self.dlg.lineEdit.text()
            version = self.dlg.lineEdit_2.text()
            logo = self.dlg.lineEdit_3.text()
            filepath = self.dlg.lineEdit_4.text()

            data = self.get_data(point_data_source, line_data_source, area_data_source)
            self.print_to_pdf(data, title, version, logo, filepath)

            self.iface.messageBar().pushMessage(
                "Success", "Output file written at " + filepath,
                level=Qgis.Success, duration=5)