import os
import json
import re
from qgis import processing

from qgis.PyQt import uic
from qgis.PyQt import QtWidgets, QtGui

from qgis.core import (QgsCoordinateReferenceSystem, QgsProject)

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'fun_reprojector_dialog_base.ui'))

CRS_DATA = []
for c_dir, sub_dir, files in os.walk(os.path.join(os.path.dirname(__file__),"crs")):
    for file in files:
        if file.endswith(".json"):
            json_path = os.path.join(c_dir, file)
            crs_dict = json.load(open(json_path))
            crs_dict["crs_path"] = c_dir
            CRS_DATA.append(crs_dict)

CRS_CODES = [crs_dict["code"] for crs_dict in CRS_DATA]

with open(os.path.join(os.path.dirname(__file__), "funreprojectorstyle.css"), "r") as f:
    TEXT_STYLE = f.read()
    TEXT_STYLE = re.sub(r"\s+", " ", TEXT_STYLE)
    
class FunReprojectorDialog(QtWidgets.QDialog, FORM_CLASS):
    LOCALE = "en"
    def __init__(self, parent=None, locale = "en"):
        """Constructor."""
        super(FunReprojectorDialog, self).__init__(parent)
        
        self.targetLayer = None
        self.LOCALE = locale
        self.setupUi(self)
        
        self.crsSelector.setFilter([QgsCoordinateReferenceSystem(code) for code in CRS_CODES])
        self.crsSelector.crsChanged.connect(self.updateCrsInfo)
        
        self.layerSelector.layerChanged.connect(self.updateTargetLayer)
        self.prevCrsButton.clicked.connect(self.prevCrs)
        self.nextCrsButton.clicked.connect(self.nextCrs)
        
        self.currentCrsEnum = CRS_CODES.index("EPSG:4326")
        self.updateCrsInfo(QgsCoordinateReferenceSystem("EPSG:4326"))

    def updateTargetLayer(self):
        self.targetLayer = self.layerSelector.currentLayer()
    
    def exec_(self):
        self.updateCrsInfo(QgsCoordinateReferenceSystem("EPSG:4326"))
        super().exec_()
    
    def updateCrsInfo(self, crs:QgsCoordinateReferenceSystem = None):
        try:
            self.currentCrsEnum = CRS_CODES.index(crs.authid())
            crs_dict = CRS_DATA[self.currentCrsEnum]
            img_url = os.path.join(crs_dict["crs_path"], crs_dict["avatar_url"])
            
        except:
            return None
        
        scene = QtWidgets.QGraphicsScene()
        pixmap = QtGui.QPixmap(img_url)
        pixmap_item = QtWidgets.QGraphicsPixmapItem(pixmap)
        scene.addItem(pixmap_item)
        
        self.avatarView.setScene(scene)
        self.avatarView.fitInView(pixmap_item,1)
        
        if self.LOCALE != "en":
            loc_crs_dict = {
                re.sub(f"_{self.LOCALE}$", "", k):v 
                for k, v in crs_dict.items() 
                if re.search(f"_{self.LOCALE}$", k)
            }
            crs_dict = crs_dict | loc_crs_dict
            
        avatar_text = self.parseAvatarText(crs_dict)
        crs_text = self.parseCrsText(crs_dict)
        
        self.avatarText.setHtml(avatar_text)
        self.crsText.setHtml(crs_text)
    
    def parseAvatarText(self, crs_dict):
        avatar_text = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>{TEXT_STYLE}</style>
        </head>
        <body>
        """
        
        if crs_dict.get("avatar_title") is not None:
            avatar_text += f"<div class=avatar_title>{crs_dict['avatar_title']}</div>"
        elif crs_dict.get("avatar_name") is not None and crs_dict.get("avatar_age") is not None:
            avatar_text += f"""
            <div class=avatar_title>
            {crs_dict['avatar_name']} 
            ({crs_dict['avatar_age']})
            </div>
            """
        elif crs_dict.get("avatar_name") is not None:
            avatar_text += f"<div class=avatar_title>{crs_dict['avatar_name']}</div>"

        if crs_dict.get("avatar_subtitle") is not None:
            avatar_text += f"<div class=avatar_subtitle>{crs_dict['avatar_subtitle']}</div>"
        elif crs_dict.get("avatar_origin") is not None:
            avatar_text += f"<div class=avatar_subtitle>{crs_dict['avatar_origin']}</div>"
        
        if crs_dict.get("avatar_description") is not None:
            avatar_text += f"<div class=avatar_text>{crs_dict['avatar_description']}</div>"
        
        avatar_text += "</body></html>"
        avatar_text = re.sub(r"\s+", " ", avatar_text)
        return avatar_text
    
    def parseCrsText(self, crs_dict):
        crs_text = ""
        if crs_dict.get("crs_title") is not None:
            crs_text += f"<div class=crs_title>{crs_dict['crs_title']}</div>"
        
        if crs_dict.get("crs_description") is not None:
            crs_text += f"<div class=crs_text>{crs_dict['crs_description']}</div>"
        crs_text = self.applyHtmlStyle(crs_text)
        return crs_text
    
    def applyHtmlStyle(self, text):        
        html_text = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>{TEXT_STYLE}</style>
        </head>
        <body>
        """ + text + " </body> </html>"
        html_text = re.sub(r"\s+", " ", html_text)
        return html_text
    
    def nextCrs(self):
        self.currentCrsEnum += 1
        if self.currentCrsEnum >= len(CRS_CODES):
            self.currentCrsEnum = 0
        self.crsSelector.setCrs(QgsCoordinateReferenceSystem(CRS_CODES[self.currentCrsEnum]))

    def prevCrs(self):
        self.currentCrsEnum -= 1
        if self.currentCrsEnum < 0:
            self.currentCrsEnum = len(CRS_CODES) - 1
        self.crsSelector.setCrs(QgsCoordinateReferenceSystem(CRS_CODES[self.currentCrsEnum]))
    
    def accept(self):
        reproject_layer = processing.run(
            "native:reprojector",
            {
                'INPUT': self.layerSelector.currentLayer(),
                'TARGET_CRS': self.crsSelector.crs(),
                'OUTPUT': "TEMPORARY_OUTPUT"
            }
        )["OUTPUT"]
        
        QgsProject.instance().addMapLayer(reproject_layer) 
        super().accept()