# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FrequencyOverlaps
                                 A QGIS plugin
 Check for overlapping frequencies in a layer
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-03-07
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Cavin Wang
        email                : kokkang_2000@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtWidgets import QAction, QMessageBox,QTableView

from qgis.core import QgsMapLayerProxyModel, QgsFieldProxyModel, QgsMessageLog, Qgis, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsVectorLayer, QgsFeature, QgsGeometry, QgsPointXY, QgsApplication
from qgis.gui import QgsMapToolEmitPoint, QgsMapTool
import processing

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .frequency_overlaps_dialog import FrequencyOverlapsDialog
import os.path


class FrequencyOverlaps:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale

        # adds the map canvas to self.canvas and add a MapTool Emitter for capturing location from mouse clicks
        self.canvas = self.iface.mapCanvas()
        self.addOverlapsLocation = QgsMapToolEmitPoint(self.canvas)

        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FrequencyOverlaps_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Frequency Overlaps')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('FrequencyOverlaps', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/frequency_overlaps/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Frequency Overlaps'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Frequency Overlaps'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = FrequencyOverlapsDialog()

            # sets the mode control of the main dialog for option center frequency or start-stop frequency to be handled by evt_cmbMode_Changed
            self.dlg.cmbMode.currentTextChanged.connect(self.evt_cmbMode_Changed)
            # sets the field selection of the main dialog to be handled by evt_cmbItemX_Changed
            self.dlg.fcbItem1.currentTextChanged.connect(self.evt_cmbItem1_Changed)
            self.dlg.fcbItem2.currentTextChanged.connect(self.evt_cmbItem2_Changed)
            # sets the unit selection Hz, kHz, MHz or GHz of the main dialog to be handled by evt_SearchAgainstRange_Changed
            self.dlg.cmbFrequencySteps1.currentTextChanged.connect(self.evt_SearchAgainstRange_Changed)
            self.dlg.cmbFrequencySteps2.currentTextChanged.connect(self.evt_SearchAgainstRange_Changed)
            # Sets the default unit for Center Frequency mode, and Center Frequency in MHz and Bandwidth in kHz
            self.dlg.cmbFrequencySteps1.setCurrentIndex(2)
            self.dlg.cmbFrequencySteps2.setCurrentIndex(1)
            # Set the main layer selection of Map Layer to allow points and polygons only.  Field Layer to allow numeric only
            self.dlg.mcbLayers.setFilters(QgsMapLayerProxyModel.PolygonLayer | QgsMapLayerProxyModel.PointLayer)
            self.dlg.fcbItem1.setFilters(QgsFieldProxyModel.Numeric)
            self.dlg.fcbItem2.setFilters(QgsFieldProxyModel.Numeric)
            # Provides the list of fields from the selected layer to the field combo boxes automatically by default
            self.dlg.fcbItem1.setLayer(self.dlg.mcbLayers.currentLayer())
            self.dlg.fcbItem2.setLayer(self.dlg.mcbLayers.currentLayer())

            # Assigns the toggling of the radio buttons to be handled by evt_rbSearch_Changed
            self.dlg.rbWithinLayer.toggled.connect(self.evt_rbSearch_Changed)
            self.dlg.rbAgainstRange.toggled.connect(self.evt_rbSearch_Changed)
            self.dlg.rbAgainstLayer.toggled.connect(self.evt_rbSearch_Changed)
            self.dlg.rbForFrequency.toggled.connect(self.evt_rbSearch_Changed)
            self.dlg.rbFrequencyGaps.toggled.connect(self.evt_rbSearch_Changed)

            # sets the unit selection Hz, kHz, MHz or GHz of the search within range panel to be handled by evt_SearchAgainstRange_Changed
            self.dlg.cmbFrequencySteps3.currentTextChanged.connect(self.evt_SearchAgainstRange_Changed)
            # Sets the dial, center, bandwidth, start, stop frequency of the search within range panel to be handled by a separate event handler
            self.dlg.dialFreqStep.valueChanged.connect(self.evt_SearchAgainstRange_Changed)
            self.dlg.dsbCenterFreq.editingFinished.connect(self.evt_CenterFreq_Changed)
            self.dlg.dsbBandwidth.editingFinished.connect(self.evt_Bandwidth_Changed)
            self.dlg.dsbStartFreq.editingFinished.connect(self.evt_StartFreq_Changed)
            self.dlg.dsbStopFreq.editingFinished.connect(self.evt_StopFreq_Changed)

            # sets up the search against layer panel to be handled by separate event handler
            self.dlg.cmbAgainstLayerMode.currentTextChanged.connect(self.evt_cmbAgainstLayerMode_Changed)
            self.dlg.fcbAgainstLayerItem1.currentTextChanged.connect(self.evt_cmbAgainstLayerItem1_Changed)
            self.dlg.fcbAgainstLayerItem2.currentTextChanged.connect(self.evt_cmbAgainstLayerItem2_Changed)
            # sets the default units Hz, kHz, MHz or GHz for the search against layer panel
            self.dlg.cmbAgainstLayerFrequencySteps1.setCurrentIndex(2)
            self.dlg.cmbAgainstLayerFrequencySteps2.setCurrentIndex(1)
            # Set the search against layer panel selection for Map Layer to allow points and polygons only.  Field Layer to allow numeric only
            self.dlg.mcbAgainstLayer.setFilters(QgsMapLayerProxyModel.PolygonLayer | QgsMapLayerProxyModel.PointLayer)
            self.dlg.fcbAgainstLayerItem1.setFilters(QgsFieldProxyModel.Numeric)
            self.dlg.fcbAgainstLayerItem2.setFilters(QgsFieldProxyModel.Numeric)
            # Provides the list of fields from the selected layer to the field combo boxes automatically by default
            self.dlg.fcbAgainstLayerItem1.setLayer(self.dlg.mcbAgainstLayer.currentLayer())
            self.dlg.fcbAgainstLayerItem2.setLayer(self.dlg.mcbAgainstLayer.currentLayer())

            # Sets the unit selection Hz, kHz, MHz or GHz of the search for frequency gaps to be handled by evt_SearchFrequencyGaps_Changed
            self.dlg.cmbFrequencySteps4.currentTextChanged.connect(self.evt_SearchFrequencyGaps_Changed)
            # Sets the default unit of the search for frequency gaps to MHz
            self.dlg.cmbFrequencySteps4.setCurrentIndex(2)
            # This connect is for tracking the mouse click location for the user when they click on the map when they choose radio button search for frequency gaps
            self.addOverlapsLocation.canvasClicked.connect(self.evt_SearchFrequencyGaps)

            # sent OK button event to be handled by runAnalysis procedure
            self.dlg.bbOK_Cancel.accepted.connect(self.evt_runAnalysis)
            self.dlg.bbOK_Cancel.rejected.connect(self.dlg.reject)

            # hides the rest of the panel except the search within layer panel
            self.dlg.frmAgainstRange.hide()
            self.dlg.frmAgainstLayer.hide()
            self.dlg.frmFrequencyGaps.hide()
            # set progress bar to 0% intially
            self.dlg.pbFrequencyOverlaps.setValue(0)

        # show the dialog
        self.dlg.show()

    def evt_runAnalysis(self):
        ###########################################################################
        #
        #   This function is the main function that runs after the user clicks OK on the dialog box
        #   The procedure to run will be based on which of the radio button is selected.
        #
        #   There are 5 options for the radio button
        #   1) Within the layer:        This option searches for any frequency overlaps of one feature against all other features in the Layer
        #   2) Within a range:          This option searches for features that the frequency overlaps other features within this frequency range
        #   3) Against another layer:   This option searches for frequency overlaps from another layer to all features of the main layer
        #   4) For Frequency:           This option searches for all features that uses a frequency within the selected range
        #   5) For frequency gaps:      This option searches for frequency that are not being used within a location and range selected by the user upon clicking on the map

        # First retrieve the field names of the main layer
        sSrc_A_Fields = self.dlg.fcbItem1.currentField()
        sSrc_B_Fields = self.dlg.fcbItem2.currentField()

        # If user did not select fields for center frequency and bandwidth or start and stop frequency, there is no way to continue
        if sSrc_A_Fields == "" or sSrc_B_Fields == "":
            QMessageBox.information(self.dlg, "Message", "Insufficient parameters provided")
            return 0

        # Retrieve the units for the field in Hz, kHz, MHz or GHz
        sSrc_A_Units = self.dlg.cmbFrequencySteps1.currentText()
        sSrc_B_Units = self.dlg.cmbFrequencySteps2.currentText()

        # if option 1 is selected - Within the layer
        if self.dlg.rbWithinLayer.isChecked():
            # Remove any previously created table from this procedure if there is one
            if len(QgsProject.instance().mapLayersByName("Overlap Locations")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Locations")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Joins")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Joins")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Gaps")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Gaps")[0].id())

            # Prepare a table to store the location for any overlaps found and prepare a line table to show the link between them
            uri = "multipolygon?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapLocations = QgsVectorLayer(uri, "Overlap Locations", "memory")
            vlOLpr = vlOverlapLocations.dataProvider()

            uri = "multilinestring?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapJoins = QgsVectorLayer(uri, "Overlap Joins", "memory")
            vlOJpr = vlOverlapJoins.dataProvider()

            # Retrieve the features from the table
            lyrLayer = self.dlg.mcbLayers.currentLayer()
            intFeatureCounts = lyrLayer.featureCount()
            lstMatrix = []
            intRecordsAnalyzed = 1

            # prepare a transformation to convert the features into a base of EPSG:3857
            new_crs = QgsCoordinateReferenceSystem('EPSG:3857')
            xtransform = QgsCoordinateTransform(lyrLayer.crs(),new_crs,QgsProject().instance())

            # loop through the layer with the layer itself to check for any overlaps of one feature with the other in frequency
            for srcLayer in lyrLayer.getFeatures():
                srcID = srcLayer.id()
                QgsApplication.processEvents()
                # if using center frequency mode, the value of both field boxes should be center and bandwidth
                if self.dlg.cmbMode.currentText() == "Center Frequency":
                    fSrcCenterFreq = srcLayer.attribute(self.dlg.fcbItem1.currentField())
                    fSrcBandwidth = srcLayer.attribute(self.dlg.fcbItem2.currentField())
                    fSrcStartFreq = 0
                    fSrcStopFreq = 0
                # if using Start-Stop frequency mode, the value of both field boxes should be start and stop frequency
                else:
                    fSrcStartFreq = srcLayer.attribute(self.dlg.fcbItem1.currentField())
                    fSrcStopFreq = srcLayer.attribute(self.dlg.fcbItem2.currentField())
                    fSrcCenterFreq = 0
                    fSrcBandwidth = 0

                for tgtLayer in lyrLayer.getFeatures():
                    tgtID = tgtLayer.id()
                    # if using center frequency mode, the value of both field boxes should be center and bandwidth
                    if self.dlg.cmbMode.currentText() == "Center Frequency":
                        fTgtCenterFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                        fTgtBandwidth = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                        fTgtStartFreq = 0
                        fTgtStopFreq = 0
                    # if using Start-Stop frequency mode, the value of both field boxes should be start and stop frequency
                    else:
                        fTgtStartFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                        fTgtStopFreq = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                        fTgtCenterFreq = 0
                        fTgtBandwidth = 0

                    # Track the source to target overlaps.  If found a source to target overlap, the other way target to source is the same.  No need to add both instance, just one instance will do
                    sSrcTgt = "{}-{}".format(srcID, tgtID)
                    sTgtSrc = "{}-{}".format(tgtID, srcID)

                    if tgtID != srcID and (sSrcTgt not in lstMatrix) and (sTgtSrc not in lstMatrix):
                        # Check for overlaps between the 2 features.  boolTest is true when overlaps occurs.  Return the Source center, start and stop frequency in MHz for input into table.  Return the Target center, start and stopp frequency in MHz for input into table.
                        boolTest, fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop =  self.calcCheckForOverlaps(sSrc_A_Units,sSrc_B_Units,fSrcCenterFreq,fSrcBandwidth,fSrcStartFreq,fSrcStopFreq,sSrc_A_Units,sSrc_B_Units,fTgtCenterFreq,fTgtBandwidth,fTgtStartFreq,fTgtStopFreq)
                        if boolTest:
                            lstMatrix.append(sSrcTgt)
                            lstMatrix.append(sTgtSrc)

                            geoSrc = srcLayer.geometry()
                            geoSrc.transform(xtransform)
                            geoTgt = tgtLayer.geometry()
                            geoTgt.transform(xtransform)
                            geoDistance = geoSrc.distance(geoTgt)

                            if geoDistance < self.dlg.sbIgnoreDistance.value() or self.dlg.sbIgnoreDistance.value() == 0:
                                geoSrcBuffer = geoSrc.buffer(300,10)
                                geoTgtBuffer = geoTgt.buffer(300,10)
                                geoCombine = geoSrcBuffer.combine(geoTgtBuffer)
                                # Add 1 circle to source and 1 circle to target if overlap occurs
                                f = QgsFeature()
                                f.setGeometry(geoCombine)
                                f.setAttributes([fSrcCenter,fSrcStart,fSrcStop,fTgtCenter,fTgtStart,fTgtStop,geoDistance])
                                vlOLpr.addFeature(f)
                                # If the source and target that overlaps is not at the same location draw a line to link the target and source feature
                                if geoDistance != 0:
                                    start_point = geoSrc.centroid().asPoint()
                                    stop_point = geoTgt.centroid().asPoint()
                                    f.setGeometry(QgsGeometry.fromPolylineXY([QgsPointXY(start_point.x(),start_point.y()),QgsPointXY(stop_point.x(),stop_point.y())]))
                                    vlOJpr.addFeature(f)

                # update the progress bar on the completion status
                intRecordsAnalyzed += 1
                self.dlg.pbFrequencyOverlaps.setValue((intRecordsAnalyzed/intFeatureCounts)*100)

            # Add the polygon and line layers to the map
            vlOverlapLocations.updateExtents()
            vlOverlapJoins.updateExtents()
            symbol = vlOverlapLocations.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.symbolLayer(0).setBrushStyle(Qt.BrushStyle(Qt.NoBrush))
            symbol.symbolLayer(0).setStrokeColor(QColor(255, 0, 0))
            symbol.symbolLayer(0).setStrokeWidth(0.86)

            symbol = vlOverlapJoins.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.setOpacity(0.2)
            symbol.setWidth(0.66)
            symbol.symbolLayer(0).setPenStyle(Qt.PenStyle(Qt.DotLine))

            vlOverlapLocations.triggerRepaint()
            vlOverlapJoins.triggerRepaint()
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapLocations.id())
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapJoins.id())
            QgsProject.instance().addMapLayer(vlOverlapLocations)
            QgsProject.instance().addMapLayer(vlOverlapJoins)

        # if option 2 is selected - Within a range
        elif self.dlg.rbAgainstRange.isChecked():
            # Remove any previously created table from this procedure if there is one
            if len(QgsProject.instance().mapLayersByName("Overlap Locations")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Locations")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Joins")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Joins")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Gaps")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Gaps")[0].id())

            # Prepare a table to store the location for any overlaps found and prepare a line table to show the link between them
            uri = "multipolygon?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapLocations = QgsVectorLayer(uri, "Overlap Locations", "memory")
            vlOLpr = vlOverlapLocations.dataProvider()

            uri = "multilinestring?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapJoins = QgsVectorLayer(uri, "Overlap Joins", "memory")
            vlOJpr = vlOverlapJoins.dataProvider()

            # Retrieve the features from the table
            lyrLayer = self.dlg.mcbLayers.currentLayer()

            # prepare a transformation to convert the features into a base of EPSG:3857
            new_crs = QgsCoordinateReferenceSystem('EPSG:3857')
            xtransform = QgsCoordinateTransform(lyrLayer.crs(), new_crs, QgsProject().instance())

            # Get the frequency search range from the search within a range panel
            fSrcStartFreq = self.dlg.dsbStartFreq.value()
            fSrcStopFreq = self.dlg.dsbStopFreq.value()
            sTgt_A_Units = sSrc_A_Units
            sTgt_B_Units = sSrc_B_Units
            sSrc_A_Units = self.dlg.cmbFrequencySteps3.currentText().lstrip("in ")
            sSrc_B_Units = self.dlg.dsbBandwidth.suffix().strip()

            # First select those features that fit inside the frequency range and get their IDs for making a selection
            lstMatrix = []
            for tgtLayer in lyrLayer.getFeatures():
                tgtID = tgtLayer.id()
                if self.dlg.cmbMode.currentText() == "Center Frequency":
                    fTgtCenterFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                    fTgtBandwidth = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                    fTgtStartFreq = 0
                    fTgtStopFreq = 0
                else:
                    fTgtStartFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                    fTgtStopFreq = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                    fTgtCenterFreq = 0
                    fTgtBandwidth = 0

                # Check for overlaps between the search range and the feature's frequency span.  boolTest is true when overlaps occurs.  Return the Source center, start and stop frequency in MHz for input into table.  Return the Target center, start and stopp frequency in MHz for input into table.
                boolTest, fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop = self.calcCheckForOverlaps(sSrc_A_Units, sSrc_B_Units, 0, 0, fSrcStartFreq, fSrcStopFreq,sTgt_A_Units, sTgt_B_Units, fTgtCenterFreq, fTgtBandwidth, fTgtStartFreq, fTgtStopFreq)

                # fills the list with the selected feature ids that meets the frequency range criteria
                if boolTest:
                    lstMatrix.append(tgtID)

            lyrLayer.select(lstMatrix)
            lyrClone = processing.run("native:saveselectedfeatures", {'INPUT': lyrLayer, 'OUTPUT': 'memory:'})['OUTPUT']
            lyrLayer.removeSelection()

            # get the feature counts for the progress bar
            intFeatureCounts = lyrClone.featureCount()
            intRecordsAnalyzed = 1

            # Start searching each feature from the selected features that meets the frequency range and against all features of the main layer
            lstMatrix = []
            for srcLayer in lyrClone.getFeatures():
                # if using center frequency mode, the value of both field boxes should be center and bandwidth
                if self.dlg.cmbMode.currentText() == "Center Frequency":
                    fSrcCenterFreq = srcLayer.attribute(self.dlg.fcbItem1.currentField())
                    fSrcBandwidth = srcLayer.attribute(self.dlg.fcbItem2.currentField())
                    fSrcStartFreq = 0
                    fSrcStopFreq = 0
                # if using Start-Stop frequency mode, the value of both field boxes should be start and stop frequency
                else:
                    fSrcStartFreq = srcLayer.attribute(self.dlg.fcbItem1.currentField())
                    fSrcStopFreq = srcLayer.attribute(self.dlg.fcbItem2.currentField())
                    fSrcCenterFreq = 0
                    fSrcBandwidth = 0

                # Loop through the main layer to see which feature overlaps in frequency with the features selected that meets the frequency range
                for tgtLayer in lyrLayer.getFeatures():
                    QgsApplication.processEvents()
                    # if using center frequency mode, the value of both field boxes should be center and bandwidth
                    if self.dlg.cmbMode.currentText() == "Center Frequency":
                        fTgtCenterFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                        fTgtBandwidth = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                        fTgtStartFreq = 0
                        fTgtStopFreq = 0
                    # if using Start-Stop frequency mode, the value of both field boxes should be start and stop frequency
                    else:
                        fTgtStartFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                        fTgtStopFreq = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                        fTgtCenterFreq = 0
                        fTgtBandwidth = 0

                    # Check for overlaps between the 2 features.  boolTest is true when overlaps occurs.  Return the Source center, start and stop frequency in MHz for input into table.  Return the Target center, start and stopp frequency in MHz for input into table.
                    boolTest, fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop =  self.calcCheckForOverlaps(sTgt_A_Units,sTgt_B_Units,fSrcCenterFreq,fSrcBandwidth,fSrcStartFreq,fSrcStopFreq,sTgt_A_Units,sTgt_B_Units,fTgtCenterFreq,fTgtBandwidth,fTgtStartFreq,fTgtStopFreq)
                    geoSrc = srcLayer.geometry()
                    geoSrc.transform(xtransform)
                    geoTgt = tgtLayer.geometry()
                    geoTgt.transform(xtransform)
                    geoDistance = geoSrc.distance(geoTgt)

                    # Track the source to target overlaps.  If found a source to target overlap, the other way target to source is the same.  No need to add both instance, just one instance will do
                    start_point = geoSrc.centroid().asPoint()
                    stop_point = geoTgt.centroid().asPoint()
                    sSrcTgt = "{}-{}|{},{}-{},{}".format(fSrcCenter, fTgtCenter, start_point.x(), start_point.y(),stop_point.x(), stop_point.y())
                    sTgtSrc = "{}-{}|{},{}-{},{}".format(fTgtCenter, fSrcCenter, stop_point.x(), stop_point.y(),start_point.x(), start_point.y())

                    if boolTest and (sSrcTgt not in lstMatrix) and (sTgtSrc not in lstMatrix):
                        lstMatrix.append(sSrcTgt)
                        lstMatrix.append(sTgtSrc)
                        if geoDistance < self.dlg.sbIgnoreDistance.value() or self.dlg.sbIgnoreDistance.value() == 0:
                            # Since the the clone layer is selected from a subset of the main layer itself, using this layer to compare against all the features of the main layer
                            # may result in the same feature in the cloned layer comparing with itself in the main layer.  This 2 should not be counted as being overlapped
                            if geoDistance == 0 and fSrcCenter != fTgtCenter and fSrcStart != fTgtStart and fSrcStop != fTgtStopFreq:
                                # this section checks for overlapping frequency in the same geographical site between itself
                                geoSrcBuffer = geoSrc.buffer(300,10)
                                geoTgtBuffer = geoTgt.buffer(300,10)
                                geoCombine = geoSrcBuffer.combine(geoTgtBuffer)
                                f = QgsFeature()
                                f.setGeometry(geoCombine)
                                f.setAttributes([fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop, geoDistance])
                                vlOLpr.addFeature(f)
                            elif geoDistance != 0:
                                # this section is for drawing 2 circles one for source and one for target feature
                                geoSrcBuffer = geoSrc.buffer(300, 10)
                                geoTgtBuffer = geoTgt.buffer(300, 10)
                                geoCombine = geoSrcBuffer.combine(geoTgtBuffer)
                                f = QgsFeature()
                                f.setGeometry(geoCombine)
                                f.setAttributes([fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop, geoDistance])
                                vlOLpr.addFeature(f)

                            if geoDistance != 0:
                                # this section draws a line between the 2 feature is they are not on the same location and have overlapping frequencies
                                f.setGeometry(QgsGeometry.fromPolylineXY([QgsPointXY(start_point.x(), start_point.y()),QgsPointXY(stop_point.x(), stop_point.y())]))
                                vlOJpr.addFeature(f)
                # update the progress bar on the completion status
                intRecordsAnalyzed += 1
                self.dlg.pbFrequencyOverlaps.setValue((intRecordsAnalyzed / intFeatureCounts) * 100)

            # Add the polygon and line layers to the map and remove the cloned layer
            QgsProject.instance().removeMapLayer(lyrClone.id())
            vlOverlapLocations.updateExtents()
            vlOverlapJoins.updateExtents()
            symbol = vlOverlapLocations.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.symbolLayer(0).setBrushStyle(Qt.BrushStyle(Qt.NoBrush))
            symbol.symbolLayer(0).setStrokeColor(QColor(255, 0, 0))
            symbol.symbolLayer(0).setStrokeWidth(0.86)

            symbol = vlOverlapJoins.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.setOpacity(0.2)
            symbol.setWidth(0.66)
            symbol.symbolLayer(0).setPenStyle(Qt.PenStyle(Qt.DotLine))

            vlOverlapLocations.triggerRepaint()
            vlOverlapJoins.triggerRepaint()
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapLocations.id())
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapJoins.id())
            QgsProject.instance().addMapLayer(vlOverlapLocations)
            QgsProject.instance().addMapLayer(vlOverlapJoins)

        # if option 3 is selected - Search for frequency
        elif self.dlg.rbAgainstLayer.isChecked():
            # Remove any previously created table from this procedure if there is one
            if len(QgsProject.instance().mapLayersByName("Overlap Locations")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Locations")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Joins")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Joins")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Gaps")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Gaps")[0].id())

            # First retrieve the field names from the search against layer
            sSrc_A_Fields = self.dlg.fcbAgainstLayerItem1.currentField()
            sSrc_B_Fields = self.dlg.fcbAgainstLayerItem2.currentField()

            # Unable to continue the search if the frequency parameters from the search against layer is not provided
            if sSrc_A_Fields == "" or sSrc_B_Fields == "":
                QMessageBox.information(self.dlg, "Message", "Missing parameters from the reference layer to search from")
                return 0

            # Retrieve the units for the field in Hz, kHz, MHz or GHz for the search against layer and the mainlayer
            sSrc_A_Units = self.dlg.cmbAgainstLayerFrequencySteps1.currentText()
            sSrc_B_Units = self.dlg.cmbAgainstLayerFrequencySteps2.currentText()
            sTgt_A_Units = self.dlg.cmbFrequencySteps1.currentText()
            sTgt_B_Units = self.dlg.cmbFrequencySteps2.currentText()

            # Prepare a table to store the location for any overlaps found and prepare a line table to show the link between them
            uri = "multipolygon?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapLocations = QgsVectorLayer(uri, "Overlap Locations", "memory")
            vlOLpr = vlOverlapLocations.dataProvider()

            uri = "multilinestring?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapJoins = QgsVectorLayer(uri, "Overlap Joins", "memory")
            vlOJpr = vlOverlapJoins.dataProvider()

            # Retrieve the features from the Search against Layer for searching against the main layer
            lyrSrcLayer = self.dlg.mcbAgainstLayer.currentLayer()
            intFeatureCounts = lyrSrcLayer.featureCount()

            # Retrieve the features from the main Layer for comparison
            lyrTgtLayer = self.dlg.mcbLayers.currentLayer()
            intRecordsAnalyzed = 1

            # prepare a transformation to convert the features into a base of EPSG:3857
            new_crs = QgsCoordinateReferenceSystem('EPSG:3857')
            xSrctransform = QgsCoordinateTransform(lyrSrcLayer.crs(),new_crs,QgsProject().instance())
            xTgttransform = QgsCoordinateTransform(lyrTgtLayer.crs(), new_crs, QgsProject().instance())

            # Loop through each feature in the Searc against layer and compare this with the main layer to check for frequency overlaps
            for srcLayer in lyrSrcLayer.getFeatures():
                QgsApplication.processEvents()
                # if using center frequency mode, the value of both field boxes should be center and bandwidth
                if self.dlg.cmbAgainstLayerMode.currentText() == "Center Frequency":
                    fSrcCenterFreq = srcLayer.attribute(self.dlg.fcbAgainstLayerItem1.currentField())
                    fSrcBandwidth = srcLayer.attribute(self.dlg.fcbAgainstLayerItem2.currentField())
                    fSrcStartFreq = 0
                    fSrcStopFreq = 0
                # if using Start-Stop frequency mode, the value of both field boxes should be start and stop frequency
                else:
                    fSrcStartFreq = srcLayer.attribute(self.dlg.fcbAgainstLayerItem1.currentField())
                    fSrcStopFreq = srcLayer.attribute(self.dlg.fcbAgainstLayerItem2.currentField())
                    fSrcCenterFreq = 0
                    fSrcBandwidth = 0

                for tgtLayer in lyrTgtLayer.getFeatures():
                    # if using center frequency mode, the value of both field boxes should be center and bandwidth
                    if self.dlg.cmbMode.currentText() == "Center Frequency":
                        fTgtCenterFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                        fTgtBandwidth = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                        fTgtStartFreq = 0
                        fTgtStopFreq = 0
                    # if using Start-Stop frequency mode, the value of both field boxes should be start and stop frequency
                    else:
                        fTgtStartFreq = tgtLayer.attribute(self.dlg.fcbItem1.currentField())
                        fTgtStopFreq = tgtLayer.attribute(self.dlg.fcbItem2.currentField())
                        fTgtCenterFreq = 0
                        fTgtBandwidth = 0

                    # Check for overlaps between the 2 features.  boolTest is true when overlaps occurs.  Return the Source center, start and stop frequency in MHz for input into table.  Return the Target center, start and stopp frequency in MHz for input into table.
                    boolTest, fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop = self.calcCheckForOverlaps(sSrc_A_Units, sSrc_B_Units, fSrcCenterFreq, fSrcBandwidth, fSrcStartFreq, fSrcStopFreq, sTgt_A_Units, sTgt_B_Units, fTgtCenterFreq, fTgtBandwidth, fTgtStartFreq, fTgtStopFreq)
                    if boolTest:
                        geoSrc = srcLayer.geometry()
                        geoSrc.transform(xSrctransform)
                        geoTgt = tgtLayer.geometry()
                        geoTgt.transform(xTgttransform)
                        geoDistance = geoSrc.distance(geoTgt)

                        if geoDistance < self.dlg.sbIgnoreDistance.value() or self.dlg.sbIgnoreDistance.value() == 0:
                            # Add 1 circle to source and 1 circle to target if overlap occurs
                            geoSrcBuffer = geoSrc.buffer(300, 10)
                            geoTgtBuffer = geoTgt.buffer(300, 10)
                            geoCombine = geoSrcBuffer.combine(geoTgtBuffer)
                            f = QgsFeature()
                            f.setGeometry(geoCombine)
                            f.setAttributes([fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop, geoDistance])
                            vlOLpr.addFeature(f)

                            # If the source and target that overlaps is not at the same location draw a line to link the target and source feature
                            if geoDistance != 0:
                                start_point = geoSrc.centroid().asPoint()
                                stop_point = geoTgt.centroid().asPoint()
                                f.setGeometry(QgsGeometry.fromPolylineXY([QgsPointXY(start_point.x(), start_point.y()), QgsPointXY(stop_point.x(), stop_point.y())]))
                                vlOJpr.addFeature(f)

                # update the progress bar on the completion status
                intRecordsAnalyzed += 1
                self.dlg.pbFrequencyOverlaps.setValue((intRecordsAnalyzed / intFeatureCounts) * 100)

            # Add the polygon and line layers to the map
            vlOverlapLocations.updateExtents()
            vlOverlapJoins.updateExtents()
            symbol = vlOverlapLocations.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.symbolLayer(0).setBrushStyle(Qt.BrushStyle(Qt.NoBrush))
            symbol.symbolLayer(0).setStrokeColor(QColor(255, 0, 0))
            symbol.symbolLayer(0).setStrokeWidth(0.86)

            symbol = vlOverlapJoins.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.setOpacity(0.2)
            symbol.setWidth(0.66)
            symbol.symbolLayer(0).setPenStyle(Qt.PenStyle(Qt.DotLine))

            vlOverlapLocations.triggerRepaint()
            vlOverlapJoins.triggerRepaint()
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapLocations.id())
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapJoins.id())
            QgsProject.instance().addMapLayer(vlOverlapLocations)
            QgsProject.instance().addMapLayer(vlOverlapJoins)

        # if option 4 is selected - Search against another layer
        elif self.dlg.rbForFrequency.isChecked():
            # Remove any previously created table from this procedure if there is one
            if len(QgsProject.instance().mapLayersByName("Overlap Locations")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Locations")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Joins")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Joins")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Gaps")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Gaps")[0].id())

            # Prepare a table to store the location for any overlaps of frequency found
            uri = "multipolygon?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapLocations = QgsVectorLayer(uri, "Overlap Locations", "memory")
            vlOLpr = vlOverlapLocations.dataProvider()

            # Retrieve the features from the table
            lyrLayer = self.dlg.mcbLayers.currentLayer()
            intFeatureCounts = lyrLayer.featureCount()
            intRecordsAnalyzed = 1

            # prepare a transformation to convert the features into a base of EPSG:3857
            new_crs = QgsCoordinateReferenceSystem('EPSG:3857')
            xtransform = QgsCoordinateTransform(lyrLayer.crs(),new_crs,QgsProject().instance())

            # Get the frequency range to search from the search for frequency panel
            fSrcStartFreq = self.dlg.dsbStartFreq.value()
            fSrcStopFreq = self.dlg.dsbStopFreq.value()
            sTgt_A_Units = sSrc_A_Units
            sTgt_B_Units = sSrc_B_Units
            sSrc_A_Units = self.dlg.cmbFrequencySteps3.currentText().lstrip("in ")
            sSrc_B_Units = self.dlg.dsbBandwidth.suffix().strip()

            # loop through the layer to check which features overlaps this frequency range
            for srcLayer in lyrLayer.getFeatures():
                QgsApplication.processEvents()
                # if using center frequency mode, the value of both field boxes should be center and bandwidth
                if self.dlg.cmbMode.currentText() == "Center Frequency":
                    fTgtCenterFreq = srcLayer.attribute(self.dlg.fcbItem1.currentField())
                    fTgtBandwidth = srcLayer.attribute(self.dlg.fcbItem2.currentField())
                    fTgtStartFreq = 0
                    fTgtStopFreq = 0
                # if using Start-Stop frequency mode, the value of both field boxes should be start and stop frequency
                else:
                    fTgtStartFreq = srcLayer.attribute(self.dlg.fcbItem1.currentField())
                    fTgtStopFreq = srcLayer.attribute(self.dlg.fcbItem2.currentField())
                    fTgtCenterFreq = 0
                    fTgtBandwidth = 0

                # Check for overlaps between the frequency range and the features.  boolTest is true when overlaps occurs.  Return the Source center, start and stop frequency in MHz for input into table.  Return the Target center, start and stopp frequency in MHz for input into table.
                boolTest, fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop = self.calcCheckForOverlaps(sSrc_A_Units, sSrc_B_Units, 0, 0, fSrcStartFreq, fSrcStopFreq, sTgt_A_Units, sTgt_B_Units, fTgtCenterFreq, fTgtBandwidth, fTgtStartFreq, fTgtStopFreq)
                if boolTest:
                    geoSrc = srcLayer.geometry()
                    geoSrc.transform(xtransform)
                    geoSrcBuffer = geoSrc.buffer(300, 10)
                    # Add 1 circle to the feature if overlaps ovvurs
                    f = QgsFeature()
                    f.setGeometry(geoSrcBuffer)
                    f.setAttributes([fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop, 0])
                    vlOLpr.addFeature(f)

                # update the progress bar on the completion status
                intRecordsAnalyzed += 1
                self.dlg.pbFrequencyOverlaps.setValue((intRecordsAnalyzed / intFeatureCounts) * 100)

            # Add the polygon to the map
            vlOverlapLocations.updateExtents()
            symbol = vlOverlapLocations.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.symbolLayer(0).setBrushStyle(Qt.BrushStyle(Qt.NoBrush))
            symbol.symbolLayer(0).setStrokeColor(QColor(255, 0, 0))
            symbol.symbolLayer(0).setStrokeWidth(0.86)

            vlOverlapLocations.triggerRepaint()
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapLocations.id())
            QgsProject.instance().addMapLayer(vlOverlapLocations)

        # if option 5 is selected - Search for frequency gaps
        elif self.dlg.rbFrequencyGaps.isChecked():
            # This function searches for frequency that is not used by clicking on a location on the map, with a distance range specified and the start and stop frequency to look for

            # Technically it is not possible for the start frequency to be smaller than the stop frequency
            if self.dlg.dsbGapsStartFreq.value() >= self.dlg.dsbGapsStopFreq.value():
                QMessageBox.information(self.dlg, "Message", "This is not a range! Start frequency should be smaller than stop frequency.")
                return 0

            # change to the map and let user click the location
            self.canvas.setMapTool(self.addOverlapsLocation)
            self.dlg.hide()

        # Reset progress bar to 0, symbolising all actions are completed
        self.dlg.pbFrequencyOverlaps.setValue(0)

    def calcCheckForOverlaps(self,sSrcUnitA,sSrcUnitB,fSrcCenterFreq,fSrcBandwidth,fSrcStartFreq,fSrcStopFreq,sTgtUnitA,sTgtUnitB,fTgtCenterFreq,fTgtBandwidth,fTgtStartFreq,fTgtStopFreq):
        ###########################################################################
        #
        #   This function takes as inputs the following parameters:
        #       sSrcUnitA       =   The source frequency unit in Hz, kHz, MHz or GHz
        #       sSrcUnitB       =   The source bandwidth unit in Hz, kHz, MHz or GHz
        #       fSrcCenter      =   The source center frequency
        #       fSrcBandwidth   =   The source bandwidth
        #       fSrcStartFreq   =   The source start frequency
        #       fSrcStopFreq    =   The source stop frequency
        #       sTgtUnitA       =   The target frequency unit in Hz, kHz, MHz or GHz
        #       sTgtUnitB       =   The target bandwidth unit in Hz, kHz, MHz or GHz
        #       fTgtCenter      =   The target center frequency
        #       fTgtBandwidth   =   The target bandwidth
        #       fTgtStartFreq   =   The target start frequency
        #       fTgtStopFreq    =   The target stop frequency
        #
        #       The purpose of the function is to search search if there is a frequency overlap between source and target.
        #       In any situation either center frequency and bandwidth or start and stop frequency is needed to know the range
        #       of frequency either for the source of the target.  The user may have different units of frequency use in Hz, kHz, MHz, GHz
        #       Thus to perform a comparison both source and target need to be normalise to a common denominator such as MHz and then to
        #       compare if there is an overlap between the source and target frequency range.  If an overlap occurs return a boolean true
        #       along with the center freq, start freq, stop freq of the source and target in MHz units.  If an overlap do not occur a boolean
        #       false is returned along with the center freq, start freq, stop freq of the source and target in MHz units.

        fSrcFreqMultiplier = 0.01

        #   This function converts the frequency portion to the equivalent value in MHz
        if sSrcUnitA == "Hz":
            fSrcFreqMultiplier = 1/1000000
        elif sSrcUnitA == "kHz":
            fSrcFreqMultiplier = 1/1000
        elif sSrcUnitA == "MHz":
            fSrcFreqMultiplier = 1
        elif sSrcUnitA == "GHz":
            fSrcFreqMultiplier = 1000

        #   initialise fBandwidthMultiplier as a float variable
        fSrcBandwidthMultiplier = 0.01

        #   This function converts the bandwidth portion to the equivalent value in MHz
        if sSrcUnitB == "Hz":
            fSrcBandwidthMultiplier = 1 / 1000000
        elif sSrcUnitB == "kHz":
            fSrcBandwidthMultiplier = 1 / 1000
        elif sSrcUnitB == "MHz":
            fSrcBandwidthMultiplier = 1
        elif sSrcUnitB == "GHz":
            fSrcBandwidthMultiplier = 1000

        #   initialise fFreqMultiplier as a float variable
        fTgtFreqMultiplier = 0.01

        #   This function converts the frequency portion to the equivalent value in MHz
        if sTgtUnitA == "Hz":
            fTgtFreqMultiplier = 1 / 1000000
        elif sTgtUnitA == "kHz":
            fTgtFreqMultiplier = 1 / 1000
        elif sTgtUnitA == "MHz":
            fTgtFreqMultiplier = 1
        elif sTgtUnitA == "GHz":
            fTgtFreqMultiplier = 1000

        #   initialise fBandwidthMultiplier as a float variable
        fTgtBandwidthMultiplier = 0.01

        #   This function converts the bandwidth portion to the equivalent value in MHz
        if sTgtUnitB == "Hz":
            fTgtBandwidthMultiplier = 1 / 1000000
        elif sTgtUnitB == "kHz":
            fTgtBandwidthMultiplier = 1 / 1000
        elif sTgtUnitB == "MHz":
            fTgtBandwidthMultiplier = 1
        elif sTgtUnitB == "GHz":
            fTgtBandwidthMultiplier = 1000

        if fSrcCenterFreq == 0 and fSrcStartFreq != 0:
            calcfSrcStartFreq = fSrcStartFreq * fSrcFreqMultiplier
            calcfSrcStopFreq = fSrcStopFreq * fSrcFreqMultiplier
            calcfSrcCenterFreq = calcfSrcStartFreq+((calcfSrcStopFreq-calcfSrcStartFreq)/2)
        elif fSrcStartFreq == 0 and fSrcCenterFreq != 0:
            calcfSrcCenterFreq = fSrcCenterFreq * fSrcFreqMultiplier
            calcfSrcBandwidth = fSrcBandwidth * fSrcBandwidthMultiplier
            calcfSrcStartFreq = calcfSrcCenterFreq - (calcfSrcBandwidth / 2)
            calcfSrcStopFreq = calcfSrcCenterFreq + (calcfSrcBandwidth / 2)

        if fTgtCenterFreq == 0 and fTgtStartFreq != 0:
            calcfTgtStartFreq = fTgtStartFreq * fTgtFreqMultiplier
            calcfTgtStopFreq = fTgtStopFreq * fTgtFreqMultiplier
            calcfTgtCenterFreq = calcfTgtStartFreq+((calcfTgtStopFreq - calcfTgtStartFreq)/2)
        elif fTgtStartFreq == 0 and fTgtCenterFreq != 0:
            calcfTgtCenterFreq = fTgtCenterFreq * fTgtFreqMultiplier
            calcfTgtBandwidth = fTgtBandwidth * fTgtBandwidthMultiplier
            calcfTgtStartFreq = calcfTgtCenterFreq - (calcfTgtBandwidth / 2)
            calcfTgtStopFreq = calcfTgtCenterFreq + (calcfTgtBandwidth / 2)

        if calcfSrcStopFreq >= calcfTgtStartFreq and calcfTgtStopFreq >= calcfSrcStartFreq:
            return True, calcfSrcCenterFreq,calcfSrcStartFreq, calcfSrcStopFreq, calcfTgtCenterFreq, calcfTgtStartFreq, calcfTgtStopFreq
        else:
            return False, calcfSrcCenterFreq,calcfSrcStartFreq, calcfSrcStopFreq, calcfTgtCenterFreq, calcfTgtStartFreq, calcfTgtStopFreq

    def evt_SearchFrequencyGaps(self,point,button):
        ###########################################################################
        #
        #   This function takes as inputs the following parameters:
        #       point           =   The location the user clicked
        #       button          =   The button clicked Left-mouse button or Right-mouse button
        #
        #       The purpose of the function is to search from a user clicked location with a bounding box size defined
        #       by distance to search for all frequency in use within this region.  User will set a range of frequency
        #       to search.  Any unused frequency range will be returned in a table, with a bounding box drawn out
        #       showing the area of search conducted.

        # Check if user clicked a location
        if button == Qt.LeftButton:
            # Remove off the Overlap Gaps table if it existed before the search
            if len(QgsProject.instance().mapLayersByName("Overlap Gaps")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Gaps")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Locations")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Locations")[0].id())
            if len(QgsProject.instance().mapLayersByName("Overlap Joins")) != 0:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayersByName("Overlap Joins")[0].id())

            # Create a table Overlap Gaps to hold the search results
            uri = "multipolygon?crs=epsg:3857&field=Avail Start Freq (MHz):double(14,7)&field=Avail Stop Freq (MHz):double(14,7)&field=Avail Bandwidth (MHz):double(14,7)"
            vlOverlapGaps = QgsVectorLayer(uri, "Overlap Gaps", "memory")
            vlOGpr = vlOverlapGaps.dataProvider()

            # Prepare a table to store the location for any overlaps of frequency found
            uri = "multipolygon?crs=epsg:3857&field=Src Center Freq (MHz):double(14,7)&field=Src Start Freq (MHz):double(14,7)&field=Src Stop Freq (MHz):double(14,7)&field=Tgt Center Freq (MHz):double(14,7)&field=Tgt Start Freq (MHz):double(14,7)&field=Tgt Stop Freq (MHz):double(14,7)&field=Distance (meters):double(14,7)"
            vlOverlapLocations = QgsVectorLayer(uri, "Overlap Locations", "memory")
            vlOLpr = vlOverlapLocations.dataProvider()

            # Retrieve the data table specified by the user
            lyrLayer = self.dlg.mcbLayers.currentLayer()
            # Convert the layer to EPSG:3857 for easier calculation of distance required in meters
            parameter = {'INPUT':lyrLayer,'TARGET_CRS':'EPSG:3857','OUTPUT':'memory:Reprojected'}
            result = processing.run('native:reprojectlayer',parameter)
            lyrClone = result['OUTPUT']
            new_crs = QgsCoordinateReferenceSystem('EPSG:3857')
            xSrctransform = QgsCoordinateTransform(self.canvas.mapSettings().destinationCrs(), new_crs, QgsProject().instance())

            # Get the location user clicked on map
            geoSrc = QgsGeometry.fromPointXY(point)
            # Convert the location to EPSG:3857
            geoSrc.transform(xSrctransform)
            iDistance = self.dlg.sbIgnoreDistance.value()
            # If distance is 0 set it to 1000 Km
            if iDistance == 0:
                iDistance = 1000000
            geoSrcBuffer = geoSrc.buffer(iDistance, 10)
            # Grab all features within this defined distance
            featureImpacted = lyrClone.getFeatures(geoSrcBuffer.boundingBox())

            # Get the units of the current search range in Hz, kHz, MHz or GHz and get the start and stop freq for the search
            sSrc_A_Units = self.dlg.cmbFrequencySteps4.currentText().lstrip("in ")
            fSrcStartFreq= self.dlg.dsbGapsStartFreq.value()
            fSrcStopFreq = self.dlg.dsbGapsStopFreq.value()
            fSrcCenterFreq = fSrcStartFreq+((fSrcStopFreq - fSrcStartFreq)/2)
            fSrcBandwidth = fSrcStopFreq - fSrcStartFreq

            # Prepare a storage array for the frequency that overlaps with the search range
            arrayFoundFreq=[]
            for feaScanFrequency in featureImpacted:
                # Retrieve the start and stop frequency from the features found.  Also converts center frequency and bandwidth to start and stop frequency.
                # vA = Center Frequency
                # vB = Bandwidth
                # vC = Start Frequency
                # vD = Stop Frequency
                QgsApplication.processEvents()
                if self.dlg.cmbMode.currentText() == "Center Frequency":
                    vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(self.dlg.cmbFrequencySteps1.currentText(), self.dlg.cmbFrequencySteps2.currentText(),feaScanFrequency.attribute(self.dlg.fcbItem1.currentField()),feaScanFrequency.attribute(self.dlg.fcbItem2.currentField()), 0, 0)
                else:
                    vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(self.dlg.cmbFrequencySteps1.currentText(), self.dlg.cmbFrequencySteps1.currentText() ,0,0,feaScanFrequency.attribute(self.dlg.fcbItem1.currentField()),feaScanFrequency.attribute(self.dlg.fcbItem2.currentField()))

                # Test if frequency of this feature intersect with the search range requested.  boolTest returns true if range intersects
                boolTest, fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop = self.calcCheckForOverlaps(sSrc_A_Units, sSrc_A_Units, fSrcCenterFreq, fSrcBandwidth, 0, 0, self.dlg.cmbFrequencySteps1.currentText(), self.dlg.cmbFrequencySteps1.currentText(), 0, 0, vC, vD)

                if boolTest:
                    arrayFoundFreq.append([vC, vD])
                    geoTgt = feaScanFrequency.geometry()
                    geoDistance = geoTgt.distance(geoSrc)
                    geoTgtBuffer = geoTgt.buffer(300, 10)
                    # Add 1 circle to the feature if the feature is in the analysis
                    f = QgsFeature()
                    f.setGeometry(geoTgtBuffer)
                    f.setAttributes([fSrcCenter, fSrcStart, fSrcStop, fTgtCenter, fTgtStart, fTgtStop, geoDistance])
                    vlOLpr.addFeature(f)

            nFreqPairs = len(arrayFoundFreq)
            arrayFreqGaps = []

            # Check if there are any frequency found within the range recorded in arrayFoundFreq.  Find the gaps between them and store in arrayFreqGaps
            if nFreqPairs >=1:
                arrayFoundFreq.sort(key=lambda aFF:aFF[0])
                for i in range(1,nFreqPairs):
                    # Previous interval end
                    prevEnd = arrayFoundFreq[i - 1][1]
                    # Current interval start
                    currStart = arrayFoundFreq[i][0]
                    # If Previous Interval is less than current Interval then we store that answer
                    if prevEnd < currStart:
                        arrayFreqGaps.append([prevEnd, currStart])

            # Show the area covered by this search on map
            f = QgsFeature()
            tempFeat = geoSrcBuffer.boundingBox().asWktPolygon()
            g = QgsGeometry.fromWkt(tempFeat)

            h = geoSrc.buffer(50, 10)
            j = g.difference(h)
            f.setGeometry(j)

            # Convert the frequency entered by user to appropriate value in MHz
            if sSrc_A_Units == "Hz":
                fMultiplier = 1/1000000
            elif sSrc_A_Units == "kHz":
                fMultiplier = 1/1000
            elif sSrc_A_Units == "MHz":
                fMultiplier = 1
            elif sSrc_A_Units == "GHz":
                fMultiplier = 1000

            # Check if any frequency gaps were found in the arrayFreqGaps
            if len(arrayFreqGaps) != 0:
                # Insert the first gap if there is a frequency gap between the start frequency the user selected and the lowest frequency found on the map within this range
                if min(arrayFoundFreq)[0] - (fSrcStartFreq*fMultiplier) > 0:
                    f.setAttributes([fSrcStartFreq*fMultiplier,min(arrayFoundFreq)[0]-(1/1000000),min(arrayFoundFreq)[0] - (fSrcStartFreq*fMultiplier)-(1/1000000)])
                    vlOGpr.addFeature(f)
                # Insert the rest of the frequency gaps into the table
                for i in arrayFreqGaps:
                    f.setAttributes([i[0]+(1/1000000),i[1]-(1/1000000),i[1]-i[0]-(2/1000000)])
                    vlOGpr.addFeature(f)

                # Insert the last gap if there is a frequency gap between the stop frequency the user selected and the highest frequency found on the map within this range
                if (fSrcStopFreq * fMultiplier) - max(arrayFoundFreq)[1] > 0:
                    f.setAttributes([max(arrayFoundFreq)[1]+(1/1000000),fSrcStopFreq * fMultiplier,(fSrcStopFreq * fMultiplier)-max(arrayFoundFreq)[1]-(1/1000000)])
                    vlOGpr.addFeature(f)
            else:
                # If there is nothing found, it means the whole range the user is searching for is available
                if len(arrayFoundFreq) == 0:
                    f.setAttributes([fSrcStartFreq*fMultiplier,fSrcStopFreq*fMultiplier,(fSrcStopFreq*fMultiplier) - (fSrcStartFreq*fMultiplier)])
                    vlOGpr.addFeature(f)
                else:
                    if min(arrayFoundFreq)[0] - (fSrcStartFreq*fMultiplier) > 0:
                        f.setAttributes([fSrcStartFreq*fMultiplier,min(arrayFoundFreq)[0]-(1/1000000),min(arrayFoundFreq)[0] - (fSrcStartFreq*fMultiplier)-(1/1000000)])
                        vlOGpr.addFeature(f)
                    if (fSrcStopFreq * fMultiplier) - max(arrayFoundFreq)[1] > 0:
                        f.setAttributes([max(arrayFoundFreq)[1]+(1/1000000),fSrcStopFreq * fMultiplier,(fSrcStopFreq * fMultiplier)-max(arrayFoundFreq)[1]-(1/1000000)])
                        vlOGpr.addFeature(f)


            # Remove the temporary layer which was converted to EPSG:3857 for the search
            QgsProject.instance().removeMapLayer(lyrClone.id())
            # Add the polygon to the map
            vlOverlapLocations.updateExtents()
            vlOverlapGaps.updateExtents()

            symbol = vlOverlapLocations.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.symbolLayer(0).setBrushStyle(Qt.BrushStyle(Qt.NoBrush))
            symbol.symbolLayer(0).setStrokeColor(QColor(255, 0, 0))
            symbol.symbolLayer(0).setStrokeWidth(0.86)

            symbol = vlOverlapGaps.renderer().symbol()
            symbol.setColor(QColor.fromRgb(255, 0, 0))
            symbol.setOpacity(0.05)

            vlOverlapLocations.triggerRepaint()
            vlOverlapGaps.triggerRepaint()
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapLocations.id())
            self.iface.layerTreeView().refreshLayerSymbology(vlOverlapGaps.id())
            QgsProject.instance().addMapLayer(vlOverlapLocations)
            QgsProject.instance().addMapLayer(vlOverlapGaps)


        # if user right click, it means a cancel action and fall back to the main dialog
        if button == Qt.RightButton:
            self.canvas.unsetMapTool(self.addOverlapsLocation)
            self.dlg.show()

    def evt_cmbMode_Changed(self,mode):
        ###########################################################################
        #
        #   This function handles the changes in the mode between use of Center Frequency basis
        #   or Start-Stop frequency
        #
        #   The purpose of the function is to switch between the values used for the source data
        #   it can only be either center frequency and bandwidth or start and stop frequency.
        #   The label will change base on the mode selected

        if mode == "Center Frequency":
            self.dlg.lblItem1.setText("Center Frequency:")
            self.dlg.lblItem2.setText("Bandwidth:")
        else:
            self.dlg.lblItem1.setText("Start Frequency:")
            self.dlg.lblItem2.setText("Stop Frequency:")
            self.dlg.cmbFrequencySteps2.setCurrentText(self.dlg.cmbFrequencySteps1.currentText())

    def evt_cmbItem1_Changed(self,item1):
        ###########################################################################
        #
        #   This function handles the changes in the main layer.  searches
        #   for any likely hint of the units that the field is in
        #
        #   The purpose of the function is to switch between the units for the main layer dialog
        #   automatically if there are any hints of words of kHz, MHz or GHz in the field

        if "KHZ" in item1.upper():
            self.dlg.cmbFrequencySteps1.setCurrentIndex(1)
        elif "MHZ" in item1.upper():
            self.dlg.cmbFrequencySteps1.setCurrentIndex(2)
        elif "GHZ" in item1.upper():
            self.dlg.cmbFrequencySteps1.setCurrentIndex(3)

    def evt_cmbItem2_Changed(self,item2):
        ###########################################################################
        #
        #   This function handles the changes in the main layer.  searches
        #   for any likely hint of the units that the field is in
        #
        #   The purpose of the function is to switch between the units for the main layer dialog
        #   automatically if there are any hints of words of kHz, MHz or GHz in the field

        if "KHZ" in item2.upper():
            self.dlg.cmbFrequencySteps2.setCurrentIndex(1)
        elif "MHZ" in item2.upper():
            self.dlg.cmbFrequencySteps2.setCurrentIndex(2)
        elif "GHZ" in item2.upper():
            self.dlg.cmbFrequencySteps2.setCurrentIndex(3)

    def evt_rbSearch_Changed(self,rbSelected):
        ###########################################################################
        #
        #   This function controls showing and hiding the various panels when toggling the radio buttons
        #

        if self.dlg.rbWithinLayer.isChecked() == True and rbSelected == True:
            self.dlg.frmWithinLayer.show()
            self.dlg.frmAgainstRange.hide()
            self.dlg.frmAgainstLayer.hide()
            self.dlg.frmFrequencyGaps.hide()
            self.dlg.sbIgnoreDistance.setEnabled(True)

        if self.dlg.rbAgainstRange.isChecked() == True and rbSelected == True:
            self.dlg.frmWithinLayer.hide()
            self.dlg.frmAgainstRange.show()
            self.dlg.frmAgainstLayer.hide()
            self.dlg.frmFrequencyGaps.hide()
            self.dlg.sbIgnoreDistance.setEnabled(True)

        if self.dlg.rbAgainstLayer.isChecked() == True and rbSelected == True:
            self.dlg.frmWithinLayer.hide()
            self.dlg.frmAgainstRange.hide()
            self.dlg.frmAgainstLayer.show()
            self.dlg.frmFrequencyGaps.hide()
            self.dlg.sbIgnoreDistance.setEnabled(True)

        if self.dlg.rbForFrequency.isChecked() == True and rbSelected == True:
            self.dlg.frmWithinLayer.hide()
            self.dlg.frmAgainstRange.show()
            self.dlg.frmAgainstLayer.hide()
            self.dlg.frmFrequencyGaps.hide()
            self.dlg.sbIgnoreDistance.setEnabled(False)
            # Copy parameters from Frequency Gaps to relieve the need to retype the frequency for Search For Frequency
            sFreqBase = self.dlg.cmbFrequencySteps4.currentText().lstrip("in ")
            sBandWidth = self.dlg.dsbBandwidth.suffix().strip()
            self.dlg.cmbFrequencySteps3.setCurrentText("in {}".format(sFreqBase))
            self.dlg.dsbStartFreq.setValue(self.dlg.dsbGapsStartFreq.value())
            self.dlg.dsbStopFreq.setValue(self.dlg.dsbGapsStopFreq.value())
            vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(sFreqBase,sBandWidth,0,0,self.dlg.dsbGapsStartFreq.value(),self.dlg.dsbGapsStopFreq.value())
            self.dlg.dsbCenterFreq.setValue(vA)
            self.dlg.dsbBandwidth.setValue(vB)
            self.dlg.dsbStartFreq.setSuffix(" {}".format(self.dlg.cmbFrequencySteps4.currentText().lstrip("in ")))
            self.dlg.dsbStopFreq.setSuffix(" {}".format(self.dlg.cmbFrequencySteps4.currentText().lstrip("in ")))

        if self.dlg.rbFrequencyGaps.isChecked() == True and rbSelected == True:
            self.dlg.frmWithinLayer.hide()
            self.dlg.frmAgainstRange.hide()
            self.dlg.frmAgainstLayer.hide()
            self.dlg.frmFrequencyGaps.show()
            self.dlg.sbIgnoreDistance.setEnabled(True)
            # Copy parameters from Search For Frequency to relieve the need to retype the frequency for Frequency Gaps
            self.dlg.dsbGapsStartFreq.setValue(self.dlg.dsbStartFreq.value())
            self.dlg.dsbGapsStopFreq.setValue(self.dlg.dsbStopFreq.value())
            self.dlg.cmbFrequencySteps4.setCurrentText("in {}".format(self.dlg.cmbFrequencySteps3.currentText().lstrip("in ")))
            self.dlg.dsbGapsStartFreq.setSuffix(" {}".format(self.dlg.cmbFrequencySteps3.currentText().lstrip("in ")))
            self.dlg.dsbGapsStopFreq.setSuffix(" {}".format(self.dlg.cmbFrequencySteps3.currentText().lstrip("in ")))

    def evt_SearchAgainstRange_Changed(self,item3):
        ###########################################################################
        #
        #   This function handles the changes in the unit Hz, kHz, MHz or GHz of the search
        #   within a range dialog, as well as changes to the dial button
        #
        #   The purpose of the function is to set the appropriate units for the search within
        #   a range dialog display

        lstFreqStep = ['Hz','kHz','MHz','GHz']
        temp = str(item3)

        # If the main data is using Start-Stop frequency the units of the stop frequency is normally the same as the start frequency
        if self.dlg.cmbMode.currentText() == "Start-Stop Frequency":
            self.dlg.cmbFrequencySteps2.setCurrentText(self.dlg.cmbFrequencySteps1.currentText())

        # Check if the changes are on the dial button of the search within a range dialog
        if temp.isnumeric():
            # Retrieve the unit in Hz, kHz, MHz or GHz from the dial position
           self.strFreqStep = lstFreqStep[item3]

           # Check the current units and convert the bandwidth to a common base in MHz first
           tempOldSuffix = self.dlg.dsbBandwidth.suffix().strip()
           tempBWValue = self.dlg.dsbBandwidth.value()
           if tempOldSuffix == "Hz":
               fBandwidthMultiplier = 1 / 1000000
           elif tempOldSuffix == "kHz":
               fBandwidthMultiplier = 1 / 1000
           elif tempOldSuffix == "MHz":
               fBandwidthMultiplier = 1
           elif tempOldSuffix == "GHz":
               fBandwidthMultiplier = 1000
           tempBWValue = tempBWValue * fBandwidthMultiplier
           self.dlg.dsbBandwidth.setSuffix(" {}".format(self.strFreqStep))

           # Check the new units and convert the bandwidth from MHz to the unit
           if self.strFreqStep == "Hz":
               fBandwidthMultiplier = 1 * 1000000
           elif self.strFreqStep == "kHz":
               fBandwidthMultiplier = 1 * 1000
           elif self.strFreqStep == "MHz":
               fBandwidthMultiplier = 1
           elif self.strFreqStep == "GHz":
               fBandwidthMultiplier = 1 / 1000
           tempBWValue = tempBWValue * fBandwidthMultiplier
           self.dlg.dsbBandwidth.setValue(tempBWValue)

           vFreqSuffix = self.dlg.dsbCenterFreq.suffix().strip()
           vBandwidthSuffix = self.dlg.dsbBandwidth.suffix().strip()
           vA,vB,vC,vD = self.calcMissingFrequencyBandwidthVariables(vFreqSuffix,vBandwidthSuffix,self.dlg.dsbCenterFreq.value(),self.dlg.dsbBandwidth.value(),0,0)
           self.dlg.dsbStartFreq.setValue(vC)
           self.dlg.dsbStopFreq.setValue(vD)
           del vA, vB, vC, vD, vFreqSuffix, vBandwidthSuffix, tempOldSuffix,tempBWValue,fBandwidthMultiplier

        # Check if the changes are on the units Hz kHz, MHz and GHz.  If yes switch the center frequency, start and stop frequency to this unit
        elif "in " in temp:
            self.strFreqStep = temp.lstrip("in ")
            self.dlg.dsbCenterFreq.setSuffix(" {}".format(self.strFreqStep))
            self.dlg.dsbStartFreq.setSuffix(" {}".format(self.strFreqStep))
            self.dlg.dsbStopFreq.setSuffix(" {}".format(self.strFreqStep))
            vFreqSuffix = self.dlg.dsbCenterFreq.suffix().strip()
            vBandwidthSuffix = self.dlg.dsbBandwidth.suffix().strip()
            vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(vFreqSuffix, vBandwidthSuffix, self.dlg.dsbCenterFreq.value(),self.dlg.dsbBandwidth.value(), 0, 0)
            self.dlg.dsbStartFreq.setValue(vC)
            self.dlg.dsbStopFreq.setValue(vD)
            del vA, vB, vC, vD, vFreqSuffix, vBandwidthSuffix

        # If the changes comes from switching the units Hz, kHz, MHz, GHz of the main data when it is toggled
        # the search within range will likely be the same units as the main data, thus this section changes the
        # units of the search within range dialog to match the units of the main search data
        else:
            self.strFreqStep = item3
            self.dlg.cmbFrequencySteps3.setCurrentText("in {}".format(self.dlg.cmbFrequencySteps1.currentText()))

            if self.dlg.cmbMode.currentText() == "Center Frequency":
                self.dlg.dialFreqStep.setValue(lstFreqStep.index(self.dlg.cmbFrequencySteps2.currentText()))
                self.dlg.dsbBandwidth.setSuffix(" {}".format(self.dlg.cmbFrequencySteps2.currentText()))

            self.dlg.dsbCenterFreq.setSuffix(" {}".format(self.dlg.cmbFrequencySteps1.currentText()))
            self.dlg.dsbStartFreq.setSuffix(" {}".format(self.dlg.cmbFrequencySteps1.currentText()))
            self.dlg.dsbStopFreq.setSuffix(" {}".format(self.dlg.cmbFrequencySteps1.currentText()))

        del temp, lstFreqStep

    def evt_CenterFreq_Changed(self):
        ###########################################################################
        #
        #   This function handles the changes in the center frequency for the search within a range
        #
        #   The purpose of the function is to calculate the start and stop frequency base on
        #   base on the current bandwidth and new center frequency and set it in the appropriate units that
        #   it is currently using.

        vFreqSuffix = self.dlg.dsbCenterFreq.suffix().strip()
        vBandwidthSuffix = self.dlg.dsbBandwidth.suffix().strip()
        vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(vFreqSuffix, vBandwidthSuffix, self.dlg.dsbCenterFreq.value(),self.dlg.dsbBandwidth.value(), 0, 0)
        self.dlg.dsbStartFreq.setValue(vC)
        self.dlg.dsbStopFreq.setValue(vD)

    def evt_Bandwidth_Changed(self):
        ###########################################################################
        #
        #   This function handles the changes in the bandwidth for the search within a range
        #
        #   The purpose of the function is to calculate the start and stop frequency base on
        #   base on the new bandwidth and current center frequency and set it in the appropriate
        #   units that it is currently using.

        vFreqSuffix = self.dlg.dsbCenterFreq.suffix().strip()
        vBandwidthSuffix = self.dlg.dsbBandwidth.suffix().strip()
        vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(vFreqSuffix, vBandwidthSuffix, self.dlg.dsbCenterFreq.value(), self.dlg.dsbBandwidth.value(), 0, 0)
        self.dlg.dsbStartFreq.setValue(vC)
        self.dlg.dsbStopFreq.setValue(vD)

        #   This section sets the spin button of the center frequency to be in steps of the bandwidth just entered
        #   Check the current units used by the bandwidth and convert it to a common base in MHz first
        tempBWSuffix = self.dlg.dsbBandwidth.suffix().strip()
        tempBWValue = self.dlg.dsbBandwidth.value()
        if tempBWSuffix == "Hz":
            fBandwidthMultiplier = 1 / 1000000
        elif tempBWSuffix == "kHz":
            fBandwidthMultiplier = 1 / 1000
        elif tempBWSuffix == "MHz":
            fBandwidthMultiplier = 1
        elif tempBWSuffix == "GHz":
            fBandwidthMultiplier = 1000
        tempBWValue = tempBWValue * fBandwidthMultiplier

        # Check the units used by the center frequency and convert the bandwidth (from MHz) to the same units used by center frequency
        tempCenterFreqSuffix = self.dlg.dsbCenterFreq.suffix().strip()
        if tempCenterFreqSuffix == "Hz":
            fBandwidthMultiplier = 1 * 1000000
        elif tempCenterFreqSuffix == "kHz":
            fBandwidthMultiplier = 1 * 1000
        elif tempCenterFreqSuffix == "MHz":
            fBandwidthMultiplier = 1
        elif tempCenterFreqSuffix == "GHz":
            fBandwidthMultiplier = 1 / 1000
        tempBWValue = tempBWValue * fBandwidthMultiplier

        #   Sets the step size of the spin button for center frequency to be equal to the bandwidth in the units used by the center frequency
        self.dlg.dsbCenterFreq.setSingleStep(tempBWValue)
        del tempBWSuffix, tempBWValue, fBandwidthMultiplier, tempCenterFreqSuffix

    def evt_StartFreq_Changed(self):
        ###########################################################################
        #
        #   This function handles the changes in the stat frequency for the search within a range
        #
        #   The purpose of the function is to calculate the center frequency and bandwidth
        #   base on the new start and current stop frequency and set it in the appropriate units that
        #   it is currently using.

        vFreqSuffix = self.dlg.dsbCenterFreq.suffix().strip()
        vBandwidthSuffix = self.dlg.dsbBandwidth.suffix().strip()
        vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(vFreqSuffix, vBandwidthSuffix, 0, 0, self.dlg.dsbStartFreq.value(), self.dlg.dsbStopFreq.value())
        self.dlg.dsbCenterFreq.setValue(vA)
        self.dlg.dsbBandwidth.setValue(vB)

    def evt_StopFreq_Changed(self):
        ###########################################################################
        #
        #   This function handles the changes in the stop frequency for the search within a range
        #
        #   The purpose of the function is to calculate the center frequency and bandwidth
        #   base on the current start and new stop frequency and set it in the appropriate units that
        #   it is currently using.

        vFreqSuffix = self.dlg.dsbCenterFreq.suffix().strip()
        vBandwidthSuffix = self.dlg.dsbBandwidth.suffix().strip()
        vA, vB, vC, vD = self.calcMissingFrequencyBandwidthVariables(vFreqSuffix, vBandwidthSuffix, 0, 0, self.dlg.dsbStartFreq.value(), self.dlg.dsbStopFreq.value())
        self.dlg.dsbCenterFreq.setValue(vA)
        self.dlg.dsbBandwidth.setValue(vB)

    def calcMissingFrequencyBandwidthVariables(self,sFreqBase,sBandwidthBase,fCenterFreq,fBandwidth,fStartFreq,fStopFreq):
        ###########################################################################
        #
        #   This function takes as inputs the following parameters:
        #       sFreqBase       =   The units for the frequency in Hz, kHz, MHz or GHz
        #       sBandwidthBase  =   The units for the bandwidth in Hz, kHz, MHz or GHz
        #       fCenterFreq     =   The value of the center frequency
        #       fBandwidth      =   The value of the bandwidth
        #       fStartFreq      =   The value of the starting frequency
        #       fStopFreq       =   The value of the stop frequency
        #
        #       The purpose of the function is to convert center frequency and bandwidth
        #       to its appropriate start and stop frequency taking note of the units for frequency
        #       and bandwidth.  It also calculates center frequency and bandwidth from the start and
        #       stop frequency taking care to return in the appropriate units for the frequency
        #       and bandwidth that is used.

        #   initialise fFreqMultiplier as a float variable
        fFreqMultiplier = 0.01

        #   This function converts the frequency portion to the equivalent value in MHz
        if sFreqBase == "Hz":
            fFreqMultiplier = 1/1000000
        elif sFreqBase == "kHz":
            fFreqMultiplier = 1/1000
        elif sFreqBase == "MHz":
            fFreqMultiplier = 1
        elif sFreqBase == "GHz":
            fFreqMultiplier = 1000

        #   initialise fBandwidthMultiplier as a float variable
        fBandwidthMultiplier = 0.01

        #   This function converts the bandwidth portion to the equivalent value in MHz
        if sBandwidthBase == "Hz":
            fBandwidthMultiplier = 1 / 1000000
        elif sBandwidthBase == "kHz":
            fBandwidthMultiplier = 1 / 1000
        elif sBandwidthBase == "MHz":
            fBandwidthMultiplier = 1
        elif sBandwidthBase == "GHz":
            fBandwidthMultiplier = 1000

        #   If only start and stop frequency is provided find center frequency and bandwidth
        if fCenterFreq == 0 and fStartFreq != 0:
            rfStartFreq = fStartFreq * fFreqMultiplier
            rfStopFreq = fStopFreq * fFreqMultiplier
            rfCenterFreq = rfStartFreq + ((rfStopFreq-rfStartFreq)/2)
            rfBandwidth = rfStopFreq - rfStartFreq
            #   Converts back frequency and bandwidth to the value of the requested unit from its current base in MHz
            return rfCenterFreq * (1 / fFreqMultiplier), rfBandwidth * (1 / fBandwidthMultiplier), rfStartFreq * (1 / fFreqMultiplier), rfStopFreq * (1 / fFreqMultiplier)
        #   If only center frequency and bandwidth is provided find start and stop frequency
        elif fStartFreq == 0 and fCenterFreq != 0:
            rfCenterFreq = fCenterFreq * fFreqMultiplier
            rfBandwidth = fBandwidth * fBandwidthMultiplier
            rfStartFreq = rfCenterFreq - (rfBandwidth/2)
            rfStopFreq = rfCenterFreq + (rfBandwidth/2)
            #   Converts back frequency and bandwidth to the value of the requested unit from its current base in MHz
            return rfCenterFreq * (1 / fFreqMultiplier), rfBandwidth * (1 / fBandwidthMultiplier), rfStartFreq * (1 / fFreqMultiplier), rfStopFreq * (1 / fFreqMultiplier)

        #   If 3 or more values is missing in (center frequency, bandwidth, start frequency or stop frequency) it is not possible to derive
        #   the rest of the variables, thus return all zeros.
        return 0,0,0,0

    def evt_cmbAgainstLayerMode_Changed(self,mode):
        ###########################################################################
        #
        #   This function handles the changes in the mode between use of Center Frequency basis
        #   or Start-Stop frequency for the search against layer panel
        #
        #   The purpose of the function is to switch between the values used for the source data
        #   it can only be either center frequency and bandwidth or start and stop frequency.
        #   The label will change base on the mode selected

        if mode == "Center Frequency":
            self.dlg.lblAgainstLayerItem1.setText("Center Frequency:")
            self.dlg.lblAgainstLayerItem2.setText("Bandwidth:")
        else:
            self.dlg.lblAgainstLayerItem1.setText("Start Frequency:")
            self.dlg.lblAgainstLayerItem2.setText("Stop Frequency:")
            # If using Start-Stop frequency the seacrh within range dialog is most likely using the same unit of Hz, kHz, MHz or GHz
            # as the source layer
            self.dlg.cmbAgainstLayerFrequencySteps2.setCurrentText(self.dlg.cmbAgainstLayerFrequencySteps1.currentText())

    def evt_cmbAgainstLayerItem1_Changed(self,item1):
        ###########################################################################
        #
        #   This function handles the changes in the search against another layer.  searches
        #   for any likely hint of the units that the field is in
        #
        #   The purpose of the function is to switch between the units for the against layer dialog
        #   automatically if there are any hints of words of kHz, MHz or GHz in the field

        if "KHZ" in item1.upper():
            self.dlg.cmbAgainstLayerFrequencySteps1.setCurrentIndex(1)
        elif "MHZ" in item1.upper():
            self.dlg.cmbAgainstLayerFrequencySteps1.setCurrentIndex(2)
        elif "GHZ" in item1.upper():
            self.dlg.cmbAgainstLayerFrequencySteps1.setCurrentIndex(3)
        # if using start-stop frequency mode, the stop frequency is likely the same unit as the start frequency
        if self.dlg.cmbAgainstLayerMode.currentText() == "Start-Stop Frequency":
            self.dlg.cmbAgainstLayerFrequencySteps2.setCurrentText(self.dlg.cmbAgainstLayerFrequencySteps1.currentText())

    def evt_cmbAgainstLayerItem2_Changed(self,item2):
        ###########################################################################
        #
        #   This function handles the changes in the search against another layer.  searches
        #   for any likely hint of the units that the field is in
        #
        #   The purpose of the function is to switch between the units for the against layer dialog
        #   automatically if there are any hints of words of kHz, MHz or GHz in the field

        if "KHZ" in item2.upper():
            self.dlg.cmbAgainstLayerFrequencySteps2.setCurrentIndex(1)
        elif "MHZ" in item2.upper():
            self.dlg.cmbAgainstLayerFrequencySteps2.setCurrentIndex(2)
        elif "GHZ" in item2.upper():
            self.dlg.cmbAgainstLayerFrequencySteps2.setCurrentIndex(3)

    def evt_SearchFrequencyGaps_Changed(self,item4):
        ###########################################################################
        #
        #   This function handles changing the units in the search for frequency gaps
        #   dialog and help to switch the suffix units to what the user selected
        #
        strGapsFreqStep = item4.lstrip("in ")
        self.dlg.dsbGapsStartFreq.setSuffix(" {}".format(strGapsFreqStep))
        self.dlg.dsbGapsStopFreq.setSuffix(" {}".format(strGapsFreqStep))