# -*- coding: utf-8 -*-
"""
/***************************************************************************
 flowTrace
                                 A QGIS plugin
 This plugin will select all upstream lines
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-05-14
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Ed B
        email                : boesiii@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
# from PyQt5.QtWidgets import QAction
from PyQt5.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .flowtrace_dialog import flowTraceDialog
from qgis.core import QgsRectangle, QgsDistanceArea, QgsUnitTypes, QgsProject, QgsSpatialIndex

import os
import math

class flowTrace:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'flowTrace_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Flow Trace')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('flowTrace', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/flowtrace/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Flow Trace'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Flow Trace'),
                action)
            self.iface.removeToolBarIcon(action)

    # get geometry from different geometry types
    def get_geometry(self, fg):
        # test for MultiLineString, MultiLineStringZ, MultiLineStringM, MultiLineStringZM, MultiLineString25D
        if fg.wkbType() in [5, 1005, 2005, 3005, -2147483643]:
            nodes = fg.asMultiPolyline()[0]
            return nodes
                        
        # test for LineString, LineStringZ, LineStringM, LineStringZM, LineString25D
        if fg.wkbType() in [2, 1002, 2002, 3002, -2147483646]:
            nodes = fg.asPolyline()
            return nodes
        
    
    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = flowTraceDialog()

        # get current selected layer
        clayer = self.iface.mapCanvas().currentLayer()

        # make sure that the selected layer is a QgsVectorLayer of
        # QgsWkbTypes.GeometryType.LineGeometry type before opening the dialog
        if clayer is None or clayer.type() != 0 or clayer.geometryType() != 1:
            return

        # set layer name in dialog
        self.dlg.labelLayer.setText(clayer.name())
        # set number of selected features in dialog
        self.dlg.labelNumFeatures.setText(str(len(clayer.selectedFeatures())))
        # print(self.iface.mapCanvas().mapUnits())
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # get direct from dialog
        direction = self.dlg.downstream_radio_button.isChecked()
        
        #setup distance
        distance = QgsDistanceArea()
        # the unit of measure will be set to the same as the layer
        # maybe it would be better to set it to the map CRS
        distance.setSourceCrs(clayer.sourceCrs(), QgsProject.instance().transformContext())
        
        if result:
            # get crs for tolerance setting
            crs = self.iface.activeLayer().crs().authid()
            # print (crs)
            if crs == 'EPSG:4269':
                tolerance = .0001
            else:
                tolerance = self.dlg.SpinBoxTolerance.value()

            #index and create sets from layer
            index = QgsSpatialIndex(clayer)
            selection_set = set(clayer.selectedFeatureIds())
            final_set = selection_set.copy()
            dict_features = {feature.id():feature for feature in clayer.getFeatures()}
            

            #loop thru selection set
            while selection_set:
                # get upstream/downstream node of next feature
                feature = dict_features[selection_set.pop()]
                nodes = self.get_geometry(feature.geometry())
                upstream_coord = nodes[0 - direction]
                
                
                # select all features around selected node 
                # using a bounding box
                upstream_coord_x = upstream_coord.x()
                upstream_coord_y = upstream_coord.y()
                
                rectangle = QgsRectangle(upstream_coord_x - tolerance, 
                                         upstream_coord_y - tolerance, 
                                         upstream_coord_x + tolerance, 
                                         upstream_coord_y + tolerance)
                
                ls_fids = index.intersects(rectangle)
                
                
                #iterate thru intersected features
                for fid in ls_fids:
                    if fid not in final_set:
                        # get downstream/upstream coordinates
                        feature = dict_features[fid]
                        nodes = self.get_geometry(feature.geometry())
                        downstream_coord = nodes[direction - 1]
                        
                        #get distance between downstream and upstream nodes
                        dist = math.sqrt((downstream_coord.x() - upstream_coord_x)**2
                                         + (downstream_coord.y() - upstream_coord_y)**2)
                        
                        if dist <= tolerance:
                            # if within tolerance, adds feature to selection and final set
                            # set values being unique, a duplicate won't be created if
                            # already present in the selection set
                            final_set.add(fid)
                            selection_set.add(fid)

            #calculate total length
            list_length = [distance.measureLength(dict_features[fid].geometry()) for fid in final_set]
            total_length = sum(list_length)
            
            
            #select features using final_set
            clayer.selectByIds(list(final_set))
            self.iface.mapCanvas().refresh()
        
            #add message bar about number of features selected and length
            message = self.tr("{} features selected totalling {} {} in length."
                              .format(len(final_set),
                                      round(total_length,2),
                                      QgsUnitTypes.toString(distance.lengthUnits())))
            self.iface.messageBar().pushMessage("Flow Trace Completed", message, 0, 10)