# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FLO2DMapCrafter
                                 A QGIS plugin
 This plugin creates maps from FLO-2D output files.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-09-21
        git sha              : $Format:%H$
        copyright            : (C) 2023 by FLO-2D
        email                : contact@flo-2d.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import numpy as np
from PyQt5.QtGui import QColor
from PyQt5.QtWidgets import QMessageBox
from PyQt5.QtXml import QDomDocument
from osgeo import gdal
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis._core import (
    QgsCoordinateReferenceSystem,
    QgsRasterBandStats,
    QgsColorRampShader,
    QgsRasterShader,
    QgsSingleBandPseudoColorRenderer,
    QgsRasterLayer,
    QgsProject,
    QgsVectorLayer,
    QgsUnitTypes,
    QgsPrintLayout,
    QgsReadWriteContext,
    QgsMessageLog, QgsApplication, QgsProcessingUtils,
)

from .mapping.flood import FloodMaps
from .mapping.hazard import HazardMaps
from .mapping.mudflow import MudflowMaps
from .mapping.scripts import set_icon
from .mapping.sediment import SedimentMaps
from .mapping.twophase import TwophaseMaps
from .resources import *
from .flo2d_mapcrafter_dialog import FLO2DMapCrafterDialog
import os.path
import processing

class FLO2DMapCrafter:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.dlg = FLO2DMapCrafterDialog()
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "FLO2DMapCrafter_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&FLO-2D MapCrafter ")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Adjust CRS
        self.crs = QgsCoordinateReferenceSystem(QgsProject.instance().crs().authid())

        # Select export folder
        self.dlg.flo2d_out_folder.fileChanged.connect(self.check_files)

        # Run Button
        self.dlg.runButton.clicked.connect(self.run_map_creator)

        # Cancel Button
        self.dlg.cancelButton.clicked.connect(self.closeDialog)

        # Check all available maps
        self.dlg.check_cw_cb.stateChanged.connect(self.check_cw)
        self.dlg.check_mf_cb.stateChanged.connect(self.check_mf)
        self.dlg.check_tp_cb.stateChanged.connect(self.check_tp)
        self.dlg.check_sd_cb.stateChanged.connect(self.check_sd)
        self.dlg.check_hm_cb.stateChanged.connect(self.check_hm)

        self.dlg.runButton_2.clicked.connect(self.run_open_layout)

        self.dlg.tab0.setEnabled(False)
        self.dlg.tab1.setEnabled(False)
        self.dlg.tab2.setEnabled(False)
        self.dlg.tab3.setEnabled(False)
        self.dlg.tab4.setEnabled(False)
        self.dlg.tab5.setEnabled(False)

        self.dlg.cg_cw_btn.clicked.connect(self.collapse_all_groups)
        self.dlg.eg_cw_btn.clicked.connect(self.expand_all_groups)
        self.dlg.cg_sd_btn.clicked.connect(self.collapse_all_groups)
        self.dlg.eg_sd_btn.clicked.connect(self.expand_all_groups)
        self.dlg.cg_md_btn.clicked.connect(self.collapse_all_groups)
        self.dlg.eg_md_btn.clicked.connect(self.expand_all_groups)
        self.dlg.cg_tp_btn.clicked.connect(self.collapse_all_groups)
        self.dlg.eg_tp_btn.clicked.connect(self.expand_all_groups)
        self.dlg.cg_hm_btn.clicked.connect(self.collapse_all_groups)
        self.dlg.eg_hm_btn.clicked.connect(self.expand_all_groups)

        set_icon(self.dlg.cg_cw_btn, "collapse_groups.svg")
        set_icon(self.dlg.eg_cw_btn, "expand_groups.svg")
        set_icon(self.dlg.cg_sd_btn, "collapse_groups.svg")
        set_icon(self.dlg.eg_sd_btn, "expand_groups.svg")
        set_icon(self.dlg.cg_md_btn, "collapse_groups.svg")
        set_icon(self.dlg.eg_md_btn, "expand_groups.svg")
        set_icon(self.dlg.cg_tp_btn, "collapse_groups.svg")
        set_icon(self.dlg.eg_tp_btn, "expand_groups.svg")
        set_icon(self.dlg.cg_hm_btn, "collapse_groups.svg")
        set_icon(self.dlg.eg_hm_btn, "expand_groups.svg")

        # DEBUG Map layouts
        # self.dlg.map_title_le.setText("Mudflow")
        # self.dlg.map_description.setPlainText(
        #     "This map is a visual representation of the areas that are likely to be submerged or covered by floodwaters during a specific flood event."
        # )

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("FLO2DMapCrafter", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/flo2d_mapcrafter/icon.png"
        self.add_action(
            icon_path,
            text=self.tr("FLO-2D MapCrafter"),
            callback=self.open,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&FLO-2D MapCrafter "), action)
            self.iface.removeToolBarIcon(action)

    # Opening the dialog
    def open(self):
        """Shows the dialog"""

        # Adjust the CRS
        self.crs = QgsCoordinateReferenceSystem(QgsProject.instance().crs().authid())
        self.dlg.crsselector.setCrs(self.crs)

        # Set the grid layer as extent, if it exists
        layer = QgsProject.instance().mapLayersByName("Grid")
        if layer:
            # Add the layer to the QgsMapLayerComboBox
            self.dlg.layer_extent_cb.setLayer(layer[0])

        self.dlg.show()
        self.dlg.activateWindow()

    def closeDialog(self):
        """Closes the Dialog"""

        checkboxes = self.dlg.tabs.findChildren(QtWidgets.QCheckBox)
        for checkBox in checkboxes:
            checkBox.setChecked(False)

        self.collapse_all_groups()

        self.dlg.close()

    def check_files(self):
        """Function to check the type of files present on the simulation"""

        # check if simulation was run
        files_in_directory = os.listdir(self.dlg.flo2d_out_folder.filePath())

        # In future version, calculate the Cell size from the DEPTH.OUT file
        if "DEPTH.OUT" in files_in_directory and "CONT.DAT" in files_in_directory:
            self.dlg.runButton.setEnabled(True)
            self.dlg.label_2.setEnabled(True)
            self.dlg.mapper_out_folder.setEnabled(True)
            self.dlg.tab4.setEnabled(True)
        else:
            msg_box = QMessageBox()
            msg_box.setIcon(QMessageBox.Warning)
            msg_box.setWindowTitle("Warning")
            msg_box.setText("No CONT.DAT and *.OUT files were found in this directory!")
            msg_box.exec_()
            return

        output_directory = self.dlg.flo2d_out_folder.filePath()

        with open(output_directory + r"\CONT.DAT", "r") as file:
            lines = file.readlines()
            elements = lines[2].split()
            units_switch = lines[0].split()[3]
            mud_switch = elements[3]
            sed_switch = elements[4]
            file.close()

        # Flood simulation
        if mud_switch == "0" and sed_switch == "0":
            self.dlg.tab0.setEnabled(False)
            self.dlg.tab1.setEnabled(True)
            self.dlg.tab2.setEnabled(False)
            self.dlg.tab3.setEnabled(False)
            self.dlg.tabs.setCurrentIndex(0)

            flood_maps = FloodMaps(units_switch)
            flood_files_dict = flood_maps.check_flood_files(output_directory)

            flood_rbs = {
                r"TOPO.DAT": self.dlg.ge_cw_cb,
                r"DEPTH.OUT": self.dlg.md_cw_cb,
                r"VELFP.OUT": self.dlg.mv_cw_cb,
                r"VELDIREC.OUT": self.dlg.mvv_cw_cb,
                r"MAXWSELEV.OUT": self.dlg.mwse_cw_cb,
                r"FINALDEP.OUT": self.dlg.fd_cw_cb,
                r"FINALVEL.OUT": self.dlg.fv_cw_cb,
                r"FINALDIR.OUT": self.dlg.fvv_cw_cb,
                r"VEL_X_DEPTH.OUT": self.dlg.dv_cw_cb,
                r"TIMEONEFT.OUT": self.dlg.t1ft_cw_cb,
                r"TIMETWOFT.OUT": self.dlg.t2ft_cw_cb,
                r"TIMETOPEAK.OUT": self.dlg.tmax_cw_cb,
                r"DEPCH.OUT": self.dlg.cd_cw_cb,
                r"VELOC.OUT": self.dlg.cv_cw_cb,
                r"VELCHFINAL.OUT": self.dlg.fcv_cw_cb,
                r"DEPCHFINAL.OUT": self.dlg.fcd_cw_cb,
                r"LEVEEDEFIC.OUT": self.dlg.ld_cw_cb,
                r"SPECENERGY.OUT": self.dlg.se_cw_cb,
                r"STATICPRESS.OUT": self.dlg.sp_cw_cb,
                r"IMPACT.OUT": self.dlg.if_cw_cb,
            }

            for key, value in flood_files_dict.items():
                if value:
                    flood_rbs[key].setEnabled(True)
                else:
                    flood_rbs[key].setEnabled(False)

        # Sediment simulation
        if mud_switch == "0" and sed_switch == "1":
            self.dlg.tab1.setEnabled(False)
            self.dlg.tab2.setEnabled(False)
            self.dlg.tab0.setEnabled(True)
            self.dlg.tab3.setEnabled(False)
            self.dlg.tabs.setCurrentIndex(1)

            sediment_maps = SedimentMaps(units_switch)
            sediment_files_dict = sediment_maps.check_sediment_files(output_directory)

            sediment_rbs = {
                r"TOPO.DAT": self.dlg.ge_sd_cb,
                r"DEPTH.OUT": self.dlg.md_sd_cb,
                r"VELFP.OUT": self.dlg.mv_sd_cb,
                r"VELDIREC.OUT": self.dlg.mvv_sd_cb,
                r"MAXWSELEV.OUT": self.dlg.mwse_sd_cb,
                r"FINALDEP.OUT": self.dlg.fd_sd_cb,
                r"FINALVEL.OUT": self.dlg.fv_sd_cb,
                r"FINALDIR.OUT": self.dlg.fvv_sd_cb,
                r"VEL_X_DEPTH.OUT": self.dlg.dv_sd_cb,
                r"TIMEONEFT.OUT": self.dlg.t1ft_sd_cb,
                r"TIMETWOFT.OUT": self.dlg.t2ft_sd_cb,
                r"TIMETOPEAK.OUT": self.dlg.tmax_sd_cb,
                r"DEPCH.OUT": self.dlg.cd_sd_cb,
                r"VELOC.OUT": self.dlg.cv_sd_cb,
                r"VELCHFINAL.OUT": self.dlg.fcv_sd_cb,
                r"DEPCHFINAL.OUT": self.dlg.fcd_sd_cb,
                r"LEVEEDEFIC.OUT": self.dlg.ld_sd_cb,
                r"SPECENERGY.OUT": self.dlg.se_sd_cb,
                r"STATICPRESS.OUT": self.dlg.sp_sd_cb,
                r"SEDFP.OUT": [
                    self.dlg.mdep_sd_cb,
                    self.dlg.msco_sd_cb,
                    self.dlg.fbd_sd_cb
                                ],
                r"IMPACT.OUT": self.dlg.if_sd_cb,
            }

            for key, value in sediment_files_dict.items():
                if value:
                    if isinstance(sediment_rbs[key], list):
                        for cb in sediment_rbs[key]:
                            cb.setEnabled(True)
                    else:
                        sediment_rbs[key].setEnabled(True)
                else:
                    if isinstance(sediment_rbs[key], list):
                        for cb in sediment_rbs[key]:
                            cb.setEnabled(False)
                    else:
                        sediment_rbs[key].setEnabled(False)

        # Mudflow simulation
        if mud_switch == "1" and sed_switch == "0":
            self.dlg.tab0.setEnabled(False)
            self.dlg.tab1.setEnabled(False)
            self.dlg.tab2.setEnabled(True)
            self.dlg.tab3.setEnabled(False)
            self.dlg.tabs.setCurrentIndex(2)

            mudflow_maps = MudflowMaps(units_switch)
            mudflow_files_dict = mudflow_maps.check_mudflow_files(output_directory)

            mudflow_rbs = {
                r"TOPO.DAT": self.dlg.ge_mf_cb,
                r"DEPTH.OUT": self.dlg.md_mf_cb,
                r"VELFP.OUT": self.dlg.mv_mf_cb,
                r"VELDIREC.OUT": self.dlg.mvv_mf_cb,
                r"MAXWSELEV.OUT": self.dlg.mwse_mf_cb,
                r"FINALDEP.OUT": self.dlg.fd_mf_cb,
                r"FINALVEL.OUT": self.dlg.fv_mf_cb,
                r"FINALDIR.OUT": self.dlg.fvv_mf_cb,
                r"VEL_X_DEPTH.OUT": self.dlg.dv_mf_cb,
                r"TIMEONEFT.OUT": self.dlg.t1ft_mf_cb,
                r"TIMETWOFT.OUT": self.dlg.t2ft_mf_cb,
                r"TIMETOPEAK.OUT": self.dlg.tmax_mf_cb,
                r"DEPCH.OUT": self.dlg.cd_mf_cb,
                r"VELOC.OUT": self.dlg.cv_mf_cb,
                r"VELCHFINAL.OUT": self.dlg.fcv_mf_cb,
                r"DEPCHFINAL.OUT": self.dlg.fcd_mf_cb,
                r"LEVEEDEFIC.OUT": self.dlg.ld_mf_cb,
                r"SPECENERGY.OUT": self.dlg.se_mf_cb,
                r"STATICPRESS.OUT": self.dlg.sp_mf_cb,
                r"CVFPMAX.OUT": self.dlg.ms_mf_cb,
                r"FINALCVFP.OUT": self.dlg.fs_mf_cb,
                r"IMPACT.OUT": self.dlg.if_mf_cb,
            }

            for key, value in mudflow_files_dict.items():
                if value:
                    mudflow_rbs[key].setEnabled(True)
                else:
                    mudflow_rbs[key].setEnabled(False)

        # Two-phase simulation
        if mud_switch == "2" and sed_switch == "0":
            self.dlg.tab0.setEnabled(False)
            self.dlg.tab1.setEnabled(False)
            self.dlg.tab2.setEnabled(False)
            self.dlg.tab3.setEnabled(True)
            self.dlg.tabs.setCurrentIndex(3)

            twophase_maps = TwophaseMaps(units_switch)
            twophase_files_dict = twophase_maps.check_twophase_files(output_directory)

            twophase_rbs = {
                r"TOPO.DAT": self.dlg.ge_tp_cb,
                r"DEPTH.OUT": self.dlg.mfd_tp_cb,
                r"DEPFPMAX_MUD.OUT": self.dlg.mmd_tp_cb,
                r"DEPTHMAX_2PHASE_COMBINED.OUT": self.dlg.cmd_tp_cb,
                r"VELFP.OUT": self.dlg.mfv_tp_cb,
                r"VELFP_MUD.OUT": self.dlg.mmv_tp_cb,
                r"VELDIREC.OUT": self.dlg.mfvv_tp_cb,
                r"VELDIREC_MUD.OUT": self.dlg.mmvv_tp_cb,
                r"CVFPMAX.OUT": self.dlg.mfsc_tp_cb,
                r"CVFPMAX_MUD.OUT": self.dlg.mmsc_tp_cb,
                #r"FINALCVFP.OUT": self.dlg.ffsc_tp_cb,
                r"FINALCVFP_MUD.OUT": self.dlg.fmsc_tp_cb,
                #r"MAXWSELEV.OUT": self.dlg.mwse_mf_cb,
                r"FINALDEP.OUT": self.dlg.ffd_tp_cb,
                r"FINALDEP_MUD.OUT": self.dlg.fmd_tp_cb,
                r"FINALDIR.OUT": self.dlg.ffvv_tp_cb,
                r"FINALDIR_MUD.OUT": self.dlg.fmvv_tp_cb,
                r"FINALDEP_COMBO.OUT": self.dlg.fcd_tp_cb,
                r"FINALVEL.OUT": self.dlg.ffv_tp_cb,
                r"FINALVEL_MUD.OUT": self.dlg.fmv_tp_cb,
                r"VEL_X_DEPTH.OUT": self.dlg.dv_tp_cb,
                r"TIMEONEFT.OUT": self.dlg.t1ft_tp_cb,
                r"TIMETWOFT.OUT": self.dlg.t2ft_tp_cb,
                r"TIMETOPEAK.OUT": self.dlg.tmax_tp_cb,
                r"DEPCH.OUT": self.dlg.mchd_tp_cb,
                r"VELOC.OUT": self.dlg.mchv_tp_cb,
                r"VELCHFINAL.OUT": self.dlg.fchv_tp_cb,
                r"DEPCHFINAL.OUT": self.dlg.fchd_tp_cb,
                r"LEVEEDEFIC.OUT": self.dlg.ld_tp_cb,
                r"SPECENERGY.OUT": self.dlg.se_tp_cb,
                r"STATICPRESS.OUT": self.dlg.sp_tp_cb,
                r"IMPACT.OUT": self.dlg.if_tp_cb,
                r"SEDFP.OUT": [
                    self.dlg.ms_tp_cb,
                    self.dlg.md_tp_cb,
                    self.dlg.fdb_tp_cb
                ],
            }

            for key, value in twophase_files_dict.items():
                if value:
                    if isinstance(twophase_rbs[key], list):
                        for cb in twophase_rbs[key]:
                            cb.setEnabled(True)
                    else:
                        twophase_rbs[key].setEnabled(True)
                else:
                    if isinstance(twophase_rbs[key], list):
                        for cb in twophase_rbs[key]:
                            cb.setEnabled(False)
                    else:
                        twophase_rbs[key].setEnabled(False)


        # Hazard Maps
        self.dlg.tab5.setEnabled(True)
        hazard_maps = HazardMaps(units_switch)
        hazard_maps_dict = hazard_maps.check_hazard_files(output_directory)

        hazard_rbs = {
            "ARR": self.dlg.fh_australian_cb,
            "Austrian": self.dlg.fh_austrian_cb,
            "FLO-2D": self.dlg.flo_hm_cb,
            "Swiss": [self.dlg.fi_swiss_cb, self.dlg.di_swiss_cb],
            "UK": self.dlg.fh_uk_cb,
            "USBR": [
                self.dlg.usbrh_hm_cb,
                self.dlg.usbrm_hm_cb,
                self.dlg.usbrv_hm_cb,
                self.dlg.usbra_hm_cb,
                self.dlg.usbrc_hm_cb,
            ],
            "FEMA": self.dlg.fema_hm_cb,
        }

        for key, value in hazard_maps_dict.items():
            if value:
                if isinstance(hazard_rbs[key], list):
                    for cb in hazard_rbs[key]:
                        cb.setEnabled(True)
                else:
                    hazard_rbs[key].setEnabled(True)
            else:
                if isinstance(hazard_rbs[key], list):
                    for cb in hazard_rbs[key]:
                        cb.setEnabled(False)
                else:
                    hazard_rbs[key].setEnabled(False)

        # Add MapCrafter to the output folder
        map_output_dir = output_directory + r"\MapCrafter"
        self.dlg.mapper_out_folder.setFilePath(map_output_dir)
        if not os.path.exists(map_output_dir):
            os.makedirs(map_output_dir)

    def run_map_creator(self):
        """Run method that performs all the real work"""

        # input & output directories
        flo2d_results_dir = self.dlg.flo2d_out_folder.filePath()
        map_output_dir = self.dlg.mapper_out_folder.filePath()
        self.crs = self.dlg.crsselector.crs()
        project_id = self.dlg.project_id.text()

        if map_output_dir == "":
            map_output_dir = QgsProcessingUtils.tempFolder()

        if not self.check_checkboxes():
            return

        with open(flo2d_results_dir + r"\CONT.DAT", "r") as file:
            lines = file.readlines()
            units_switch = lines[0].split()[3]
            elements = lines[2].split()
            mud_switch = elements[3]
            sed_switch = elements[4]
            file.close()

        """
        GROUPS CREATION
        """

        root = QgsProject.instance().layerTreeRoot()

        mapping_group_name = "FLO-2D MapCrafter"
        if root.findGroup(mapping_group_name):
            mapping_group = root.findGroup(mapping_group_name)
        else:
            mapping_group = root.insertGroup(0, mapping_group_name)

        """        
        FLOOD MAPS        
        """

        if mud_switch == "0" and sed_switch == "0":
            flood_rbs = {
                r"TOPO.DAT": self.dlg.ge_cw_cb.isChecked(),
                r"DEPTH.OUT": self.dlg.md_cw_cb.isChecked(),
                r"VELFP.OUT": self.dlg.mv_cw_cb.isChecked(),
                r"VELDIREC.OUT": self.dlg.mvv_cw_cb.isChecked(),
                r"MAXWSELEV.OUT": self.dlg.mwse_cw_cb.isChecked(),
                r"FINALDEP.OUT": self.dlg.fd_cw_cb.isChecked(),
                r"FINALVEL.OUT": self.dlg.fv_cw_cb.isChecked(),
                r"FINALDIR.OUT": self.dlg.fvv_cw_cb.isChecked(),
                r"VEL_X_DEPTH.OUT": self.dlg.dv_cw_cb.isChecked(),
                r"TIMEONEFT.OUT": self.dlg.t1ft_cw_cb.isChecked(),
                r"TIMETWOFT.OUT": self.dlg.t2ft_cw_cb.isChecked(),
                r"TIMETOPEAK.OUT": self.dlg.tmax_cw_cb.isChecked(),
                r"DEPCH.OUT": self.dlg.cd_cw_cb.isChecked(),
                r"VELOC.OUT": self.dlg.cv_cw_cb.isChecked(),
                r"VELCHFINAL.OUT": self.dlg.fcv_cw_cb.isChecked(),
                r"DEPCHFINAL.OUT": self.dlg.fcd_cw_cb.isChecked(),
                r"LEVEEDEFIC.OUT": self.dlg.ld_cw_cb.isChecked(),
                r"SPECENERGY.OUT": self.dlg.se_cw_cb.isChecked(),
                r"STATICPRESS.OUT": self.dlg.sp_cw_cb.isChecked(),
                r"IMPACT.OUT": self.dlg.if_cw_cb.isChecked(),
            }

            flood_maps = FloodMaps(units_switch)
            flood_maps.create_maps(
                flood_rbs, flo2d_results_dir, map_output_dir, mapping_group, self.crs, project_id
            )

        """
        SEDIMENT MAPS
        """

        if mud_switch == "0" and sed_switch == "1":
            sediment_rbs = {
                r"TOPO.DAT": self.dlg.ge_sd_cb.isChecked(),
                r"DEPTH.OUT": self.dlg.md_sd_cb.isChecked(),
                r"VELFP.OUT": self.dlg.mv_sd_cb.isChecked(),
                r"VELDIREC.OUT": self.dlg.mvv_sd_cb.isChecked(),
                r"MAXWSELEV.OUT": self.dlg.mwse_sd_cb.isChecked(),
                r"FINALDEP.OUT": self.dlg.fd_sd_cb.isChecked(),
                r"FINALVEL.OUT": self.dlg.fv_sd_cb.isChecked(),
                r"FINALDIR.OUT": self.dlg.fvv_sd_cb.isChecked(),
                r"VEL_X_DEPTH.OUT": self.dlg.dv_sd_cb.isChecked(),
                r"TIMEONEFT.OUT": self.dlg.t1ft_sd_cb.isChecked(),
                r"TIMETWOFT.OUT": self.dlg.t2ft_sd_cb.isChecked(),
                r"TIMETOPEAK.OUT": self.dlg.tmax_sd_cb.isChecked(),
                r"DEPCH.OUT": self.dlg.cd_sd_cb.isChecked(),
                r"VELOC.OUT": self.dlg.cv_sd_cb.isChecked(),
                r"VELCHFINAL.OUT": self.dlg.fcv_sd_cb.isChecked(),
                r"DEPCHFINAL.OUT": self.dlg.fcd_sd_cb.isChecked(),
                r"LEVEEDEFIC.OUT": self.dlg.ld_sd_cb.isChecked(),
                r"SPECENERGY.OUT": self.dlg.se_sd_cb.isChecked(),
                r"STATICPRESS.OUT": self.dlg.sp_sd_cb.isChecked(),
                r"SEDFP.OUT": [
                    self.dlg.mdep_sd_cb.isChecked(),
                    self.dlg.msco_sd_cb.isChecked(),
                    self.dlg.fbd_sd_cb.isChecked()
                ],
                r"IMPACT.OUT": self.dlg.if_sd_cb.isChecked(),
            }

            sediment_maps = SedimentMaps(units_switch)
            sediment_maps.create_maps(
                sediment_rbs, flo2d_results_dir, map_output_dir, mapping_group, self.crs, project_id
            )

        """"
        MUDFLOW MAPS
        """

        if mud_switch == "1" and sed_switch == "0":
            mudflow_rbs = {
                r"TOPO.DAT": self.dlg.ge_mf_cb.isChecked(),
                r"DEPTH.OUT": self.dlg.md_mf_cb.isChecked(),
                r"VELFP.OUT": self.dlg.mv_mf_cb.isChecked(),
                r"VELDIREC.OUT": self.dlg.mvv_mf_cb.isChecked(),
                r"MAXWSELEV.OUT": self.dlg.mwse_mf_cb.isChecked(),
                r"FINALDEP.OUT": self.dlg.fd_mf_cb.isChecked(),
                r"FINALVEL.OUT": self.dlg.fv_mf_cb.isChecked(),
                r"FINALDIR.OUT": self.dlg.fvv_mf_cb.isChecked(),
                r"VEL_X_DEPTH.OUT": self.dlg.dv_mf_cb.isChecked(),
                r"TIMEONEFT.OUT": self.dlg.t1ft_mf_cb.isChecked(),
                r"TIMETWOFT.OUT": self.dlg.t2ft_mf_cb.isChecked(),
                r"TIMETOPEAK.OUT": self.dlg.tmax_mf_cb.isChecked(),
                r"DEPCH.OUT": self.dlg.cd_mf_cb.isChecked(),
                r"VELOC.OUT": self.dlg.cv_mf_cb.isChecked(),
                r"VELCHFINAL.OUT": self.dlg.fcv_mf_cb.isChecked(),
                r"DEPCHFINAL.OUT": self.dlg.fcd_mf_cb.isChecked(),
                r"LEVEEDEFIC.OUT": self.dlg.ld_mf_cb.isChecked(),
                r"SPECENERGY.OUT": self.dlg.se_mf_cb.isChecked(),
                r"STATICPRESS.OUT": self.dlg.sp_mf_cb.isChecked(),
                r"CVFPMAX.OUT": self.dlg.ms_mf_cb.isChecked(),
                r"FINALCVFP.OUT": self.dlg.fs_mf_cb.isChecked(),
                r"IMPACT.OUT": self.dlg.if_mf_cb.isChecked(),
            }

            mudflow_maps = MudflowMaps(units_switch)
            mudflow_maps.create_maps(
                mudflow_rbs, flo2d_results_dir, map_output_dir, mapping_group, self.crs, project_id
            )

        """"
        TWO-PHASE MAPS
        """

        if mud_switch == "2" and sed_switch == "0":
            twophase_rbs = {
                r"TOPO.DAT": self.dlg.ge_tp_cb.isChecked(),
                r"DEPTH.OUT": self.dlg.mfd_tp_cb.isChecked(),
                r"DEPFPMAX_MUD.OUT": self.dlg.mmd_tp_cb.isChecked(),
                r"DEPTHMAX_2PHASE_COMBINED.OUT": self.dlg.cmd_tp_cb.isChecked(),
                r"VELFP.OUT": self.dlg.mfv_tp_cb.isChecked(),
                r"VELFP_MUD.OUT": self.dlg.mmv_tp_cb.isChecked(),
                r"VELDIREC.OUT": self.dlg.mfvv_tp_cb.isChecked(),
                r"VELDIREC_MUD.OUT": self.dlg.mmvv_tp_cb.isChecked(),
                r"CVFPMAX.OUT": self.dlg.mfsc_tp_cb.isChecked(),
                r"CVFPMAX_MUD.OUT": self.dlg.mmsc_tp_cb.isChecked(),
                # r"FINALCVFP.OUT": self.dlg.ffsc_tp_cb,
                r"FINALCVFP_MUD.OUT": self.dlg.fmsc_tp_cb.isChecked(),
                # r"MAXWSELEV.OUT": self.dlg.mwse_mf_cb,
                r"FINALDEP.OUT": self.dlg.ffd_tp_cb.isChecked(),
                r"FINALDEP_MUD.OUT": self.dlg.fmd_tp_cb.isChecked(),
                r"FINALDIR.OUT": self.dlg.ffvv_tp_cb.isChecked(),
                r"FINALDIR_MUD.OUT": self.dlg.fmvv_tp_cb.isChecked(),
                r"FINALDEP_COMBO.OUT": self.dlg.fcd_tp_cb.isChecked(),
                r"FINALVEL.OUT": self.dlg.ffv_tp_cb.isChecked(),
                r"FINALVEL_MUD.OUT": self.dlg.fmv_tp_cb.isChecked(),
                r"VEL_X_DEPTH.OUT": self.dlg.dv_tp_cb.isChecked(),
                r"TIMEONEFT.OUT": self.dlg.t1ft_tp_cb.isChecked(),
                r"TIMETWOFT.OUT": self.dlg.t2ft_tp_cb.isChecked(),
                r"TIMETOPEAK.OUT": self.dlg.tmax_tp_cb.isChecked(),
                r"DEPCH.OUT": self.dlg.mchd_tp_cb.isChecked(),
                r"VELOC.OUT": self.dlg.mchv_tp_cb.isChecked(),
                r"VELCHFINAL.OUT": self.dlg.fchv_tp_cb.isChecked(),
                r"DEPCHFINAL.OUT": self.dlg.fchd_tp_cb.isChecked(),
                r"LEVEEDEFIC.OUT": self.dlg.ld_tp_cb.isChecked(),
                r"SPECENERGY.OUT": self.dlg.se_tp_cb.isChecked(),
                r"STATICPRESS.OUT": self.dlg.sp_tp_cb.isChecked(),
                r"IMPACT.OUT": self.dlg.if_tp_cb.isChecked(),
                r"SEDFP.OUT": [
                self.dlg.ms_tp_cb.isChecked(),
                self.dlg.md_tp_cb.isChecked(),
                self.dlg.fdb_tp_cb.isChecked()
                ],
            }

            twophase_maps = TwophaseMaps(units_switch)
            twophase_maps.create_maps(
                twophase_rbs, flo2d_results_dir, map_output_dir, mapping_group, self.crs, project_id
            )

        """
        HYDRODYNAMIC RISK MAPS
        """

        hazard_rbs = {
            "ARR": self.dlg.fh_australian_cb.isChecked(),
            "Austrian": self.dlg.fh_austrian_cb.isChecked(),
            "Swiss": [
                self.dlg.fi_swiss_cb.isChecked(),
                self.dlg.di_swiss_cb.isChecked()
            ],
            "UK": self.dlg.fh_uk_cb.isChecked(),
            "USBR": [
                self.dlg.usbrh_hm_cb.isChecked(),
                self.dlg.usbrm_hm_cb.isChecked(),
                self.dlg.usbrv_hm_cb.isChecked(),
                self.dlg.usbra_hm_cb.isChecked(),
                self.dlg.usbrc_hm_cb.isChecked(),
            ],
            "FEMA": self.dlg.fema_hm_cb.isChecked()
        }

        at_least_one_checked = any(
            value if not isinstance(value, list) else any(value) for value in hazard_rbs.values())

        if at_least_one_checked:

            hazard_maps = HazardMaps(units_switch)
            hazard_maps.create_maps(
                hazard_rbs, flo2d_results_dir, map_output_dir, mapping_group, self.crs, project_id
            )

        # remove empty groups
        groups = mapping_group.findGroups()
        for group in groups:
            subgroups = group.findGroups()
            for subgroup in subgroups:
                all_subgrup_layers = subgroup.findLayers()
                if len(all_subgrup_layers) == 0:
                    group.removeChildNode(subgroup)
            all_group_layers = group.findLayers()
            if len(all_group_layers) == 0:
                mapping_group.removeChildNode(group)

        msg_box = QMessageBox()
        msg_box.setIcon(QMessageBox.Information)
        msg_box.setWindowTitle("Mapping complete!")
        msg_box.setText("The selected maps were created and added to the Map Canvas.")
        msg_box.exec_()

        self.closeDialog()

    def set_raster_style(self, layer, style):
        """Define the raster styles"""
        colDic = {
            "white": "#ffffff",
            "lightblue": "#9ecae1",
            "blue": "#4292c6",
            "darkblue": "#08306b",
            "lightgreen": "#a1d99b",
            "green": "#41ab5d",
            "darkgreen": "#006d2c",
            "black": "#000000",
            "grey": "#808080",
            "red": "#FF0000",
            "yellow": "#FFFF00",
            "risk_red": "#FF0000",
            "risk_orange": "#FFC000",
            "risk_lightgreen": "#92D050",
            "risk_green": "#006600",
            "risk_lightblue": "#BDD6EE",
            "risk_blue": "#0033CC",
            "mud_lightbrown": "#be4d24",
            "mud_brown": "#752c12",
            "mud_darkbrown": "#2c0c00",
        }

        provider = layer.dataProvider()
        extent = layer.extent()

        script_directory = os.path.dirname(os.path.realpath(__file__))
        style_directory = script_directory + r"/raster_styles"

        # Hydrodynamic Risk
        if style == 2:
            layer.loadNamedStyle(style_directory + r"/hydro_risk.qml")
        elif style == 3:
            layer.loadNamedStyle(style_directory + r"/time.qml")
        # Other styles
        else:
            stats = provider.bandStatistics(1, QgsRasterBandStats.All, extent, 0)
            if stats.minimumValue <= 0.001:
                min = 0.001
            else:
                min = stats.minimumValue

            max = stats.maximumValue
            range = max - min
            add = range / 2
            interval = min + add
            valueList = [min, interval, max]

            dep_lst = [
                QgsColorRampShader.ColorRampItem(
                    valueList[0], QColor(colDic["lightblue"])
                ),
                QgsColorRampShader.ColorRampItem(valueList[1], QColor(colDic["blue"])),
                QgsColorRampShader.ColorRampItem(
                    valueList[2], QColor(colDic["darkblue"])
                ),
            ]

            vel_lst = [
                QgsColorRampShader.ColorRampItem(
                    valueList[0], QColor(colDic["lightgreen"])
                ),
                QgsColorRampShader.ColorRampItem(valueList[1], QColor(colDic["green"])),
                QgsColorRampShader.ColorRampItem(
                    valueList[2], QColor(colDic["darkgreen"])
                ),
            ]

            time_lst = [
                QgsColorRampShader.ColorRampItem(valueList[0], QColor(colDic["green"])),
                QgsColorRampShader.ColorRampItem(
                    valueList[1], QColor(colDic["yellow"])
                ),
                QgsColorRampShader.ColorRampItem(valueList[2], QColor(colDic["red"])),
            ]

            q_lst = [
                QgsColorRampShader.ColorRampItem(valueList[0], QColor(colDic["white"])),
                QgsColorRampShader.ColorRampItem(
                    valueList[1], QColor(colDic["lightblue"])
                ),
                QgsColorRampShader.ColorRampItem(valueList[2], QColor(colDic["blue"])),
            ]

            mud_lst = [
                QgsColorRampShader.ColorRampItem(
                    valueList[0], QColor(colDic["mud_lightbrown"])
                ),
                QgsColorRampShader.ColorRampItem(
                    valueList[1], QColor(colDic["mud_brown"])
                ),
                QgsColorRampShader.ColorRampItem(
                    valueList[2], QColor(colDic["mud_darkbrown"])
                ),
            ]

            style_dict = {
                0: dep_lst,
                1: vel_lst,
                5: mud_lst,
                3: time_lst,
                4: q_lst,
            }

            myRasterShader = QgsRasterShader()
            myColorRamp = QgsColorRampShader(minimumValue=min, maximumValue=max)

            myColorRamp.setColorRampItemList(style_dict[style])
            myColorRamp.setColorRampType(QgsColorRampShader.Interpolated)
            myColorRamp.setClip(True)

            myRasterShader.setRasterShaderFunction(myColorRamp)

            myPseudoRenderer = QgsSingleBandPseudoColorRenderer(
                layer.dataProvider(), layer.type(), myRasterShader
            )

            layer.setRenderer(myPseudoRenderer)

        layer.triggerRepaint()

    def set_vector_style(self, layer, style):
        """Define the vector styles"""

        script_directory = os.path.dirname(os.path.realpath(__file__))
        style_directory = script_directory + r"/vector_styles"

        # Extent red
        if style == 0:
            layer.loadNamedStyle(style_directory + r"/extent.qml")

        # Extent blue
        if style == 1:
            layer.loadNamedStyle(style_directory + r"/fluid_extent.qml")

        # Extent brown
        if style == 2:
            layer.loadNamedStyle(style_directory + r"/mud_extent.qml")


    def remove_layer(self, layer_name):
        """Function to remove layer name based on name"""
        for layer in QgsProject.instance().mapLayers().values():
            if layer.name() == layer_name:
                QgsProject.instance().removeMapLayers([layer.id()])

    def get_extent(self, raster, flood_extent_vector, name):
        """Function to get the extent of a raster layer"""
        vectorized = processing.run(
            "gdal:polygonize",
            {
                "INPUT": raster,
                "BAND": 1,
                "FIELD": "DN",
                "EIGHT_CONNECTEDNESS": False,
                "EXTRA": "",
                "OUTPUT": "TEMPORARY_OUTPUT",
            },
        )["OUTPUT"]

        processing.run(
            "native:dissolve",
            {
                "INPUT": vectorized,
                "FIELD": [],
                "SEPARATE_DISJOINT": False,
                "OUTPUT": flood_extent_vector,
            },
        )
        extent = QgsVectorLayer(flood_extent_vector, name)

        return extent

    def check_checkboxes(self):
        """Function to check if at least one map checkbox was checked"""

        checkboxes = self.dlg.tabs.findChildren(QtWidgets.QCheckBox)

        none_checked = not any(checkbox.isChecked() for checkbox in checkboxes)

        if none_checked:
            msg_box = QMessageBox()
            msg_box.setIcon(QMessageBox.Warning)
            msg_box.setWindowTitle("Warning")
            msg_box.setText("Check at least one map option!")
            msg_box.exec_()
            return False
        else:
            return True

    def run_open_layout(self):
        """Function to open the selected layout"""

        map_title = self.dlg.map_title_le.text()
        map_description = self.dlg.map_description.toPlainText()
        layer_extent = self.dlg.layer_extent_cb.currentLayer()

        lm = QgsProject.instance().layoutManager()
        l = QgsPrintLayout(QgsProject.instance())
        l.initializeDefaults()

        script_directory = os.path.dirname(os.path.realpath(__file__))
        template_directory = script_directory + r"/layout_templates"
        template_source = ""
        if self.dlg.a4_land.isChecked():
            template_source = template_directory + r"/FLO-2D A4 Landscape.qpt"
            layout_name = self.layout_exists("FLO-2D A4 Landscape")

        if self.dlg.a4_port.isChecked():
            template_source = template_directory + r"/FLO-2D A4 Portrait.qpt"
            layout_name = self.layout_exists("FLO-2D A4 Portrait")

        if self.dlg.a3_land.isChecked():
            template_source = template_directory + r"/FLO-2D A3 Landscape.qpt"
            layout_name = self.layout_exists("FLO-2D A3 Landscape")

        if self.dlg.a3_port.isChecked():
            template_source = template_directory + r"/FLO-2D A3 Portrait.qpt"
            layout_name = self.layout_exists("FLO-2D A3 Portrait")

        if template_source == "":
            msg = QMessageBox()
            msg.setIcon(QMessageBox.Warning)
            msg.setText("Please, select a layout.")
            msg.exec_()
            return

        # template_source = template_directory + r"/FLO-2D A4 Landscape.qpt"

        template_file = open(template_source, "r+", encoding="utf-8")
        template_content = template_file.read()
        template_file.close()
        document = QDomDocument()
        document.setContent(template_content)
        context = QgsReadWriteContext()
        l.loadFromTemplate(document, context)

        l.setName(layout_name)

        # canvas = self.iface.mapCanvas()
        for item in l.items():
            # QgsMessageLog.logMessage(str(item))
            if item.type() == 65639:  # Map
                item.zoomToExtent(layer_extent.extent())
            if item.type() == 65641:  # Label
                item.setText(item.text().replace("{{title}}", map_title))
                item.setText(item.text().replace("{{description}}", map_description))

        # Add layout to layout manager
        l.refresh()
        lm.addLayout(l)

        # Open and show the layout in designer
        # try:
        self.iface.openLayoutDesigner(l)
        # except:
        #     msg = QMessageBox()
        #     msg.setIcon(QMessageBox.Warning)
        #     msg.setText(self.tr(
        #         f'Error trying to open the layout ({l.name()}) returned errors.'
        #             ))
        #     msg.exec_()

    def layout_exists(self, layout_name):
        """Check if a layout already exists and return a correct name"""
        lm = QgsProject.instance().layoutManager()
        layouts = []
        n_layouts = 0
        for l in lm.layouts():
            layouts.append(l.name())
        if layout_name in layouts:
            n_layouts = sum(layout_name in s for s in layouts)

        if n_layouts == 0:
            return layout_name
        else:
            return layout_name + f" ({n_layouts})"

    def check_cw(self):
        """
        Function to check all available flood maps
        """

        flood_rbs = [
            self.dlg.ge_cw_cb,
            self.dlg.md_cw_cb,
            self.dlg.mv_cw_cb,
            self.dlg.mwse_cw_cb,
            self.dlg.fd_cw_cb,
            self.dlg.fv_cw_cb,
            self.dlg.dv_cw_cb,
            self.dlg.t1ft_cw_cb,
            self.dlg.t2ft_cw_cb,
            self.dlg.tmax_cw_cb,
            self.dlg.cd_cw_cb,
            self.dlg.cv_cw_cb,
            self.dlg.fcd_cw_cb,
            self.dlg.fcv_cw_cb,
            self.dlg.ld_cw_cb,
            self.dlg.se_cw_cb,
            self.dlg.sp_cw_cb,
            self.dlg.mvv_cw_cb,
            self.dlg.fvv_cw_cb,
            self.dlg.if_cw_cb,
        ]

        if self.dlg.check_cw_cb.isChecked():
            for cb in flood_rbs:
                if cb.isEnabled():
                    cb.setChecked(True)
                else:
                    cb.setChecked(False)
        else:
            for cb in flood_rbs:
                cb.setChecked(False)

    def check_sd(self):
        """
        Function to check all available sediment maps
        """
        sediment_rbs = [
            self.dlg.ge_sd_cb,
            self.dlg.md_sd_cb,
            self.dlg.mv_sd_cb,
            self.dlg.mwse_sd_cb,
            self.dlg.fd_sd_cb,
            self.dlg.fv_sd_cb,
            self.dlg.dv_sd_cb,
            self.dlg.t1ft_sd_cb,
            self.dlg.t2ft_sd_cb,
            self.dlg.tmax_sd_cb,
            self.dlg.cd_sd_cb,
            self.dlg.cv_sd_cb,
            self.dlg.fcv_sd_cb,
            self.dlg.fcd_sd_cb,
            self.dlg.ld_sd_cb,
            self.dlg.se_sd_cb,
            self.dlg.sp_sd_cb,
            self.dlg.mdep_sd_cb,
            self.dlg.msco_sd_cb,
            self.dlg.fbd_sd_cb,
            self.dlg.mvv_sd_cb,
            self.dlg.fvv_sd_cb,
            self.dlg.if_sd_cb,
            ]

        if self.dlg.check_sd_cb.isChecked():
            for cb in sediment_rbs:
                if cb.isEnabled():
                    cb.setChecked(True)
                else:
                    cb.setChecked(False)
        else:
            for cb in sediment_rbs:
                cb.setChecked(False)

    def check_hm(self):
        """
        Function to check all available hazard maps
        """
        hazard_rbs = [
            self.dlg.fh_australian_cb,
            self.dlg.fh_austrian_cb,
            self.dlg.flo_hm_cb,
            self.dlg.fi_swiss_cb,
            self.dlg.di_swiss_cb,
            self.dlg.fh_uk_cb,
            self.dlg.usbrh_hm_cb,
            self.dlg.usbrm_hm_cb,
            self.dlg.usbrv_hm_cb,
            self.dlg.usbra_hm_cb,
            self.dlg.usbrc_hm_cb,
            self.dlg.fema_hm_cb,
        ]

        if self.dlg.check_hm_cb.isChecked():
            for cb in hazard_rbs:
                if cb.isEnabled():
                    cb.setChecked(True)
                else:
                    cb.setChecked(False)
        else:
            for cb in hazard_rbs:
                cb.setChecked(False)

    def check_mf(self):
        """
        Function to check all available mudflow maps
        """
        mudflow_rbs = [
            self.dlg.ge_mf_cb,
            self.dlg.md_mf_cb,
            self.dlg.mv_mf_cb,
            self.dlg.mvv_mf_cb,
            self.dlg.mwse_mf_cb,
            self.dlg.fd_mf_cb,
            self.dlg.fv_mf_cb,
            self.dlg.fvv_mf_cb,
            self.dlg.dv_mf_cb,
            self.dlg.t1ft_mf_cb,
            self.dlg.t2ft_mf_cb,
            self.dlg.tmax_mf_cb,
            self.dlg.cd_mf_cb,
            self.dlg.cv_mf_cb,
            self.dlg.fcd_mf_cb,
            self.dlg.fcv_mf_cb,
            self.dlg.ld_mf_cb,
            self.dlg.se_mf_cb,
            self.dlg.sp_mf_cb,
            self.dlg.ms_mf_cb,
            self.dlg.fs_mf_cb,
            self.dlg.if_mf_cb,
        ]

        if self.dlg.check_mf_cb.isChecked():
            for cb in mudflow_rbs:
                if cb.isEnabled():
                    cb.setChecked(True)
                else:
                    cb.setChecked(False)
        else:
            for cb in mudflow_rbs:
                cb.setChecked(False)

    def check_tp(self):
        """
        Function to check all available twophase maps
        """
        twophase_rbs = [
            self.dlg.ge_tp_cb,
            self.dlg.mfd_tp_cb,
            self.dlg.mmd_tp_cb,
            self.dlg.cmd_tp_cb,
            self.dlg.mfv_tp_cb,
            self.dlg.mmv_tp_cb,
            self.dlg.mfsc_tp_cb,
            self.dlg.mmsc_tp_cb,
            # self.dlg.ffsc_tp_cb,
            self.dlg.fmsc_tp_cb,
            self.dlg.mwse_mf_cb,
            self.dlg.ffd_tp_cb,
            self.dlg.fmd_tp_cb,
            self.dlg.fcd_tp_cb,
            self.dlg.ffv_tp_cb,
            self.dlg.fmv_tp_cb,
            self.dlg.dv_tp_cb,
            self.dlg.t1ft_tp_cb,
            self.dlg.t2ft_tp_cb,
            self.dlg.tmax_tp_cb,
            self.dlg.mchd_tp_cb,
            self.dlg.mchv_tp_cb,
            self.dlg.fchd_tp_cb,
            self.dlg.fchv_tp_cb,
            self.dlg.ld_tp_cb,
            self.dlg.se_tp_cb,
            self.dlg.sp_tp_cb,
            self.dlg.md_tp_cb,
            self.dlg.ms_tp_cb,
            self.dlg.fdb_tp_cb,
            self.dlg.mfvv_tp_cb,
            self.dlg.ffvv_tp_cb,
            self.dlg.fmvv_tp_cb,
            self.dlg.mmvv_tp_cb,
            self.dlg.if_tp_cb,
        ]

        if self.dlg.check_tp_cb.isChecked():
            for cb in twophase_rbs:
                if cb.isEnabled():
                    cb.setChecked(True)
                else:
                    cb.setChecked(False)
        else:
            for cb in twophase_rbs:
                cb.setChecked(False)

    def collapse_all_groups(self):
        """
        Function to collapse all groups
        """
        cw_grps = [
            self.dlg.sc_cw_cgb,
            self.dlg.bv_cw_cgb,
            self.dlg.dv_cw_cgb,
            self.dlg.tv_cw_cgb,
            self.dlg.ch_cw_cgb,
            self.dlg.sv_cw_cgb,
            self.dlg.hp_cw_cgb,
        ]
        sd_grps = [
            self.dlg.sc_sd_cgb,
            self.dlg.bv_sd_cgb,
            self.dlg.dv_sd_cgb,
            self.dlg.tv_sd_cgb,
            self.dlg.ch_sd_cgb,
            self.dlg.sd_sd_cgb,
            self.dlg.sv_sd_cgb,
            self.dlg.hp_sd_cgb,
        ]
        md_grps = [
            self.dlg.sc_mf_cgb,
            self.dlg.bv_mf_cgb,
            self.dlg.dv_mf_cgb,
            self.dlg.tv_mf_cgb,
            self.dlg.ch_mf_cgb,
            self.dlg.mf_mf_cgb,
            self.dlg.sv_mf_cgb,
            self.dlg.hp_mf_cgb,
        ]
        tp_grps = [
            self.dlg.sc_tp_cgb,
            self.dlg.bv_tp_cgb,
            self.dlg.dv_tp_cgb,
            self.dlg.tv_tp_cgb,
            self.dlg.ch_tp_cgb,
            self.dlg.sv_tp_cgb,
            self.dlg.mv_tp_cgb,
            self.dlg.sdv_tp_cgb,
            self.dlg.hp_tp_cgb,
        ]
        hm_grps = [
            self.dlg.australian_hm_cgb,
            self.dlg.austrian_hm_cgb,
            self.dlg.flo_hm_cgb,
            self.dlg.swiss_hm_cgb,
            self.dlg.uk_hm_cgb,
            self.dlg.usbr_hm_cgb,
            self.dlg.fema_hz_cgb,
        ]

        if self.dlg.tab0.isEnabled():
            for grp in sd_grps:
                if grp.isEnabled():
                    grp.setCollapsed(True)
        if self.dlg.tab1.isEnabled():
            for grp in cw_grps:
                if grp.isEnabled():
                    grp.setCollapsed(True)
        if self.dlg.tab2.isEnabled():
            for grp in md_grps:
                if grp.isEnabled():
                    grp.setCollapsed(True)
        if self.dlg.tab3.isEnabled():
            for grp in tp_grps:
                if grp.isEnabled():
                    grp.setCollapsed(True)
        if self.dlg.tab5.isEnabled():
            for grp in hm_grps:
                if grp.isEnabled():
                    grp.setCollapsed(True)

    def expand_all_groups(self):
        """
        Function to expand all groups
        """
        cw_grps = [
            self.dlg.sc_cw_cgb,
            self.dlg.bv_cw_cgb,
            self.dlg.dv_cw_cgb,
            self.dlg.tv_cw_cgb,
            self.dlg.ch_cw_cgb,
            self.dlg.sv_cw_cgb,
            self.dlg.hp_cw_cgb,
        ]
        sd_grps = [
            self.dlg.sc_sd_cgb,
            self.dlg.bv_sd_cgb,
            self.dlg.dv_sd_cgb,
            self.dlg.tv_sd_cgb,
            self.dlg.ch_sd_cgb,
            self.dlg.sd_sd_cgb,
            self.dlg.sv_sd_cgb,
            self.dlg.hp_sd_cgb,
        ]
        md_grps = [
            self.dlg.sc_mf_cgb,
            self.dlg.bv_mf_cgb,
            self.dlg.dv_mf_cgb,
            self.dlg.tv_mf_cgb,
            self.dlg.ch_mf_cgb,
            self.dlg.mf_mf_cgb,
            self.dlg.sv_mf_cgb,
            self.dlg.hp_mf_cgb,
        ]
        tp_grps = [
            self.dlg.sc_tp_cgb,
            self.dlg.bv_tp_cgb,
            self.dlg.dv_tp_cgb,
            self.dlg.tv_tp_cgb,
            self.dlg.ch_tp_cgb,
            self.dlg.sv_tp_cgb,
            self.dlg.mv_tp_cgb,
            self.dlg.sdv_tp_cgb,
            self.dlg.hp_tp_cgb,
        ]
        hm_grps = [
            self.dlg.australian_hm_cgb,
            self.dlg.austrian_hm_cgb,
            self.dlg.flo_hm_cgb,
            self.dlg.swiss_hm_cgb,
            self.dlg.uk_hm_cgb,
            self.dlg.usbr_hm_cgb,
            self.dlg.fema_hz_cgb,
        ]

        if self.dlg.tab0.isEnabled():
            for grp in sd_grps:
                if grp.isEnabled():
                    grp.setCollapsed(False)
        if self.dlg.tab1.isEnabled():
            for grp in cw_grps:
                if grp.isEnabled():
                    grp.setCollapsed(False)
        if self.dlg.tab2.isEnabled():
            for grp in md_grps:
                if grp.isEnabled():
                    grp.setCollapsed(False)
        if self.dlg.tab3.isEnabled():
            for grp in tp_grps:
                if grp.isEnabled():
                    grp.setCollapsed(False)
        if self.dlg.tab5.isEnabled():
            for grp in hm_grps:
                if grp.isEnabled():
                    grp.setCollapsed(False)
