#! python3  # noqa: E265

"""Main plugin module."""

# standard
import os
from functools import partial
from pathlib import Path

# PyQGIS
from qgis.core import Qgis, QgsApplication, QgsSettings
from qgis.gui import QgisInterface
from qgis.PyQt.QtCore import QCoreApplication, QLocale, Qt, QTranslator, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction, QDockWidget, QToolBar, QWidget

# project
from flipflop.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
    __title__,
    __uri_homepage__,
)
from flipflop.gui.dlg_settings import PlgOptionsFactory
from flipflop.gui.wdg_select_layer import WdgSelectLayer
from flipflop.toolbelt import PlgLogger

# ############################################################################
# ########## Classes ###############
# ##################################


class FlipflopPlugin:
    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.log = PlgLogger().log

        # translation
        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[
            0:2
        ]
        locale_path: Path = (
            DIR_PLUGIN_ROOT
            / "resources"
            / "i18n"
            / f"{__title__.lower()}_{self.locale}.qm"
        )
        self.log(
            message=f"Translation: {self.locale}, {locale_path}",
            log_level=Qgis.MessageLevel.NoLevel,
        )

        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

        self.display_selected_layer = None
        self.docks = []

    def initGui(self):
        """Set up plugin UI elements."""

        # settings page within the QGIS preferences menu
        self.options_factory = PlgOptionsFactory()
        self.iface.registerOptionsWidgetFactory(self.options_factory)

        # -- Actions
        self.action_help = QAction(
            QgsApplication.getThemeIcon("mActionHelpContents.svg"),
            self.tr("Help"),
            self.iface.mainWindow(),
        )
        self.action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self.iface.mainWindow(),
        )
        self.action_settings.triggered.connect(
            lambda: self.iface.showOptionsDialog(
                currentPage="mOptionsPage{}".format(__title__)
            )
        )

        # -- Menu
        self.iface.addPluginToMenu(__title__, self.action_settings)
        self.iface.addPluginToMenu(__title__, self.action_help)

        # -- Help menu

        # documentation
        self.iface.pluginHelpMenu().addSeparator()
        self.action_help_plugin_menu_documentation = QAction(
            QIcon(str(__icon_path__)),
            f"{__title__} - Documentation",
            self.iface.mainWindow(),
        )
        self.action_help_plugin_menu_documentation.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.iface.pluginHelpMenu().addAction(
            self.action_help_plugin_menu_documentation
        )
        # FlipFlop Toolbar
        self.toolbar = QToolBar("FlipFlopToolbar")
        self.toolbar.setObjectName("FlopFlopToolbar")
        self.iface.addToolBar(self.toolbar)

        self.display_selected_layer = self.add_dock_widget_and_create_action(
            title=self.tr("Select layers for Flipflop"),
            name="select_layer",
            widget=WdgSelectLayer(self.iface.mainWindow()),
            add_in_toolbar=True,
        )

        self.display_selected_layer.setIcon(
            QIcon(os.path.join(DIR_PLUGIN_ROOT, "resources/images/default_icon.png")),
        )

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        # -- Clean up menu
        self.iface.removePluginMenu(__title__, self.action_help)
        self.iface.removePluginMenu(__title__, self.action_settings)

        # -- Clean up preferences panel in QGIS settings
        self.iface.unregisterOptionsWidgetFactory(self.options_factory)

        # remove from QGIS help/extensions menu
        if self.action_help_plugin_menu_documentation:
            self.iface.pluginHelpMenu().removeAction(
                self.action_help_plugin_menu_documentation
            )

        # remove actions
        del self.action_settings
        del self.action_help

        # Clean up docks
        for dock in self.docks:
            self.iface.removeDockWidget(dock)
            dock.deleteLater()

        self.docks.clear()

        # Toolbar
        if self.toolbar:
            self.toolbar.deleteLater()
            self.toolbar = None

    def run(self):
        """Main process.

        :raises Exception: if there is no item in the feed
        """
        try:
            self.log(
                message=self.tr("Everything ran OK."),
                log_level=Qgis.MessageLevel.Success,
                push=False,
            )
        except Exception as err:
            self.log(
                message=self.tr("Houston, we've got a problem: {}".format(err)),
                log_level=Qgis.MessageLevel.Critical,
                push=True,
            )

    def add_dock_widget_and_create_action(
        self, title: str, name: str, widget: QWidget, add_in_toolbar: bool = False
    ) -> QAction:
        """Add widget display as QDockWidget with an QAction in plugin toolbar


        :param name: dockwidget name for position save
        :type name: str
        :param widget: widget to insert
        :type widget: QWidget
        """

        # Create dockwidget
        dock = QDockWidget(title, self.iface.mainWindow())
        dock.setObjectName(name)
        dock.setWindowIcon(widget.windowIcon())

        # Add widget
        dock.setWidget(widget)

        # Connecter le signal de fermeture du dock pour nettoyer la barre de statut
        dock.visibilityChanged.connect(
            lambda visible: self.onDockVisibilityChanged(widget, visible)
        )

        # Add to QGIS
        self.iface.addDockWidget(Qt.DockWidgetArea.RightDockWidgetArea, dock)

        # Default close
        dock.close()

        # Append to dock list for unload
        self.docks.append(dock)
        dock.toggleViewAction().setIcon(widget.windowIcon())

        # Append to action list for unload
        action = dock.toggleViewAction()

        # Add action to toolbar
        if add_in_toolbar:
            self.toolbar.addAction(action)

        return action

    def onDockVisibilityChanged(self, widget, visible):
        """Appelé quand un dock change de visibilité"""
        if not visible and isinstance(widget, WdgSelectLayer):
            # Le dock est fermé, nettoyer la barre de statut
            widget.clearStatusBar()
