# Standard
from pathlib import Path

# QGIS
from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QIcon, QKeySequence, QShortcut
from qgis.PyQt.QtWidgets import QAbstractItemView, QLabel, QWidget

# Project
from flipflop.gui.mdl_layer_list import LayerListModel
from flipflop.gui.mdl_selected_layer_list import SelectedLayerListModel
from flipflop.toolbelt.preferences import PlgOptionsManager


class WdgSelectLayer(QWidget):
    """Sélectionner les layers pour la fonctionnalité de flipflop"""

    def __init__(self, parent=None):
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        # Modèle pour les couches disponibles
        self.model_available = LayerListModel()
        self.lv_available_layer.setModel(self.model_available)

        # Configuration de la liste source (drag seulement)
        self.lv_available_layer.setSelectionMode(
            QAbstractItemView.SelectionMode.ExtendedSelection
        )
        self.lv_available_layer.setDragEnabled(True)
        self.lv_available_layer.setDefaultDropAction(Qt.DropAction.CopyAction)

        # Modèle pour les couches sélectionnées
        self.model_selected = SelectedLayerListModel()
        self.lv_selected_layer.setModel(self.model_selected)

        # Configuration de la liste destination (drag & drop pour réordonnancement)
        self.lv_selected_layer.setSelectionMode(
            QAbstractItemView.SelectionMode.SingleSelection
        )
        self.lv_selected_layer.setAcceptDrops(True)
        self.lv_selected_layer.setDropIndicatorShown(True)
        self.lv_selected_layer.setDragEnabled(True)
        self.lv_selected_layer.setDragDropMode(QAbstractItemView.DragDropMode.DragDrop)
        self.lv_selected_layer.setDefaultDropAction(Qt.DropAction.MoveAction)

        # Connecter les boutons
        self.btn_add_layer.clicked.connect(self.addSelectedLayers)
        self.btn_remove_layer.clicked.connect(self.removeSelectedLayers)
        self.btn_clear.clicked.connect(self.clearAllLayers)

        # icon
        self.btn_add_layer.setIcon(
            QIcon(":images/themes/default/mActionArrowRight.svg")
        )
        self.btn_remove_layer.setIcon(
            QIcon(":images/themes/default/mActionArrowLeft.svg")
        )
        self.btn_clear.setIcon(
            QIcon(":images/themes/default/console/iconClearConsole.svg")
        )

        # Navigation cyclique
        self.current_visible_index = (
            -1
        )  # Index du layer actuellement visible (-1 = tous visibles)

        # Keyboard shortcut
        settings = PlgOptionsManager().get_plg_settings()
        letter = settings.letter_keybord_shortcut

        self.shortcut_cycle = QShortcut(QKeySequence(f"Ctrl+{letter}"), self)
        self.shortcut_cycle.activated.connect(self.cycleVisibleLayer)

        # Widget pour la barre de statut
        self.status_label = None

        # Connecter au signal du modèle pour détecter les changements
        self.model_selected.modelReset.connect(self.onModelReset)
        self.model_selected.rowsRemoved.connect(self.onModelChanged)

    def onModelReset(self):
        """Appelé quand le modèle est réinitialisé (nouveau projet, etc.)"""
        if self.model_selected.rowCount() == 0:
            self.current_visible_index = -1
            self.clearStatusBar()

    def onModelChanged(self):
        """Appelé quand des lignes sont supprimées du modèle"""
        if self.model_selected.rowCount() == 0:
            self.current_visible_index = -1
            self.clearStatusBar()

    def addSelectedLayers(self):
        """Ajoute les couches sélectionnées à la liste de droite (via bouton)"""
        selected_indexes = self.lv_available_layer.selectedIndexes()

        for index in selected_indexes:
            layer = self.model_available.getLayer(index)
            if layer:
                self.model_selected.addLayer(layer)

    def removeSelectedLayers(self):
        """Retire les couches sélectionnées de la liste de droite"""
        selected_indexes = self.lv_selected_layer.selectedIndexes()

        # Trier en ordre inverse pour supprimer du bas vers le haut
        rows_to_remove = sorted(
            [index.row() for index in selected_indexes], reverse=True
        )

        for row in rows_to_remove:
            self.model_selected.removeLayerAt(row)

        # Réinitialiser l'index de navigation et nettoyer la barre de statut
        self.current_visible_index = -1
        self.clearStatusBar()

    def clearStatusBar(self):
        """Nettoie le widget de la barre de statut"""
        from qgis.utils import iface

        if self.status_label and iface:
            iface.mainWindow().statusBar().removeWidget(self.status_label)
            self.status_label.deleteLater()
            self.status_label = None

    def getSelectedLayers(self):
        """Retourne la liste des couches sélectionnées dans l'ordre"""
        return self.model_selected.getLayers()

    def cycleVisibleLayer(self):
        """Cycle à travers les layers sélectionnés, affichant un seul à la fois dans le canvas"""
        from qgis.core import QgsProject
        from qgis.utils import iface

        row_count = self.model_selected.rowCount()

        # Si aucune couche sélectionnée, nettoyer et ne rien faire
        if row_count == 0:
            self.clearStatusBar()
            if iface:
                iface.messageBar().pushMessage(
                    "FlipFlop",
                    self.tr("No layer selected for the flip-flop"),
                    level=1,
                    duration=3,
                )
            return

        # Passer au layer suivant (cyclique)
        self.current_visible_index = (self.current_visible_index + 1) % row_count

        # Masquer tous les layers dans le canvas sauf celui à l'index courant
        for row in range(row_count):
            index = self.model_selected.index(row, 0)
            layer_id = self.model_selected.data(index, Qt.ItemDataRole.UserRole)
            layer_obj = QgsProject.instance().mapLayer(layer_id)

            if layer_obj:
                if row == self.current_visible_index:
                    # Afficher ce layer dans le canvas
                    QgsProject.instance().layerTreeRoot().findLayer(
                        layer_obj
                    ).setItemVisibilityChecked(True)
                else:
                    # Masquer les autres dans le canvas
                    QgsProject.instance().layerTreeRoot().findLayer(
                        layer_obj
                    ).setItemVisibilityChecked(False)

        # Récupérer et afficher les infos du layer visible
        visible_index = self.model_selected.index(self.current_visible_index, 0)
        layer_id = self.model_selected.data(visible_index, Qt.ItemDataRole.UserRole)
        layer_name = self.model_selected.data(
            visible_index, Qt.ItemDataRole.DisplayRole
        )
        layer_obj = QgsProject.instance().mapLayer(layer_id)

        if iface:
            # Créer le label si il n'existe pas encore
            if self.status_label is None:
                self.status_label = QLabel()
                self.status_label.setStyleSheet("QLabel { padding: 0 10px; }")
                # Fixer une largeur minimale pour éviter les décalages
                self.status_label.setMinimumWidth(250)
                self.status_label.setMaximumWidth(250)
                # Ajouter à gauche de la barre de statut, permanent
                iface.mainWindow().statusBar().insertPermanentWidget(
                    0, self.status_label
                )

            # Mettre à jour le texte
            message = f"FlipFlop: Layer {self.current_visible_index + 1}/{row_count} - {layer_name}"
            self.status_label.setText(message)
            self.status_label.setToolTip(f"id layer: {layer_id}")

    def closeEvent(self, event):
        """Nettoie le widget de la barre de statut à la fermeture"""
        self.clearStatusBar()
        super().closeEvent(event)

    def clearAllLayers(self):
        """Vide toutes les couches sélectionnées"""
        self.model_selected.clearLayers()

        # Réinitialiser l'index de navigation et nettoyer la barre de statut
        self.current_visible_index = -1
        self.clearStatusBar()
