from typing import Optional

from qgis.core import QgsMapLayer, QgsProject, QgsRasterLayer, QgsVectorLayer
from qgis.PyQt.QtCore import QMimeData, Qt
from qgis.PyQt.QtGui import QIcon, QStandardItemModel


class LayerListModel(QStandardItemModel):
    """Modèle pour les couches disponibles dans le projet"""

    def __init__(self, parent=None):
        super().__init__(0, 1, parent)
        self.setHorizontalHeaderLabels([self.tr("Nom")])

        self.loadLayers()
        self.connectProjectSignals()

    def reconnectLayerSignal(self, layer):
        """Déconnecte puis reconnecte le signal d'un layer"""
        try:
            layer.nameChanged.disconnect(self.refreshLayers)
        except Exception:
            pass

        try:
            layer.nameChanged.connect(self.refreshLayers)
        except Exception:
            pass

    def loadLayers(self):
        """Charge toutes les couches du projet"""
        self.removeRows(0, self.rowCount())
        for layer in QgsProject.instance().mapLayers().values():
            self.addLayer(layer)

    def addLayer(self, layer) -> None:
        row = self.rowCount()
        self.insertRow(row)
        self.setData(self.index(row, 0), layer.name())
        if isinstance(layer, QgsVectorLayer):
            self.setData(
                self.index(row, 0),
                QIcon(":/images/themes/default/mIconVector.svg"),
                Qt.ItemDataRole.DecorationRole,
            )
        elif isinstance(layer, QgsRasterLayer):
            self.setData(
                self.index(row, 0),
                QIcon(":/images/themes/default/mIconRaster.svg"),
                Qt.ItemDataRole.DecorationRole,
            )

        self.setData(self.index(row, 0), layer.id(), Qt.ItemDataRole.UserRole)
        self.setData(self.index(row, 0), layer, Qt.ItemDataRole.UserRole + 1)

        # Déconnecter puis reconnecter le signal nameChanged
        self.reconnectLayerSignal(layer)

    def onLayerAdded(self, layer):
        """Appelé quand un layer est ajouté au projet"""
        self.refreshLayers()

    def onLayersRemoved(self, layer_ids):
        """Appelé quand des layers sont supprimés du projet"""
        self.refreshLayers()

    def refreshLayers(self):
        """Rafraîchit la liste des couches"""
        self.loadLayers()

    def connectProjectSignals(self):
        """Connecte les signaux du projet QGIS"""
        project = QgsProject.instance()
        project.layerWasAdded.connect(self.onLayerAdded)

        # Utiliser layersRemoved au lieu de layersWillBeRemoved
        try:
            project.layersRemoved.connect(self.onLayersRemoved)
        except AttributeError:
            # Fallback
            project.layersWillBeRemoved.connect(self.onLayersRemoved)

        # Utiliser 'cleared' au lieu de 'removeAll'
        try:
            project.cleared.connect(self.refreshLayers)
        except AttributeError:
            # Fallback pour anciennes versions
            try:
                project.removeAll.connect(self.refreshLayers)
            except AttributeError:
                pass

    def disconnectProjectSignals(self):
        """Déconnecte les signaux du projet QGIS"""
        project = QgsProject.instance()
        try:
            project.layerWasAdded.disconnect(self.onLayerAdded)
            project.layersRemoved.disconnect(self.onLayersRemoved)
            project.layersWillBeRemoved.disconnect(self.onLayersRemoved)
            project.cleared.disconnect(self.refreshLayers)
            project.removeAll.disconnect(self.refreshLayers)
        except (TypeError, RuntimeError, AttributeError):
            # Ignorer toutes les erreurs de déconnexion
            pass

    def getLayer(self, index) -> Optional[QgsMapLayer]:
        """Retourne la couche à l'index donné"""
        if not index.isValid():
            return None

        return self.data(index, Qt.ItemDataRole.UserRole + 1)

    def flags(self, index):
        """Active le drag"""
        default_flags = super().flags(index)
        if index.isValid():
            return default_flags | Qt.ItemFlag.ItemIsDragEnabled
        return default_flags

    def mimeTypes(self):
        """Définit le type MIME pour le drag"""
        return ["application/x-qgis-layer-ids"]

    def mimeData(self, indexes):
        """Crée les données MIME pour le drag (supporte la sélection multiple)"""
        mime_data = QMimeData()

        if indexes:
            layer_ids = []
            for index in indexes:
                if index.isValid():
                    layer_ids.append(self.data(index, Qt.ItemDataRole.UserRole))
            if layer_ids:
                mime_data.setData(
                    "application/x-qgis-layer-ids", ";".join(layer_ids).encode()
                )

        return mime_data

    def __del__(self):
        """Déconnecte les signaux à la destruction"""
        self.disconnectProjectSignals()
