from qgis.core import QgsProject, QgsRasterLayer, QgsVectorLayer
from qgis.PyQt.QtCore import QAbstractListModel, QMimeData, QModelIndex, Qt
from qgis.PyQt.QtGui import QIcon


class SelectedLayerListModel(QAbstractListModel):
    """Modèle pour la liste des couches sélectionnées (avec réordonnancement)"""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.selected_layers = []
        self.connectProjectSignals()

    def connectProjectSignals(self):
        """Connecte les signaux du projet QGIS"""
        project = QgsProject.instance()
        project.layersWillBeRemoved.connect(self.onLayersWillBeRemoved)

        try:
            project.cleared.connect(self.onProjectCleared)
        except AttributeError:
            try:
                project.removeAll.connect(self.onProjectCleared)
            except AttributeError:
                pass

    def disconnectProjectSignals(self):
        """Déconnecte les signaux du projet QGIS"""
        project = QgsProject.instance()
        try:
            project.layersWillBeRemoved.disconnect(self.onLayersWillBeRemoved)
            project.cleared.disconnect(self.onProjectCleared)
        except (TypeError, RuntimeError, AttributeError):
            pass

        try:
            project.removeAll.disconnect(self.onProjectCleared)
        except (TypeError, RuntimeError, AttributeError):
            pass

    def onLayerNameChanged(self):
        """Appelé quand un layer change de nom"""

        sender_layer = self.sender()
        if sender_layer:
            try:
                row = self.selected_layers.index(sender_layer.id())
                index = self.index(row, 0)
                self.dataChanged.emit(index, index, [Qt.ItemDataRole.DisplayRole])
            except (ValueError, AttributeError):
                pass

    def onLayersWillBeRemoved(self, layer_ids):
        """Retire automatiquement les couches supprimées du projet"""
        self.removeLayersByIds(layer_ids)

    def onProjectCleared(self):
        """Vide la liste quand le projet est vidé"""
        self.clearLayers()

    def rowCount(self, parent=QModelIndex()):
        return len(self.selected_layers)

    def data(self, index, role=Qt.ItemDataRole.DisplayRole):
        if not index.isValid() or index.row() >= len(self.selected_layers):
            return None

        layer_id = self.selected_layers[index.row()]
        layer = QgsProject.instance().mapLayer(layer_id)

        if not layer:
            return None

        if role == Qt.ItemDataRole.DisplayRole:
            return layer.name()

        elif role == Qt.ItemDataRole.DecorationRole:
            if isinstance(layer, QgsVectorLayer):
                return QIcon(":/images/themes/default/mIconVector.svg")
            elif isinstance(layer, QgsRasterLayer):
                return QIcon(":/images/themes/default/mIconRaster.svg")

        elif role == Qt.ItemDataRole.UserRole:
            return layer_id

        return None

    def reconnectLayerSignal(self, layer):
        """Déconnecte puis reconnecte le signal nameChanged d'un layer"""
        if not layer:
            return

        try:
            layer.nameChanged.disconnect(self.onLayerNameChanged)
        except Exception:
            pass

        try:
            layer.nameChanged.connect(self.onLayerNameChanged)
        except Exception:
            pass

    def addLayer(self, layer):
        """Ajoute une couche à la liste (évite les doublons)"""
        if layer and layer.id() not in self.selected_layers:
            row = len(self.selected_layers)
            self.beginInsertRows(QModelIndex(), row, row)
            self.selected_layers.append(layer.id())
            self.endInsertRows()
            # Connecter le signal nameChanged du nouveau layer
            self.reconnectLayerSignal(layer)
            return True
        return False

    def addLayers(self, layers):
        """Ajoute plusieurs couches d'un coup"""
        added = False
        for layer in layers:
            if self.addLayer(layer):
                added = True
        return added

    def removeLayerAt(self, row):
        """Retire une couche à l'index donné"""
        if 0 <= row < len(self.selected_layers):
            self.beginRemoveRows(QModelIndex(), row, row)
            del self.selected_layers[row]
            self.endRemoveRows()
            return True
        return False

    def removeLayersByIds(self, layer_ids):
        """Retire les couches correspondant aux IDs donnés"""
        rows_to_remove = []
        for i, layer_id in enumerate(self.selected_layers):
            if layer_id in layer_ids:
                rows_to_remove.append(i)

        # Supprimer en ordre inverse pour ne pas décaler les indices
        for row in reversed(rows_to_remove):
            self.removeLayerAt(row)

    def clearLayers(self):
        """Vide la liste"""
        if self.selected_layers:
            self.beginResetModel()
            self.selected_layers.clear()
            self.endResetModel()

    def getLayers(self):
        """Retourne la liste des objets couches"""
        layers = []
        for layer_id in self.selected_layers:
            layer = QgsProject.instance().mapLayer(layer_id)
            if layer:
                layers.append(layer)

        return layers

    def supportedDropActions(self):
        """Actions de drop supportées"""
        return Qt.DropAction.MoveAction

    def canDropMimeData(self, data, action, row, column, parent):
        """Vérifie si on peut accepter le drop"""
        if data.hasFormat("application/x-qgis-layer-ids"):
            return True
        if data.hasFormat("application/x-qgis-selected-layer-index"):
            return True
        return False

    def moveRow(self, source_row, dest_row):
        """
        Déplace une ligne de source_row vers dest_row.
        dest_row est la position finale souhaitée APRÈS le déplacement.
        """
        if source_row == dest_row:
            return False

        if not (0 <= source_row < len(self.selected_layers)):
            return False

        if not (0 <= dest_row < len(self.selected_layers)):
            return False

        # Pour beginMoveRows, on doit calculer la position d'insertion
        # AVANT que l'élément source soit retiré
        if dest_row > source_row:
            # Déplacement vers le bas
            # La position Qt est dest_row + 1 car l'élément source est encore présent
            qt_dest = dest_row + 1
        else:
            # Déplacement vers le haut
            qt_dest = dest_row

        self.beginMoveRows(
            QModelIndex(), source_row, source_row, QModelIndex(), qt_dest
        )

        # Effectuer le déplacement dans la liste Python
        layer_id = self.selected_layers.pop(source_row)

        # Après le pop, recalculer la position d'insertion
        if dest_row > source_row:
            # L'élément source a été retiré, donc dest_row est maintenant décalé
            self.selected_layers.insert(dest_row, layer_id)
        else:
            # Pas de décalage nécessaire
            self.selected_layers.insert(dest_row, layer_id)

        self.endMoveRows()
        return True

    def dropMimeData(self, data, action, row, column, parent):
        """Gère le drop des couches (externe ou réordonnancement interne)"""

        # CAS 1 : Réordonnancement interne
        if data.hasFormat("application/x-qgis-selected-layer-index"):
            try:
                source_row = int(
                    data.data("application/x-qgis-selected-layer-index").data().decode()
                )
            except (ValueError, AttributeError):
                return False

            # Vérifier que source_row est valide
            if not (0 <= source_row < len(self.selected_layers)):
                return False

            # Déterminer la position de destination
            if row != -1:
                # Drop ENTRE deux items
                dest_row = row
            elif parent.isValid():
                # Drop SUR un item
                dest_row = parent.row() + 1
            else:
                # Drop dans le vide (à la fin)
                dest_row = len(self.selected_layers)

            # Ajuster si on drop après la fin
            if dest_row > len(self.selected_layers):
                dest_row = len(self.selected_layers)

            # Calculer la position finale après le retrait de source
            # Si on drag vers le bas, la position réelle sera dest_row - 1
            if dest_row > source_row:
                final_dest = dest_row - 1
            else:
                final_dest = dest_row

            # Pas de déplacement si on est déjà à la bonne position
            if source_row == final_dest:
                return True

            return self.moveRow(source_row, final_dest)

        # CAS 2 : Ajout depuis la liste disponible
        if data.hasFormat("application/x-qgis-layer-ids"):
            layer_ids_str = data.data("application/x-qgis-layer-ids").data().decode()
            layer_ids = layer_ids_str.split(";")

            added = False
            for layer_id in layer_ids:
                layer = QgsProject.instance().mapLayer(layer_id)
                if layer and self.addLayer(layer):
                    added = True

            return added

        return False

    def flags(self, index):
        """Définit les flags pour le drag & drop"""
        default_flags = super().flags(index)

        if index.isValid():
            # Les items peuvent être draggés ET recevoir des drops
            return (
                default_flags
                | Qt.ItemFlag.ItemIsDragEnabled
                | Qt.ItemFlag.ItemIsDropEnabled
            )
        else:
            # La zone vide peut recevoir des drops
            return default_flags | Qt.ItemFlag.ItemIsDropEnabled

    def mimeTypes(self):
        """Types MIME supportés"""
        return [
            "application/x-qgis-layer-ids",
            "application/x-qgis-selected-layer-index",
        ]

    def mimeData(self, indexes):
        """Crée les données MIME pour le drag interne (réordonnancement)"""
        mime_data = QMimeData()

        if indexes:
            source_row = indexes[0].row()
            mime_data.setData(
                "application/x-qgis-selected-layer-index", str(source_row).encode()
            )

        return mime_data
