# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FlightPlannerDialog
                                 A QGIS plugin
 Flight Planner allows you to plan photogrammetric flight and control it.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2019-09-22
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Jakub Gruca
        email                : jakubmgruca@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
import json
import traceback
from math import sqrt, ceil, fabs, pi, atan2, atan

import processing
from osgeo import gdal
from pyproj import Transformer
from PyQt5.QtWidgets import QMessageBox, QInputDialog
from PyQt5.QtCore import pyqtSlot, QVariant, QThread
from qgis.PyQt import uic, QtWidgets
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsField,
    QgsFieldProxyModel,
    QgsMapLayerProxyModel,
    QgsPointXY
)

from .camera import Camera
from .worker import Worker
from .functions import (
    add_layers_to_canvas,
    change_layer_style,
    corridor_flight_numbering,
    create_waypoints,
    create_flight_line,
    bounding_box_at_angle,
    projection_centres,
    line,
    transf_coord,
    minmaxheight,
    save_error
)

# Load .ui file so that PyQt can populate plugin
# with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'flight_planner_dialog_base.ui'))


class FlightPlannerDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(FlightPlannerDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        self.setupUi(self)
        
        self.tabBlock = True
        self.tabCorridor = False
        self.mMapLayerComboBoxAoI.setLayer(None)
        self.mMapLayerComboBoxCorridor.setLayer(None)
        self.mMapLayerComboBoxDTM.setLayer(None)
        self.mMapLayerComboBoxProjectionCentres.setLayer(None)

        self.mGroupBox.setToolTip("* Maximum and minimum height"
            " (together with camera parameters and GSD/Alt. AGL)"
            " are used to calculate increased Overlap/Sidelap."
            "\nThe heights are also used to calculate average "
            "terrain height, but only in 'One altitude for entire"
            " flight' mode. Read more about it in the Guide.")
        self.mGroupBox.setToolTipDuration(20000)

        # Set up filters for ComboBoxes
        self.mMapLayerComboBoxProjectionCentres.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.mFieldComboBoxAltControl.setFilters(QgsFieldProxyModel.Numeric)
        self.mFieldComboBoxOmega.setFilters(QgsFieldProxyModel.Numeric)
        self.mFieldComboBoxPhi.setFilters(QgsFieldProxyModel.Numeric)
        self.mFieldComboBoxKappa.setFilters(QgsFieldProxyModel.Numeric)
        self.mMapLayerComboBoxDTM.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.mMapLayerComboBoxAoI.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.mMapLayerComboBoxCorridor.setFilters(QgsMapLayerProxyModel.LineLayer)

        self.comboBoxAltitudeType.addItems(["One Altitude ASL For Entire Flight",
            "Separate Altitude ASL For Each Strip",
            "Terrain Following"])

        # Set up ComboBox of camera
        self.cameras_file = os.path.join(
            os.path.dirname(os.path.abspath(__file__)), 'cameras.json')
        with open(self.cameras_file, 'r', encoding='utf-8') as file:
            self.cameras = [Camera(**camera) for camera in sorted(json.load(file), key=lambda x : x['name'])]
            self.comboBoxCamera.addItems([camera.name for camera in self.cameras])
        if self.comboBoxCamera.count() > 0:
            self.comboBoxCamera.setItemText(0, 'Select camera or set parameters')
        else:
            self.comboBoxCamera.addItem('No cameras listed')
            self.comboBoxCamera.setCurrentText('No cameras listed')

        self.camera = Camera(name='',
                            focal_length=self.doubleSpinBoxFocalLength.value(),
                            sensor_size=self.doubleSpinBoxSensorSize.value(),
                            pixels_along_track=self.spinBoxPixelsAlongTrack.value(),
                            pixels_across_track=self.spinBoxPixelsAcrossTrack.value())

        self.control_run_counter = 1
        self.design_run_counter = 1

    def startWorker_control(self, **params):
        """Start worker for control module of plugin."""
        # Create a new worker instance
        worker = Worker(**params)

        self.pushButtonStopControl.clicked.connect(worker.kill)
        # Start the worker in a new thread
        thread = QThread(self)
        worker.moveToThread(thread)
        worker.finished.connect(self.workerFinished)
        worker.error.connect(self.workerError)
        worker.progress.connect(self.progressBarControl.setValue)
        worker.enabled.connect(self.pushButtonRunControl.setEnabled)
        worker.enabled.connect(self.pushButtonRunDesign.setEnabled)
        thread.started.connect(worker.run_control)
        thread.start()
        self.thread = thread
        self.worker = worker

    def startWorker_updateAltitude(self, **params):
        """Start a worker for update altitude of flight in 'altitude for
        each strip' or 'terraing following' mode."""
        worker = Worker(**params)
        # Create a new worker instance
        # Start the worker in a new thread
        thread = QThread(self)
        worker.moveToThread(thread)
        worker.finished.connect(self.workerFinished)
        worker.error.connect(self.workerError)
        worker.progress.connect(self.progressBar.setValue)
        worker.enabled.connect(self.pushButtonRunDesign.setEnabled)
        worker.enabled.connect(self.pushButtonRunControl.setEnabled)

        if self.comboBoxAltitudeType.currentText() == 'Separate Altitude ASL For Each Strip':
            thread.started.connect(worker.run_altitudeStrip)
        elif self.comboBoxAltitudeType.currentText() == 'Terrain Following':
            thread.started.connect(worker.run_followingTerrain)

        thread.start()
        self.thread = thread
        self.worker = worker

    def workerFinished(self, result, group_name):
        # clean up the worker and thread
        self.worker.deleteLater()
        self.thread.quit()
        self.thread.wait()
        self.thread.deleteLater()
        if result is not None:
            # report the result
            if group_name == 'flight_design':
                add_layers_to_canvas(result, group_name, self.design_run_counter)
                self.design_run_counter += 1
            elif group_name == 'quality_control':
                add_layers_to_canvas(result, group_name, self.control_run_counter)
                self.control_run_counter += 1
        else:
            # notify the user that something went wrong
            print('Something went wrong!')

    def workerError(self, e, exception_string):
        print(f'Worker thread raised an exception: {exception_string}')
        save_error()
        QMessageBox.about(self, 'Error', 'See error log file in plugin folder')


    def check_set_gsd(self):
        gsd = (self.doubleSpinBoxAltAGL.value()*100) \
            / (self.doubleSpinBoxFocalLength.value()/10) \
            * (self.doubleSpinBoxSensorSize.value()/10_000)
        if gsd > self.doubleSpinBoxGSD.maximum():
            self.doubleSpinBoxGSD.setValue(-1)
            self.doubleSpinBoxGSD.setSpecialValueText(f"> {self.doubleSpinBoxGSD.maximum()} [cm/px]")
        elif gsd < self.doubleSpinBoxGSD.minimum():
            self.doubleSpinBoxGSD.setValue(-1)
            self.doubleSpinBoxGSD.setSpecialValueText(f"< {self.doubleSpinBoxGSD.minimum()} [cm/px]")
        else:
            self.doubleSpinBoxGSD.setSpecialValueText("")
            self.doubleSpinBoxGSD.setValue(gsd)

    def check_set_altitude(self):
        w = (self.doubleSpinBoxGSD.value()/100) \
            / (self.doubleSpinBoxSensorSize.value()/1_000_000) \
            * (self.doubleSpinBoxFocalLength.value()/1000)
        if w < self.doubleSpinBoxAltAGL.minimum():
            self.doubleSpinBoxAltAGL.setValue(-1)
            self.doubleSpinBoxAltAGL.setSpecialValueText(f"< {self.doubleSpinBoxAltAGL.minimum()} [m]")
        else:
            self.doubleSpinBoxAltAGL.setSpecialValueText("")
            self.doubleSpinBoxAltAGL.setValue(w)

    def on_pushButtonStopControl_clicked(self):
        pass

    def on_progressBar_valueChanged(self):
        pass

    def on_progressBarControl_valueChanged(self):
        pass

    def on_comboBoxCamera_highlighted(self):
        camera_names = [camera.name for camera in self.cameras]
        items_list = [self.comboBoxCamera.itemText(i) for i in range(self.comboBoxCamera.count())]
        if ("Select camera or set parameters" in items_list 
            or "No cameras listed" in items_list) and camera_names:
            self.comboBoxCamera.clear()
            self.comboBoxCamera.addItems(camera_names)
        elif ("Select camera or set parameters" in items_list) and not camera_names:
            self.comboBoxCamera.clear()
            self.comboBoxCamera.addItem("No cameras listed")

    def on_comboBoxCamera_activated(self, i):
        camera_names = [camera.name for camera in self.cameras]
        if isinstance(i, int) and self.comboBoxCamera.currentText() in camera_names:
            self.doubleSpinBoxFocalLength.setValue(self.cameras[i].focal_length * 1_000)
            self.doubleSpinBoxSensorSize.setValue(self.cameras[i].sensor_size * 1_000_000)
            self.spinBoxPixelsAlongTrack.setValue(self.cameras[i].pixels_along_track)
            self.spinBoxPixelsAcrossTrack.setValue(self.cameras[i].pixels_across_track)
            self.camera.name = self.comboBoxCamera.currentText()

    def on_comboBoxAltitudeType_activated(self, text):
        if isinstance(text, str):
            if text in ['Separate Altitude ASL For Each Strip', 'Terrain Following'] \
                and not self.mMapLayerComboBoxDTM.currentLayer():
                QMessageBox.about(self, 'DTM needed', 'You must select DTM to use this option')
                self.comboBoxAltitudeType.setCurrentText("One Altitude ASL For Entire Flight")
            elif text == 'Terrain Following' and self.mMapLayerComboBoxDTM.currentLayer():
                self.doubleSpinBoxTolerance.setEnabled(True)
            else:
                self.doubleSpinBoxTolerance.setEnabled(False)

            if self.comboBoxAltitudeType.currentText() != 'One Altitude ASL For Entire Flight':
                self.checkBoxIncreaseOverlap.setChecked(False)
                self.checkBoxIncreaseOverlap.setEnabled(False)
                self.pushButtonGetHeights.setEnabled(False)
                self.doubleSpinBoxMaxHeight.setEnabled(False)
                self.doubleSpinBoxMinHeight.setEnabled(False)
            else:
                self.checkBoxIncreaseOverlap.setEnabled(True)
                self.pushButtonGetHeights.setEnabled(True)
                self.doubleSpinBoxMaxHeight.setEnabled(True)
                self.doubleSpinBoxMinHeight.setEnabled(True)

            if self.comboBoxAltitudeType.currentText() == 'Terrain Following':
              self.radioButtonAltAGL.setEnabled(True)
            else:
               self.radioButtonAltAGL.setEnabled(False)
               self.radioButtonGSD.setChecked(True)

    def on_doubleSpinBoxFocalLength_valueChanged(self):
        self.camera.focal_length = self.doubleSpinBoxFocalLength.value() / 1_000
        if self.radioButtonGSD.isChecked():
            self.check_set_altitude()
        elif self.radioButtonAltAGL.isChecked():
            self.check_set_gsd()

    def on_doubleSpinBoxSensorSize_valueChanged(self):
        self.camera.sensor_size = self.doubleSpinBoxSensorSize.value() / 1_000_000
        if self.radioButtonGSD.isChecked():
            self.check_set_altitude()
        elif self.radioButtonAltAGL.isChecked():
            self.check_set_gsd()

    def on_doubleSpinBoxGSD_valueChanged(self):
        if self.radioButtonGSD.isChecked():
            self.check_set_altitude()

    def on_doubleSpinBoxAltAGL_valueChanged(self):
        if self.radioButtonAltAGL.isChecked():
            self.check_set_gsd()

    def on_spinBoxPixelsAlongTrack_valueChanged(self):
        self.camera.pixels_along_track = self.spinBoxPixelsAlongTrack.value()

    def on_spinBoxPixelsAcrossTrack_valueChanged(self):
        self.camera.pixels_across_track = self.spinBoxPixelsAcrossTrack.value()

    def on_doubleSpinBoxMaxHeight_valueChanged(self):
        pass

    def on_doubleSpinBoxMinHeight_valueChanged(self):
        pass

    def on_doubleSpinBoxOverlap_valueChanged(self):
        pass

    def on_doubleSpinBoxSidelap_valueChanged(self):
        pass

    def on_doubleSpinBoxBuffer_valueChanged(self):
        pass

    def on_doubleSpinBoxIterationThreshold_valueChanged(self):
        pass

    def on_checkBoxOverlapImages_stateChanged(self):
        pass

    def on_checkBoxFootprint_stateChanged(self):
        pass

    def on_checkBoxGSDmap_stateChanged(self):
        pass

    def on_checkBoxIncreaseOverlap_stateChanged(self):
        try:
            gsd = self.doubleSpinBoxGSD.value() / 100
            max_h = self.doubleSpinBoxMaxHeight.value()
            min_h = self.doubleSpinBoxMinHeight.value()
            p0 = self.doubleSpinBoxOverlap.value()
            q0 = self.doubleSpinBoxSidelap.value()
            W = gsd / self.camera.sensor_size * self.camera.focal_length
            if self.checkBoxIncreaseOverlap.isChecked():
                # remember old values of overlap p, sidelap q
                self.p0_prev = self.doubleSpinBoxOverlap.value()
                self.q0_prev = self.doubleSpinBoxSidelap.value()
                # new values of overlap p, sidelap q
                self.p = p0 / 100 + 0.5 * ((max_h - min_h) / 2) / W
                self.q = q0 / 100 + 0.7 * ((max_h - min_h) / 2) / W
            else:
                self.p = self.p0_prev / 100
                self.q = self.q0_prev / 100
        except:
            QMessageBox.about(self, 'Error', 'Overlap/sidelap increasing failed')
        else:
            self.doubleSpinBoxOverlap.setValue(self.p * 100)
            self.doubleSpinBoxSidelap.setValue(self.q * 100)

    def on_spinBoxMultipleBase(self):
        pass

    def on_mMapLayerComboBoxProjectionCentres_layerChanged(self):
        try:
            if self.mMapLayerComboBoxProjectionCentres.currentLayer():
                proj_cent_layer = self.mMapLayerComboBoxProjectionCentres.currentLayer()
                self.crs_vct_ctrl = proj_cent_layer.sourceCrs().authid()
                crs = QgsCoordinateReferenceSystem(self.crs_vct_ctrl)
                if crs.isGeographic():
                    QMessageBox.about(self, 'Error', 'CRS of layer cannot be' \
                                      + 'geographic')
                    self.mFieldComboBoxAltControl.setLayer(None)
                    self.mFieldComboBoxOmega.setLayer(None)
                    self.mFieldComboBoxPhi.setLayer(None)
                    self.mFieldComboBoxKappa.setLayer(None)
                else:
                    self.mFieldComboBoxAltControl.setLayer(proj_cent_layer)
                    self.mFieldComboBoxOmega.setLayer(proj_cent_layer)
                    self.mFieldComboBoxPhi.setLayer(proj_cent_layer)
                    self.mFieldComboBoxKappa.setLayer(proj_cent_layer)
        except:
            QMessageBox.about(self, 'Error', 'See error log file in plugin'
                              ' folder')
            save_error()

    def on_hFieldComboBox_fieldChanged(self):
        pass

    def on_mFieldComboBoxOmega_fieldChanged(self):
        pass

    def on_mFieldComboBoxPhi_fieldChanged(self):
        pass

    def on_mFieldComboBoxKappa_fieldChanged(self):
        pass

    @pyqtSlot()
    def on_pushButtonGetHeights_clicked(self):
        attributes_exist = True
        if not hasattr(self, 'DTM'):
            QMessageBox.about(self, 'DTM needed',
                'You have to load DTM layer')
            attributes_exist = False

        if self.tabBlock:
            if not hasattr(self, 'AreaOfInterest'):
                QMessageBox.about(self, 'AoI needed',
                    'You have to load Area of Interest layer')
                attributes_exist = False
        else:
            if not hasattr(self, 'pathLine'):
                QMessageBox.about(self, 'Corridor line needed',
                    'You have to load Corridor line layer')
                attributes_exist = False

        if attributes_exist:
            try:
                if self.tabBlock:
                    h_min, h_max = minmaxheight(self.AreaOfInterest, self.DTM)
                else:
                    g_rst = self.raster.GetGeoTransform()
                    pix_width = g_rst[1]
                    pix_height = -g_rst[5]
                    uplx = g_rst[0]
                    uply = g_rst[3]
                    uplx_n = uplx + pix_width
                    uply_n = uply + pix_height
                    xo, yo = uplx, uply
                    xo1, yo1 = uplx_n, uply_n
                    # setting minimum buffer size to be able to get heights
                    if self.crs_rst != self.crs_vct:
                        transf_rst_vct = Transformer.from_crs(self.crs_rst,
                                                            self.crs_vct,
                                                            always_xy=True)
                        xo, yo = transf_coord(transf_rst_vct, uplx, uply)
                        xo1, yo1 = transf_coord(transf_rst_vct, uplx_n, uply_n)

                    min_buff_size = max(ceil(fabs(xo1 - xo)), ceil(fabs(yo1 - yo)))
                    self.doubleSpinBoxBuffer.setMinimum(min_buff_size / 2)
                    buffLine = processing.run("native:buffer",
                                            {'INPUT': self.pathLine,
                                            'DISTANCE': self.doubleSpinBoxBuffer.value(),
                                            'SEGMENTS': 5, 'END_CAP_STYLE': 0, 'JOIN_STYLE': 0,
                                            'MITER_LIMIT': 2, 'DISSOLVE': False,
                                            'OUTPUT': 'TEMPORARY_OUTPUT'})
                    self.bufferedLine = buffLine['OUTPUT']
                    h_min, h_max = minmaxheight(self.bufferedLine, self.DTM)
            except:
                QMessageBox.about(self, 'Error', 'Get heights from DTM failed')
                save_error()
            else:
                self.doubleSpinBoxMinHeight.setValue(h_min)
                self.doubleSpinBoxMaxHeight.setValue(h_max)

    def on_mMapLayerComboBoxCorridor_layerChanged(self):
        if self.mMapLayerComboBoxCorridor.currentLayer():
            self.CorLine = self.mMapLayerComboBoxCorridor.currentLayer()
            if self.CorLine.sourceCrs().mapUnits() == 0:
                self.crs_vct = self.CorLine.sourceCrs().authid()
                self.pathLine = self.CorLine.dataProvider().dataSourceUri()
            else:
                QMessageBox.about(self, 'Incorrect CRS', 'The layer units must be meters')
                self.mMapLayerComboBoxCorridor.setLayer(None)

    def on_mMapLayerComboBoxAoI_layerChanged(self):
        if self.mMapLayerComboBoxAoI.currentLayer():
            self.AreaOfInterest = self.mMapLayerComboBoxAoI.currentLayer()
            if self.AreaOfInterest.sourceCrs().mapUnits() == 0:
                self.crs_vct = self.AreaOfInterest.sourceCrs().authid()
                features = self.AreaOfInterest.getFeatures()
                for feature in features:
                    self.geom_AoI = feature.geometry()
            else:
                QMessageBox.about(self, 'Incorrect CRS', 'The layer units must be meters')
                self.mMapLayerComboBoxAoI.setLayer(None)

    def on_mMapLayerComboBoxDTM_layerChanged(self):
        if self.mMapLayerComboBoxDTM.currentLayer():
            self.DTM = self.mMapLayerComboBoxDTM.currentLayer()
            self.crs_rst = self.DTM.crs().authid()
            pathDTM = self.DTM.source()
            self.raster = gdal.Open(pathDTM)

    def on_tabWidgetBlockCorridor_currentChanged(self):
        if self.tabWidgetBlockCorridor.currentIndex() == 0:
            self.tabBlock = True
            self.tabCorridor = False
        else:
            self.tabBlock = False
            self.tabCorridor = True

    def on_dial_valueChanged(self):
        if self.dial.value() > 180:
            self.spinBoxDirection.setValue(self.dial.value() - 180)
        else:
            self.spinBoxDirection.setValue(self.dial.value() + 180)

    def on_spinBoxDirection_valueChanged(self):
        if self.spinBoxDirection.value() > 180:
            self.dial.setValue(self.spinBoxDirection.value() - 180)
        else:
            self.dial.setValue(self.spinBoxDirection.value() + 180)

    def on_spinBoxExceedExtremeStrips(self):
        pass

    def on_radioButtonGSD_toggled(self):
        if self.radioButtonGSD.isChecked():
            self.doubleSpinBoxAltAGL.setEnabled(False)
            self.doubleSpinBoxGSD.setEnabled(True)
            if self.doubleSpinBoxGSD.specialValueText() == f"> {self.doubleSpinBoxGSD.maximum()} [cm/px]":
                self.doubleSpinBoxGSD.setSpecialValueText("")
                self.doubleSpinBoxGSD.setValue(self.doubleSpinBoxGSD.maximum())
            elif self.doubleSpinBoxGSD.specialValueText() == f"< {self.doubleSpinBoxGSD.minimum()} [cm/px]":
                self.doubleSpinBoxGSD.setSpecialValueText("")
                self.doubleSpinBoxGSD.setValue(self.doubleSpinBoxGSD.minimum())
            self.check_set_altitude()
        elif self.radioButtonAltAGL.isChecked():
            self.doubleSpinBoxAltAGL.setEnabled(True)
            self.doubleSpinBoxGSD.setEnabled(False)
            if self.doubleSpinBoxAltAGL.specialValueText() == f"< {self.doubleSpinBoxAltAGL.minimum()} [m]":
                self.doubleSpinBoxAltAGL.setSpecialValueText("")
                self.doubleSpinBoxAltAGL.setValue(self.doubleSpinBoxAltAGL.minimum())
            self.check_set_gsd()

    @pyqtSlot()
    def on_pushButtonRunDesign_clicked(self):
        """Push Button to make a flight plan."""
        attributes_exist = True
        if self.comboBoxAltitudeType.currentText() in ['Separate Altitude ASL For Each Strip', 'Terrain Following']:
            if not hasattr(self, 'DTM'):
                QMessageBox.about(self, 'DTM needed', 'You have to load DTM layer')
                attributes_exist = False

        if self.tabBlock:
            if not hasattr(self, 'AreaOfInterest'):
                QMessageBox.about(self, 'AoI needed', 'You have to load Area of Interest layer')
                attributes_exist = False
        else:
            if not hasattr(self, 'pathLine'):
                QMessageBox.about(self, 'Corridor line needed', 'You have to load Corridor line layer')
                attributes_exist = False

        if attributes_exist:
            try:
                gsd = self.doubleSpinBoxGSD.value() / 100
                max_h = self.doubleSpinBoxMaxHeight.value()
                min_h = self.doubleSpinBoxMinHeight.value()
                p0 = self.doubleSpinBoxOverlap.value()
                q0 = self.doubleSpinBoxSidelap.value()
                mult_base = self.spinBoxMultipleBase.value()
                x_percent = self.spinBoxExceedExtremeStrips.value()

                avg_terrain_height = (max_h + min_h) / 2
                if self.radioButtonGSD.isChecked():
                    altitude_AGL = gsd / self.camera.sensor_size * self.camera.focal_length
                elif self.radioButtonAltAGL.isChecked():
                    altitude_AGL = self.doubleSpinBoxAltAGL.value()
                altitude_ASL = avg_terrain_height + altitude_AGL

                if not self.checkBoxIncreaseOverlap.isChecked():
                    self.p = p0 / 100
                    self.q = q0 / 100
                # image length along and across flight direction[m]
                len_along = self.camera.pixels_along_track * gsd
                len_across = self.camera.pixels_across_track * gsd
                # longitudinal base Bx, transverse base By
                Bx = len_along * (1 - self.p)
                By = len_across * (1 - self.q)
                strip = 0
                photo = 0

                if self.tabBlock:
                    angle = 90 - self.spinBoxDirection.value()
                    if 90 - self.spinBoxDirection.value() < 0:
                        angle = 90 - self.spinBoxDirection.value() + 360
                    # bounding box equotations and dimensions Dx, Dy
                    a, b, a2, b2, Dx, Dy = bounding_box_at_angle(angle,
                                                                self.geom_AoI)
                    pc_lay, photo_lay, s_nr, p_nr = projection_centres(
                        angle, self.geom_AoI, self.crs_vct, a, b, a2, b2, Dx, Dy,
                        Bx, By, len_along, len_across, x_percent, mult_base, altitude_ASL,
                        strip, photo)

                elif self.tabCorridor:
                    exploded_lines = processing.run("native:explodelines",
                                                    {'INPUT': self.pathLine,
                                                    'OUTPUT': 'TEMPORARY_OUTPUT'})
                    exp_lines = exploded_lines['OUTPUT']
                    # buffer for each exp_lines
                    buffered_exp_lines = processing.run("native:buffer",
                                                        {'INPUT': exp_lines, 'DISTANCE': self.doubleSpinBoxBuffer.value(),
                                                        'SEGMENTS': 5, 'END_CAP_STYLE': 1, 'JOIN_STYLE': 1,
                                                        'MITER_LIMIT': 2, 'DISSOLVE': False,
                                                        'OUTPUT': 'TEMPORARY_OUTPUT'})
                    buff_exp_lines = buffered_exp_lines['OUTPUT']
                    feats_exp_lines = exp_lines.getFeatures()
                    pc_lay_list = []
                    photo_lay_list = []
                    line_buf_list = []

                    segments = len([segment for segment in exp_lines.getFeatures()])
                    ordered_segments = corridor_flight_numbering(exp_lines.getFeatures(),
                        buff_exp_lines, Bx, By, len_across, mult_base, x_percent, segments)

                    segment_nr = 1
                    # building projection centres and photos layer for each line
                    for feat_exp in feats_exp_lines:
                        x_start = feat_exp.geometry().asPolyline()[0].x()
                        y_start = feat_exp.geometry().asPolyline()[0].y()
                        x_end = feat_exp.geometry().asPolyline()[1].x()
                        y_end = feat_exp.geometry().asPolyline()[1].y()
                        # equation of corridor line
                        a_line, b_line = line(y_start, y_end, x_start, x_end)
                        angle = atan(a_line) * 180 / pi

                        if angle < 0:
                            angle = angle + 180
                        if y_end - y_start < 0:
                            angle = angle + 180

                        featbuff_exp = buff_exp_lines.getFeature(segment_nr)
                        # geometry object of line buffer
                        geom_line_buf = featbuff_exp.geometry()
                        line_buf_list.append(geom_line_buf)
                        a, b, a2, b2, Dx, Dy = bounding_box_at_angle(angle,
                                                                    geom_line_buf)
                        # projection centres layer and photos layer for given line
                        pc_lay, photo_lay, s_nr, p_nr = projection_centres(
                            angle, geom_line_buf, self.crs_vct, a, b, a2, b2, Dx,
                            Dy, Bx, By, len_along, len_across, x_percent,
                            mult_base, altitude_ASL, strip, photo)
                        # adding helping field for function 'alt. for each strip'
                        pc_lay.startEditing()
                        photo_lay.startEditing()
                        pc_lay.addAttribute(QgsField("BuffNr", QVariant.Int))

                        segment = ordered_segments[f'segment_{segment_nr}']
                        feature_id = 1
                        for strip_nr, photos_nr in segment.items():
                            for photo_nr in photos_nr:
                                st_nr = '%(s_nr)04d' % {'s_nr': strip_nr}
                                ph_nr = '%(p_nr)05d' % {'p_nr': photo_nr}
                                pc_lay.changeAttributeValue(feature_id, 0, st_nr)
                                pc_lay.changeAttributeValue(feature_id, 1, ph_nr)
                                pc_lay.changeAttributeValue(feature_id, 9, segment_nr)
                                photo_lay.changeAttributeValue(feature_id, 0, st_nr)
                                photo_lay.changeAttributeValue(feature_id, 1, ph_nr)
                                feature_id += 1
                        segment_nr = segment_nr + 1

                        pc_lay.commitChanges()
                        photo_lay.commitChanges()
                        pc_lay_list.append(pc_lay)
                        photo_lay_list.append(photo_lay)
                        strip = s_nr
                        photo = p_nr

                    # merging results for every line
                    merged_pnt_lay = processing.run("native:mergevectorlayers",
                                                    {'LAYERS': pc_lay_list,
                                                    'CRS': None, 'OUTPUT': 'TEMPORARY_OUTPUT'})
                    pc_lay = merged_pnt_lay['OUTPUT']
                    merged_poly_lay = processing.run("native:mergevectorlayers",
                                                    {'LAYERS': photo_lay_list, 'CRS': None,
                                                    'OUTPUT': 'TEMPORARY_OUTPUT'})
                    photo_lay = merged_poly_lay['OUTPUT']
                theta = fabs(atan2(len_across / 2, len_along / 2))
                dist = sqrt((len_along / 2) ** 2 + (len_across / 2) ** 2)
            except Exception as e:
                QMessageBox.about(self, 'Error', 'Flight design failed')
                print(e, traceback.format_exc())
                save_error()
            else:
                if self.comboBoxAltitudeType.currentText() == 'Separate Altitude ASL For Each Strip':
                    if self.tabCorridor:
                        self.startWorker_updateAltitude(pointLayer=pc_lay,
                                                        theta=theta, 
                                                        distance=dist,
                                                        crsVectorLayer=self.crs_vct,
                                                        crsRasterLayer=self.crs_rst,
                                                        DTM=self.DTM, 
                                                        altitude_AGL=altitude_AGL,
                                                        polygonLayer=photo_lay,
                                                        tabWidg=self.tabCorridor,
                                                        LineRangeList=line_buf_list)
                    else:
                        self.startWorker_updateAltitude(pointLayer=pc_lay,
                                                        theta=theta,
                                                        distance=dist,
                                                        crsVectorLayer=self.crs_vct,
                                                        crsRasterLayer=self.crs_rst,
                                                        DTM=self.DTM,
                                                        altitude_AGL=altitude_AGL,
                                                        polygonLayer=photo_lay,
                                                        tabWidg=self.tabCorridor,
                                                        Range=self.geom_AoI)
                    self.pushButtonRunDesign.setEnabled(False)
                    self.pushButtonRunControl.setEnabled(False)

                elif self.comboBoxAltitudeType.currentText() == 'Terrain Following':
                    self.startWorker_updateAltitude(pointLayer=pc_lay,
                                                    theta=theta,
                                                    distance=dist,
                                                    crsVectorLayer=self.crs_vct,
                                                    crsRasterLayer=self.crs_rst, 
                                                    raster=self.raster,
                                                    altitude_AGL=altitude_AGL,
                                                    polygonLayer=photo_lay,
                                                    tolerance = self.doubleSpinBoxTolerance.value()
                                                    )
                    self.pushButtonRunDesign.setEnabled(False)
                    self.pushButtonRunControl.setEnabled(False)

                else:
                    if hasattr(self, 'DTM'):
                        if self.crs_rst != self.crs_vct:
                            transf_vct_rst = Transformer.from_crs(self.crs_vct,
                                                                self.crs_rst,
                                                                always_xy=True)
                        feats = pc_lay.getFeatures()
                        for f in feats:
                            # projection center coordinates
                            x = f.geometry().asPoint().x()
                            y = f.geometry().asPoint().y()
                            if self.crs_rst != self.crs_vct:
                                x, y = transf_coord(transf_vct_rst, x, y)

                            altitude_ASL = f.attribute('Alt. ASL [m]')
                            terrain_height, res = self.DTM.dataProvider().sample(QgsPointXY(x, y,), 1)
                            altitude_AGL = altitude_ASL - terrain_height
                            pc_lay.startEditing()
                            pc_lay.changeAttributeValue(f.id(), 5, round(altitude_AGL, 2))
                            pc_lay.commitChanges()

                    waypoints_layer = create_waypoints(pc_lay, self.crs_vct)
                    flight_line = create_flight_line(waypoints_layer, self.crs_vct)
                    style_path = os.path.join(
                        os.path.dirname(os.path.abspath(__file__)), 'flight_line_style.qml')
                    flight_line.loadNamedStyle(style_path)
                    
                    # delete redundant fields
                    pc_lay.startEditing()
                    pc_lay.deleteAttributes([9, 10, 11])
                    pc_lay.commitChanges()
                    photo_lay.startEditing()
                    photo_lay.deleteAttributes([2, 3])
                    photo_lay.commitChanges()

                    # change layers style
                    prop = {'color': '200,200,200,30', 'color_border': '#000000',
                            'width_border': '0.2'}
                    change_layer_style(photo_lay, prop)
                    change_layer_style(pc_lay, {'size': '1.0'})
                    photo_lay.setName('photos')
                    pc_lay.setName('projection centres')

                    layers = [pc_lay, flight_line, waypoints_layer, photo_lay]
                    add_layers_to_canvas(layers, "flight_design", self.design_run_counter)
                    self.design_run_counter += 1

    @pyqtSlot()
    def on_pushButtonRunControl_clicked(self):
        """Push Button to execute all control activites."""
        proj_centres = self.mMapLayerComboBoxProjectionCentres.currentLayer()
        h_field = self.mFieldComboBoxAltControl.currentField()
        o_field = self.mFieldComboBoxOmega.currentField()
        p_field = self.mFieldComboBoxPhi.currentField()
        k_field = self.mFieldComboBoxKappa.currentField()

        attributes_exist = True
        if not hasattr(self, 'DTM'):
            QMessageBox.about(self, 'DTM needed', 'You have to load DTM layer')
            attributes_exist = False
        elif not proj_centres:
            QMessageBox.about(self, 'Error', 'You have to load Projection centres layer')
            attributes_exist = False
        elif not h_field:
            QMessageBox.about(self, 'Error', 'You have to indicate Altitude field')
            attributes_exist = False
        elif not o_field:
            QMessageBox.about(self, 'Error', 'You have to indicate Omega field')
            attributes_exist = False
        elif not p_field:
            QMessageBox.about(self, 'Error', 'You have to indicate Phi field')
            attributes_exist = False
        elif not k_field:
            QMessageBox.about(self, 'Error', 'You have to indicate Kappa field')
            attributes_exist = False

        if attributes_exist:
            try:
                threshold = self.doubleSpinBoxIterationThreshold.value()
                self.startWorker_control(pointLayer=proj_centres,
                                        hField=h_field,
                                        omegaField=o_field,
                                        phiField=p_field,
                                        kappaField=k_field,
                                        camera=self.camera,
                                        crsVectorLayer=self.crs_vct_ctrl,
                                        crsRasterLayer=self.crs_rst,
                                        DTM=self.DTM,
                                        raster = self.raster,
                                        overlap=self.checkBoxOverlapImages.isChecked(),
                                        gsd=self.checkBoxGSDmap.isChecked(),
                                        footprint=self.checkBoxFootprint.isChecked(),
                                        threshold=threshold,
                                        height_is_ASL = self.radioButtonSeaLevel.isChecked())
                # disable GUI elements to prevent thread from starting
                # a second time
                self.pushButtonRunControl.setEnabled(False)
                self.pushButtonRunDesign.setEnabled(False)
            except:
                QMessageBox.about(self, 'Error', 'Quality control failed')
                save_error()

    @pyqtSlot()
    def on_pushButtonSaveCamera_clicked(self):
        """Push Button to add camera to camera list."""
        try:
            camera_name, pressed_ok = QInputDialog.getText(self, 'Save camera',
                                                        'Enter camera name:')
            if pressed_ok:
                new_camera = Camera(camera_name,
                                    self.doubleSpinBoxFocalLength.value() / 1000,
                                    self.doubleSpinBoxSensorSize.value() / 1000000,
                                    self.spinBoxPixelsAlongTrack.value(),
                                    self.spinBoxPixelsAcrossTrack.value())
                new_camera.save()
                self.cameras.append(new_camera)
                self.comboBoxCamera.addItem(new_camera.name)
                self.comboBoxCamera.setCurrentText(self.cameras[-1].name)
        except:
            QMessageBox.about(self, 'Error', 'Saving camera failed')
            save_error()

    @pyqtSlot()
    def on_pushButtonDeleteCamera_clicked(self):
        """Push Button to delete camera from camera list."""
        try:
            camera_names = [camera.name for camera in self.cameras]
            option, pressed = QInputDialog.getItem(None, "Delete camera",
                                            "Select camera to delete:",
                                            camera_names, 0, False)
            if pressed:
                selected_camera = next(camera for camera in self.cameras if camera.name == option)
                selected_camera.delete()
                self.cameras.remove(selected_camera)
                selected_camera_index = self.comboBoxCamera.findText(selected_camera.name)
                self.comboBoxCamera.removeItem(selected_camera_index)

                if self.comboBoxCamera.currentText() in camera_names:
                    self.doubleSpinBoxFocalLength.setValue(self.cameras[self.comboBoxCamera.currentIndex()].focal_length * 1_000)
                    self.doubleSpinBoxSensorSize.setValue(self.cameras[self.comboBoxCamera.currentIndex()].sensor_size * 1_000_000)
                    self.spinBoxPixelsAlongTrack.setValue(self.cameras[self.comboBoxCamera.currentIndex()].pixels_along_track)
                    self.spinBoxPixelsAcrossTrack.setValue(self.cameras[self.comboBoxCamera.currentIndex()].pixels_across_track)
                else:
                    self.doubleSpinBoxFocalLength.setValue(self.doubleSpinBoxFocalLength.minimum())
                    self.doubleSpinBoxSensorSize.setValue(self.doubleSpinBoxSensorSize.minimum())
                    self.spinBoxPixelsAlongTrack.setValue(self.spinBoxPixelsAlongTrack.minimum())
                    self.spinBoxPixelsAcrossTrack.setValue(self.spinBoxPixelsAcrossTrack.minimum())
                    self.comboBoxCamera.addItem('No cameras listed')
        except:
            QMessageBox.about(self, 'Error', 'Deleting camera failed')
            save_error()
