# -*- coding: utf-8 -*-

"""
/***************************************************************************
 FlightCalculator
                                 A QGIS plugin
 This plugin calculates the distance, time take and estimated fuel burn for a line dataset.  
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-07-19
        copyright            : (C) 2024 by Elena Field/British Antarctic Survey
        email                : eleeld@bas.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Elena Field/British Antarctic Survey'
__date__ = '2024-07-19'
__copyright__ = '(C) 2024 by Elena Field/British Antarctic Survey'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsProcessing,
                       QgsVectorLayer,
                       QgsField,
                       QgsProject,
                       QgsProcessingAlgorithm,
                       QgsProcessingMultiStepFeedback,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingUtils,
                       QgsVectorFileWriter,
                       QgsExpression)

import processing

import time
from time import strftime

#Define the calculations for Fuel Burn and Time Estimates

def calculate_fuel_burn(value1, value2, value3):
    time_hours =  (float(value1)/float(value3))
    fuel_burn_journey = (int(value2))*time_hours
    return fuel_burn_journey


def calculate_time_taken(value1, value2):
    time_sec = ((60*60*(int(float(value1))/int(float((value2))))))
    time_convert = time.gmtime(time_sec)
    result_time = time.strftime("%H:%M",time_convert)
    return result_time

class FlightCalculatorAlgorithm(QgsProcessingAlgorithm):

    def registerExpressions():
        QgsExpression.registerFunction(calculate_time_taken)
        QgsExpression.registerFunction(calculate_fuel_burn)

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterNumber('speed_knots', 'Speed (knots)', type=QgsProcessingParameterNumber.Integer, minValue=0, defaultValue=135))
        # Route dataset
        self.addParameter(QgsProcessingParameterVectorLayer('route_layer', 'Route layer', types=[QgsProcessing.TypeVectorLine], defaultValue=None))
        self.addParameter(QgsProcessingParameterNumber('fuel_burn_lbshour', 'Fuel burn (lbs/hour)', type=QgsProcessingParameterNumber.Integer, minValue=0, defaultValue=600))
        self.addParameter(QgsProcessingParameterFeatureSink('Network_calculated', 'Output', type=QgsProcessing.TypeVectorAnyGeometry, createByDefault=True, supportsAppend=True, defaultValue=None))

    def processAlgorithm(self, parameters, context, model_feedback):
        # Use a multi-step feedback, so that individual child algorithm progress reports are adjusted for the
        # overall progress through the model
        feedback = QgsProcessingMultiStepFeedback(9, model_feedback)
        results = {}
        outputs = {}

        # Refactor fields
        alg_params = {
            'FIELDS_MAPPING': [{'alias': '','comment': '','expression': '"origin"','length': 80,'name': 'origin','precision': 0,'sub_type': 0,'type': 10,'type_name': 'text'},{'alias': '','comment': '','expression': '"destinatio"','length': 80,'name': 'destinatio','precision': 0,'sub_type': 0,'type': 10,'type_name': 'text'}],
            'INPUT': parameters['route_layer'],
            'OUTPUT': QgsProcessingUtils.generateTempFilename('fields_refactored.gpkg')
        }
        outputs['RefactorFields'] = processing.run('native:refactorfields', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(1)
        if feedback.isCanceled():
            return {}
        print("Refactored Fields:  ", outputs['RefactorFields'])

        # Calculate Distance (km)
        alg_params = {
            'FIELD_LENGTH': 6,
            'FIELD_NAME': 'dist_km',
            'FIELD_PRECISION': 3,
            'FIELD_TYPE': 0,  # Decimal (double)
            'FORMULA': '$length  / 1000',
            'INPUT': outputs['RefactorFields']['OUTPUT'],
            'OUTPUT': QgsProcessingUtils.generateTempFilename('calc_distance_km.gpkg')
        }
        outputs['CalculateDistanceKm'] = processing.run('native:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(2)
        if feedback.isCanceled():
            return {}
        print("Calculated Distance Km", outputs['CalculateDistanceKm'])

        # Calculate Distance (nm)
        alg_params = {
            'FIELD_LENGTH': 6,
            'FIELD_NAME': 'dist_nm',
            'FIELD_PRECISION': 3,
            'FIELD_TYPE': 0,  # Decimal (double)
            'FORMULA': ' $length  * 0.000539957',
            'INPUT': outputs['CalculateDistanceKm']['OUTPUT'],
            'OUTPUT': QgsProcessingUtils.generateTempFilename('distcalc_km.gpkg')
        }
        outputs['CalculateDistanceNm'] = processing.run('native:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(3)
        if feedback.isCanceled():
            return {}

        # Origin Latitude
        alg_params = {
            'FIELD_LENGTH': 20,
            'FIELD_NAME': 'o_lat',
            'FIELD_PRECISION': 6,
            'FIELD_TYPE': 0,  # Decimal (double)
            'FORMULA': 'y( start_point($geometry))',
            'INPUT': outputs['CalculateDistanceNm']['OUTPUT'],
            'OUTPUT': QgsProcessingUtils.generateTempFilename('distcalc_nm.gpkg')
        }
        outputs['OriginLatitude'] = processing.run('native:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(4)
        if feedback.isCanceled():
            return {}
        print("Calculated Distance Nm", outputs['CalculateDistanceNm'])


        # Origin Longitude
        alg_params = {
            'FIELD_LENGTH': 20,
            'FIELD_NAME': 'o_lon',
            'FIELD_PRECISION': 6,
            'FIELD_TYPE': 0,  # Decimal (double)
            'FORMULA': 'x( start_point($geometry))',
            'INPUT': outputs['OriginLatitude']['OUTPUT'],
            'OUTPUT': QgsProcessingUtils.generateTempFilename('orig_long.gpkg')
        }
        outputs['OriginLongitude'] = processing.run('native:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(5)
        if feedback.isCanceled():
            return {}

        # Destination Latitude
        alg_params = {
            'FIELD_LENGTH': 20,
            'FIELD_NAME': 'd_lat',
            'FIELD_PRECISION': 6,
            'FIELD_TYPE': 0,  # Decimal (double)
            'FORMULA': 'y( end_point($geometry))',
            'INPUT': outputs['OriginLongitude']['OUTPUT'],
            'OUTPUT': QgsProcessingUtils.generateTempFilename('orig_lat.gpkg')
        }
        outputs['DestinationLatitude'] = processing.run('native:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(6)
        if feedback.isCanceled():
            return {}

        # Destination Longitude
        alg_params = {
            'FIELD_LENGTH': 20,
            'FIELD_NAME': 'd_lon',
            'FIELD_PRECISION': 6,
            'FIELD_TYPE': 0,  # Decimal (double)
            'FORMULA': 'x(end_point($geometry))',
            'INPUT': outputs['DestinationLatitude']['OUTPUT'],
            'OUTPUT': QgsProcessingUtils.generateTempFilename('dest_lon.gpkg')
        }
        outputs['DestinationLongitude'] = processing.run('native:fieldcalculator', alg_params, context=context, feedback=feedback, is_child_algorithm=True)

        feedback.setCurrentStep(7)
        if feedback.isCanceled():
            return {}
        print("Start/End Coordinates Calculated", outputs['DestinationLatitude'])

        # Calculate time taken
                
        speed_knots = self.parameterAsDouble(parameters, 'speed_knots', context) 
        fuel_burn_lbshour = self.parameterAsDouble(parameters, 'fuel_burn_lbshour', context) 

        # Create an edit session on the final output layer and update attributes
        temp_layer = QgsProcessingUtils.mapLayerFromString(outputs['DestinationLongitude']['OUTPUT'], context)
        temp_layer_provider = temp_layer.dataProvider()

        # Add new fields for time_taken and fuel_cost
        time_taken_field = QgsField('time_taken', QVariant.String)
        fuel_cost_field = QgsField('fuel_lbs', QVariant.Int)

        temp_layer_provider.addAttributes([time_taken_field, fuel_cost_field])
        temp_layer.updateFields()

        temp_layer.startEditing()

        for feature in temp_layer.getFeatures():
            length_nm = feature['dist_nm']  # Ensure this field exists in your layer
            time_taken = calculate_time_taken(length_nm, speed_knots)
            fuel_burn = calculate_fuel_burn(length_nm, fuel_burn_lbshour, speed_knots)

            # Update the feature with the calculated values
            feature['time_taken'] = time_taken
            feature['fuel_lbs'] = fuel_burn
            temp_layer.updateFeature(feature)
        
        # Commit the changes
        temp_layer.commitChanges()

        # Add the modified layer to the results
        # Define the path for the final output shapefile
        final_output_path = QgsProcessingUtils.generateTempFilename('final_output.gpkg')

        # Save the modified temporary layer to the final output shapefile
        QgsVectorFileWriter.writeAsVectorFormat(temp_layer, final_output_path, 'utf-8', temp_layer.crs())
        results['Network_calculated'] = final_output_path

        final_layer = QgsVectorLayer(final_output_path, 'Network Calculated', 'ogr') 
        QgsProject.instance().addMapLayer(final_layer)

        return results

    def name(self):
        return 'route_calcs_working24'

    def displayName(self):
        return 'route_calcs_working24'

    def group(self):
        return ''

    def groupId(self):
        return ''

    def shortHelpString(self):
        return self.tr(
            "<p>This tool calculates the flight time and fuel burn based on input values. The result is a new file containing start/end coordinates, distances in kilometres and nautical miles, time taken (hh:mm:ss), and fuel burn</p>"
            "<h3>Speed (knots)</h3>"
            "<p>Please input the average flight speed</p>"
            "<h3>Route Layer</h3>"
            "<p>Selecte the polyline feature that you wish to calculate the values from.</p>"
            "<h3>Fuel burn</h3>"
            "<p>Please input the average fuel burn (lbs/hour)</p>" 
            "<h3>Output</h3>"
            "<p>Choose where the output is saved. If left blank, it will produce a temporary layer within your QGIS Project.</p>"
            "<h3>ToolTip</h3>"
            "<p>Ensure your project is using an appropriate projection for distance calculations (e.g. Equidistant)</p>"
            )

    
    def createInstance(self):
        return Route_calcs_working24()

    def name(self):
        
        return 'Flight Calculator'

    def displayName(self):
       
        return self.tr(self.name())

    def group(self):
        
        return self.tr(self.groupId())

    def groupId(self):
        
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return FlightCalculatorAlgorithm()
