# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FlickrdlDialog
                                 A QGIS plugin
 This plugin helps downloading metadata of geotagged Flickr photos
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2018-06-04
        git sha              : $Format:%H$
        copyright            : (C) 2018-2026 by Mátyás Gede
        email                : gedematyas@inf.elte.hu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import requests
import qgis.utils

from PyQt5 import uic
from PyQt5 import QtWidgets
from PyQt5.QtWidgets import QAction, QMessageBox, QWidget
from PyQt5.QtCore import *
from PyQt5 import QtSql
from PyQt5.QtSql import *
from collections import deque
from datetime import *

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'flickrdl_dialog_base.ui'))


class FlickrdlDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(FlickrdlDialog, self).__init__(parent)
        # Set up the user interface from Designer.
        # After setupUI you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.fwDBFile.setFilter("SQLite files (*.sqlite)")
        # event handlers
        self.pbStart.clicked.connect(self.startDlThread) # Start button
        self.pbClose.clicked.connect(self.close) # Close button
        self.pbHelp.clicked.connect(self.help) # Close button
        self.WT=None
    
    def close(self):
        """Close dialog"""
        if self.WT is not None:
            self.WT.stop()
        self.reject()
    
    def help(self):
        """Help 'dialog'"""
        QMessageBox.information(self,"Help",'This plugin requires a Flickr API key.<br/> Please obtain a key at <br/>'+
            '<a href="https://www.flickr.com/services/api/misc.api_keys.html">https://www.flickr.com/services/api/misc.api_keys.html</a>'+
            '<h3>Usage</h3>Create a Spatialite database file. Select the database file, and set the bounding latitudes/longitudes of the area to download, '+
            'then press "Start".<br/>Depending on the number of photos in the area, download may take several minutes.')
        
    def startDlThread(self):
        """Starts downloading thread"""
        
        # return to lat/lon boxes if number is not valid
        def numberError(le):
            QMessageBox.warning(self,"Error","Enter a valid number here")
            le.setFocus()
            le.selectAll()
            
        if self.pbStart.text()=="Start":
            # get values from ui
            key=self.leApiKey.text()
            if key=="tudod":
                key="ee27f5b7187c0c765d3c81f32b5488ee"
                self.leApiKey.setText(key)
                # this is the key for testing :)
            dbFile=self.fwDBFile.filePath()
            tblName=self.leTblName.text()
            # check BBox lats/lons
            N=self.leNLat.text()
            try:
                nv=float(N)
            except ValueError:
                numberError(self.leNLat)
                return
            S=self.leSLat.text()
            try:
                sv=float(S)
            except ValueError:
                numberError(self.leSLat)
                return
            W=self.leWLon.text()
            try:
                wv=float(W)
            except ValueError:
                numberError(self.leWLon)
                return
            E=self.leELon.text()
            try:
                ev=float(E)
            except ValueError:
                numberError(self.leELon)
                return
            
            # swap values if necessary
            if nv<sv:
                x=N;N=S;S=x
            if ev<wv:
                x=E;E=W;W=x
            
            initialBB=[W,S,E,N]
            # change button text to stop
            self.pbStart.setText("Stop");
            # clear log box
            self.teLog.clear()
            # create and start thread
            self.WT=WorkerThread(qgis.utils.iface.mainWindow(),key,dbFile,tblName,initialBB)
            self.WT.jobFinished.connect(self.jobFinishedFromThread)
            self.WT.addMsg.connect(self.msgFromThread)
            self.WT.setTotal.connect(self.setTotal)
            self.WT.setProgress.connect(self.setProgress)
            self.WT.start()
        else:
            # change button text to start
            self.pbStart.setText("Start")
            # stop working thread
            self.WT.stop()
            self.teLog.append("Downloading stopped")
            
    def jobFinishedFromThread( self, success ):
        self.progressBar.setValue(self.progressBar.maximum())
        self.WT.stop()

    def msgFromThread( self, msg ):
        self.teLog.append(msg)        
    
    def setTotal( self, total ):
        self.progressBar.setMaximum(int(total))
        
    def setProgress( self, p ):
        self.progressBar.setValue(p)

class WorkerThread( QThread ):
    # signals
    addMsg=pyqtSignal(str)
    jobFinished=pyqtSignal(bool)
    setTotal=pyqtSignal(str)
    setProgress=pyqtSignal(int)
       
    def __init__( self, parentThread,key,dbFile,tblName,initialBB):
        QThread.__init__( self, parentThread )
        self.key=key
        self.dbFile=dbFile
        self.tblName=tblName
        self.initialBB=initialBB
    def run( self ):
        self.running = True
        success = self.doWork()
        self.jobFinished.emit(success)
    def stop( self ):
        self.running = False
        pass
    def doWork( self ):
        """Starts download process"""
        key=self.key
        dbFile=self.dbFile
        tblName=self.tblName
        initialBB=self.initialBB
        # minimum date (2000-01-01) and current time (may be required for temporal division)
        minDate=datetime(2000,1,1)
        maxDate=datetime.now()
        # check key validity
        self.addMsg.emit('Checking connection to Flickr API...')
        url='https://api.flickr.com/services/rest/?api_key='+key+'&method=flickr.test.echo&format=json&nojsoncallback=1'
        data=requests.get(url).json()
        if data['stat']=='fail':
            self.addMsg.emit('Error: '+data['message'])
            return
        else:    
            self.addMsg.emit('Connection OK.')
        
        # connect to spatialite
        con=qgis.utils.spatialite_connect(dbFile)
        cur=con.cursor()
        
        # create table
        cur.execute("drop table if exists "+tblName) 
        self.addMsg.emit("old table dropped if there was one")
        cur.execute("create table "+tblName+" (p_id integer primary key autoincrement, lat real, lon real, o_id text, p_date text, accuracy int, title text, tags text, url text)")        
        self.addMsg.emit(tblName+" table created")
        cur.execute("select AddGeometryColumn('"+tblName+"', 'geom', 4326, 'POINT', 'XY');")
        cur.execute("select CreateSpatialIndex('"+tblName+"', 'geom');")
                  
        # fifo list of bboxes to get
        bboxes=deque()
        bb=None
        
        # escapes 's for sqlite
        def escquotes(s):
            return s.replace("'","''")
            
        # send request to Flickr and get response
        def getPage(bbx,page):
            bbox=bbx[0]+','+bbx[1]+','+bbx[2]+','+bbx[3]
            url='https://api.flickr.com/services/rest/?api_key='+key+'&method=flickr.photos.search&bbox='+bbox+'&accuracy=1&format=json&nojsoncallback=1&page='+str(page)+'&perpage=250&extras=geo%2Cdate_taken%2Ctags%2Curl_s'
            if len(bbx)==6:
                # bbox contains datetime limits
                d1=bbx[4]
                d2=bbx[5]
                url+='&min_taken_date='+d1+'&max_taken_date='+d2
            r=requests.get(url)
            if r.status_code!=200:
                return {'stat':'fail','message':'http status: '+str(r.status_code)+' '+r.reason}
            try:
                j=r.json()
                return r.json()
            except Exception as e:
                return {'stat':'fail','message':'error in response: '+r.text}
        
        # push photo data do DB
        def pushData(data):
            q="replace into "+tblName+" (p_id,lat,lon,o_id,p_date,accuracy,title,tags,url,geom) values "
            qv=''
            for p in data['photos']['photo']:
                k=p.keys()
                if p['latitude']!=0 and p['longitude']!=0 and 'id' in k and 'owner' in k and 'datetaken' in k and 'accuracy' in k and 'title' in k and 'tags' in k and 'url_s' in k:
                    if qv!='':
                        qv+=','
                    qv+='('+p['id']+','+p['latitude']+','+p['longitude']+",'"+p['owner']+"','"+p['datetaken']+"',"+p['accuracy']+",'"+escquotes(p['title'])+"','"+escquotes(p['tags'])+"','"+escquotes(p['url_s'])+"',PointFromText('point("+p['longitude']+' '+p['latitude']+")',4326))"
            if qv!='':
                cur.execute(q+qv)
                self.addMsg.emit('page '+str(pg)+' from '+str(pages)+' inserted')
                # get number of records for setting progress bar
                res=cur.execute("select count(*) from "+tblName) 
                con.commit()
                self.setProgress.emit(res.fetchone()[0])
                
        # put initial bbox into queue
        bboxes.append(initialBB)
        
        first=True;
        
        # main downloading loop
        while len(bboxes)>0:
            # exit if thread stopped
            if not self.running:
                return False
            bb=bboxes.popleft() # next bbox to download
            self.addMsg.emit("next BBox: "+str(bb))
            pp=1 # start with first page
            data=getPage(bb,pp)
            # if there is a problem...
            if data['stat']=='fail':
                self.addMsg.emit('Error: '+data['message'])
                rd-=1
                return False
            # number of pages
            pages=data['photos']['pages']
            pg=data['photos']['page']
            if first:
                first=False
                self.setTotal.emit(str(data['photos']['total']))
            if pages>16:
                # too much data, dividing bbox
                # TODO: skip data when too much photos at the same time and space (when dividing does not help)
                if (float(bb[2])-float(bb[0])>1e-4) and (float(bb[2])-float(bb[0])>1e-4):
                    # bbox big enough to divide
                    self.addMsg.emit(str(pages)+" pages, dividing...")
                    mlon=str((float(bb[0])+float(bb[2]))/2)
                    mlat=str((float(bb[1])+float(bb[3]))/2)
                    bboxes.append([bb[0],bb[1],mlon,mlat]);
                    bboxes.append([mlon,bb[1],bb[2],mlat]);
                    bboxes.append([bb[0],mlat,mlon,bb[3]]);
                    bboxes.append([mlon,mlat,bb[2],bb[3]]);
                else:
                    # bbox too small - temporal division
                    if len(bb)==6:
                        # bbox contains datetime limits
                        d1=datetime.strptime((bb[4]),'%Y-%m-%d %H:%M:%S')
                        d2=datetime.strptime((bb[5]),'%Y-%m-%d %H:%M:%S')
                    else:
                        # using global date limits
                        d1=minDate
                        d2=maxDate
                    d3=datetime.fromtimestamp((d1.timestamp()+d2.timestamp())/2)
                    d1s=d1.strftime('%Y-%m-%d %H:%M:%S')
                    d2s=d2.strftime('%Y-%m-%d %H:%M:%S')
                    d3s=d3.strftime('%Y-%m-%d %H:%M:%S')
                    bboxes.append([bb[0],bb[1],bb[2],bb[3],d1s,d3s]);
                    bboxes.append([bb[0],bb[1],bb[2],bb[3],d3s,d2s]);
                    self.addMsg.emit("temporal division: "+d1s+" - "+d3s+" - "+d2s)
                    
            else:
                # push first page
                pushData(data)
                # get and push remaining pages
                while pp<pages:
                    # exit if thread stopped
                    if not self.running:
                        return False
                    pp+=1
                    data=getPage(bb,pp)
                    while not 'photos' in data.keys():
                        self.addMsg.emit("we have problem: "+data['message']+" - retrying...")
                        data=getPage(bb,pp)
                    pages=data['photos']['pages']
                    pg=data['photos']['page']
                    pushData(data)
        # finished
        # cur.execute("SELECT CreateIsoMetadataTables();")
        # con.commit()
        self.addMsg.emit("I think it's ready...")    
        return True
        
    def cleanUp( self):
        pass