# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FlaiQgisPlugin
                                 A QGIS plugin
 Provides GUI access to Flai's Open Lidar Data Hub and local control over the Flai CLI for AI-powered geospatial classification.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-11
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Flai d.o.o
        email                : vid.tk@flai.ai
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore    import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui     import QIcon
from qgis.PyQt.QtWidgets import QAction

# initialize Qt resources from file resources.py
from .resources                   import *
# import the code for the dialog
from .flai_qgis_plugin_dockwidget import FlaiQgisPluginDockWidget

import os.path


class FlaiQgisPlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FlaiQgisPlugin{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # declare instance attributes
        self.menu_actions = []
        self.plugin_menu = self.tr(u'&Flai')
        self.main_menu = self.tr(u'&Flai')

        self.toolbar_actions = []
        self.toolbar = None # self.iface.addToolBar('Flai')

        # declare instance variables for toggleable actions
        self.pin_menu_action = None
        self.display_first_time_warning_action = None
        # self.is_menu_pinned = False  # Initially, menu is not pinned


    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Flai', message)


    def add_action(
        self,
        text,
        callback,
        icon_path='',
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=False,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.plugin_menu,
                action)

        self.menu_actions.append(action)

        return action


    def initGui(self):
        
        # build main plugin interface so that we can maniplulate it with menu entries and so on
        self.dck = FlaiQgisPluginDockWidget(iface=self.iface)
        self.iface.addDockWidget(Qt.DockWidgetArea.BottomDockWidgetArea, self.dck)

        # # 1. grab its existing font
        # font = self.dck.font()

        # # 2. bump the point-size to whatever you like
        # font.setPointSize(11)       # e.g. 14pt

        # # 3. re-apply to the dialog
        # self.dck.setFont(font)

        # # does not work, look at histroy, there should be a working example
        # self.pin_menu_action = QAction(self.tr(u'&Pin it to Menu Bar'), self.iface.mainWindow())
        # self.pin_menu_action.setCheckable(True)               # make it a checkbox
        # self.pin_menu_action.setChecked(self.is_menu_pinned)  # set initial state
        # self.pin_menu_action.triggered.connect(self.toggle_pin_menu)

        # plugin menu
        self.display_first_time_warning_action = QAction(self.tr(u'&Display welcome warning'), self.iface.mainWindow())
        self.display_first_time_warning_action.setCheckable(True)                                       # make it a checkbox
        self.display_first_time_warning_action.setChecked(not self.dck.get_show_welcome_warning_value())    # set initial state
        self.display_first_time_warning_action.triggered.connect(self.on_welcome_warning_changed)

        self.add_action(
            icon_path=":/plugins/flai/media/FLAI-sign-gradient-cropped.png",
            text=self.tr(u'&Show'),
            callback=self.run,
            parent=self.iface.mainWindow())
        
        self.add_action(
            icon_path=":/plugins/flai/media/icons8-hard-hat-48.png",
            text=self.tr(u'&Hard reset'),
            callback=self.hard_reset_ui,
            parent=self.iface.mainWindow())
        
        # add the checkbox action to both the plugin and main menus
        # self.iface.addPluginToMenu(self.plugin_menu, self.pin_menu_action)
        # self.iface.addPluginToMenu(self.main_menu, self.pin_menu_action)

        self.iface.addPluginToMenu(self.plugin_menu, self.display_first_time_warning_action)

        # toolbar
        if self.toolbar is None:
            self.toolbar = self.iface.addToolBar('Flai')
            self.toolbar.setObjectName('FlaiToolbar')

        # now add your actions
        action_show = QAction(
            QIcon(":/plugins/flai/media/FLAI-sign-gradient-cropped.png"),
            "<b>Flai - show</b>",
            self.iface.mainWindow()
        )
        action_show.triggered.connect(self.run)

        action_hard_reset = QAction(
            QIcon(":/plugins/flai/media/icons8-hard-hat-48.png"),
            "<b>Flai - hard reset</b>",
            self.iface.mainWindow()
        )
        action_hard_reset.triggered.connect(self.hard_reset_ui)

        self.toolbar.addAction(action_show)
        self.toolbar.addAction(action_hard_reset)

        self.toolbar_actions = [action_show, action_hard_reset]

        # connect the signal from the dialog to the slot to update the checkbox
        self.dck.value_changed_for_welcome_warning.connect(self.update_checkbox_state)
    

    def unload(self):
        # remove the plugin menu
        for action in self.menu_actions:
            self.iface.removePluginMenu(self.plugin_menu, action)
            self.iface.removeToolBarIcon(action)

        self.iface.removePluginMenu(self.plugin_menu, self.display_first_time_warning_action)

        # also remove the 'Pin it to Menu Bar' action
        # if self.pin_menu_action:
            # self.iface.removePluginMenu(self.plugin_menu, self.pin_menu_action)
            # self.iface.removePluginMenu(self.main_menu, self.pin_menu_action)

        # remove toolbar actions
        if self.toolbar:
            for a in self.toolbar_actions:
                try:
                    self.toolbar.removeAction(a)
                except Exception:
                    pass
                self.iface.removeToolBarIcon(a)
            self.toolbar_actions.clear()

            # remove the toolbar itself
            self.iface.mainWindow().removeToolBar(self.toolbar)
            try:
                self.toolbar.deleteLater()
            except Exception:
                pass
            self.toolbar = None


    def run(self):
        self.dck.show()


    def hard_reset_ui(self):
        self.dck.stop_threaded_processes()
        self.dck.close()

        # now safely delete the dialog reference
        del self.dck
        
        # recreate the dialog and show it
        self.dck = FlaiQgisPluginDockWidget(iface=self.iface)
        self.iface.addDockWidget(Qt.DockWidgetArea.BottomDockWidgetArea, self.dck)
        self.run()


    # def toggle_pin_menu(self):
    #     """Toggles the 'Pin to Menu Bar' action."""
    #     if self.is_menu_pinned:
    #         # remove the menu from both the main menu and the plugin menu
    #         self.iface.removePluginMenu(self.main_menu, self.pin_menu_action)
    #         self.iface.removePluginMenu(self.plugin_menu, self.pin_menu_action)
    #         self.is_menu_pinned = False
    #     else:
    #         # add the menu to both the main menu and the plugin menu
    #         self.iface.addPluginToMenu(self.main_menu, self.pin_menu_action)
    #         self.iface.addPluginToMenu(self.plugin_menu, self.pin_menu_action)
    #         self.is_menu_pinned = True

    #     # force QGIS interface to refresh
    #     self.iface.mainWindow().repaint()


    def on_welcome_warning_changed(self):
        self.dck.set_show_welcome_warning_value(not self.dck.get_show_welcome_warning_value())


    def update_checkbox_state(self, new_value):
        """Slot to update the checkbox state based on the dialog's signal."""
        self.display_first_time_warning_action.setChecked(not new_value)
