# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FlaiQgisPlugin
                                 A QGIS plugin
 Provides GUI access to Flai's Open Lidar Data Hub and local control over the Flai CLI for AI-powered geospatial classification.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-06-11
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Flai d.o.o
        email                : vid.tk@flai.ai
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os, time, re, platform, sys, importlib, threading, requests, html, concurrent.futures, json
import webbrowser   # python native lib for browser opening links, could use also QDesktopServices.openUrl(QUrl("https://www.example.com"))
from functools          import partial
from datetime           import datetime
from typing             import Tuple, Optional
from concurrent.futures import ThreadPoolExecutor, as_completed
from pathlib            import Path

from .file_selection_dialog import FileSelectionDialog
from .download_dialog       import DownloadDialog
from .constants             import *

from qgis.PyQt.QtCore    import Qt, QUrl, pyqtSlot, QTimer, QSettings, QProcess, pyqtSignal, QStandardPaths, QFile, QIODevice, QJsonDocument, QJsonParseError, QSize, QEvent, QObject
from qgis.PyQt.QtGui     import QDesktopServices, QPixmap, QFont, QColor, QFontMetrics
from qgis.PyQt.QtWidgets import QDialog, QFileDialog, QCheckBox, QVBoxLayout, QPushButton, QLabel, QMessageBox, QPlainTextEdit, QHBoxLayout, QTableWidgetItem, QWidget, QAbstractItemView, QHeaderView, QSizePolicy, QWIDGETSIZE_MAX
from qgis.PyQt           import uic, QtWidgets
from qgis.core           import QgsProject, QgsLayerTreeLayer, QgsVectorLayer, QgsRasterLayer, QgsPointCloudLayer, QgsVectorTileLayer, QgsPointCloudCategory, QgsPointCloudClassifiedRenderer, QgsCoordinateTransform, QgsCoordinateReferenceSystem, QgsLayerTreeGroup, QgsVectorTileBasicRenderer


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'flai_qgis_plugin_dockwidget_base.ui'))



class FlaiQgisPluginDockWidget(QtWidgets.QDockWidget, FORM_CLASS):
    value_changed_for_welcome_warning = pyqtSignal(bool)


    def __init__(self, iface, parent=None):
        """Constructor."""
        super().__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        self.iface = iface
        self.setupUi(self)

        # had to set global text size for QDockWidget (for QDialog it worked with designer - maybe some properites were not correctly copied when converting from QDialog to QDockWidget)
        # also when Processing Toolkil was placed over / in the same group it overwrote our font size !!!
        self.dockWidgetContents.setStyleSheet(f"QWidget {{ font-size: {TEXT_SIZE_INSIDE_MSG_BOX}pt; }}")

        # capture size from the .ui (configured through QtDesigner)
        self._floating_size_max = self.size()

        if self._floating_size_max.isEmpty():
            self._floating_size_max = self.geometry().size()      # also fine
        
        if self._floating_size_max.isEmpty():
            self._floating_size_max = self.sizeHint()             # fallback

        # constants / state for size
        self._floating_size_min = QSize(396, 204)
        self._docked_min = self.minimumSize()   # to restore on docking
        self._docked_max = QSize(QWIDGETSIZE_MAX, QWIDGETSIZE_MAX)   # to restore on docking

        # watch for state changes and double-clicks
        self.topLevelChanged.connect(self._on_top_level_changed)
        self.installEventFilter(self)
        if self.titleBarWidget():
            self.titleBarWidget().installEventFilter(self)  # needed if you use a custom title bar

        # create a new QFont for message box text and its buttons
        self._font_inside_msg_box = QFont(self.label_wasCLIFound.font())

        self.label_logoFlaiGeneral.setText('')
        self.label_logoFlaiGeneral.setPixmap(QPixmap(":/plugins/flai/media/icon.png"))      # this path has to be the same as in the resources.qrc

        self.label_logoDataHub.setText('')
        self.label_logoDataHub.setPixmap(QPixmap(":/plugins/flai/media/icon_with_DataHub.png"))      # this path has to be the same as in the resources.qrc

        # important variables
        plugin_dir = os.path.dirname(__file__)  # this file lives in .../plugins/flai-cli-interface/
        
        self._cfg_path = os.path.join(plugin_dir, HIDDEN_SETTINGS_FILE_NAME)    # create a QSettings that reads/writes program settings, to make life easier for user
        self._settings = QSettings(self._cfg_path, QSettings.IniFormat)

        # setting paths (for CLI and for DataHub)
        self._default_cli_dir     = os.path.join(plugin_dir, MAIN_SAVING_FOLDER_NAME, CLI_FOLDER_NAME)
        os.makedirs(self._default_cli_dir, exist_ok=True)
        self._set_saving_location_for_cli()

        self._default_datahub_dir = os.path.join(plugin_dir, MAIN_SAVING_FOLDER_NAME, DATAHUB_FOLDER_NAME)
        os.makedirs(self._default_datahub_dir, exist_ok=True)
        self._set_saving_location_for_datahub()

        self._current_system = platform.system()

        # make minimal flai config file so that user does not need to configure it for manually for datahub
        home = QStandardPaths.writableLocation(QStandardPaths.HomeLocation)
        self._hidden_flai_config_file_path = Path(home) / HIDDEN_FLAI_CONFIG_FILE_NAME

        if not self._hidden_flai_config_file_path.exists():
            self._hidden_flai_config_file_path.parent.mkdir(parents=True, exist_ok=True)

            # atomic-ish write: write to temp file then replace
            temp = self._hidden_flai_config_file_path.with_suffix(self._hidden_flai_config_file_path.suffix + ".tmp")
            with open(temp, "w", encoding="utf-8") as f:
                json.dump(
                    obj          = {"flai_host": "https://api.flai.ai",}, 
                    fp           = f, 
                    ensure_ascii = False, 
                    indent       = 2
                )
                f.flush()
                os.fsync(f.fileno())  # ensure data is durable
            
            # on most OSes this is atomic if on same filesystem
            os.replace(temp, self._hidden_flai_config_file_path)

        # we need to save this variables only if we are on Windows
        self._windows_flai_access_token = ''
        self._windows_flai_host         = ''
        
        self._is_show_welcome_warning_disabled = False
        self._loading_and_setting_show_welcome_warning_from_settings()

        self._flai_sdk_executor = concurrent.futures.ThreadPoolExecutor(max_workers=1)     # for async sdk calls (making a bit smoother experience); also do not use ProcessPoolExecutor - can cause problems on Windows
        self._running_flow_process = QProcess()

        self.flow_templates = None
        self._selected_flow_template_id = ''

        self._flai_semantic_schemas = {}
        self._asprs_semantic_schema_id = '12e72edc-811d-4677-8bb4-67eaf0e53fc5'  # it should be 'the generic' ASPRS schema
        self._asprs_hardcoded_schema = [{'code': '0', 'name': '[0] Never Classified', 'color': [0.3, 0.6, 0.6, 0.5]}, {'code': '1', 'name': '[1] Unclassified', 'color': [0.5, 0.5, 0.5, 1]}, {'code': '2', 'name': '[2] Ground', 'color': [0.984313725490196, 0.9019607843137255, 0.6745098039215687, 1]}, {'code': '3', 'name': '[3] Low Vegetation', 'color': [0.68235294117647, 0.83921568627451, 0.68235294117647, 1]}, {'code': '4', 'name': '[4] Medium Vegetation', 'color': [0.33725490196078, 0.64313725490196, 0.33725490196078, 1]}, {'code': '5', 'name': '[5] High Vegetation', 'color': [0.13725490196078, 0.43921568627451, 0.13725490196078, 1]}, {'code': '6', 'name': '[6] Building', 'color': [0.71764705882353, 0.003921568627451, 0.003921568627451, 1]}, {'code': '7', 'name': '[7] Low Point (Noise)', 'color': [0, 1, 0.9843137254902, 1]}, {'code': '8', 'name': '[8] Reserved', 'color': [1, 0, 0.8156862745098, 1]}, {'code': '9', 'name': '[9] Water', 'color': [0.007843137254902, 0.12941176470588, 0.73333333333333, 1]}, {'code': '10', 'name': '[10] Rail', 'color': [0.14117647058824, 0.14117647058824, 0.14117647058824, 1]}, {'code': '11', 'name': '[11] Road Surface', 'color': [0.34117647058824, 0.34117647058824, 0.34117647058824, 1]}, {'code': '12', 'name': '[12] Reserved', 'color': [0.87058823529412, 0.87058823529412, 0.87058823529412, 1]}, {'code': '13', 'name': '[13] Wire - Guard (Shield)', 'color': [0, 0.3843137254902, 1, 1]}, {'code': '14', 'name': '[14] Wire - Conductor (Phase)', 'color': [0.76470588235294, 0.30196078431373, 0.79607843137255, 1]}, {'code': '15', 'name': '[15] Transmission Tower', 'color': [0.13333333333333, 0.76470588235294, 0.8078431372549, 1]}, {'code': '16', 'name': '[16] Wire-Structure Connector', 'color': [0.6156862745098, 0.3843137254902, 0.3843137254902, 1]}, {'code': '17', 'name': '[17] Bridge Deck', 'color': [0.007843137254902, 0.89019607843137, 0.71372549019608, 1]}, {'code': '18', 'name': '[18] High Noise', 'color': [0.11764705882353, 0.086274509803922, 0.57254901960784, 1]}, {'code': '19', 'name': '[19] Overhead Structure', 'color': [0.78823529411765, 0.78823529411765, 0.78823529411765, 1]}, {'code': '20', 'name': '[20] Ignored Ground', 'color': [0.72156862745098, 0.41960784313725, 0, 1]}, {'code': '21', 'name': '[21] Snow', 'color': [0.88235294117647, 0.27843137254902, 0.27843137254902, 1]}, {'code': '22', 'name': '[22] Temporal Exclusion', 'color': [0.20392156862745, 0.34117647058824, 0, 1]}]

        self._original_readers, self._original_writers = [], []           # data types
        self._output_manager_paths, self._input_manager_paths = {}, {}    # dictionary paths
        
        self._first_time_showing_automatically_generated_output_paths = True

        self._temp_cli_output_folder = ""

        self._processing_start_time = None
        self._output_paths = {}

        self._whitespace_regex = re.compile(r"\s")
        self._md_link_regex = re.compile(r"\[([^\]]+)\]\(([^)]+)\)")

        self._default_input_pc_semantic_schema = []
        self._output_pc_semantic_schema = []

        self._datahub_in_qgis = None
        self._canvas = self.iface.mapCanvas()

        self._dot_index_for_dataset_loader_info_timer = 0
        self._dataset_loader_info_timer = QTimer(self)
        self._dataset_loader_info_timer.setInterval(1000)  # 1 second
        self._dataset_loader_info_timer.timeout.connect(self._animate_label_dataset_loader_info)

        # hidden variables, only modified by setters and getters
        self._is_flow_template_selected = False  # checks for button that shows command  
        self._are_output_paths_ok = False        # all have to be True for it to be enabled
        self._are_input_paths_ok = False
        self._are_measurement_units_ok = False

        # do not forget about automatic scaling !!!!!
        # make program automagically remember last location when user loads it, save it as .flai_cli_interface_settings
        # warning about closing the gui but not killing the running process, it will still run in background (only button Reset and entry from dropdown Hard reset will kill running flow)

        self._was_cli_loaded_on_first_show_menu_button_click = False

        self._vt_uri = 'type=xyz&url=https://api.flai.ai/public/vectors/open-data/tiles/{z}/{x}/{y}.pbf'
        self._vt_uri = 'type=xyz&url=https://api.flai.ai/public/vectors/open-data/tiles/{z}/{x}/{y}.pbf?public=1'
        self._osm_uri = 'type=xyz&url=https://tile.openstreetmap.org/{z}/{x}/{y}.png&zmax=19&zmin=0'

        self._datahub_query_pool = concurrent.futures.ThreadPoolExecutor(max_workers=1)
        self._datahub_poller = QTimer(self)
        self._datahub_poller.setInterval(100)
        self._datahub_poller.timeout.connect(self._check_region_future)

        # setting up "Settings" tab, this vars do not need to be in `reset_ui_and_variables(...)` function
        self.pshBtn_resetUI.setText('Reset UI')
        self.pshBtn_findFile.clicked.connect(self._select_cli)
        self.pshBtn_resetUI.clicked.connect(self._reset_ui_and_variables)
        self.pshBtn_resetCLIPath.clicked.connect(self._reset_cli_path)
        self.pshBtn_downloadFile.clicked.connect(self._get_cli_link)
        self.pshBtn_openCLIOutputFolder.clicked.connect(
            lambda: QDesktopServices.openUrl(
                QUrl.fromLocalFile(self._default_cli_dir + os.sep + CLI_PROCESSED_DATA_FOLDER_NAME)
            )
        )
        self.pshBtn_changeCLIOutputFolder.clicked.connect(self._updated_saving_location_for_cli)
        self.pshBtn_changeDataHubFolder.clicked.connect(self._updated_saving_location_for_datahub)
        self.pshBtn_updateSDK.clicked.connect(
            lambda: self._flai_sdk_installer(reinstalling_package=True)
        )
        self.label_disabledTabInformation.show()    # redundant but ok

        # setting up "Processing engine" tab, this vars do not need to be in `reset_ui_and_variables(...)` function
        self._reset_flow_template_combobox()
        self.cmbBx_flowTemplate.currentIndexChanged.connect(self._on_flow_template_changed)
        self._on_flow_template_changed(self.cmbBx_flowTemplate.currentIndex())

        self.pshBtn_createOrEdit.clicked.connect(self._on_create_or_edit_flow_template)
        
        self.pshBtn_selectInput.clicked.connect(
            partial(self._get_file_paths, self.pshBtn_selectInput)
        )
        self.pshBtn_selectOutput.clicked.connect(
            partial(self._get_file_paths, self.pshBtn_selectOutput)
        )

        self.cmbBx_measurementUnits.addItems([
            "",                 # default at index 0
            "meter",
            "foot",
            "us-foot",
        ])
        self.cmbBx_measurementUnits.currentIndexChanged.connect(self._on_measurement_unit_changed)
        self._on_measurement_unit_changed(self.cmbBx_measurementUnits.currentIndex())

        self.pshBtn_selectTmpFolder.clicked.connect(self._updated_tmp_location_for_cli)
        
        self.cmbBx_dataDisplayingBehaviour.addItems([
            DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_NON,
            DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH,
            DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT,   # default at index 2
            # DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_REALTIME               # disabled for now, problematic because file listener can be faster than CLI can finish creating an output file (there can be missing data and file will not load correctly)
        ])
        self.cmbBx_dataDisplayingBehaviour.currentIndexChanged.connect(self._on_displaying_input_output_behaviour_changed)
        self._set_displaying_input_output_behaviour()

        self.pshBtn_showCommand.clicked.connect(self._show_executable_command)
        self.pshBtn_runFlow.clicked.connect(self._run_flow)

        self.warningLabel_flowTemplate.setStyleSheet("color: #D1AE00;") # #F0D25A
        self.warningLabel_inputFolder.setStyleSheet("color: #D1AE00;")
        self.warningLabel_outputFolder.setStyleSheet("color: #D1AE00;")
        self.warningLabel_measurementUnits.setStyleSheet("color: #D1AE00;")
        self.warningLabel_tmpFolder.setStyleSheet("color: #D1AE00;")

        # setting up "DataHub" tab, this vars do not need to be in `reset_ui_and_variables(...)` function
        self.pshBtn_showAllAvailableRegions.clicked.connect(self._show_available_datahub_regions)
        self.pshBtn_searchCurrentView.clicked.connect(self._get_datasets_in_current_region)
        self.pshBtn_downloadCheckedDatasets.clicked.connect(self._on_downloadCheckedDatasets_pressed)
        self.pshBtn_preloadDataHubSavedDatasets.clicked.connect(self._load_preloaded_datasets_from_datahub)
        self.pshBtn_openDataHubFolder.clicked.connect(
            lambda: QDesktopServices.openUrl(
                QUrl.fromLocalFile(self._default_datahub_dir)
            )
        )
        self.checkBox_addOSM.stateChanged.connect(self._checkbox_add_osm_changed)
        self._set_display_osm_for_datahub()
        self.label_datasetGatheringInfo.hide()

        # self.pshBtn_drawPolygon.clicked.connect(self.on_draw_polygon_toggled)

        self._reset_ui_and_variables()
        
        # hardcoded view is set to Settings tab, will be overwritten with function self._choose_focus_tab(...)
        self.tabWidget.setCurrentIndex(
            self.tabWidget.indexOf(self.tab_settings)
        )
        self.tabWidget_processing_engine.setCurrentIndex(
            self.tabWidget_processing_engine.indexOf(self.tab_parameters_in_processing_engine)
        )


    def _effective_max(self):
        # never let detached windows size be smaller than the min.
        return QSize(
            max(self._floating_size_max.width(),  self._floating_size_min.width()),
            max(self._floating_size_max.height(), self._floating_size_min.height()),
        )


    def _on_top_level_changed(self, floating: bool):
        # apply constraints right away when we become floating
        if floating:
            self.setMinimumSize(self._floating_size_min)
            self.setMaximumSize(self._effective_max())
        
        # restore unconstrained when docked
        else:
            self.setMaximumSize(QSize(QWIDGETSIZE_MAX, QWIDGETSIZE_MAX))
            self.setMinimumSize(self._docked_min)


    def eventFilter(self, obj, ev):
        tbw = self.titleBarWidget()
        is_self_or_title = (obj is self) or (tbw is not None and obj is tbw)
        et = ev.type()

        # double-click on native/custom title bar -> let Qt toggle, then maximize if floating
        if is_self_or_title and et in (
            QEvent.NonClientAreaMouseButtonDblClick,  # native title bar
            QEvent.MouseButtonDblClick,               # custom title bar widget
        ):
            QTimer.singleShot(0, self._maximize_if_floating_with_retries)
            return False  # don't eat it; default undock/redock should still run

        return super().eventFilter(obj, ev)


    def _maximize_if_floating_with_retries(self, tries: int = 8):
        """if floating, grow to the configured max; retry briefly if WM resists."""
        if self.isFloating() and self.isVisible():
            min_sz = self._floating_size_min
            max_sz = self._effective_max()

            # apply constraints to both QWidget and (if present) the native window.
            self.setMinimumSize(min_sz)
            self.setMaximumSize(max_sz)
            wh = self.windowHandle()
            if wh:
                wh.setMinimumSize(min_sz)
                wh.setMaximumSize(max_sz)

            # try to snap to max.
            self.resize(max_sz)

            # if it didn't stick (common right after undock), try again shortly.
            if self.size() != max_sz and tries > 0:
                QTimer.singleShot(100, lambda: self._maximize_if_floating_with_retries(tries - 1))
        elif tries > 0:
            # we are likely still in the toggle; wait a bit and check again.
            QTimer.singleShot(100, lambda: self._maximize_if_floating_with_retries(tries - 1))


    def _on_install_clicked(self, reinstalling_package = False):
        # swap UI into installing mode
        if not reinstalling_package:
            self._promptLabel.setText("Installing flai-sdk, please wait... It might take some time...")
        
        self._installBtn.setEnabled(False)
        self._cancelBtn.setEnabled(False)
        self._installer_log.show()

        # clear any stale caches so import will work after install
        importlib.invalidate_caches()

        # launch pip
        base_args = ["-m", "pip", "install", "flai-sdk"]              # production PyPI
        # base_args = ["-m", "pip", "install", "-e", "/home/vtk/Flai/code/flai-sdk"] # idk what is happening anymore with python, cursed !!!!!!
        # base_args = ["-m", "pip", "install", "--index-url", "https://test.pypi.org/simple/", "--extra-index-url",
                    #  "https://pypi.org/simple", "flai-sdk==0.0.3"]  # testing PyPI

        # we look for keyword "install" and insert after it flag for upgrade
        if reinstalling_package:
            try:
                idx = base_args.index("install")
            except ValueError:
                pass  # no "install" in args, nothing to do
            else:
                # desired flags in order
                extra_flags = [
                    "--upgrade",                        # upgrade to latest flai-sdk package
                    "--upgrade-strategy", "eager",      # upgrade all dependencies of the package
                    "--no-cache-dir",                   # dont use pip cache
                    "--force-reinstall",                # overwrite old version of the package(s)
                ]

                # insert in reverse so the final order after "install" is as in `flags`
                for flag in reversed(extra_flags):
                    # only insert if not already present right after install (or further down)
                    if flag not in base_args[idx + 1: idx + 1 + len(extra_flags)]:
                        base_args.insert(idx + 1, flag)

        # use correct QGIS's python path based on user's OS
        if self._current_system == SYSTEM_WINDOWS:
            # find real python.exe (or pip.exe wrapper)…
            program = os.path.join(os.path.dirname(sys.executable), "python.exe")

        elif self._current_system == SYSTEM_MACOS:
            # path to QGIS's internal Python interpreter
            program = sys.executable
            # convert the launcher path to the embedded python3 path(s)
            contents = program.rsplit("/Contents/MacOS/QGIS", 1)[0] + "/Contents"
            
            for p in (
                os.path.join(contents, "MacOS/bin/python3"),
                os.path.join(contents, "Frameworks/Python.framework/Versions/Current/bin/python3"),
            ):
                if os.path.exists(p):
                    program = p
                    break

        else:
            program = sys.executable

        self._installer_log.appendHtml(f"<b>Running command</b>: {program} {' '.join(base_args)}<br>")

        self._running_flow_process.readyReadStandardOutput.connect(self._installer_log_append_stdout)
        self._running_flow_process.readyReadStandardError.connect(self._installer_log_append_stderr)
        self._running_flow_process.finished.connect(self._on_install_finished)
        self._running_flow_process.start(program, base_args)


    def _installer_log_append_stdout(self):
        out = bytes(self._running_flow_process.readAllStandardOutput()).decode()
        # collapse multiple newlines into one
        cleaned = re.sub(r'\n+', '\n', out).rstrip('\n')
        self._installer_log.appendPlainText(cleaned)


    def _installer_log_append_stderr(self):
        err = bytes(self._running_flow_process.readAllStandardError()).decode()
        cleaned = re.sub(r'\n+', '\n', err).rstrip('\n')
        self._installer_log.appendHtml(f'<span style="color:red;">{cleaned}</span>')


    def _on_install_finished(self, exit_code, exit_status):
        if exit_code == 0:
            self._promptLabel.setText("flai-sdk installed successfully. No need to restart QGIS.")
        else:
            self._promptLabel.setText(
                f"Installation failed (exit code {exit_code}). "
            )
        # repurpose Cancel into a Close button now that we are done
        self._cancelBtn.setEnabled(True)
        self._cancelBtn.setText("Close")

        # finally, if success, load the SDK so the rest of your plugin can run
        if exit_code == 0:
            self._load_flai_sdk()


    def _load_flai_sdk(self):
        # if you just installed it, clear import caches
        importlib.invalidate_caches()

        # imports for sdk
        from flai_sdk.api.organizations               import FlaiOrganization
        from flai_sdk.api.flow_templates              import FlaiFlowTemplates
        from flai_sdk.api.flows                       import FlaiFlows
        from flai_sdk.models.flow_templates           import CreateLocalFlow
        from flai_sdk.api.semantic_definition_schemas import SemanticDefinitionSchemasModel
        from flai_sdk.api.ai_models                   import FlaiAiModel
        from flai_sdk.api.opendata                    import FlaiOpenData

        # stash them as attributes
        self.FlaiOrganization               = FlaiOrganization
        self.FlaiFlowTemplates              = FlaiFlowTemplates
        self.FlaiFlows                      = FlaiFlows
        self.CreateLocalFlow                = CreateLocalFlow
        self.SemanticDefinitionSchemasModel = SemanticDefinitionSchemasModel
        self.FlaiAiModel                    = FlaiAiModel
        self.FlaiOpenData                   = FlaiOpenData

    
    def show(self):
        """Override the show method to display a message box before showing the dialog."""
        # show a message box before showing the dialog
        if not self._is_show_welcome_warning_disabled:
            # create a custom message box (QDialog)
            msg_box = QDialog(self)
            msg_box.setWindowTitle("Important information about usage")

            # create a vertical layout to hold the message and checkbox
            layout = QVBoxLayout()

            # set label for the message
            message = \
            """
            <p>In order for plugin to work as expected, it needs connection to the internet
            to gather data from our servers (to get data for flows).</p>

            <p>For more information about plugin usage and know-how see:
            <a href="https://www.flai.ai/post/flai-in-qgis-open-data-and-ai-analysis-one-click-away">
            this post on flai.ai</a>.</p>
            """

            message_label = QLabel(message, msg_box)
            message_label.setTextFormat(Qt.RichText)                          # treat as HTML
            message_label.setTextInteractionFlags(Qt.TextBrowserInteraction)  # allow clicking & selection
            message_label.setOpenExternalLinks(True)                          # open in default browser
            message_label.setWordWrap(True)
            layout.addWidget(message_label)

            # create the checkbox
            self.checkbox = QCheckBox("Do not show again", msg_box)
            self.checkbox.setChecked(self._is_show_welcome_warning_disabled)
            layout.addWidget(self.checkbox)

            # create OK button
            ok_button = QPushButton("OK", msg_box)
            ok_button.clicked.connect(msg_box.accept)  # Accept the dialog on OK button click
            layout.addWidget(ok_button)

            msg_box.setLayout(layout)

            # unify font
            msg_box.setFont(self._font_inside_msg_box)

            result = msg_box.exec_()

            if result != QDialog.Accepted:
                return

            # after the dialog is closed, we can check the state of the checkbox
            checkbox_value = self.checkbox.isChecked()

            if self._is_show_welcome_warning_disabled != checkbox_value:
                self._settings.setValue(HIDDEN_SETTING_FIELD_SHOW_WELCOME_WARNING, int(checkbox_value))
                self._is_show_welcome_warning_disabled = checkbox_value

        # install flai-sdk if missing
        try:
            self._load_flai_sdk()
        except ImportError:
            is_install_successful = self._flai_sdk_installer()

            if not is_install_successful:
                return

        # call the base class's show method to actually display the dialog
        super().show()
        self._choose_focus_tab()


    def _flai_sdk_installer(self, reinstalling_package = False):
            # build the one-and-only dialog
            self._installDialog = QDialog(self)

            if not reinstalling_package:
                self._installDialog.setWindowTitle("Flai's plugin - missing dependency")
            else:
                self._installDialog.setWindowTitle("Flai's plugin - FLAI SDK managment")

            # make it bigger up front
            self._installDialog.resize(600, 250)

            # apply font-size to everything in one go
            self._installDialog.setStyleSheet(
                f"* {{ font-size: {TEXT_SIZE_INSIDE_MSG_BOX}pt; }}"
            )
            layout = QVBoxLayout(self._installDialog)

            # prompt label
            if not reinstalling_package:
                self._promptLabel = QLabel(
                    "The Python package <b>flai-sdk</b> is not installed. If you get error even after successful installation, go to \"Getting started with CLI\" tab and there click on button \"Update flai-sdk package\".",
                    self._installDialog
                )
            else:
                self._promptLabel = QLabel(
                    "The Python package <b>flai-sdk</b> will be updated...",
                    self._installDialog
                )

            self._installDialog.setWindowFlags(
                  Qt.Dialog
                | Qt.CustomizeWindowHint
                | Qt.WindowTitleHint
            )

            layout.addWidget(self._promptLabel)

            # buttons
            btn_layout = QHBoxLayout()
            if not reinstalling_package:
                self._installBtn = QPushButton("Install flai-sdk", self._installDialog)
            else:
                self._installBtn = QPushButton("Reinstalling flai-sdk...", self._installDialog)

            self._cancelBtn  = QPushButton("Close plugin",   self._installDialog)
            btn_layout.addWidget(self._installBtn)
            btn_layout.addWidget(self._cancelBtn)
            layout.addLayout(btn_layout)

            # hidden log console
            self._installer_log = QPlainTextEdit(self._installDialog)
            self._installer_log.setReadOnly(True)
            self._installer_log.hide()
            layout.addWidget(self._installer_log)

            # connections
            if not reinstalling_package:
                self._installBtn.clicked.connect(self._on_install_clicked)
            else:
                self._on_install_clicked(reinstalling_package=True)

            self._cancelBtn.clicked.connect(lambda: (setattr(self._installDialog, "_cancel_clicked", True) or self._installDialog.reject()))
            self._installDialog.exec_()

            # super nasty solution
            # should not show our dialog if our sdk was not installed and user clicked `Close plugin` button
            try:
                import flai_sdk
            except ImportError:
                if self._installDialog._cancel_clicked:
                    return False
            
            return True

    
    def stop_threaded_processes(self):          # somehow this does not work correctly, fix it
        return
    
        # stop the QProcess + its entire tree
        if hasattr(self, "_running_flow_process"):
            proc = self._running_flow_process
            if proc.state() != QProcess.NotRunning:
                try:
                    # first, try the graceful Qt shutdown
                    proc.terminate()
                    proc.waitForFinished(2000)
                finally:
                    # then forcibly kill everything
                    pid = int(proc.processId() or proc.pid())
                    parent = psutil.Process(pid)
                    for child in parent.children(recursive=True):
                        child.kill()
                    parent.kill()

            self._running_flow_process = QProcess()

        # then your ThreadPoolExecutor shutdown as before
        if hasattr(self, "_flai_sdk_executor"):
            try:
                self._flai_sdk_executor.shutdown(
                    wait=False,
                    cancel_futures=True
                )
            except TypeError:
                self._flai_sdk_executor.shutdown(wait=False)
            # note: this will not forcibly kill already running threads;
            # tasks themselves must cooperate if you need immediate cancellation.

            self._flai_sdk_executor = concurrent.futures.ThreadPoolExecutor(max_workers=1)


    def show_generic_sdk_warning(self, 
                                 message: str,
                                 title: str = "flai-sdk encountered a problem"
                                 ) -> QMessageBox.StandardButton:
        box = QMessageBox(self)                     # parent=this dialog
        box.setIcon(QMessageBox.Warning)            # yellow warning icon
        box.setWindowTitle(title)
        box.setText(message)
        box.setStandardButtons(QMessageBox.Ok)
        box.setDefaultButton(QMessageBox.Ok)
        box.setWindowModality(Qt.WindowModal)       # only block this dialog
        return box.exec_()


    def _choose_focus_tab(self):

        # when user already set cli path we load plugin config auto
        if self._settings.contains(HIDDEN_SETTING_FIELD_CLI_PATH):

            # this will happen only for the first time select CLI was successfully set
            if not self._was_cli_loaded_on_first_show_menu_button_click:
                self._select_cli()

                self.tabWidget.setCurrentIndex(
                    self.tabWidget.indexOf(self.tab_processingEngine)
                )

            # on other occurrences we check whether user wants to have processing or get started tab in focus
            # here we assume user is not interested in datahub tab
            else:
                # if processing is disabled, show getting starter tab
                if not self.tab_processingEngine.isEnabled():
                    self.tabWidget.setCurrentIndex(
                        self.tabWidget.indexOf(self.tab_settings)
                    )
                # otherwise processing tab
                else:
                    self.tabWidget.setCurrentIndex(
                        self.tabWidget.indexOf(self.tab_processingEngine)
                    )
        
        # user is only interested in datahub
        else:
            self.tabWidget.setCurrentIndex(
                self.tabWidget.indexOf(self.tab_datahub)
            )


    def _reset_cli_path(self):
        self._settings.remove(HIDDEN_SETTING_FIELD_CLI_PATH)
        self._settings.sync();              # forces an immediate write
        self._reset_ui_and_variables()


    def _reset_ui_and_variables(self):
        self._cli_path    = ''
        self._cli_version = ''
        self._org_name    = ''

        self._output_manager_paths, self._input_manager_paths = {}, {}
        self._original_readers, self._original_writers = [], []
        
        self._first_time_showing_automatically_generated_output_paths = True
        
        self.is_flow_template_selected = False
        self.are_output_paths_ok = False
        self.are_input_paths_ok = False
        self.are_measurement_units_ok = False

        self.label_belowTab.setText('')
        self.label_flowSummary.clear()

        # setting up "Settings" tab
        self.label_wasCLIFound.setText(
            '<strong>Flai CLI not found.</strong><br>' +
            'The Processing tab is disabled because the Flai CLI is not installed or configured on this system.<br>' +
            'Please install the Flai CLI and then click the button below to locate its executable.'
        )
        self.label_aboutCLIVersion.setText('')
        self.label_aboutYourOrganization.setText('')

        self.pshBtn_findFile.setEnabled(True)           # this two might be set to true if user has successfully activated CLI before clicking "reset" button
        self.pshBtn_downloadFile.setEnabled(True)

        # setting up "Processing engine" tab
        self.tab_processingEngine.setEnabled(False)     # until user is not loged in, this will be disabled
        
        self.cmbBx_flowTemplate.setCurrentIndex(0)      # setting default value for flow template
        self.pshBtn_createOrEdit.setText("Create")      # and setting correct string to button next to flow template GUI object
        
        self.cmbBx_measurementUnits.setCurrentIndex(0)  # setting default value for mesurement units
        
        if self._settings.contains(HIDDEN_SETTING_FIELD_MEASUREMENTS):  # preload measurement unit from settings file
            saved_measurement_unit = self._settings.value(HIDDEN_SETTING_FIELD_MEASUREMENTS)

            idx = self.cmbBx_measurementUnits.findText(
                saved_measurement_unit, 
                Qt.MatchExactly
            )

            # if we found it set it as current index
            if idx != -1:
                self.cmbBx_measurementUnits.setCurrentIndex(idx)
                self.are_measurement_units_ok = True

        self.cmbBx_dataDisplayingBehaviour.setEnabled(False)
        self._set_displaying_input_output_behaviour()

        self._clean_variables_of_processing_tab()
        self._disable_buttons_and_clean_warnings_on_processing_tab()
        self.label_disabledTabInformation.show()    # redundant but ok

        self.pshBtn_downloadCheckedDatasets.setEnabled(False)

        # self.stop_threaded_processes()


    def _set_displaying_input_output_behaviour(self):

        if self._settings.contains(HIDDEN_SETTING_FIELD_DISPLAYING_INPUT_AND_OUTPUT_BEHAVIOR):
            saved_displaying_input_and_output_behavior = self._settings.value(HIDDEN_SETTING_FIELD_DISPLAYING_INPUT_AND_OUTPUT_BEHAVIOR)
            
            if saved_displaying_input_and_output_behavior == DATA_DISPLAYING_BEHAVIOUR_NON:
                saving_type = DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_NON

            elif saved_displaying_input_and_output_behavior == DATA_DISPLAYING_BEHAVIOUR_ON_FINISH:
                saving_type = DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH

            elif saved_displaying_input_and_output_behavior == DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT:
                saving_type = DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT

            elif saved_displaying_input_and_output_behavior == DATA_DISPLAYING_BEHAVIOUR_REALTIME:
                saving_type = DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_REALTIME

            idx = self.cmbBx_dataDisplayingBehaviour.findText(
                saving_type, 
                Qt.MatchExactly
            )

            # if we found it set it as current index
            if idx != -1:
                self.cmbBx_dataDisplayingBehaviour.setCurrentIndex(idx)

        else:
            self.cmbBx_dataDisplayingBehaviour.setCurrentIndex(2)
            self._on_displaying_input_output_behaviour_changed()


    def _on_displaying_input_output_behaviour_changed(self):
        cmbox_text = self.cmbBx_dataDisplayingBehaviour.currentText()

        if cmbox_text == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_NON:
            saving_type = DATA_DISPLAYING_BEHAVIOUR_NON

        elif cmbox_text == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH:
            saving_type = DATA_DISPLAYING_BEHAVIOUR_ON_FINISH

        elif cmbox_text == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT:
            saving_type = DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT

        elif cmbox_text == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_REALTIME:
            saving_type = DATA_DISPLAYING_BEHAVIOUR_REALTIME

        self._settings.setValue(HIDDEN_SETTING_FIELD_DISPLAYING_INPUT_AND_OUTPUT_BEHAVIOR, saving_type)


    def _reset_flow_template_combobox(self):
        self.cmbBx_flowTemplate.clear()                       # nuke all items
        self.cmbBx_flowTemplate.addItem("Non selected")       # back to default
        self.cmbBx_flowTemplate.setCurrentIndex(0)            # make sure it is selected
        self.flow_templates_cmbBx_mapping = {
            0: {'title': self.cmbBx_flowTemplate.currentText(), 'id': 'null'}
        }


    # this might not be the best solution, but it will do for now
    @property
    def is_flow_template_selected(self): 
        return self._is_flow_template_selected

    @is_flow_template_selected.setter
    def is_flow_template_selected(self, val: bool):
        self._is_flow_template_selected = bool(val)
        self._update_show_command_button()


    @property
    def are_output_paths_ok(self): 
        return self._are_output_paths_ok

    @are_output_paths_ok.setter
    def are_output_paths_ok(self, val: bool):
        self._are_output_paths_ok = bool(val)
        self._update_show_command_button()


    @property
    def are_input_paths_ok(self): 
        return self._are_input_paths_ok

    @are_input_paths_ok.setter
    def are_input_paths_ok(self, val: bool):
        self._are_input_paths_ok = bool(val)
        self._update_show_command_button()


    @property
    def are_measurement_units_ok(self): 
        return self._are_measurement_units_ok

    @are_measurement_units_ok.setter
    def are_measurement_units_ok(self, val: bool):
        self._are_measurement_units_ok = bool(val)
        self._update_show_command_button()



###################
### DATAHUB tab ###
###################

    def _animate_label_dataset_loader_info(self):
        # cycles: "", ".", "..", "..."
        dots = "." * (self._dot_index_for_dataset_loader_info_timer % 4)
        self.label_datasetGatheringInfo.setText(f"Gathering data, please wait{dots}")
        self._dot_index_for_dataset_loader_info_timer += 1


    def _set_display_osm_for_datahub(self):

        if self._settings.contains(HIDDEN_SETTING_FIELD_DISPLAY_OSM_FOR_DATAHUB):
            status = self._settings.value(HIDDEN_SETTING_FIELD_DISPLAY_OSM_FOR_DATAHUB)
            self.checkBox_addOSM.setChecked(int(status))

        else:
            self._checkbox_add_osm_changed()


    def _checkbox_add_osm_changed(self):
        checkbox_status = self.checkBox_addOSM.isChecked()

        if int(STATE_DISPLAY_OSM_FOR_DATAHUB_ON) == checkbox_status:
            self._settings.setValue(HIDDEN_SETTING_FIELD_DISPLAY_OSM_FOR_DATAHUB, int(STATE_DISPLAY_OSM_FOR_DATAHUB_ON))
        elif int(STATE_DISPLAY_OSM_FOR_DATAHUB_OFF) == checkbox_status:
            self._settings.setValue(HIDDEN_SETTING_FIELD_DISPLAY_OSM_FOR_DATAHUB, int(STATE_DISPLAY_OSM_FOR_DATAHUB_OFF))


    def _human_readable(self, num_bytes: int) -> str:
        if num_bytes < 0:   # negative value
            return "?"
        
        num_bytes = float(num_bytes)
        for unit in ("B", "KiB", "MiB", "GiB", "TiB", "PiB"):
            if num_bytes < 1024 or unit == "PiB":
                dec = 0 if unit == "B" else 2        # floating point number 
                return f"{num_bytes:.{dec}f} {unit}"
            num_bytes /= 1024


    def _fetch_size(self, url: str, timeout: float = 10.0) -> Tuple[str, Optional[int], Optional[str]]:

        """
        Returns (url, size_in_bytes or None if unknown, error_message or None).
        """
        try:
            # try HEAD first
            resp = requests.head(url, allow_redirects=True, timeout=timeout)
            cl = resp.headers.get("Content-Length")
            if cl is not None:
                return url, int(cl), None

            # fallback: ask for first byte to get Content-Range
            resp2 = requests.get(url, headers={"Range": "bytes=0-0"}, allow_redirects=True, timeout=timeout)
            content_range = resp2.headers.get("Content-Range")  # e.g., "bytes 0-0/12345678"
            if content_range and "/" in content_range:
                total_size = int(content_range.split("/")[-1])
                return url, total_size, None

            # some servers may respond with size in an alternate way, give up gracefully
            return url, None, "no size header"
        except Exception as e:
            return url, None, str(e)


    def _total_download_size(self, urls: list[str], max_workers: int = 8) -> tuple[int, list[tuple[str, str]]]:
        """
        Returns (total_bytes, list_of_missing[(url, reason)])
        """
        total = 0
        missing = []

        with ThreadPoolExecutor(max_workers=max_workers) as exe:
            futures = {exe.submit(self._fetch_size, u): u for u in urls}
            for fut in as_completed(futures):
                url, size, err = fut.result()
                if size is None:
                    missing.append((url, err or "unknown"))
                else:
                    total += size

        return total, missing


    def _link_md_to_html_single_line(self, md: str) -> str:
        # normalize to one line first
        md = " ".join(md.split())

        parts = []
        last_idx = 0

        for m in self._md_link_regex.finditer(md):
            parts.append(html.escape(md[last_idx:m.start()]))
            text = html.escape(m.group(1))
            url = html.escape(m.group(2), quote=True)
            parts.append(f'<a href="{url}">{text}</a>')
            last_idx = m.end()

        parts.append(html.escape(md[last_idx:]))
        html_with_links = "".join(parts)

        # nowrap to keep it as single line; overflow will show as needed
        return f"<div style='margin:0;padding:0;text-align:center;white-space:nowrap;overflow:hidden;'>{html_with_links}</div>"


    def _parse_semantic_schema(self, schema: dict, fields_to_keep=[SEMANTIC_DEFINITION_VALUE_FIELD, SEMANTIC_DEFINITION_LABEL_FIELD, SEMANTIC_DEFINITION_COLOR_FIELD]):
        final_variable = []

        # filter out desired information
        for semantic_definition in schema.values():
            tmp = {}
            
            for field in fields_to_keep:
                tmp[field] = semantic_definition[field]

            final_variable.append(tmp)

        for i in range(len(final_variable)):
            final_variable[i][SEMANTIC_DEFINITION_LABEL_FIELD] = f'[{final_variable[i][SEMANTIC_DEFINITION_VALUE_FIELD]}] {final_variable[i][SEMANTIC_DEFINITION_LABEL_FIELD]}'

        # fiter out non-numeric and sort ascending
        # we get rid of the "DEFAULT" semantic_definition
        numeric_only = [
            item for item in final_variable
            if isinstance(item.get("code"), (int))
                or (isinstance(item.get("code"), str) and item["code"].isdigit())
        ]
        numeric_only.sort(key=lambda it: int(it["code"]))

        return numeric_only


    def _get_datasets_in_current_region(self):
        if self.checkBox_addOSM.isChecked():
            self._datahub_in_qgis = self._get_or_create_group(QgsProject.instance().layerTreeRoot(), 'Flai\'s DataHub')

            existing = self._find_layer_node_in_group_by_source(self._datahub_in_qgis, self._osm_uri)
        
            if not existing:
                osm_layer = QgsRasterLayer(self._osm_uri, 'OpenStreetMap', 'wms')
                
                if osm_layer.isValid():
                    self._insert_layer_node(osm_layer, self._datahub_in_qgis)


        # get the current extent in the canvas CRS
        extent = self._canvas.extent()  # QgsRectangle

        # set up a transform from the canvas CRS → EPSG:3857
        source_crs = self._canvas.mapSettings().destinationCrs()
        mercator_crs = QgsCoordinateReferenceSystem("EPSG:3857")
        xform = QgsCoordinateTransform(source_crs, mercator_crs, QgsProject.instance())

        # transform the two opposite corners
        bl = xform.transform(extent.xMinimum(), extent.yMinimum())  # bottom‑left
        tr = xform.transform(extent.xMaximum(), extent.yMaximum())  # top‑right

        min_x, min_y = bl.x(), bl.y()
        max_x, max_y = tr.x(), tr.y()

        # store view coords
        self._current_bbox_3857 = [min_x, min_y, max_x, max_y]

        # start animation
        self.label_datasetGatheringInfo.show()
        self._dot_index_for_dataset_loader_info_timer = 0
        self._dataset_loader_info_timer.start()
        self._animate_label_dataset_loader_info()
        
        self.pshBtn_searchCurrentView.setEnabled(False)
        self.tableWidget_availableRegionDatasets.clear()                # remove any old content
        self.tableWidget_availableRegionDatasets.setRowCount(0)
        self.tableWidget_availableRegionDatasets.setColumnCount(0)

        # run your heavy function in a background THREAD
        self._datahub_future = self._datahub_query_pool.submit(self._fetch_region_data)
        self._datahub_poller.start()


    def _fetch_region_data(self):
        # get region subsets
        future = self._flai_sdk_executor.submit(
            self.FlaiOpenData().get_possible_region_ids,
            geom_filter = self._current_bbox_3857
        )

        # wait for slow call to finish
        try:
            region_subsets_raw = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
        except concurrent.futures.TimeoutError:
            # still running after TIMEOUT_TIME_IN_SECONDS -> cancel & fallback
            future.cancel()
            self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
            return
        except Exception as e:
            self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
            return
            
        # filtering of data
        region_subsets                 = []
        point_clouds_in_current_region = {}

        for item_region in region_subsets_raw['items']:
            current_dataset_id = item_region['id']

            future = self._flai_sdk_executor.submit(
                self.FlaiOpenData().get_dataset_files,
                dataset_id  = current_dataset_id,
                geom_filter = self._current_bbox_3857
            )

            # wait for slow call to finish
            try:
                point_clouds_in_current_region_raw = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
            except concurrent.futures.TimeoutError:
                # still running after TIMEOUT_TIME_IN_SECONDS -> cancel & fallback
                future.cancel()
                self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
                continue    # we do not add new entry to both region_subsets and point_clouds_in_current_region if we cannot get regions
            except Exception as e:
                self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
                continue    # we do not add new entry to both region_subsets and point_clouds_in_current_region if we cannot get regions
            
            # also get schema for current region
            current_item_semantic_schema_id = item_region['semantic_definition_schema_id']

            # as of 2025/07/31 some opendata records for semantic schema are missing (will be null)
            if current_item_semantic_schema_id == None:
                current_item_semantic_schema_id = self._asprs_semantic_schema_id + '-hardcoded' # we use hardcode schema which is asprs schema saved on 2025.07.31
                self._flai_semantic_schemas[current_item_semantic_schema_id] = self._asprs_hardcoded_schema

            # schema is not set
            elif self._flai_semantic_schemas.get(current_item_semantic_schema_id) == None:
                semantic_schema = dict(enumerate(item_region['semantic_labels_definition']))    # i had to gave it key of ascending ints otherwise i would need to change _parse_semantic_schema below
                self._flai_semantic_schemas[current_item_semantic_schema_id] = self._parse_semantic_schema(semantic_schema)
                
            # base for our point clouds
            current_dataset_point_clouds = { 
                "downloadable_paths": [], 
                "file_names": [],
                "total_items": point_clouds_in_current_region_raw['pagination']['total_items']
            }

            # get downloadable path for all point clouds
            for item_point_cloud in point_clouds_in_current_region_raw['items']:
                downloadable_path = item_point_cloud['datasource_host'] + '/' + item_point_cloud['path']
                current_dataset_point_clouds['downloadable_paths'].append(downloadable_path)    # a bo to delal
                current_dataset_point_clouds['file_names'].append(item_point_cloud['filename']) 
                
            # save point clouds download paths for download
            point_clouds_in_current_region[current_dataset_id] = current_dataset_point_clouds

            # year filtering
            region_time_frame = item_region['start_acquisition_date'][:4]                                 \
                if   item_region['start_acquisition_date'][:4] == item_region['end_acquisition_date'][:4] \
                else f"{item_region['start_acquisition_date'][:4]} - {item_region['end_acquisition_date'][:4]}"
            
            total_bytes, missing = self._total_download_size(current_dataset_point_clouds['downloadable_paths'])

            # save all region details
            region_subsets.append({
                "id": current_dataset_id,
                "dataset_name": item_region['dataset_name'],
                "time_frame": region_time_frame,
                "srid": item_region['srid'],
                "pointcloud_dataset_point_density": item_region['pointcloud_dataset_point_density'],
                "total_items": current_dataset_point_clouds['total_items'],
                "total_bytes": total_bytes,
                "license": item_region['licences'],
                "semantic_definition_schema_id": current_item_semantic_schema_id,
            })

        return {
            "regions": region_subsets,
            "point_clouds": point_clouds_in_current_region
        }
    

    def _hook_table_auto_fill(self, tbl):
        header = tbl.horizontalHeader()
        n = tbl.columnCount()
        if n == 0:
            return

        # modes & scrollbars (scroll only if truly too narrow to fit header minima)
        for c in range(n):
            header.setSectionResizeMode(c, QHeaderView.Interactive)
        header.setCascadingSectionResizes(True)
        header.setStretchLastSection(False)
        tbl.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        tbl.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)

        # per-column minimums based on header text so labels never clip
        fm = QFontMetrics(header.font())
        PADDING = 24  # margin + sort indicator
        labels = [(tbl.horizontalHeaderItem(i).text() if tbl.horizontalHeaderItem(i) else "") for i in range(n)]
        per_min = [max(40, fm.horizontalAdvance(txt) + PADDING) for txt in labels]

        # ensure any QWidget cell can actually expand (do this once per build)
        for r in range(tbl.rowCount()):
            for c in range(n):
                w = tbl.cellWidget(r, c)
                if w is not None:
                    # expanding horizontally lets it consume the column width
                    policy = w.sizePolicy()
                    policy.setHorizontalPolicy(QSizePolicy.Expanding)
                    w.setSizePolicy(policy)
                    if hasattr(w, "setMinimumWidth"):
                        w.setMinimumWidth(1)

        # keep proportional weights (from current widths) and redistribute to fill viewport
        def current_weights():
            widths = [max(per_min[i], tbl.columnWidth(i)) for i in range(n)]
            total = sum(widths) or 1
            return [w / total for w in widths]

        # store on the table so re-calling this function won’t duplicate state
        tbl._aff_weights = current_weights()

        def redistribute():
            # if structure changed, bail (caller will re-hook after rebuild)
            if tbl.columnCount() != n:
                return

            # viewport width already accounts for scrollbars and style paddings
            avail = max(0, tbl.viewport().width())
            min_sum = sum(per_min)

            if avail < min_sum:
                # too narrow: pin each column to its minimum and show horizontal scrollbar
                for i in range(n):
                    tbl.setColumnWidth(i, per_min[i])
                return

            # allocate proportionally, then fix rounding to exactly match viewport width
            targets = [max(int(round(avail * w)), per_min[i]) for i, w in enumerate(tbl._aff_weights)]
            delta = avail - sum(targets)

            if delta != 0:
                # give/take pixels to eliminate any residual gap or overflow
                order = sorted(range(n), key=lambda i: targets[i], reverse=True)
                k = 0
                while delta != 0 and order:
                    j = order[k % len(order)]
                    nxt = targets[j] + (1 if delta > 0 else -1)
                    if nxt >= per_min[j]:
                        targets[j] = nxt
                        delta += (-1 if delta > 0 else 1)
                    k += 1

            for i, w in enumerate(targets):
                tbl.setColumnWidth(i, w)

        def on_section_resized(_logical, _old, _new):
            # user dragged a header: capture new proportions, then fill to viewport
            tbl._aff_weights = current_weights()
            QTimer.singleShot(0, redistribute)

        # hook events ONCE (safe if this function is called again)
        try:
            header.sectionResized.disconnect(tbl._aff_section_cb)
        except Exception:
            pass
        tbl._aff_section_cb = on_section_resized
        header.sectionResized.connect(tbl._aff_section_cb)

        class _ResizeFilter(QObject):
            def eventFilter(self, obj, ev):
                if ev.type() in (QEvent.Resize, QEvent.Show):
                    QTimer.singleShot(0, redistribute)
                return False

        if not hasattr(tbl, "_aff_viewport_filter"):
            tbl._aff_viewport_filter = _ResizeFilter(tbl)
            tbl.viewport().installEventFilter(tbl._aff_viewport_filter)

        if not hasattr(tbl, "_aff_table_filter"):
            tbl._aff_table_filter = _ResizeFilter(tbl)
            tbl.installEventFilter(tbl._aff_table_filter)

        # scrollbar appearance can change viewport width; reflow on range change
        try:
            tbl.verticalScrollBar().rangeChanged.disconnect(tbl._aff_vscroll_cb)
        except Exception:
            pass
        tbl._aff_vscroll_cb = lambda *_: QTimer.singleShot(0, redistribute)
        tbl.verticalScrollBar().rangeChanged.connect(tbl._aff_vscroll_cb)

        try:
            header.geometriesChanged.disconnect(tbl._aff_geom_cb)
        except Exception:
            pass
        tbl._aff_geom_cb = lambda: QTimer.singleShot(0, redistribute)
        header.geometriesChanged.connect(tbl._aff_geom_cb)

        # initial pass (also ensures current widths >= minima and fills to the right edge)
        for i in range(n):
            tbl.setColumnWidth(i, max(tbl.columnWidth(i), per_min[i]))
        QTimer.singleShot(0, redistribute)


    def _check_region_future(self):
        # get data from thread
        if not self._datahub_future.done():
            return

        self._datahub_poller.stop()
        result = self._datahub_future.result()

        self._region_subsets                 = result["regions"]
        self._point_clouds_in_current_region = result["point_clouds"]

        # stop animation
        self._dataset_loader_info_timer.stop()
        self.label_datasetGatheringInfo.hide()

        # reenable button
        self.pshBtn_searchCurrentView.setEnabled(True)

        if len(self._region_subsets) == 0:
            self.label_datasetGatheringInfo.show()
            self.label_datasetGatheringInfo.setText('No data found on Datahub for this area!')
            return

        # populate table headers
        self.pshBtn_downloadCheckedDatasets.setEnabled(False)   # suspend signals
        tbl = self.tableWidget_availableRegionDatasets
        tbl.blockSignals(True)

        tbl.setColumnCount(len(TABEL_AVAILABLE_REGIONS_ITEMS))
        tbl.setHorizontalHeaderLabels(TABEL_AVAILABLE_REGIONS_ITEMS)
        tbl.setRowCount(len(self._region_subsets))
        tbl.setEditTriggers(QAbstractItemView.NoEditTriggers)   # make table non-editable except widgets (checkboxes are separate)
        tbl.verticalHeader().setVisible(False)                  # hide row numbers
        tbl.setHorizontalScrollMode(QAbstractItemView.ScrollPerPixel)

        header = tbl.horizontalHeader()
        last_col = tbl.columnCount() - 1

        # header modes: user-resizable everywhere (no platform quirks)
        # first column small, rest interactive; we manage the "fill" logic ourselves.
        header.setSectionResizeMode(0, QHeaderView.ResizeToContents)
        for c in range(1, last_col + 1):
            header.setSectionResizeMode(c, QHeaderView.Interactive)

        # avoid odd jumps when a section grows: let Qt cascade space properly
        header.setCascadingSectionResizes(True)
        # reasonable minimum so columns don't collapse to zero while dragging
        header.setMinimumSectionSize(30)

        # populate rows
        for row, entry in enumerate(self._region_subsets):
            # checkbox cell in column 0
            checkbox = QCheckBox()
            checkbox.toggled.connect(self._on_region_dataset_tabel_checkboxes_changed)
            container = QWidget()
            layout = QHBoxLayout(container)
            layout.setContentsMargins(0, 0, 0, 0)
            layout.setAlignment(Qt.AlignCenter)
            layout.addWidget(checkbox)
            # keep this column tight
            container.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
            tbl.setCellWidget(row, 0, container)

            # other columns except last
            values = [
                entry["dataset_name"],
                entry["time_frame"],
                entry["srid"],
                f'{float(entry["pointcloud_dataset_point_density"]):.2f} pts / m²',
                f'{entry["total_items"]} ({self._human_readable(entry["total_bytes"])})',
                entry["license"]  # markdown -> QLabel below
            ]
            for col, val in enumerate(values, start=1):
                if col == last_col:
                    # last column: has markdown-style link; use QLabel widget
                    label = QLabel()
                    label.setTextFormat(Qt.RichText)
                    label.setText(self._link_md_to_html_single_line(str(val)))
                    label.setAlignment(Qt.AlignCenter)
                    label.setWordWrap(False)
                    label.setTextInteractionFlags(Qt.TextBrowserInteraction)
                    label.setOpenExternalLinks(True)
                    # CRUCIAL: let label stretch/shrink with the column
                    label.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Preferred)
                    label.setMinimumWidth(1)
                    tbl.setCellWidget(row, col, label)
                else:
                    item = QTableWidgetItem(str(val))
                    item.setTextAlignment(Qt.AlignCenter)
                    # ensure not editable (redundant due to NoEditTriggers, but safe)
                    item.setFlags(item.flags() & ~Qt.ItemIsEditable)
                    tbl.setItem(row, col, item)

        # let Qt choose a sensible starting size (esp. for col 0 / contents)
        tbl.resizeColumnsToContents()
        self._hook_table_auto_fill(self.tableWidget_availableRegionDatasets)

        # keep last column filling leftover width without "collision"
        def _sum_other_columns(exclude):
            return sum(tbl.columnWidth(c) for c in range(tbl.columnCount()) if c != exclude)

        def _fill_last_column():
            if tbl.columnCount() == 0:
                return
            viewport_w = tbl.viewport().width()
            other_w = _sum_other_columns(last_col)
            remaining = viewport_w - other_w
            # only expand last column if there's slack; don't force shrink below min size
            min_last = max(header.minimumSectionSize(), 20)
            if remaining >= min_last:
                # -1 avoids off-by-one scrollbars on macOS styles
                tbl.setColumnWidth(last_col, remaining - 1)

        # connect once; avoid duplicate connections if this function runs often
        try:
            header.sectionResized.disconnect(self._region_table_section_resized_cb)
        except Exception:
            pass

        def _on_section_resized(logical_index, old_size, new_size):
            # if a non-last column changed, let the last column absorb/release slack.
            if logical_index != last_col:
                _fill_last_column()

        self._region_table_section_resized_cb = _on_section_resized
        header.sectionResized.connect(self._region_table_section_resized_cb)

        # also react to viewport/table size changes (splitter moves, window resize, DPI)
        class _ViewportFilter(QObject):
            def eventFilter(self, obj, ev):
                if ev.type() in (QEvent.Resize, QEvent.Show):
                    QTimer.singleShot(0, _fill_last_column)
                return False

        if not hasattr(self, "_region_tbl_vp_filter"):
            self._region_tbl_vp_filter = _ViewportFilter(tbl)
            tbl.viewport().installEventFilter(self._region_tbl_vp_filter)

        # queue one pass after pending layouts
        QTimer.singleShot(0, _fill_last_column)

        tbl.blockSignals(False) # reenable signaling


    def _on_downloadCheckedDatasets_pressed(self):
        checked_rows = []

        # get index of rows that are selected for download
        for row in range(self.tableWidget_availableRegionDatasets.rowCount()):
            container = self.tableWidget_availableRegionDatasets.cellWidget(row, 0)
            checkbox = container.findChild(QCheckBox)
            
            if checkbox and checkbox.isChecked():
                checked_rows.append(row)
        
        paths_for_download = []
        final_filenames = []
        final_saving_folder_names = []
        region_schematic_schemas = []
        srids = []
        total_downloadable_size = 0

        # save downloadable paths, folder names, schemas and final download size (will be used also as layers in QGIS)
        for index in checked_rows:
            # first get filepaths
            region_id = self._region_subsets[index]['id']
            paths = self._point_clouds_in_current_region[region_id]['downloadable_paths']  
            # paths_for_download.append(paths)
            paths_for_download.append(paths[:10])
            
            filenames = self._point_clouds_in_current_region[region_id]['file_names']
            # final_filenames.append(filenames)
            final_filenames.append(filenames[:10])

            # secondly generate folder names
            region_name = self._region_subsets[index]['dataset_name']
            region_capture_time = self._region_subsets[index]['time_frame']
            final_saving_folder_names.append(f'{region_name} [{region_capture_time}]')

            # thirdly get region schema to be used for all point clouds
            region_schematic_schemas.append(
                self._flai_semantic_schemas[
                    self._region_subsets[index]['semantic_definition_schema_id']
                ]
            )

            srids.append(self._region_subsets[index]['srid'])

            total_downloadable_size += self._region_subsets[index]["total_bytes"]

        # make worker list for download_worker class
        tasks = []
        for i in range(len(final_saving_folder_names)):
            tmp = {
                "folder_where_to_save": self._default_datahub_dir + os.sep + final_saving_folder_names[i],
                "urls": paths_for_download[i],
                "file_names": final_filenames[i],
                "semantic_schema": region_schematic_schemas[i],
                "srid": srids[i]
            }
            tasks.append(tmp)

        # inform the user about successful selection of CLI
        msg = QtWidgets.QMessageBox(self)
        msg.setIcon(QtWidgets.QMessageBox.Information)
        msg.setWindowTitle("Notification before download")
        msg.setText(
            f'You have selected to download {len(tasks)} datasets from desired area. Each dataset will be downloaded in parallel. '
            f'This will download about {self._human_readable(total_downloadable_size)} of data.\n\n'
            'If any of this data already exists locally it will not be downloaded.\n\n'
            'Do you wish to continue?\n\n'
            'TIPS AND TRICKS:\n'
            ' - since files can be big, QGIS might need some time to index all of them before showing.'
        )

        # remove default buttons and add custom ones
        start_btn = msg.addButton("Start download", QtWidgets.QMessageBox.AcceptRole)
        cancel_btn = msg.addButton("Cancel", QtWidgets.QMessageBox.RejectRole)

        msg.setFont(self._font_inside_msg_box)
        for btn in msg.buttons():
            btn.setFont(self._font_inside_msg_box)

        msg.exec_()

        if msg.clickedButton() is start_btn:
            # check if Datahub layer is created
            self._datahub_in_qgis = self._get_or_create_group(QgsProject.instance().layerTreeRoot(), 'Flai\'s DataHub')

            # create layers with region / dataset names
            for i in range(len(final_saving_folder_names)):
                self._get_or_create_group(self._datahub_in_qgis, final_saving_folder_names[i], put_on_top=True)
                
                current_dataset_path = Path(
                    self._default_datahub_dir + os.sep +
                      final_saving_folder_names[i] + os.sep + 
                      DATAHUB_DATASET_SAVED_SCHEME_FILE
                )
                current_dataset_path.parent.mkdir(parents=True, exist_ok=True)  # ensure directory exists

                # atomic-ish write: write to temp file then replace
                temp = current_dataset_path.with_suffix(current_dataset_path.suffix + ".tmp")
                json_content = {
                    'srid'           : tasks[i]["srid"],
                    'semantic_schema': tasks[i]["semantic_schema"]
                }

                with open(temp, "w", encoding="utf-8") as f:
                    json.dump(json_content, f, ensure_ascii=False, indent=2)
                    f.flush()
                    os.fsync(f.fileno())  # ensure data is durable
            
                # on most OSes this is atomic if on same filesystem
                os.replace(temp, current_dataset_path)

            # spawn downloader
            downloader = DownloadDialog(tasks, parent=self)
            downloader.show()


    def load_laz_with_schema(self, path, semantic_schema, srid=None):
        layer = self._create_layer(path, semantic_schema, srid=srid)

        tmp = Path(path)               # cannot join them with the line below, funny lib
        last_folder = tmp.parent.name  # "/this/is/my/path/with/subfolder/item.txt" -> it gets out subfolder
        grp = self._datahub_in_qgis.findGroup(last_folder)

        if layer and layer.isValid():
            self._insert_layer_node(layer, grp)
        else:
            self.report_error(f"Failed to load open-data file: {path}")


    def _load_preloaded_datasets_from_datahub(self):
        self._datahub_in_qgis = self._get_or_create_group(QgsProject.instance().layerTreeRoot(), 'Flai\'s DataHub')
        root = Path(self._default_datahub_dir)

        # loop through all subfolders
        for sub in root.iterdir():
            if not sub.is_dir():
                continue  # skip files at top level

            # determine schema
            dataset_metadata_path = sub / DATAHUB_DATASET_SAVED_SCHEME_FILE
            if dataset_metadata_path.is_file():
                try:
                    with open(dataset_metadata_path, "r", encoding="utf-8") as f:
                        metadata = json.load(f)
                except (json.JSONDecodeError, OSError) as e:
                    # fallback to hardcoded if file is unreadable or invalid
                    # e.g., report_error(f"Warning: failed to load {schema_path}: {e}, using hardcoded schema")
                    metadata = {
                        'srid'           : 3857,
                        'semantic_schema': self._asprs_hardcoded_schema
                    } 
            else:
                metadata = {
                    'srid'           : 3857,
                    'semantic_schema': self._asprs_hardcoded_schema
                }

            # process all .laz files in this subfolder
            for laz_file in sub.glob("*.laz"):
                try:
                    # assuming load_file takes (path, schema) or adjust if signature differs
                    self._get_or_create_group(self._datahub_in_qgis, laz_file.parent.name, put_on_top=True)
                    self.load_laz_with_schema(str(laz_file), metadata['semantic_schema'], srid=metadata['srid'],)
                except Exception as e:
                    # handle per-file errors as appropriate (log, continue, aggregate, etc.)
                    # e.g., report_error(f"Error loading {laz_file}: {e}")
                    continue


    def _on_region_dataset_tabel_checkboxes_changed(self):
        # only react when the checkbox column changes
        any_checked = False

        for r in range(self.tableWidget_availableRegionDatasets.rowCount()):
            container = self.tableWidget_availableRegionDatasets.cellWidget(r, 0)
            
            if not container:
                continue
            
            checkbox = container.findChild(QCheckBox)
            
            if checkbox and checkbox.isChecked():
                any_checked = True
                break
        
        self.pshBtn_downloadCheckedDatasets.setEnabled(any_checked)


    def _find_layer_node_in_group_by_source(self, group_node, source_uri):
        for child in group_node.children():
            if isinstance(child, QgsLayerTreeLayer):
                layer = child.layer()
                if layer and layer.source() == source_uri:
                    return child  # return the node, not the layer
            elif isinstance(child, QgsLayerTreeGroup):
                found = self._find_layer_node_in_group_by_source(child, source_uri)
                if found:
                    return found
        return None
    

    def _show_available_datahub_regions(self):
        self._datahub_in_qgis = self._get_or_create_group(QgsProject.instance().layerTreeRoot(), 'Flai\'s DataHub')

        # vector tiles
        existing = self._find_layer_node_in_group_by_source(self._datahub_in_qgis, self._vt_uri)
        
        if not existing:
            vector_tiles_layer = QgsVectorTileLayer(self._vt_uri, 'Available regions')
            
            if vector_tiles_layer.isValid():
                fill = QColor(AVAILABLE_REGIONS_COLOR)
                fill.setAlphaF(0.15)                        # 15% opacity (0.0–1.0)
                stroke = QColor(AVAILABLE_REGIONS_COLOR)    # keep outlines fully opaque
                line   = QColor(AVAILABLE_REGIONS_COLOR)
                ptfill = QColor(AVAILABLE_REGIONS_COLOR)
                ptline = QColor(AVAILABLE_REGIONS_COLOR)

                styles = QgsVectorTileBasicRenderer.simpleStyle(
                    fill,        # polygon fill (with reduced alpha)
                    stroke,      # polygon outline
                    0.6,         # polygon outline width (mm)
                    line,        # line stroke
                    0.8,         # line width (mm)
                    ptfill,      # point fill
                    ptline,      # point outline
                    3.0          # point size (mm)
                )

                renderer = QgsVectorTileBasicRenderer()
                renderer.setStyles(styles)
                vector_tiles_layer.setRenderer(renderer) 

                self._insert_layer_node(vector_tiles_layer, self._datahub_in_qgis)

        # osm
        if self.checkBox_addOSM.isChecked():
            existing = self._find_layer_node_in_group_by_source(self._datahub_in_qgis, self._osm_uri)
            
            if not existing:
                osm_layer = QgsRasterLayer(self._osm_uri, 'OpenStreetMap', 'wms')
                
                if osm_layer.isValid():
                    self._insert_layer_node(osm_layer, self._datahub_in_qgis)

        # make sure there is correct order
        osm_node = self._find_layer_node_in_group_by_source(self._datahub_in_qgis, self._osm_uri)
        vt_node = self._find_layer_node_in_group_by_source(self._datahub_in_qgis, self._vt_uri)

        if not osm_node or not vt_node:
            return  # need both to reorder

        siblings = self._datahub_in_qgis.children()
        try:
            idx_osm = siblings.index(osm_node)
        except ValueError:
            return

        cloned_vt = vt_node.clone()
        self._datahub_in_qgis.insertChildNode(idx_osm, cloned_vt)

        # Remove the original vt_node from its parent (safe after cloning)
        parent_of_original = vt_node.parent()
        if parent_of_original:
            parent_of_original.removeChildNode(vt_node)



#############################
### PROCESSING ENGINE tab ###
#############################

    def _get_file_paths(self, button_object_name):
        window_title = ''

        # minor settings
        if button_object_name == self.pshBtn_selectInput:
            window_title = 'Select inputs'
            required_types = self._original_readers # ['vector_1', 'point_cloud_2', 'vector_3', 'raster_4']
        elif button_object_name == self.pshBtn_selectOutput:
            window_title = 'Select outputs'
            required_types = self._original_writers
        else:
            print('Hold your horses, this should not happen')

        # show file manager window
        dialog = FileSelectionDialog(
            parent            = self,
            window_title      = window_title,
            given_types       = required_types, 
            font_reference    = self._font_inside_msg_box,
            btn_reference     = self.pshBtn_resetUI,
            remebered_paths   = self._input_manager_paths if button_object_name == self.pshBtn_selectInput else self._output_manager_paths,
            is_output_manager = True if button_object_name == self.pshBtn_selectOutput else False
        )

        dialog.exec_()

        # check if user did not successfully set input / ouput folders
        if button_object_name == self.pshBtn_selectInput:
            self._input_manager_paths = dialog.selected_paths()
            empty_paths_count = sum(1 for files in self._input_manager_paths.values() if not files)

        elif button_object_name == self.pshBtn_selectOutput:
            self._output_manager_paths = dialog.selected_paths()
            empty_paths_count = sum(1 for files in self._output_manager_paths.values() if not files)
        
        # clearing warnings
        if empty_paths_count == 0:

            if button_object_name == self.pshBtn_selectInput:
                self.warningLabel_inputFolder.setText('')
                self.are_input_paths_ok = True

            elif button_object_name == self.pshBtn_selectOutput:
                self.warningLabel_outputFolder.setText('')
                self.are_output_paths_ok = True
        
        # setting warrnings
        else:
            if button_object_name == self.pshBtn_selectInput:
                self.warningLabel_inputFolder.setText(f'There are {empty_paths_count} empty inputs fields !')
                self.are_input_paths_ok = False

            elif button_object_name == self.pshBtn_selectOutput:
                self.warningLabel_outputFolder.setText(f'There are {empty_paths_count} empty outputs fields !')
                self.are_output_paths_ok = False


    @pyqtSlot(bool)
    def _update_show_command_button(self, _=None):
        # we ignore the boolean payload - we only care about the four attrs
        enable_button_if_all_true = all((
            self._is_flow_template_selected, 
            self._are_output_paths_ok, 
            self._are_input_paths_ok, 
            self._are_measurement_units_ok
        ))
        self.pshBtn_showCommand.setEnabled(enable_button_if_all_true)

        if enable_button_if_all_true:

            chosen_template = self.flow_templates_cmbBx_mapping[self.cmbBx_flowTemplate.currentIndex()]
            local_flow_creator = self.CreateLocalFlow()

            future = self._flai_sdk_executor.submit(
                self.FlaiFlowTemplates().convert_template_to_local_flow,
                create_local_flow_status = local_flow_creator,
                flow_template_id         = chosen_template['id']
            )

            # wait for slow call to finish
            try:
                raw = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
                self.flow_id_with_local_io = raw["flow_id"] 
            except concurrent.futures.TimeoutError:
                # still running after TIMEOUT_TIME_IN_SECONDS -> cancel & fallback
                future.cancel()
                self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
                return
            except Exception as e:
                self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
                return

            # add local_ if it does not already exist
            self._local_reader = [
                item if item.startswith('local_') else 'local_' + item
                for item in self._original_readers
            ]
            self._local_writer = [     
                item if item.startswith('local_') else 'local_' + item
                for item in self._original_writers
            ]

            # for PC we will use our semantic schema 
            if any('point_cloud' in item for item in self._original_writers):

                detected_type = ''
                detected_id = ''

                for node in self._selected_flow_template_info['flow_nodes']:

                    if POSSIBLE_NODE_KEY_SCHEMA_ID in node['options']:
                        detected_id = node['options'][POSSIBLE_NODE_KEY_SCHEMA_ID]
                        detected_type = POSSIBLE_NODE_KEY_SCHEMA_ID

                    elif POSSIBLE_NODE_KEY_MODEL_ID in node['options']:
                        detected_id = node['options'][POSSIBLE_NODE_KEY_MODEL_ID]
                        detected_type = POSSIBLE_NODE_KEY_MODEL_ID


                if detected_type == POSSIBLE_NODE_KEY_MODEL_ID:
                    future = self._flai_sdk_executor.submit(
                        self.FlaiAiModel().get_semantic_definitions,
                        model_id = detected_id,
                    )

                    # wait for slow call to finish
                    try:
                        semantic_schema = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
                        self._output_pc_semantic_schema = self._parse_semantic_schema(semantic_schema)
                        self._flai_semantic_schemas[detected_id] = self._output_pc_semantic_schema
                    except concurrent.futures.TimeoutError:
                        # still running after TIMEOUT_TIME_IN_SECONDS -> cancel & fallback
                        future.cancel()
                        self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
                        return
                    except Exception as e:
                        self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
                        return

                elif detected_type == POSSIBLE_NODE_KEY_SCHEMA_ID:

                    # schema is set
                    if self._flai_semantic_schemas.get(detected_id) != None:
                        self._output_pc_semantic_schema = self._flai_semantic_schemas[detected_id]

                    else:
                        # get output schema
                        future = self._flai_sdk_executor.submit(
                            self.SemanticDefinitionSchemasModel().get_semantic_definitions,
                            semantic_definition_schema_id = detected_id
                        )

                        # wait for slow call to finish
                        try:
                            semantic_schema = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
                            self._output_pc_semantic_schema = self._parse_semantic_schema(semantic_schema)
                            self._flai_semantic_schemas[detected_id] = self._output_pc_semantic_schema
                        except concurrent.futures.TimeoutError:
                            # still running after TIMEOUT_TIME_IN_SECONDS s -> cancel & fallback
                            future.cancel()
                            self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
                            return
                        except Exception as e:
                            self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
                            return


    def _disable_buttons_and_clean_warnings_on_processing_tab(self):
        for b in (self.pshBtn_selectInput, self.pshBtn_selectOutput):
            b.setEnabled(False)

        self.cmbBx_measurementUnits.setEnabled(False)    # we will only show it when all data from processing tab is populated
        self.pshBtn_selectTmpFolder.setEnabled(False)
        self.checkBox_overwrite_files.setEnabled(False)

        # we also clean all warnings that appeared
        for l in (self.warningLabel_flowTemplate, 
                  self.warningLabel_inputFolder, 
                  self.warningLabel_outputFolder, 
                  self.warningLabel_measurementUnits,
                  self.warningLabel_tmpFolder
                ):
            l.setText('') 


    def _clean_variables_of_processing_tab(self):
        # reset input and output path managers
        self.are_output_paths_ok = False
        self.are_input_paths_ok  = False

        # dictionary paths
        self._output_manager_paths, self._input_manager_paths = {}, {}

        # data types
        self._original_readers, self._original_writers = [], []

        # temporary folder
        self._temp_cli_output_folder = ""
         

    @pyqtSlot(int)
    def _on_flow_template_changed(self, index: int):
        """
        Handles both:
        1. Toggling Create/Edit button text based on whether
           the user picked "Non selected" (index 0) or something else.
        2. Enabling/disabling the other three buttons and updating
           the warning label whenever the selection changes.
        """
        self._clean_variables_of_processing_tab()
        text = self.cmbBx_flowTemplate.currentText()

        # update button next to flow template lineEdit (Create vs. Edit)
        if text != "Non selected":
            self.pshBtn_createOrEdit.setText("Edit")
            self.cmbBx_dataDisplayingBehaviour.setEnabled(True)
        else:
            self.pshBtn_createOrEdit.setText("Create")
            self.cmbBx_dataDisplayingBehaviour.setEnabled(False)

        is_not_default = (index > 0)

        # fetch data with sdk about the selected flow
        if is_not_default:
            self._selected_flow_template_id = self.flow_templates_cmbBx_mapping[index]['id']
            status = self._get_data_from_selected_flow(self._selected_flow_template_id)
            
            # did we got an error when fetching data?
            if not self.is_flow_template_selected:
                msg = QtWidgets.QMessageBox(self)
                msg.setWindowTitle(f"Error getting flow template data")
                msg.setText(f'{status}\n\nReseting back to empty flow template. Please try again or select different template.')
                msg.setStandardButtons(QtWidgets.QMessageBox.Ok)

                msg.setFont(self._font_inside_msg_box)
                for btn in msg.buttons():
                    btn.setFont(self._font_inside_msg_box)

                msg.exec_()

                # reseting combobox back to default value
                self.cmbBx_flowTemplate.setCurrentIndex(0)
                self.pshBtn_createOrEdit.setText("Create")
                return

        else:
            self._selected_flow_template_id = ''
            self.is_flow_template_selected = False

        # enable/disable the four action buttons, combobox and checkbox
        for btn in (self.pshBtn_selectInput,
                    self.pshBtn_selectOutput):
            btn.setEnabled(is_not_default)

        self.cmbBx_measurementUnits.setEnabled(is_not_default) 
        self.pshBtn_selectTmpFolder.setEnabled(is_not_default)
        self.checkBox_overwrite_files.setEnabled(is_not_default)

        # warning label
        if is_not_default:
            if self._first_time_showing_automatically_generated_output_paths:
                self.warningLabel_outputFolder.setText(OUTPUT_PATH_MANAGER_DEFAULT_VALUE_MESSAGE)
                self._first_time_showing_automatically_generated_output_paths = False

            self._automatically_populate_output_manager()
            self.are_output_paths_ok = True
        
        else:
            self.warningLabel_outputFolder.clear()
            self.are_output_paths_ok = False


    def _get_data_from_selected_flow(self, id):
        flai = self.FlaiFlows()
        future = self._flai_sdk_executor.submit(
            flai.get_flow,
            flow_id   = id,
            get_nodes = True
        )

        # wait for slow call to finish
        try:
            self._selected_flow_template_info = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)

            flow_nodes = self._selected_flow_template_info.get("flow_nodes", [])
            self._original_readers, self._original_writers = [], []

            for item in flow_nodes:
                if isinstance(item, dict):
                    flow_node_key = item.get("flow_node_key", "")
                    if isinstance(flow_node_key, str) and "reader" in flow_node_key:
                        self._original_readers.append(flow_node_key)
                    elif isinstance(flow_node_key, str) and "writer" in flow_node_key:
                        self._original_writers.append(flow_node_key)
            
            self.warningLabel_flowTemplate.setText('')
            self.is_flow_template_selected = True
            return ''
        
        except concurrent.futures.TimeoutError:
            # still running after TIMEOUT_TIME_IN_SECONDS -> cancel & fallback
            future.cancel()
            self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
            return
        
        except Exception as e:
            self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
            return  

    
    def _automatically_populate_output_manager(self):
        base_dir = os.path.join(self._default_cli_dir, CLI_PROCESSED_DATA_FOLDER_NAME, DATETIME_PLACEHOLDER)
        line_edits_text = []

        for writer in self._original_writers:
            full_path = os.path.join(base_dir, writer)
            line_edits_text.append(full_path)

        self._output_manager_paths = { i: line_edits_text[i] for i in range(len(line_edits_text)) } 


    def _on_create_or_edit_flow_template(self):
        
        if "Edit" == self.pshBtn_createOrEdit.text():
            field_text = (
                'Plugin will now open a new browser window where you will be able to edit chosen flow. '
                '\n\nDo not forget to save the flow otherwise plugin will not be able to get updated flow.'
                '\n\nOnce satisfied with the change click on the button "Fetch updates" which will '
                'update plugin with the latest data for selected flow template.'
            )
            url = (
                "https://app.flai.ai/#/admin/pages:flows/"
                f"{self._selected_flow_template_id}/edit?"
                "utm_source=qgis&utm_medium=qgis-flai-plugin&utm_campaigne=flai-pages:flows"
            )

        elif "Create" == self.pshBtn_createOrEdit.text():
            field_text = (
                'Plugin open link to our App where you can create a new project called "QGIS plugin" '
                'where you will be able to create and edit flow templates.'
                '\n\nOnce satisfied with the new flow click on the button "Fetch updates" which will '
                'update plugin with the latest data for all flow templates.'
            )
            url = (
                "https://app.flai.ai/#/admin/pages:projects/?"
                "utm_source=qgis&utm_medium=qgis-flai-plugin&utm_campaigne=pages:projects"
            )

        msg = QtWidgets.QMessageBox(self)
        msg.setWindowTitle(f"{self.pshBtn_createOrEdit.text()} flow template")
        msg.setText(field_text)

        # add a custom fetch update button
        msg.addButton("Fetch updates", QtWidgets.QMessageBox.AcceptRole)

        msg.setFont(self._font_inside_msg_box)
        for btn in msg.buttons():
            btn.setFont(self._font_inside_msg_box)

        # schedule URL to open in 3 seconds
        QTimer.singleShot(3000, lambda: webbrowser.open(url, new=1))

        msg.exec_()

        # reset processing variables so that we dont keep old data
        self._clean_variables_of_processing_tab()

        # we update all data related to flows
        if "Create" == self.pshBtn_createOrEdit.text():
            self._reset_flow_template_combobox()     # including UI/UX
            self._populate_flow_template_combobox()

        elif "Edit" == self.pshBtn_createOrEdit.text():
            self._get_data_from_selected_flow(self._selected_flow_template_id)

        # automatically populate output path manager
        self._automatically_populate_output_manager()
        self.are_output_paths_ok = True

        self.warningLabel_inputFolder.setText("Update was fetched, please reselect inputs")
        self.warningLabel_outputFolder.setText("Update was fetched, you might need to reselect outputs")


    @pyqtSlot(int)
    def _on_measurement_unit_changed(self, index: int):
        # if not default, user selected unit, which is ok
        if index == 0:
            self.are_measurement_units_ok = False
        else:
            self.are_measurement_units_ok = True
            self.warningLabel_measurementUnits.setText('')
            self._settings.setValue(HIDDEN_SETTING_FIELD_MEASUREMENTS, self.cmbBx_measurementUnits.currentText())


    def _updated_tmp_location_for_cli(self):
        selected_path = self._select_folder(self._default_cli_dir, "Select temporary folder for CLI outputs")

        if selected_path == "":
            self._temp_cli_output_folder = ""
            self.warningLabel_tmpFolder.setText('Temporary folder flag was cleared!')
        else:
            selected_path = Path(selected_path)
            self._temp_cli_output_folder = str(selected_path) + os.sep + 'tmp'
            self.warningLabel_tmpFolder.setText('Folder set. Clear it with click on button below, press Cancel or X on top right corner.')

            
    def _show_executable_command(self):
        command_with_args = self._get_executable_command()[0]

        # if value contains any whitespace, wrap it in double-quotes
        program_location = f'"{command_with_args[0]}"' if self._whitespace_regex.search(command_with_args[0]) else command_with_args[0]
        
        # windows specific command
        if self._current_system == SYSTEM_WINDOWS:
            program_location = (
                'wsl \\\n'                                      +
                f'      {self._windows_flai_access_token} \\\n' +
                f'      {self._windows_flai_host} \\\n'         +
                f'      {command_with_args[0]}'
            )

        # first line: the program
        lines = [ f'{program_location} \\' ]

        # every subsequent pair: flag + value
        for flag, val in zip(command_with_args[1::2], command_with_args[2::2]):
            arg = f'"{val}"' if self._whitespace_regex.search(val) else val    # if value contains any whitespace, wrap it in double-quotes
            lines.append(f'      {flag} {arg} \\')
        
        # strip the trailing backslash from the last line
        lines[-1] = lines[-1].rstrip(" \\")
        shown_command = "\n".join(lines)

        msg = QtWidgets.QMessageBox(self)
        msg.setIcon(QtWidgets.QMessageBox.Information)
        msg.setWindowTitle("Command to be executed")
        msg.setText(shown_command)
        msg.setStandardButtons(QtWidgets.QMessageBox.Ok)

        msg.setFont(self._font_inside_msg_box)
        for btn in msg.buttons():
            btn.setFont(self._font_inside_msg_box)

        msg.exec_()


    def _get_executable_command(self):
        # parse string DATETIME_PLACEHOLDER if in self.output_manager_paths
        current_datetime = datetime.now()
        str_current_datetime = current_datetime.strftime(f"%Y.%m.%d{os.sep}%H.%M.%S") # cannot use %H:%M:%S, otherwise windows complains

        # needed for later processing, cleanup of DATETIME_PLACEHOLDER
        local_output_type_and_folder = {
            local_io: folder.replace(DATETIME_PLACEHOLDER, str_current_datetime)
            for local_io, folder in zip(self._local_writer, list(self._output_manager_paths.values()))
        }

        # needed for later processing, copying of paths
        local_input_type_and_files = {
            local_key: folder
            for local_key, folder in zip(self._local_reader, self._input_manager_paths.values())
        }

        # keep only path to a single folder for each reader (instead of item paths)
        input_folder = []
        for lst in local_input_type_and_files.values():
            input_folder.append(os.path.dirname(lst[0]))
        
        command = [
            f'{self._cli_path}',
            '--flow_id',     str(self.flow_id_with_local_io),
            '--length_unit', self.cmbBx_measurementUnits.currentText(),
        ]

        # add overwrite flag if checkbox is checked
        if self.checkBox_overwrite_files.isChecked():
            command.append('--overwrite')

        # add temp folder if set
        if self._temp_cli_output_folder != "":
            command.extend(['--tmp_path', f'{self._temp_cli_output_folder}'])

        # add reader flags + values
        for rdr_cmd, val in zip(self._local_reader, input_folder):
            command.extend([f'--{rdr_cmd}', f'{val}'])

        local_output = list(local_output_type_and_folder.items())

        # add all but last writer
        for wrtr_cmd, val in local_output[:-1]:
            command.extend([f'--{wrtr_cmd}', f'{val}'])

        # add last writer
        wrtr_cmd, val = local_output[-1]
        command.extend([f'--{wrtr_cmd}', f'{val}'])

        # on windows paths paths will not be ok for wsl, we do a little bit of parsing
        if self._current_system == SYSTEM_WINDOWS:
            command = [self._parse_windows_path(arg) for arg in command]

        return command, current_datetime, local_input_type_and_files, local_output_type_and_folder


    def _parse_windows_path(self, path):
        # normalize *any* backslashes, for example `\\\\wsl.localhost\\Ubuntu\\home\\alice\\file.txt`
        if '\\' in path:
            # 1) Leading \\…            -> //
            #    - r"^\\\\+"            -> match two or more backslashes at the start
            #    - replace with "//"
            # 2) Any remaining \        -> /
            path = re.sub(r"^\\\\+", "//", path).replace("\\", "/")

        path_located_inside_wsl_pattern = re.compile(r"^//wsl[^/]+/[^/]+(/.*)$")
        # ^//wsl    -> only if the string starts with //wsl
        # [^/]+/    -> match the rest of the server name (e.g. .localhost or IP) plus a slash
        # [^/]+     -> consume the distro name (Ubuntu, Debian, etc.)
        # (/.*)$    -> capture the slash plus everything after (our "real" path)
        m = path_located_inside_wsl_pattern.match(path)
        if m:
            path = m.group(1)

        windows_drive_pattern = re.compile(r'^([A-Za-z]):/(.*)$')
        # ^([A-Za-z])  -> capture one ASCII letter (drive letter A-Z or a-z), sometime it could be lower case (redundant but just in case)
        # :/           -> literal colon then forward‐slash
        # (.*)$        -> capture the rest of the string (zero or more characters)
        m = windows_drive_pattern.match(path)
        if m:
            drive = m.group(1).lower()
            rest  = m.group(2)
            path  = f"/mnt/{drive}/{rest}"

        return path


    def _run_flow(self):
        is_everthing_set_correctly = True

        if not self.is_flow_template_selected or self._selected_flow_template_id == '':
            self.warningLabel_flowTemplate.setText('Please select flow template.')
            is_everthing_set_correctly = False

        if not self.are_input_paths_ok or self._input_manager_paths == {}:
            self.warningLabel_inputFolder.setText('Please select input folders.')
            is_everthing_set_correctly = False

        if not self.are_output_paths_ok or self._output_manager_paths == {}:
            self.warningLabel_outputFolder.setText('Please select output folders.')
            is_everthing_set_correctly = False

        if not self.are_measurement_units_ok or self.cmbBx_measurementUnits.currentIndex() == 0:
            self.warningLabel_measurementUnits.setText('Please select measurement units.')
            is_everthing_set_correctly = False

        if not is_everthing_set_correctly:
            return
        
        msg = QtWidgets.QMessageBox(self)
        msg.setIcon(QtWidgets.QMessageBox.Warning)
        msg.setWindowTitle("Confirming action")
        msg.setText(
            "Before we run the given command, make sure everything is set as expected flow-wise "
            "since costs can occur on your side." \
            "\n\n" \
            "TIPS AND TRICKS:" \
            "\n - on tab Logs you can see what is happening (you will be automatically redirected to it)." \
            "\n - at bottom right corner you can see latest line output from CLI."
        )
        # remove any default buttons:
        msg.setStandardButtons(QtWidgets.QMessageBox.NoButton)

        # add your two custom buttons with appropriate roles:
        btn_start = msg.addButton("Start processing", QtWidgets.QMessageBox.AcceptRole)
        btn_back  = msg.addButton("Take me back",  QtWidgets.QMessageBox.RejectRole)

        msg.setFont(self._font_inside_msg_box)
        for btn in msg.buttons():
            btn.setFont(self._font_inside_msg_box)

        msg.exec_()

        if msg.clickedButton() == btn_start:

            # clear message if _temp_cli_output_folder was cleared from previous run
            if self._temp_cli_output_folder == "":
                self.warningLabel_tmpFolder.setText('')

            # disabling UI so that it cannot be pressed while flow is executing
            self.cmbBx_flowTemplate.setEnabled(False)
            self.pshBtn_createOrEdit.setEnabled(False)
            self.pshBtn_selectInput.setEnabled(False)
            self.pshBtn_selectOutput.setEnabled(False)
            self.cmbBx_measurementUnits.setEnabled(False)
            self.pshBtn_selectTmpFolder.setEnabled(False)
            self.checkBox_overwrite_files.setEnabled(False)
            self.cmbBx_dataDisplayingBehaviour.setEnabled(False)
            self.pshBtn_runFlow.setEnabled(False)

            # force immediate update otherwise disabling of UI will not happen before executing the code below
            QtWidgets.QApplication.processEvents()

            # parsing and creating output folders
            self._executing_command,     \
            self._processing_start_time, \
            self._local_input_type_and_files, self._local_output_type_and_folder = self._get_executable_command()

            self._create_folders_from_args(self._executing_command)

            # setting up windows specific command
            if self._current_system == SYSTEM_WINDOWS:
                self._executing_command = [
                    'wsl', 
                    self._windows_flai_access_token, 
                    self._windows_flai_host
                ] + self._executing_command

            # create the layer-tree structure, where paths / files will be grouped
            # for option where we do not display anything we do not need to be calls this
            if self.cmbBx_dataDisplayingBehaviour.currentText() != DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_NON:
                self._setup_layer_tree()

            # load files as they are generated
            if self.cmbBx_dataDisplayingBehaviour.currentText() == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_REALTIME:
                # load all inputs right away
                self._load_input_layers()

                # start watching each writer folder for newly-created files
                self._reset_output_helper_dictonary()

                # had problems with QFileSystemWatcher() (with and without moveToThread), this solution worked (not most elegant because of time polling)
                self._output_folder_watcher_poll_stop,  \
                self._output_folder_watcher_poll_thread = self._start_polling_output_folders_watcher(
                    paths    = list(self._local_output_type_and_folder.values()),
                    callback = self._on_directory_changed,
                    interval = OUTPUT_FOLDER_WATCHER_POOLING_RATE_IN_SEC
                )

            parsed_cmd = ''

            for arg in self._executing_command:
                string_arg = f'"{arg}"' if self._whitespace_regex.search(arg) else arg
                parsed_cmd += string_arg + ' '

            # we add few new lines to make output more readable
            if not self.plainTextEdit_logsProcessing.document().isEmpty():
                self.plainTextEdit_logsProcessing.appendPlainText('\n\n')

            self.plainTextEdit_logsProcessing.appendHtml(f'<b>Running command:</b> {parsed_cmd}<br>')
            self.label_flowSummary.setText('Launching CLI...')
            
            # use QProcess for running CLI
            self._proc = QProcess(self)
            # merge stderr into stdout
            self._proc.setProcessChannelMode(QProcess.MergedChannels)
            # whenever there is new data, append it to the text widget
            self._proc.readyReadStandardOutput.connect(self._read_output)
            self._proc.finished.connect(self._on_finished)  # this is the last thing that will happen when CLI stops

            # start the external command
            program, *args = self._executing_command
            self._proc.start(program, args)

            # for nicer transition
            QTimer.singleShot(
                1200, 
                lambda: self.tabWidget_processing_engine.setCurrentIndex(
                    self.tabWidget_processing_engine.indexOf(self.tab_logs_in_processing_engine)
                )
            )


    def _start_polling_output_folders_watcher(self, paths, callback, interval=1.0):
        """
        paths:    iterable of directory paths
        callback: fn(dirpath) - e.g. self._on_directory_changed
        interval: seconds between polls
        returns:  (stop_event, thread)
        """
        stop_event = threading.Event()

        def _poll_loop():
            while not stop_event.is_set():
                for p in paths:
                    callback(p)
                time.sleep(interval)

        t = threading.Thread(target=_poll_loop, daemon=True)
        t.start()
        return stop_event, t
    

    def _read_output(self):
        data = self._proc.readAllStandardOutput()
        text = bytes(data).decode('utf-8', errors='replace')

        # sometimes we could get None in text (not clear why) - can be a single CLI line output which has multiple line breaks
        for line in text.splitlines():
            if line.strip() == "None":
                continue

            self.plainTextEdit_logsProcessing.appendPlainText(line)
            self.label_flowSummary.setText(line)


    def _on_finished(self, exitCode, exitStatus):
        out = f"\n Process finished with exit code {exitCode}"
        # out += ' (' + f'❌' if exitCode != 0 else f'✅' + ')'
        self.plainTextEdit_logsProcessing.appendPlainText(out)

        # load files when flow finishes
        if self.cmbBx_dataDisplayingBehaviour.currentText() == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH or \
            self.cmbBx_dataDisplayingBehaviour.currentText() == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT:
            
            # if CLI returned error, we will not be looking for output
            if exitCode == 0:

                # load all inputs if using correct data displaying behavior 
                if self.cmbBx_dataDisplayingBehaviour.currentText() != DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT:
                    self._load_input_layers()

                # load all outputs
                self._reset_output_helper_dictonary()
                for folder_paths in list(self._local_output_type_and_folder.values()):
                    self._on_directory_changed(folder_paths)

        elif self.cmbBx_dataDisplayingBehaviour.currentText() == DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_REALTIME:
            self._output_folder_watcher_poll_stop.set()
            self._output_folder_watcher_poll_thread.join()

        # reenabling back what was disabled
        self.cmbBx_flowTemplate.setEnabled(True)
        self.pshBtn_createOrEdit.setEnabled(True)
        self.pshBtn_selectInput.setEnabled(True)
        self.pshBtn_selectOutput.setEnabled(True)
        self.cmbBx_measurementUnits.setEnabled(True)
        self.pshBtn_selectTmpFolder.setEnabled(True)
        self.checkBox_overwrite_files.setEnabled(True)
        self.cmbBx_dataDisplayingBehaviour.setEnabled(True)
        self.pshBtn_runFlow.setEnabled(True)
        
        if self._temp_cli_output_folder != "":
            self.warningLabel_tmpFolder.setText("Folder path was cleared!")
            self._temp_cli_output_folder = ""

        self.label_flowSummary.setText('processing ended (˶ᵔ ᵕ ᵔ˶)')
        QTimer.singleShot(5000, lambda: self.label_flowSummary.clear()) # in ms, non blocking

        # force immediate update just in case
        QtWidgets.QApplication.processEvents()


    def _create_folders_from_args(self, args):
        """
        Process command-line arguments to create folders next to 'writer' arguments.
        """
        # loop through each argument
        i = 1   # first arg will always be CLI-exe-name thus skipping it
        while i < len(args):

            # check if the argument contains "writer"
            if "writer" in args[i]:
                folder_name = args[i + 1]

                # create the folder recursively
                try:
                    os.makedirs(folder_name, exist_ok=True)
                except Exception as e:
                    self.show_generic_sdk_warning(message=f"Error creating folder {folder_name}: {str(e)}", title="Problem with creation of a folder")
                # skip the next argument as it's the folder name
                i += 2
                continue

            i += 1

        # make tmp folder if selected / enabled
        if self._temp_cli_output_folder != "":
            os.makedirs(self._temp_cli_output_folder, exist_ok=True)
            
        return


    def _get_paths_from_command(self, searched_string):
        # skip the program name, then slice into flag/value pairs
        arg_val_map = {
            flag.lstrip('-'): val
            for flag, val in zip(self._executing_command[1::2], self._executing_command[2::2])
            if searched_string in flag
        }

        return arg_val_map
    

    def _safe_name(self, flag):
        # remove leading dashes, replace other dashes with underscores
        return flag.lstrip('-').replace('-', '_')


    def _move_group_to_top(self, grp: QgsLayerTreeGroup):
        parent = grp.parent()
        if not isinstance(parent, QgsLayerTreeGroup):
            return  # already at root or weird state

        # if it's already first, bail early
        try:
            if parent.indexOfChild(grp) == 0:
                return
        except AttributeError:
            pass  # indexOfChild not in very old versions; harmless to proceed

        # prefer native move when available
        if hasattr(grp, "moveTo"):
            grp.moveTo(parent, 0)
            return

        # fallback: clone-then-replace
        clone = grp.clone()
        parent.insertChildNode(0, clone)
        parent.removeChildNode(grp)


    def _get_or_create_group(self, parent: QgsLayerTreeGroup, name: str, put_on_top=False):
        grp = parent.findGroup(name)
        if grp:
            if put_on_top:
                self._move_group_to_top(grp)
            return grp
        return parent.insertGroup(0, name) if put_on_top else parent.addGroup(name)


    def _setup_layer_tree(self):
        """Ensure top-level 'input' & 'output' and one subgroup per flag."""
        cli_outputs_in_qgis = self._get_or_create_group(QgsProject.instance().layerTreeRoot(), 'Flai\'s CLI')

        # make one subgroup under input per reader-flag
        if self.cmbBx_dataDisplayingBehaviour.currentText() != DROPDOWN_MENU_DATA_DISPLAYING_BEHAVIOUR_ON_FINISH_ONLY_OUTPUT:
            self._grp_in  = self._get_or_create_group(cli_outputs_in_qgis, 'input')
            
            for flag in self._local_input_type_and_files:
                self._get_or_create_group(self._grp_in, self._safe_name(flag))

        # make one subgroup under output per writer-flag
        self._grp_out = self._get_or_create_group(cli_outputs_in_qgis, 'output')

        for flag in self._local_output_type_and_folder:
            self._get_or_create_group(self._grp_out, self._safe_name(flag))


    def _crs_from_any(self, srid) -> Optional[QgsCoordinateReferenceSystem]:
        if not srid:
            return None
        # accept 3857, "3857", "EPSG:3857", or even a WKT string
        if isinstance(srid, int) or (isinstance(srid, str) and srid.isdigit()):
            auth = f"EPSG:{int(srid)}"
            crs = QgsCoordinateReferenceSystem(auth)
        elif isinstance(srid, str) and ":" in srid:
            crs = QgsCoordinateReferenceSystem(srid)
        else:
            crs = QgsCoordinateReferenceSystem(f"EPSG:{srid}")
        if not crs.isValid():
            # last-ditch: treat srid as WKT
            crs = QgsCoordinateReferenceSystem.fromWkt(str(srid))
        return crs if crs.isValid() else None


    def _srs_for_pdal(qgis_crs: QgsCoordinateReferenceSystem) -> str:
        # prefer an auth ID (e.g., "EPSG:3857"); fall back to WKT if custom
        auth = qgis_crs.authid()
        if auth:
            return auth
        return qgis_crs.toWkt(QgsCoordinateReferenceSystem.WKT_PREFERRED)


    def _create_layer(self, path, current_semantic_scheme = None, srid = None):
        """Pick the right Qgs*Layer by extension."""
        bn = os.path.basename(path)
        ext = bn.lower()

        if ext.endswith(('.shp', '.gpkg', '.geojson')):
            return QgsVectorLayer(path, bn, 'ogr')
        
        if ext.endswith(('.tif','.tiff', '.jp2', '.jp2000')):
            return QgsRasterLayer(path, bn, 'gdal')
        
        if ext.endswith(('.las', '.laz')):
            opts = QgsPointCloudLayer.LayerOptions(QgsProject.instance().transformContext())
            opts.skipCrsValidation = True   # just in case
            
            # reduce duplication when .copc.laz is detected
            if path.lower().endswith('.copc.laz'):
                layer = QgsPointCloudLayer(path, bn, 'copc', opts)
            else:
                layer = QgsPointCloudLayer(path, bn, 'pdal', opts)

            # if we pass srid we convert it
            if srid != None:
                src_crs = self._crs_from_any(srid)
                layer.setCrs(src_crs)

            # update classification renderer
            renderer = layer.renderer()
            if not isinstance(renderer, QgsPointCloudClassifiedRenderer):
                # no classified renderer yet? make one and attach it
                renderer = QgsPointCloudClassifiedRenderer("Classification")
                layer.setRenderer(renderer)

            # by default will be using input_pc_semantic_schema which should be ASPRS
            if current_semantic_scheme == None:
                current_semantic_scheme = self._default_input_pc_semantic_schema

            categories = []

            # convert current_schema to QgsPointCloudCategory
            for entry in current_semantic_scheme:
                if entry[SEMANTIC_DEFINITION_VALUE_FIELD] == 'DEFAULT':
                    continue

                r, g, b, a = entry[SEMANTIC_DEFINITION_COLOR_FIELD]

                categories.append( 
                    QgsPointCloudCategory(
                        int(   entry[SEMANTIC_DEFINITION_VALUE_FIELD]),
                        color = QColor.fromRgbF(r, g, b, a),
                        label = str(   entry[SEMANTIC_DEFINITION_LABEL_FIELD])
                    )
                )

            # build a classified renderer on the 'Classification' attribute and add it to layer
            # swap in only your new categories
            renderer.setAttribute("Classification")            # make sure we're on the right field
            renderer.setCategories(categories)                 # replaces just the Classification classes

            # refresh
            layer.triggerRepaint()
            return layer
        
        return None


    def _insert_layer_node(self, layer, group):
        """Add layer to project (hidden from root) and into our tree group."""
        QgsProject.instance().addMapLayer(layer, addToLegend=False)
        node = QgsLayerTreeLayer(layer)
        group.addChildNode(node)


    def _load_input_layers(self):
        """Load all pre-existing reader files into 'input' -> flag groups."""
        for flag, paths in self._local_input_type_and_files.items():
            grp = self._grp_in.findGroup(self._safe_name(flag))
            for p in paths:
                layer = self._create_layer(p)
                if layer and layer.isValid():
                    self._insert_layer_node(layer, grp)
                else:
                    self.report_error(f"Failed to load input {p}")


    def _reset_output_helper_dictonary(self):
        self._output_paths = {}
        for key in self._local_output_type_and_folder:
            self._output_paths[key] = []


    def _on_directory_changed(self, dirpath):
        """Called whenever any watched dir is modified."""
        # figure out which flag this dir belongs to
        flag = next(
            (f for f, d in self._local_output_type_and_folder.items() if d == dirpath),
            None
        )
        if not flag:
            return

        grp = self._grp_out.findGroup(self._safe_name(flag))
        dirpath = dirpath if self._current_system != SYSTEM_WINDOWS else dirpath.replace("\\", "/")
        
        # time.sleep(2.0) # quick and dirty workaround when (realtime) file listener is faster than CLI finishing making a file
                          # but will not work for large file
                          # maybe try using synhronization (e.g. semaphores, mutex) from CLI (making endpoint which can tell when file is not locked or not)
                          # and maybe replace this kind of file listener with synhronization listener (can check lock status)

        for fn in os.listdir(dirpath):
            full = dirpath + os.sep + fn

            # skip if already recorded
            if full in self._output_paths[flag]:
                continue
            # only accept files newer than time user started CLI
            try:
                mtime = datetime.fromtimestamp(os.path.getmtime(full))
            except OSError:
                continue
            if mtime >= self._processing_start_time:
                self._output_paths[flag].append(full)
                layer = self._create_layer(path = full, current_semantic_scheme = self._output_pc_semantic_schema)
                if layer and layer.isValid():
                    self._insert_layer_node(layer, grp)
                else:
                    self.report_error(f"Failed to load output {full}")


    def report_error(self, msg):
        if self.iface:
            self.iface.messageBar().pushMessage("Error", msg, level=3)
        else:
            print(msg)



####################
### SETTINGS tab ###
####################

    def _select_cli(self):
        # get path of user's "installed" CLI
        if self._settings.contains(HIDDEN_SETTING_FIELD_CLI_PATH):
            file_path = self._settings.value(HIDDEN_SETTING_FIELD_CLI_PATH)
        else:
            file_path, _ = QFileDialog.getOpenFileName(
                self,
                "Select flai-cli executable",
                self._default_cli_dir,
                "All Files (*)"
            )
            if not file_path:
                return  # user cancelled

        version, status = self._get_cli_version(file_path)

        # if anything was wrong with CLI selection, display error
        if status != '':
            msg = QtWidgets.QMessageBox(self)
            msg.setIcon(QtWidgets.QMessageBox.Critical)
            msg.setWindowTitle("Error")
            msg.setText(status)
            msg.setStandardButtons(QtWidgets.QMessageBox.Ok)

            msg.setFont(self._font_inside_msg_box)
            for btn in msg.buttons():
                btn.setFont(self._font_inside_msg_box)

            msg.exec_()
            return

        # store in instance
        self._cli_path = file_path
        self._cli_version = version

        self._settings.setValue(HIDDEN_SETTING_FIELD_CLI_PATH, self._cli_path)

        # updating "Getting started" tab
        # kick off the slow call in a separate process
        future = self._flai_sdk_executor.submit(
            self.FlaiOrganization().get_organization_name_and_address    # if you get stuck here, you are missing $HOME/.flai file or you have no internet
        )

        # inform the user about successful selection of CLI
        msg = QtWidgets.QMessageBox(self)
        msg.setIcon(QtWidgets.QMessageBox.Information)
        msg.setWindowTitle("Flai CLI detected")
        msg.setText('CLI program successfully selected.\n\nProgram might seem stuck for a few second. It needs to fetch required data for to be able to use it.\n\nClick OK to proceed.')
        msg.setStandardButtons(QtWidgets.QMessageBox.Ok)

        msg.setFont(self._font_inside_msg_box)
        for btn in msg.buttons():
            btn.setFont(self._font_inside_msg_box)

        msg.exec_()

        if self._current_system == SYSTEM_MACOS:
            msg = QtWidgets.QMessageBox(self)
            msg.setIcon(QtWidgets.QMessageBox.Information)
            msg.setWindowTitle("MacOS limitation")
            # Note: triple-quoted string preserves newlines exactly as shown
            message = \
"""Our CLI is not supported on MacOS yet, but there is a workaround (less performant).

One-time setup:
  docker buildx create --name emu-builder --driver docker-container --use
  docker buildx inspect --bootstrap

Template for executing:
  docker run -it --rm --platform=linux/amd64 --shm-size 32G \\
    -v "$HOME/.flai":/root/.flai \\
    -v "$HOME/path/to/user/CLI/folder":/Flai/CLI/ \\
    -v "$HOME/path/to/input/folder":/Flai/inputs/ \\
    -v "$HOME/path/to/output/folder":/Flai/outputs/ \\
    nvidia/cuda:12.8.0-runtime-ubuntu20.04 /bin/bash -c \\
      "cd /Flai/CLI/; chmod +x ./run_flai_flow; ./run_flai_flow \\
        --flow_id ID \\
        --length_unit meter \\
        --local_X_reader_Y /Flai/inputs/ \\
        --local_Z_writer_A /Flai/outputs/"

TIPS AND TRICKS:
 - to get flow-id, fill out fields on "Processing" tab until "Show me" button is enabled (there you will also get information which local_readers and local_writters to use)
"""
            msg.setText(message)
            msg.setStandardButtons(QtWidgets.QMessageBox.Ok)

            msg.setFont(self._font_inside_msg_box)
            for btn in msg.buttons():
                btn.setFont(self._font_inside_msg_box)

            msg.exec_()

        # wait for slow call to finish
        try:
            self._org_name = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
        except concurrent.futures.TimeoutError:
            # still running after TIMEOUT_TIME_IN_SECONDS s -> cancel & fallback
            future.cancel()
            self._org_name = 'CONNECTION ERROR'
            self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
        except Exception as e:
            self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
            self._org_name = 'GENERAL_ERROR ENCOUNTERED'
        
        # parsing data from .flai on user's home path, since WSL cannot access it
        if self._current_system == SYSTEM_WINDOWS:
            home = QStandardPaths.writableLocation(QStandardPaths.HomeLocation)
            file_path = home + '/.flai'

            f = QFile(file_path)
            if not f.open(QIODevice.ReadOnly | QIODevice.Text):
                print("Cannot open file:", f.errorString())
            else:
                raw = f.readAll()
                doc = QJsonDocument.fromJson(raw, _err := QJsonParseError())
                if _err.error != QJsonParseError.NoError:
                    print("JSON parse error:", _err.errorString())
                else:
                    obj = doc.object()
                    self._windows_flai_access_token = f"FLAI_ACCESS_TOKEN='{obj['flai_access_token'].toString()}'"
                    self._windows_flai_host = f"FLAI_HOST='{obj['flai_host'].toString()}'"

                    if '|' in self._windows_flai_access_token:
                        self._windows_flai_access_token = self._windows_flai_access_token.replace('|', r'\|')

                f.close()


        # get default semantic schema for PC inputs
        future = self._flai_sdk_executor.submit(
            self.SemanticDefinitionSchemasModel().get_semantic_definitions,
            semantic_definition_schema_id = self._asprs_semantic_schema_id
        )

        # updating labels
        self.label_wasCLIFound.setText('<b>Flai CLI detected</b>. Processing tab unlocked.')
        self.label_aboutCLIVersion.setText(
            f'You have Flai CLI version '
            f'<span style="color: #498FFF;"><b>{self._cli_version}</b></span>'
            f' installed.'
            )
        filtered_org_name = re.sub(r"[\"\'\`]", "", self._org_name)    # strip out any ASCII single-quote, double-quote or backtick characters
        self.label_aboutYourOrganization.setText(
            f'You are currently logged in organization: '
            f'<span style="color: #498FFF;"><b>{filtered_org_name}</b></span>'
        )

        # wait for slow call to finish
        try:
            semantic_schema = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
            self._default_input_pc_semantic_schema = self._parse_semantic_schema(semantic_schema)
            self._flai_semantic_schemas[self._asprs_semantic_schema_id] = self._default_input_pc_semantic_schema
        except concurrent.futures.TimeoutError:
            # still running after TIMEOUT_TIME_IN_SECONDS s -> cancel & fallback
            future.cancel()
            self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
            return
        except Exception as e:
            self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
            return

        # disabling buttons
        self.pshBtn_findFile.setEnabled(False)      # user "logged in", we do not need this two enabled
        self.pshBtn_downloadFile.setEnabled(False)
        self.label_belowTab.setText('Information about selected Flow')
        self._was_cli_loaded_on_first_show_menu_button_click = True

        # updating "Processing engine" tab
        self.tab_processingEngine.setEnabled(True)  # user "logged in", enable it
        self._clean_variables_of_processing_tab()
        self._disable_buttons_and_clean_warnings_on_processing_tab()
        self._populate_flow_template_combobox()
        self.label_disabledTabInformation.hide()    # redundant but ok
        self._choose_focus_tab()


    def _populate_flow_template_combobox(self):
        future = self._flai_sdk_executor.submit(
            self.FlaiFlowTemplates().get_flow_templates    # if you get stuck here, you are missing $HOME/.flai file
        )

        # wait for slow call to finish
        try:
            flow_templates = future.result(timeout=TIMEOUT_TIME_IN_SECONDS)
        except concurrent.futures.TimeoutError:
            # still running after TIMEOUT_TIME_IN_SECONDS -> cancel & fallback
            future.cancel()
            self.show_generic_sdk_warning(message=TIMEOUT_ERROR)
            return
        except Exception as e:
            self.show_generic_sdk_warning(message=GENERAL_ERROR + f"{str(e)}")
            return
        
        # sort values in ascending alphabet order 
        flow_templates.sort(key=lambda x: x["title"])

        self.flow_templates_cmbBx_mapping.update({
            i+1: {
                'title': entry['title'],
                'id':    entry['id']
            }
            for i, entry in enumerate(flow_templates)
        })

        current_index = self.cmbBx_flowTemplate.currentIndex()
        self.cmbBx_flowTemplate.clear()
        
        for entry in self.flow_templates_cmbBx_mapping.values():
            self.cmbBx_flowTemplate.addItem(entry['title'])

        self.cmbBx_flowTemplate.setCurrentIndex(current_index)


    def _get_cli_version(self, cli_path):
        issue = ''
        tag = ''
        
        # finding _internal folder
        base_dir = os.path.dirname(cli_path)
        internal_dir = os.path.join(base_dir, "_internal")

        if not os.path.isdir(internal_dir):
            issue = f"Could not find folder _internal on path:\n{base_dir}"
            return tag, issue

        # look for config.env
        config_path = os.path.join(internal_dir, "config.env")
        if not os.path.isfile(config_path):
            issue = f"Could not find file config.env on path:\n{internal_dir}"
            return tag, issue

        # parsing out BITBUCKET_TAG=
        pattern = re.compile(r"^BITBUCKET_TAG=(.+)$")
        try:
            with open(config_path, "r") as cfg:
                for line in cfg:
                    m = pattern.match(line.strip())
                    if m:
                        tag = m.group(1)
                        break
        except Exception as e:
            issue =  f"Failed to read {config_path}:\n{e}"
            return tag, issue

        if not tag:
            issue =  f"No line beginning with BITBUCKET_TAG= found in config.env on path:\n{config_path}"
            return tag, issue

        return tag, issue


    def _get_cli_link(self):
        msg = QtWidgets.QMessageBox(self)
        msg.setIcon(QtWidgets.QMessageBox.Information)
        msg.setWindowTitle("Instructions")
        msg.setText(
            "After clicking OK button below, plugin will open file explorer and two tabs.\n\n"
            "One tab will have documentation on how to install CLI while the other one will open a page where CLI can be download.\n\n"
            "Plugin will also open default folder where CLI can be saved by using system file explorer. Once download is finished you can copy your CLI to this folder."
        )

        # default setting for QMessageBox was buggy, even if user clicked on Close (x) button this line "if ret == QtWidgets.QMessageBox.Ok" was True
        # give it both OK and Cancel...
        msg.setStandardButtons(
            QtWidgets.QMessageBox.Ok |
            QtWidgets.QMessageBox.Cancel
        )
        # but hide the Cancel button so the user only sees OK
        msg.button(QtWidgets.QMessageBox.Cancel).setVisible(False)

        msg.setFont(self._font_inside_msg_box)
        for btn in msg.buttons():
            btn.setFont(self._font_inside_msg_box)

        ret = msg.exec_()
        if ret == QtWidgets.QMessageBox.Ok:
            # on Windows this should call Explorer, on macOS "open", on Linux "xdg-open"
            QDesktopServices.openUrl(
                QUrl.fromLocalFile(self._default_cli_dir)
            )
            time.sleep(.5)  # to be more smooth

            # hard-coded URL
            webbrowser.open("https://app.flai.ai/#/admin/pages:cli-dashboard/?utm_source=qgis&utm_medium=qgis-flai-plugin&utm_campaigne=flai-pages:cli-dashboard", new=1)   # open new window
            time.sleep(.2)  # to be more smooth
            webbrowser.open("https://docs.flai.ai/?utm_source=qgis&utm_medium=qgis-flai-plugin&utm_campaigne=flai-docs.flai.ai#/command-line-interface", new=2)             # open new tab


    def get_show_welcome_warning_value(self): 
        return self._is_show_welcome_warning_disabled


    def set_show_welcome_warning_value(self, value):
        """Setter method to set the show_welcome_warning_value."""
        self._is_show_welcome_warning_disabled = value
        # self._loading_and_setting_show_welcome_warning_from_settings()
        self.value_changed_for_welcome_warning.emit(self._is_show_welcome_warning_disabled)  # emit the signal to notify listeners


    def _loading_and_setting_show_welcome_warning_from_settings(self):
        if self._settings.contains(HIDDEN_SETTING_FIELD_SHOW_WELCOME_WARNING):
            int_value = int(self._settings.value(HIDDEN_SETTING_FIELD_SHOW_WELCOME_WARNING))
            self._is_show_welcome_warning_disabled = bool(int_value)
        else:
            self._settings.setValue(HIDDEN_SETTING_FIELD_SHOW_WELCOME_WARNING, int(self._is_show_welcome_warning_disabled))


    def _set_saving_location_for_cli(self):
        if self._settings.contains(HIDDEN_SETTING_SAVING_PATH_FOR_CLI_OUTPUTS):
            self._default_cli_dir = self._settings.value(HIDDEN_SETTING_SAVING_PATH_FOR_CLI_OUTPUTS)
        else:
            self._settings.setValue(HIDDEN_SETTING_SAVING_PATH_FOR_CLI_OUTPUTS, self._default_cli_dir)
        

    def _select_folder(self, path, dialog_tile):
        # Show native folder picker; returns "" if cancelled
        options = QFileDialog.Option.ShowDirsOnly | QFileDialog.Option.DontResolveSymlinks
        result = QFileDialog.getExistingDirectory(
            self,
            dialog_tile,
            str(path),
            options=options
        )
        return result


    def _updated_saving_location_for_cli(self):
        selected_path = self._select_folder(self._default_cli_dir, "Select folder where CLI output folder will be generated")

        if selected_path != "":
            selected_path = Path(selected_path)
            self._default_cli_dir = str(selected_path)


    def _set_saving_location_for_datahub(self):
        if self._settings.contains(HIDDEN_SETTING_SAVING_PATH_FOR_DATAHUB_OUTPUTS):
            self._default_datahub_dir = self._settings.value(HIDDEN_SETTING_SAVING_PATH_FOR_DATAHUB_OUTPUTS)
        else:
            self._settings.setValue(HIDDEN_SETTING_SAVING_PATH_FOR_DATAHUB_OUTPUTS, self._default_datahub_dir)
        

    def _updated_saving_location_for_datahub(self):
        selected_path = self._select_folder(self._default_datahub_dir, "Select folder where DataHub outputs will be saved")

        if selected_path != "":
            selected_path = Path(selected_path)
            self._default_datahub_dir = str(selected_path)
