import os
from datetime  import date
from functools import partial

from .constants import DATETIME_PLACEHOLDER 

from qgis.PyQt.QtWidgets import QVBoxLayout, QHBoxLayout, QPushButton, QDialog, QLabel, QLineEdit, QFileDialog, QGridLayout, QWidget
from qgis.PyQt.QtGui     import QFont, QFontMetrics
from qgis.PyQt.QtCore    import QStandardPaths, Qt


class FileSelectionDialog(QDialog):
    def __init__(self, 
                 given_types, 
                 window_title,
                 parent            = None, 
                 font_reference    = None,
                 btn_reference     = None,
                 remebered_paths   = {},
                 is_output_manager = False
        ):
        super().__init__(parent)
        self.setWindowTitle(window_title)
        self.setWindowModality(Qt.ApplicationModal)
        self.setMinimumWidth(600)

        # pull in the font (or default) and apply it to the whole dialog
        inherited_font = font_reference or QFont()
        self.setFont(inherited_font)

        # grab a uniform height from your reference button
        if btn_reference:
            # use sizeHint in case the widget isn’t yet shown
            self.uniform_btn_height = btn_reference.sizeHint().height()
        else:
            self.uniform_btn_height = None

        self.is_output_manager = is_output_manager
        
        possible_types        = ['vector', 'raster', 'point_cloud']
        label_displaying_name = ['vector', 'raster', 'point cloud']
        self.final_types = []

        for type in given_types:
            for i in range(len(possible_types)):
                if possible_types[i] in type:
                    self.final_types.append(label_displaying_name[i])
                    break

        # pull ext_map into __init__ so we can refer to it everywhere
        self.ext_map = {
            "vector":      (".shp", ".geojson", ".kml"),
            "raster":      (".tif", ".tiff", ".png", ".jpg", ".jpeg"),
            "point cloud": (".las", ".laz"),
        }

        # initialize scanned_files with one empty list per index
        if remebered_paths == {}:
            self.scanned_files = {i: [] for i in range(len(self.final_types))}
        else:
            self.scanned_files = remebered_paths

        # fill line edits with text
        line_edits_text = ['' for i in range(len(self.final_types))]

        if remebered_paths != {}:
            if is_output_manager == True:
                line_edits_text = [folder for folder in self.scanned_files.values()]
            else:
                line_edits_text = []
                
                for file_paths in self.scanned_files.values():
                    if file_paths == []:
                        line_edits_text.append('')
                    else:
                        line_edits_text.append(os.path.dirname(file_paths[0]))

        # layout helpers
        fm = QFontMetrics(inherited_font)
        label_texts = [f"{t.capitalize()} folder:" for t in self.final_types]
        max_label_w = max(fm.horizontalAdvance(txt) for txt in label_texts)
        warn_h      = fm.height()
        extra_space = 12

        self.edits = []
        self.warning_labels = []

        grid = QGridLayout()
        grid.setColumnStretch(1, 1)
        grid.setContentsMargins(0, 8, 0, 0)
        grid.setHorizontalSpacing(20)

        for i, constant in enumerate(self.final_types):
            #–– warning row
            warn = QLabel(" ")
            warn.setStyleSheet("color: #D1AE00;")
            warn.setFixedHeight(warn_h)
            warn.setFont(inherited_font)
            self.warning_labels.append(warn)
            grid.addWidget(QWidget(), i*3,   0)
            grid.addWidget(warn,          i*3,   1, Qt.AlignLeft)

            #–– label + line edit row
            lbl = QLabel(label_texts[i])
            lbl.setFont(inherited_font)
            lbl.setFixedWidth(max_label_w)
            grid.addWidget(lbl, i*3+1, 0, Qt.AlignVCenter)

            le = QLineEdit()
            # apply uniform height and font
            if self.uniform_btn_height:
                le.setFixedHeight(self.uniform_btn_height)
            le.setFont(inherited_font)
            exts = ", ".join(self.ext_map.get(constant, ()))
            le.setPlaceholderText(
                f"Click to select {constant} folder (allowed: {exts})"
            )
            le.setReadOnly(True)
            le.mousePressEvent = partial(
                self.handle_folder_click,
                index=i,
                original_event=le.mousePressEvent
            )
            le.setText(line_edits_text[i])
            
            # we do not need to check this condition for output since folders can be generated by code 
            if le.text() != '' and not self.is_output_manager:
                self._scan_folder(i, le.text())

            self.edits.append(le)
            grid.addWidget(le, i*3+1, 1)

            #–– spacer
            spacer = QWidget()
            spacer.setFixedHeight(extra_space)
            grid.addWidget(spacer, i*3+2, 0, 1, 2)

        main_layout = QVBoxLayout(self)
        main_layout.addLayout(grid)
        main_layout.addStretch()

        btn_hbox = QHBoxLayout()
        btn_hbox.addStretch()
        confirm_btn = QPushButton("Confirm")
        confirm_btn.clicked.connect(self.accept)

        # match height & font of all other buttons / edits
        if self.uniform_btn_height:
            confirm_btn.setFixedHeight(self.uniform_btn_height)
        confirm_btn.setFont(inherited_font)

        btn_hbox.addWidget(confirm_btn)
        main_layout.addLayout(btn_hbox)


    def handle_folder_click(self, event, index, original_event):
        current_text = self.edits[index].text()

        # input was already set (from before)
        if current_text and os.path.isdir(current_text):
            start_dir = current_text
        else:
            # for outputs we do a little bit of parsing
            if DATETIME_PLACEHOLDER in current_text:
                marker = "current-date/current-time"
                base, sep, tail = current_text.partition(marker)
                # -> base == ".../outputs/", sep == "current-date/current-time", tail == "/raster_writer_1"

                # 2) prepare the date‐folder name
                today = date.today()
                date_str = today.strftime("%Y.%m.%d")  # e.g. "2025.07.14"

                # 3) build the candidate path and test for existence
                candidate = os.path.join(base.rstrip(os.sep), date_str)
                if os.path.isdir(candidate):
                    start_dir = candidate
                else:
                    start_dir = base.rstrip(os.sep)

            # empty input
            else:
                start_dir = QStandardPaths.writableLocation(
                                QStandardPaths.DesktopLocation)

        """Allow picking a directory while still showing its files."""
        dlg = QFileDialog(
            self,
            f"Select folder for {self.final_types[index]} files",
            start_dir
        )
        # 1) use Qt’s own dialog so ShowDirsOnly=False is respected
        dlg.setOption(QFileDialog.Option.DontUseNativeDialog, True)
        # 2) only directories may be *selected*
        dlg.setFileMode(QFileDialog.FileMode.Directory)
        # 3) but show both files and folders in the list
        dlg.setOption(QFileDialog.Option.ShowDirsOnly, False)

        if not dlg.exec_():
            self.warning_labels[index].setText("No folder selected.")
            return original_event(event)

        folder = dlg.selectedFiles()[0]

        if not self.is_output_manager:
            self._scan_folder(index, folder)
            if self.scanned_files[index]:
                if 'unsupported file' not in self.warning_labels[index].text():
                    self.warning_labels[index].setText(" ")
                self.edits[index].setText(folder)
            else:
                # warning + clear line edit
                self.warning_labels[index].setText("No supported files in folder.")
                self.edits[index].clear()
        else:
            self.scanned_files[index] = folder
            self.edits[index].setText(folder)


    def _scan_folder(self, index: int, folder: str):
        """Populate self.scanned_files for this index, set warnings."""
        current_type = self.final_types[index]
        all_entries = os.listdir(folder)
        files = [f for f in all_entries
                 if os.path.isfile(os.path.join(folder, f))]

        if not files:
            self.warning_labels[index].setText("Folder is empty.")
            self.scanned_files[index] = []
            return

        allowed = self.ext_map.get(current_type, ())
        matched = [
            os.path.join(folder, f)
            for f in files
            if os.path.splitext(f)[1].lower() in allowed
        ]

        skipped = len(files) - len(matched)
        if skipped:
            self.warning_labels[index].setText(
                f"{skipped} unsupported file(s) skipped."
            )
        else:
            self.warning_labels[index].setText(" ")

        self.scanned_files[index] = matched


    def accept(self):
        all_ok = True
        for i in range(len(self.final_types)):
            if not self.edits[i].text().strip():
                self.warning_labels[i].setText("This field is required")
                all_ok = False
                continue
            if not self.scanned_files.get(i):
                self.warning_labels[i].setText("No supported files in folder.")
                all_ok = False
        if all_ok:
            super().accept()


    def selected_paths(self) -> dict:
        """Returns { index: [matched file paths] }. """
        return self.scanned_files
