"""
FilterMate Core Module.

Pure Python business logic with no QGIS dependencies.
This layer follows the Hexagonal Architecture pattern:

- domain/: Value objects, entities, and domain logic
- ports/: Abstract interfaces for external dependencies
- services/: Business logic orchestration

Usage:
    from ..core.domain import FilterExpression, FilterResult, LayerInfo  # noqa: F401
    from ..core.ports import BackendPort, CachePort  # noqa: F401
    from ..core.services import FilterService, ExpressionService  # noqa: F401
"""

# Re-export commonly used types for convenience
from .domain import (  # noqa: F401
    FilterExpression,
    FilterResult,
    FilterStatus,
    LayerInfo,
    GeometryType,
    ProviderType,
    SpatialPredicate,
    OptimizationConfig,
)

from .ports import (  # noqa: F401
    BackendPort,
    BackendInfo,
    BackendCapability,
    CachePort,
    CacheStats,
    LayerRepositoryPort,
)

from .services import (  # noqa: F401
    FilterService,
    FilterRequest,
    FilterResponse,
    ExpressionService,
    ValidationResult,
    HistoryService,
    HistoryEntry,
)

# v4.0 EPIC-1 Phase E1: Export module
from .export import (  # noqa: F401
    LayerExporter,
    ExportConfig,
    ExportResult,
    ExportFormat,
    StyleExporter,
    StyleFormat,
    save_layer_style,
    validate_export_parameters,
    ExportValidationResult,
)

# v4.0 EPIC-1 Phase E2: Geometry module
from .geometry import (  # noqa: F401
    apply_qgis_buffer,
    create_buffered_memory_layer,
    BufferConfig,
    aggressive_geometry_repair,
    repair_invalid_geometries,
    convert_geometry_collection_to_multipolygon,
)

# v4.0 EPIC-1 Phase E3: Filter/Expression module
from .filter import (  # noqa: F401
    build_feature_id_expression,
    build_combined_filter_expression,
    sanitize_subset_string,
    optimize_duplicate_in_clauses,
    is_pk_numeric,
    format_pk_values_for_sql,
    apply_combine_operator,
    combine_with_old_subset,
    CombineOperator,
)

__all__ = [
    # Domain
    'FilterExpression',
    'FilterResult',
    'FilterStatus',
    'LayerInfo',
    'GeometryType',
    'ProviderType',
    'SpatialPredicate',
    'OptimizationConfig',
    # Ports
    'BackendPort',
    'BackendInfo',
    'BackendCapability',
    'CachePort',
    'CacheStats',
    'LayerRepositoryPort',
    # Services
    'FilterService',
    'FilterRequest',
    'FilterResponse',
    'ExpressionService',
    'ValidationResult',
    'HistoryService',
    'HistoryEntry',
    # Export (v4.0 EPIC-1 Phase E1)
    'LayerExporter',
    'ExportConfig',
    'ExportResult',
    'ExportFormat',
    'StyleExporter',
    'StyleFormat',
    'save_layer_style',
    'validate_export_parameters',
    'ExportValidationResult',
    # Geometry (v4.0 EPIC-1 Phase E2)
    'apply_qgis_buffer',
    'create_buffered_memory_layer',
    'BufferConfig',
    'aggressive_geometry_repair',
    'repair_invalid_geometries',
    'convert_geometry_collection_to_multipolygon',
    # Filter/Expression (v4.0 EPIC-1 Phase E3)
    'build_feature_id_expression',
    'build_combined_filter_expression',
    'sanitize_subset_string',
    'optimize_duplicate_in_clauses',
    'is_pk_numeric',
    'format_pk_values_for_sql',
    'apply_combine_operator',
    'combine_with_old_subset',
    'CombineOperator',
]
