#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Test script for ColorType in Qt JSON View.

This script demonstrates the QgsColorButton integration for hex color values
in the JSON tree view.
"""

import sys
import json
from qgis.PyQt.QtWidgets import QApplication, QMainWindow, QVBoxLayout, QWidget
from qgis.core import QgsApplication

# Import qt_json_view modules
from modules.qt_json_view import view, model


def create_test_data():
    """Create sample JSON data with color values."""
    return {
        "app": {
            "name": "FilterMate",
            "version": "2.3.0",
            "themes": {
                "default": {
                    "description": "Default light theme with Material Design colors",
                    "background": [
                        "#F5F5F5",  # Frame background
                        "#FFFFFF",  # Widget background
                        "#E0E0E0",  # Borders/selected
                        "#2196F3"   # Accent color
                    ],
                    "font": [
                        "#212121",  # Primary text
                        "#616161",  # Secondary text
                        "#BDBDBD"   # Disabled text
                    ],
                    "accent": {
                        "primary": "#1976D2",
                        "hover": "#2196F3",
                        "pressed": "#0D47A1",
                        "light_bg": "#E3F2FD",
                        "dark": "#01579B"
                    }
                },
                "dark": {
                    "description": "Dark theme inspired by VS Code",
                    "background": [
                        "#1E1E1E",  # Frame background
                        "#2D2D30",  # Widget background
                        "#3E3E42",  # Borders/selected
                        "#007ACC"   # Accent color
                    ],
                    "font": [
                        "#EFF0F1",  # Primary text
                        "#D0D0D0",  # Secondary text
                        "#808080"   # Disabled text
                    ],
                    "accent": {
                        "primary": "#007ACC",
                        "hover": "#1E90FF",
                        "pressed": "#005A9E",
                        "light_bg": "#1E3A5F",
                        "dark": "#003D66"
                    }
                },
                "custom_example": {
                    "description": "Example with transparency",
                    "overlay": "#000000CC",      # Semi-transparent black
                    "highlight": "#FFFF00",      # Pure yellow
                    "glass_effect": "#FFFFFF44"  # Very transparent white
                }
            }
        },
        "non_color_strings": {
            "url": "https://github.com/sducournau/filter_mate",
            "path": "/path/to/file.json",
            "normal_text": "This is not a color"
        }
    }


class ColorPickerTestWindow(QMainWindow):
    """Test window for ColorType functionality."""
    
    def __init__(self):
        super().__init__()
        self.setWindowTitle("FilterMate Color Picker Test")
        self.setGeometry(100, 100, 800, 600)
        
        # Create central widget
        central_widget = QWidget()
        self.setCentralWidget(central_widget)
        layout = QVBoxLayout(central_widget)
        
        # Create JSON model and view
        test_data = create_test_data()
        self.json_model = model.JsonModel(test_data, editable_keys=False, editable_values=True)
        self.json_view = view.JsonView(self.json_model)
        
        layout.addWidget(self.json_view)
        
        # Expand all to show colors
        self.json_view.expandAll()
        
        print("ColorType Test Window created successfully!")
        print("Instructions:")
        print("1. Click on any hex color value (e.g., #F5F5F5)")
        print("2. A QgsColorButton should appear for color selection")
        print("3. Change the color and it should update in the JSON")


def main():
    """Main entry point for testing."""
    # Initialize QGIS application
    qgs = QgsApplication([], True)
    qgs.initQgis()
    
    # Create and show test window
    window = ColorPickerTestWindow()
    window.show()
    
    # Run application
    exit_code = qgs.exec_()
    
    # Cleanup
    qgs.exitQgis()
    return exit_code


if __name__ == "__main__":
    sys.exit(main())
