# Diagnostic et Correction du Crash QGIS au Lancement de FilterMate

**Date:** 7 décembre 2025  
**Version:** FilterMate v2.1.0  
**Statut:** ✅ CORRIGÉ

---

## 🔴 PROBLÈME

QGIS crashe lors du lancement du plugin FilterMate.

---

## 🔍 ANALYSE

### Causes Identifiées

#### 1. **CAUSE PRINCIPALE : Race Condition d'Initialisation** (CRITIQUE)

**Localisation:** `filter_mate_app.py` ligne 160

**Problème:**
```python
def __init__(self, plugin_dir):
    # ... initialisation ...
    self.run()  # ⚠️ APPEL IMMÉDIAT dans __init__
```

`self.run()` était appelé **immédiatement à la fin du `__init__`**, créant la dockwidget et connectant les signaux Qt AVANT que QGIS soit complètement stable. Cela causait des **violations d'accès mémoire** (access violations) et des crashes.

**Impact:** Crash systématique au lancement sur certaines configurations.

---

#### 2. **Conflit PyQt5 vs qgis.PyQt** (HAUTE PRIORITÉ)

**Localisation:** `filter_mate_dockwidget_base.py` ligne 11

**Problème:**
```python
from PyQt5 import QtCore, QtGui, QtWidgets  # ❌ Mauvais
```

Le fichier UI compilé utilisait `PyQt5` directement au lieu de `qgis.PyQt`, causant des incompatibilités potentielles avec l'environnement Qt de QGIS.

**Impact:** Instabilité, conflits de versions Qt, crashes intermittents.

---

#### 3. **Imports Wildcard Excessifs** (MOYENNE PRIORITÉ)

**Localisation:** Multiples fichiers

**Problème:**
```python
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *
from qgis.core import *
```

Ces imports wildcard peuvent causer des conflits de noms et des références circulaires.

**Impact:** Augmente le risque de bugs subtils et de crashes.

---

#### 4. **Accès Prématuré à `iface`** (FAIBLE PRIORITÉ)

**Localisation:** `filter_mate.py` et `filter_mate_app.py`

**Problème:**
```python
from qgis.utils import iface  # Accès lors de l'import
```

L'accès à `iface` au moment de l'import peut échouer si l'interface n'est pas encore disponible.

**Impact:** Potentiellement déstabilisant lors du chargement initial.

---

## ✅ CORRECTIONS APPLIQUÉES

### 1. **Déplacement de l'Appel `run()`** ✅

**Fichier:** `filter_mate_app.py`

**Modification:**
```python
def __init__(self, plugin_dir):
    # ... initialisation ...
    # Note: Do NOT call self.run() here - it will be called from filter_mate.py
    # when the user actually activates the plugin to avoid QGIS initialization race conditions
```

**Résultat:** `run()` n'est plus appelé dans `__init__`, mais uniquement depuis `filter_mate.py` quand l'utilisateur active le plugin.

---

### 2. **Protection et Gestion d'Erreurs** ✅

**Fichier:** `filter_mate.py`

**Modification:**
```python
def run(self):
    """Run method that loads and starts the plugin"""
    if not self.pluginIsActive:
        self.pluginIsActive = True
        try:
            if not self.app:
                # Create app WITHOUT calling run() automatically
                self.app = FilterMateApp(self.plugin_dir)
                # NOW call run() after QGIS is stable
                self.app.run()
                self.app.dockwidget.closingPlugin.connect(self.onClosePlugin)
            else:
                self.app.run()
                self.app.dockwidget.closingPlugin.connect(self.onClosePlugin)
                self.app.dockwidget.show()
        except Exception as e:
            iface.messageBar().pushCritical(
                "FilterMate",
                f"Error loading plugin: {str(e)}. Check QGIS Python console for details."
            )
            import traceback
            print(f"FilterMate Error: {traceback.format_exc()}")
            self.pluginIsActive = False
```

**Résultat:** 
- Gestion d'erreurs robuste
- Messages clairs à l'utilisateur
- Traceback dans la console Python pour le debug

---

### 3. **Correction des Imports PyQt5** ✅

**Fichier:** `filter_mate_dockwidget_base.py`

**Modification:**
```python
# MODIFIED: Use qgis.PyQt instead of PyQt5 for QGIS compatibility
from qgis.PyQt import QtCore, QtGui, QtWidgets
```

**Résultat:** Utilise maintenant les modules Qt fournis par QGIS.

---

### 4. **Script de Post-Compilation** ✅

**Nouveau Fichier:** `fix_compiled_ui.py`

**Fonction:** Script automatique qui corrige les imports PyQt5 après chaque compilation UI.

**Usage:**
```bash
python3 fix_compiled_ui.py [file_path]
```

**Intégration:** Ajouté à `compile_ui.sh` pour application automatique.

---

## 🧪 TESTS RECOMMANDÉS

### Test 1: Lancement Initial
```
1. Redémarrer QGIS complètement
2. Activer le plugin FilterMate via le menu Plugins
3. Vérifier: Plugin se charge sans crash
4. Vérifier: Dockwidget s'affiche correctement
```

### Test 2: Rechargement
```
1. Désactiver le plugin
2. Réactiver le plugin
3. Vérifier: Pas de crash, pas de message d'erreur
```

### Test 3: Avec Projet Existant
```
1. Ouvrir un projet QGIS avec des couches vectorielles
2. Activer FilterMate
3. Vérifier: Les couches sont listées correctement
4. Vérifier: Pas d'erreur dans la console Python
```

### Test 4: Projet Vide
```
1. Créer un nouveau projet vide
2. Activer FilterMate
3. Vérifier: Plugin s'active sans crash
4. Vérifier: Interface correctement désactivée (aucune couche)
```

---

## 📋 CHECKLIST DE VÉRIFICATION

- [x] Code modifié dans `filter_mate_app.py`
- [x] Code modifié dans `filter_mate.py`
- [x] Imports PyQt5 corrigés dans `filter_mate_dockwidget_base.py`
- [x] Script `fix_compiled_ui.py` créé
- [x] Script `compile_ui.sh` mis à jour
- [ ] Tests manuels effectués dans QGIS
- [ ] Validation sur Windows
- [ ] Validation sur Linux
- [ ] Tests avec projets réels
- [ ] Mise à jour CHANGELOG.md

---

## 🔄 WORKFLOW DE COMPILATION

**Nouveau workflow avec correction automatique:**

```bash
# 1. Modifier l'UI dans Qt Designer (filter_mate_dockwidget_base.ui)

# 2. Compiler l'UI
bash compile_ui.sh

# 3. Le script applique automatiquement:
#    - Compilation pyuic5
#    - Correction PyQt5 -> qgis.PyQt
#    - Backup automatique

# 4. Le fichier .py est prêt à l'emploi
```

---

## 📝 NOTES TECHNIQUES

### Ordre d'Initialisation Correct

```
1. filter_mate.py: classFactory(iface) appelé par QGIS
2. FilterMate.__init__() créé
3. FilterMate.initGui() appelé
4. Utilisateur clique sur l'icône du plugin
5. FilterMate.run() appelé
6. FilterMateApp.__init__() appelé (sans run())
7. FilterMateApp.run() appelé explicitement
8. Dockwidget créé et affiché
9. Signaux Qt connectés
```

### Protection Contre les Crashes Futurs

1. **try/except robuste** dans `filter_mate.py:run()`
2. **Messages d'erreur clairs** avec contexte
3. **Traceback dans console** pour debugging
4. **Flag `pluginIsActive`** réinitialisé en cas d'erreur
5. **Imports Qt standardisés** via `qgis.PyQt`

---

## 🚀 PROCHAINES ÉTAPES

### Immédiat
1. ✅ Tester le lancement dans QGIS
2. ✅ Vérifier les logs Python Console
3. ✅ Valider fonctionnalité de base

### Court Terme
1. ⏳ Réviser tous les imports wildcard
2. ⏳ Ajouter plus de logging pour tracking
3. ⏳ Mettre à jour la documentation

### Moyen Terme
1. 📋 Audit complet de la gestion des signaux Qt
2. 📋 Optimisation de l'ordre d'initialisation
3. 📋 Tests automatisés pour prévenir régressions

---

## 📚 RÉFÉRENCES

- [QGIS Plugin Development Best Practices](https://docs.qgis.org/latest/en/docs/pyqgis_developer_cookbook/)
- [Qt Signal/Slot Thread Safety](https://doc.qt.io/qt-5/threads-qobject.html)
- FilterMate Issue: Crash au lancement (7 déc 2025)

---

## 👤 CONTRIBUTEURS

- **Analyse et Correction:** GitHub Copilot (Claude Sonnet 4.5)
- **Validation:** [À compléter après tests]

---

**FIN DU DIAGNOSTIC**
