# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FieldStats
                                 A QGIS plugin
 This pluggins calculate basic stats and graph histogram and boxplot
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-07
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Manuel Alejandro Montealegre Martínez
        email                : manuel.montealegre@udea.edu.co
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .field_stats_dialog import FieldStatsDialog
import os.path

# Import qgis core complete, matplotlib and pandas
from qgis.core import *
import matplotlib.pyplot as plt
from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
import pandas as pd


class FieldStats:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FieldStats_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Field Stats')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('FieldStats', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/field_stats/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'calculate stats'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Field Stats'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False

        self.dlg = FieldStatsDialog()
        # Filter layers to show vector type only
        self.dlg.cmbCapas.setFilters(QgsMapLayerProxyModel.VectorLayer)
        # Add ComboBoxField Conector
        self.dlg.cmbCapas.layerChanged.connect(self.dlg.cmbCampos.setLayer)
        # Filter numeric fields only
        self.dlg.cmbCampos.setFilters(QgsFieldProxyModel.Numeric)
        # Check if there is selected rows
        self.dlg.chcSeleccion.stateChanged.connect(self.obtener_valores_de_campo)
        # Calculate stats when click on button
        self.dlg.btnCalcular.clicked.connect(self.btn_calcular_click)
        # round decimal places
        self.dlg.spBoxDecimales.valueChanged.connect(self.num_decimales)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
    
    # Methods to get info
    # To get values from field
    def obtener_valores_de_campo(self):
        """
        obtener_valores_de_campo
        return a Pandas data series with values from selected field
        """
        capa = self.dlg.cmbCapas.currentLayer()
        campo_seleccionado = self.dlg.cmbCampos.currentField()
        # if only selected is enable
        if self.dlg.chcSeleccion.isChecked() == True:
            # list of values from selected rows of selected field
            datos = QgsVectorLayerUtils.getValues(layer = capa,
                                                  fieldOrExpression = campo_seleccionado,
                                                  selectedOnly = True)[0] # True for selected Only
            # Make an empty dataFrame
            df_datos = pd.DataFrame()
            # Add values and name field to empty dataFrame
            df_datos[campo_seleccionado] = datos
            # Change de Pandas DataFrame to Pandas numeric DataSeries
            datos = pd.to_numeric(df_datos[campo_seleccionado], errors='coerce')
            # Return the Pandas DataSeries
            return datos
        # If only selected is dissable
        else:
            # list of values from selected rows of selected field
            datos = QgsVectorLayerUtils.getValues(layer = capa,
                                                  fieldOrExpression = campo_seleccionado,
                                                  selectedOnly = False)[0] #False to take all values
            # Make an empty dataFrame
            df_datos = pd.DataFrame()
            # Add values and name field to empty dataFrame
            df_datos[campo_seleccionado] = datos
            # Change de Pandas DataFrame to Pandas numeric DataSeries
            datos = pd.to_numeric(df_datos[campo_seleccionado], errors='coerce')
            # Return the Pandas DataSeries
            return datos

    # Method to make the stats when push button
    def btn_calcular_click(self):
        """
        btn_calcular_click
        display a message with stats and make the graphs
        """
        # Set to 6 the decimal places
        self.dlg.spBoxDecimales.setValue(6)
        # Get values from selected field
        datos = self.obtener_valores_de_campo()
        # make stats
        if datos is None:
            mensaje = 'No hay datos'
            self.dlg.lbResultados.setText(mensaje)
            self.actualizar_grafica()
        else:
            estadisticas_campo = self.estadisticas_num(datos)
            # Round to 6 decimal places by default
            redondeo_estadisticas = [round(valor, 6) for valor in estadisticas_campo]
            # Message with stats results
            mensaje = 'Registros: {}'.format(str(int(redondeo_estadisticas[0])))
            mensaje += '\nNulos: {}'.format(str(int(redondeo_estadisticas[1])))
            mensaje += '\nSuma: {}'.format(str(redondeo_estadisticas[2]))
            mensaje += '\nMedia: {}'.format(str(redondeo_estadisticas[3]))
            mensaje += '\nSD (muestra): {}'.format(str(redondeo_estadisticas[4]))
            mensaje += '\nMín: {}'.format(str(redondeo_estadisticas[5]))
            mensaje += '\n(Q1: 0.25): {}'.format(str(redondeo_estadisticas[6]))
            mensaje += '\n(Q2: 0.5): {}'.format(str(redondeo_estadisticas[7]))
            mensaje += '\n(Q3: 0.75): {}'.format(str(redondeo_estadisticas[8]))
            mensaje += '\nMáx: {}'.format(str(redondeo_estadisticas[9]))
            # Refresh message
            self.dlg.lbResultados.setText(mensaje)
            # Refresh histogram and boxplot
            self.actualizar_grafica()

    # Method to round to given decimal place
    def num_decimales(self):
        """
        num_decimales
        Refresh message and decimal places from stats
        """
        # Get the value from spBoxDecimales
        decimales = self.dlg.spBoxDecimales.value()
        # Get values from selected field
        datos = self.obtener_valores_de_campo()
        if datos is None:
            mensaje = 'No hay datos'
            self.dlg.lbResultados.setText(mensaje)
        else:
            # Calculate stats
            estadisticas_campo = self.estadisticas_num(datos)
            # Round stats to given decimal place
            redondeo_estadisticas = [round(valor, decimales) for valor in estadisticas_campo]
            # Message with stats results
            mensaje = 'Registros: {}'.format(str(int(redondeo_estadisticas[0])))
            mensaje += '\nNulos: {}'.format(str(int(redondeo_estadisticas[1])))
            mensaje += '\nSuma: {}'.format(str(redondeo_estadisticas[2]))
            mensaje += '\nMedia: {}'.format(str(redondeo_estadisticas[3]))
            mensaje += '\nSD (muestra): {}'.format(str(redondeo_estadisticas[4]))
            mensaje += '\nMín: {}'.format(str(redondeo_estadisticas[5]))
            mensaje += '\n(Q1: 0.25): {}'.format(str(redondeo_estadisticas[6]))
            mensaje += '\n(Q2: 0.5): {}'.format(str(redondeo_estadisticas[7]))
            mensaje += '\n(Q3: 0.75): {}'.format(str(redondeo_estadisticas[8]))
            mensaje += '\nMáx: {}'.format(str(redondeo_estadisticas[9]))
            # Refresh message
            self.dlg.lbResultados.setText(mensaje)

    # Method to update stats
    def estadisticas_num(self, data_series):
        """
        estadisticas_num
        Get a pandas DataSeries to calc stats
        Return a list with stats
        """
        # Empty list
        lista_resultados = []
        # add total values (count)
        lista_resultados.append(data_series.size)
        # add count null or nan values
        lista_resultados.append(data_series.isnull().sum())
        # add Sum
        lista_resultados.append(data_series.sum())
        # Add media
        lista_resultados.append(data_series.mean())
        # Add standar deviation, calculate for 
        lista_resultados.append(data_series.std())
        # Add min
        lista_resultados.append(data_series.min())
        # Add Q1 quantil 0.25
        lista_resultados.append(data_series.quantile(q=0.25))
        # Add Q2 quantil 0.5
        lista_resultados.append(data_series.quantile(q=0.5))
        # Add Q3 quantil 0.75
        lista_resultados.append(data_series.quantile(q=0.75))
        # Add max
        lista_resultados.append(data_series.max())
        # Return list of stats
        return lista_resultados
    
    # Method to make a histogram and boxplot
    def agregar_grafica(self):
        """
        agregar_grafica
        Add axes to canvas to make histogram and boxplot
        """
        # clear all graph
        self.limpiar_grafica()
        # graph style
        estilos = plt.style.available
        # look for proper style
        if 'seaborn-v0_8-whitegrid' in estilos:
            plt.style.use('seaborn-v0_8-whitegrid')
        elif 'seaborn-whitegrid' in estilos:
            plt.style.use('seaborn-whitegrid')
        elif 'ggplot' in estilos:
            plt.style.use('ggplot')
        elif 'bmh' in estilos:
            plt.style.use('bmh')
        else:
            plt.style.use('classic')
        # add figure to canvas
        self.figura = plt.figure(figsize=(4, 2.8), tight_layout=True)
        # make an instance of FigureCanvas and add figure without axes
        self.canvas = FigureCanvas(self.figura)
        # add an instance of canvas
        self.dlg.graficaEstadistica.addWidget(self.canvas)        
        # add axes to make graph
        self.axes = self.canvas.figure.subplots(nrows = 2, # two rows
                                                ncols=1, # one column
                                                sharex=True, # share values of x axis
                                                gridspec_kw={"height_ratios": (.3, .7)} # boxplot in 30% and histogram in 70%
                                                )

    # Method to refresh boxplot and histogram
    def actualizar_grafica(self):
        """
        actualizar_grafica
        make a graph with boxplot and histogram
        """
        # add graph
        self.agregar_grafica()
        # get values to graph
        datos = self.obtener_valores_de_campo()
        # check if data is not empty
        if datos is None:
            self.agregar_grafica()
            # refreh graph
            self.canvas.draw()
        # check if values is not empty
        elif datos.size == 0:
            #add graph
            self.agregar_grafica()
            # refresh canvas
            self.canvas.draw()
        elif datos.isnull().all():
            # add graph
            self.agregar_grafica()
            # refresh canvas
            self.canvas.draw()
        else:
            # Make axes with boxplot
            self.axes[0].boxplot(datos, # values to boxplot
                                 vert=False, # horizontal boxplot
                                 labels= [''], # erase labels from boxplot
                                 widths=(0.7), # set wide for boxplot
                                 showmeans=True, # show media value in boxplot
                                 meanprops={'marker': 'o',
                                            'markersize': 3,
                                            'markerfacecolor':'red',
                                            'markeredgecolor':'none'}, # set color and size to media point
                                 patch_artist=True, # fill box, if false the box is empty
                                 boxprops = dict(facecolor = "yellowgreen"), # set box color
                                 flierprops={'marker': 'o',
                                             'markersize': 3,
                                             'markerfacecolor': 'black',
                                             'markeredgecolor':'none'}, # set outliers color and size
                                 showfliers=True # show outliers, if False hide
                                 )
            
            # Make axes with histogram
            self.axes[1].hist(datos, # values to histogram
                              bins='sqrt', # method for bins size
                              color = "yellowgreen" # color graph
                              )
            
            # Add vertical line with mean value
            self.axes[1].axvline(x = datos.mean(), # value to graph: mean
                                 color = 'black',
                                 linestyle='dashed',
                                 linewidth= 1,
                                 label='Media'
                                 )
            
            # Add vertical line with Q1 value
            self.axes[1].axvline(x = datos.quantile(0.25), # value to graph: Q1
                                 color = 'blue',
                                 linestyle='dashed',
                                 linewidth= 1,
                                 label='Q1'
                                 )
            
            # Add vertical line with Q3 value
            self.axes[1].axvline(x = datos.quantile(0.75), # value to graph: Q3
                                 color = 'red',
                                 linestyle='dashed',
                                 linewidth= 1,
                                 label='Q3'
                                 )
            
            # Add title to histogram
            self.axes[1].set_title('Distribución de frecuencias', fontsize=9)
            self.axes[1].set_xlabel(datos.name, fontsize=9)
            self.axes[1].set_ylabel('Conteo', fontsize=9)
            self.axes[1].tick_params(axis='both', which='major', labelsize=9)
            self.axes[1].tick_params(axis='both', which='minor', labelsize=9)
            self.axes[1].legend(loc='best', fontsize='8')
            
            # Refresh plot
            self.canvas.draw()
    
    # Method to clean plot
    def limpiar_grafica(self):
        graficas_previas = []
        for i in range(self.dlg.graficaEstadistica.count()):
            grafica = self.dlg.graficaEstadistica.itemAt(i).widget()
            if grafica:
                graficas_previas.append(grafica)
        for grafica in graficas_previas:
            grafica.deleteLater()

