# pyright: reportMissingImports=false, reportMissingModuleSource=false
"""FiberQ Locator Dialog.

This module contains the dialog for finding addresses on map using OSM Nominatim.
"""

from qgis.PyQt.QtWidgets import (
    QDialog,
    QVBoxLayout,
    QLineEdit,
    QLabel,
    QDialogButtonBox,
    QFormLayout,
    QPushButton,
    QMessageBox,
)

# Phase 5.2: Logging
from ..utils.logger import get_logger
logger = get_logger(__name__)


class LocatorDialog(QDialog):
    """Dialog for entering address and positioning on map (OSM Nominatim)."""
    
    def __init__(self, core):
        super().__init__(core.iface.mainWindow())
        self.core = core
        self.setWindowTitle("Locator - Find Address on Map")
        self.setMinimumWidth(380)

        layout = QVBoxLayout(self)

        form = QFormLayout()
        self.edit_country = QLineEdit()
        self.edit_city = QLineEdit()
        self.edit_municipality = QLineEdit()
        self.edit_street = QLineEdit()
        self.edit_number = QLineEdit()

        form.addRow("Country:", self.edit_country)
        form.addRow("City:", self.edit_city)
        form.addRow("Municipality (optional):", self.edit_municipality)
        form.addRow("Street:", self.edit_street)
        form.addRow("Number:", self.edit_number)
        layout.addLayout(form)

        btns = QDialogButtonBox(QDialogButtonBox.Close)
        self.btn_find = QPushButton("Find on map")
        btns.addButton(self.btn_find, QDialogButtonBox.ActionRole)
        self.btn_find.clicked.connect(self._on_find_clicked)
        btns.rejected.connect(self.reject)
        layout.addWidget(btns)

    def _on_find_clicked(self):
        """Handle find button click - geocode the address."""
        # Build query
        s = self.edit_street.text().strip()
        n = self.edit_number.text().strip()
        parts = []
        if s:
            parts.append(s + (f" {n}" if n else ""))

        m = self.edit_municipality.text().strip()
        if m:
            parts.append(m)

        c = self.edit_city.text().strip()
        if c:
            parts.append(c)

        co = self.edit_country.text().strip()
        if co:
            parts.append(co)

        if not parts:
            QMessageBox.warning(self, "Locator", "Please enter at least City and Country (Street/Number recommended).")
            return

        query = ", ".join(parts)

        # Call Nominatim (OSM) – without additional libraries
        try:
            import urllib.parse, urllib.request, json, ssl

            url = "https://nominatim.openstreetmap.org/search?format=json&limit=1&q=" + urllib.parse.quote(query)
            headers = {"User-Agent": "FiberQ/1.0 (contact: vukovicvl@fiberq.net)"}
            req = urllib.request.Request(url, headers=headers)
            context = ssl.create_default_context()
            with urllib.request.urlopen(req, context=context, timeout=15) as resp:
                data = json.loads(resp.read().decode("utf-8"))

            if not data:
                QMessageBox.information(self, "Locator", f"No location found for: {query}")
                return

            lat = float(data[0].get("lat"))
            lon = float(data[0].get("lon"))
            self.core._center_and_mark_wgs84(lon, lat, label=query)
            self.accept()
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Error during geocoding: {e}")


__all__ = ['LocatorDialog']
