"""
FiberQ v2 - Manhole Tool

Tool for placing manholes on the route network.
"""

from .base import (
    Qt,
    QgsProject, QgsVectorLayer, QgsFeature, QgsGeometry,
    QgsPointXY, QgsWkbTypes,
    QgsMapToolEmitPoint
)

# Phase 5.2: Logging
from ..utils.logger import get_logger
logger = get_logger(__name__)


class ManholePlaceTool(QgsMapToolEmitPoint):
    """
    Tool for placing manholes on the map.
    
    Snaps to route vertices and segment midpoints.
    Uses attributes previously set via the manhole dialog.
    """
    
    def __init__(self, iface, plugin):
        super().__init__(iface.mapCanvas())
        self.iface = iface
        self.plugin = plugin
    
    def _find_route_layer(self):
        """Find the Route layer in the project."""
        for lyr in QgsProject.instance().mapLayers().values():
            try:
                if (isinstance(lyr, QgsVectorLayer) and
                    lyr.name() in ('Route', 'Trasa') and
                    lyr.geometryType() == QgsWkbTypes.LineGeometry):
                    return lyr
            except Exception as e:
                logger.debug(f"Error in ManholePlaceTool._find_route_layer: {e}")
        return None
    
    def _snap_to_route(self, point):
        """
        Snap a point to route vertices or segment midpoints.
        
        Returns:
            Tuple of (snapped_point, min_distance) or (None, None)
        """
        route_layer = self._find_route_layer()
        if route_layer is None or route_layer.featureCount() == 0:
            return None, None
        
        snap_point = None
        min_dist = None
        
        for feat in route_layer.getFeatures():
            geom = feat.geometry()
            if geom.isMultipart():
                lines = geom.asMultiPolyline()
            else:
                lines = [geom.asPolyline()]
            
            for line in lines:
                if not line:
                    continue
                
                # Check all vertices
                for p in line:
                    d = QgsPointXY(point).distance(QgsPointXY(p))
                    if min_dist is None or d < min_dist:
                        min_dist = d
                        snap_point = QgsPointXY(p)
                
                # Check segment midpoints
                for i in range(len(line) - 1):
                    mid = QgsPointXY(
                        (line[i].x() + line[i + 1].x()) / 2,
                        (line[i].y() + line[i + 1].y()) / 2
                    )
                    d = QgsPointXY(point).distance(mid)
                    if min_dist is None or d < min_dist:
                        min_dist = d
                        snap_point = mid
        
        return snap_point, min_dist
    
    def canvasReleaseEvent(self, event):
        """Handle mouse release - place manhole."""
        point = self.toMapCoordinates(event.pos())
        
        # Snap to route
        snap_point, min_dist = self._snap_to_route(point)
        tolerance = self.iface.mapCanvas().mapUnitsPerPixel() * 20
        
        if snap_point is not None and min_dist is not None and min_dist < tolerance:
            point = snap_point
        
        # Get or create manhole layer
        layer = self.plugin._ensure_okna_layer()
        
        # Create feature
        f = QgsFeature(layer.fields())
        f.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(point)))
        
        # Get pending attributes from plugin
        attrs = getattr(self.plugin, '_manhole_pending_attrs', {}) or {}
        
        # Set manhole ID
        broj = attrs.get('broj_okna') or ''
        f['broj_okna'] = broj
        
        # Set other attributes
        attr_fields = [
            'tip_okna', 'vrsta_okna', 'polozaj_okna', 'adresa', 'stanje',
            'god_ugrad', 'opis', 'dimenzije', 'mat_zida', 'mat_poklop',
            'odvodnj', 'poklop_tes', 'poklop_lak', 'br_nosaca', 'debl_zida', 'lestve'
        ]
        
        for key in attr_fields:
            if key in attrs and key in layer.fields().names():
                f[key] = attrs.get(key)
        
        # Add feature
        try:
            layer.startEditing()
            layer.addFeature(f)
            layer.commitChanges()
        except Exception as e:
            layer.dataProvider().addFeatures([f])
        
        layer.updateExtents()
        layer.triggerRepaint()
        
        # Apply style
        self.plugin._apply_manhole_style(layer)
        self.plugin._move_layer_to_top(layer)
        
        # Show message
        self.iface.messageBar().pushInfo(
            "Manholes",
            f"Manhole placed: {broj or '(no ID)'}"
        )
    
    def keyPressEvent(self, event):
        """Handle ESC key to cancel tool."""
        if event.key() == Qt.Key_Escape:
            try:
                self.iface.mapCanvas().unsetMapTool(self)
            except Exception as e:
                logger.debug(f"Error in ManholePlaceTool.keyPressEvent: {e}")
    
    def canvasPressEvent(self, event):
        """Handle right-click to cancel tool."""
        if event.button() == Qt.RightButton:
            try:
                self.iface.mapCanvas().unsetMapTool(self)
            except Exception as e:
                logger.debug(f"Error in ManholePlaceTool.canvasPressEvent: {e}")


__all__ = ['ManholePlaceTool']
