# pyright: reportMissingImports=false, reportMissingModuleSource=false
"""FiberQ Settings Dialog.

This module contains the main plugin settings dialog.
"""

from qgis.PyQt.QtWidgets import (
    QDialog,
    QVBoxLayout,
    QLineEdit,
    QLabel,
    QComboBox,
    QCheckBox,
    QPushButton,
)

from qgis.core import QgsSettings

# Phase 5.2: Logging
from ..utils.logger import get_logger
logger = get_logger(__name__)


class FiberQSettingsDialog(QDialog):
    """Dialog for configuring FiberQ plugin settings."""
    
    def __init__(self, parent=None):
        super().__init__(parent)

        self.setWindowTitle("FiberQ Settings")
        self.settings = QgsSettings()

        layout = QVBoxLayout()

        # Default cable type
        self.cb_default_cable = QComboBox()
        self.cb_default_cable.addItems([
            "Optical – 12F", "Optical – 24F", "Optical – 48F",
            "Optical – 96F", "Optical – 144F"
        ])

        saved = self.settings.value("FiberQ/default_cable_type", "", type=str)
        if saved:
            idx = self.cb_default_cable.findText(saved)
            if idx >= 0:
                self.cb_default_cable.setCurrentIndex(idx)

        layout.addWidget(QLabel("Default Cable Type:"))
        layout.addWidget(self.cb_default_cable)

        # Default slack length (m)
        self.ed_slack = QLineEdit()
        self.ed_slack.setPlaceholderText("10")
        self.ed_slack.setText(self.settings.value("FiberQ/default_slack_length", "10"))
        layout.addWidget(QLabel("Default Slack Length (m):"))
        layout.addWidget(self.ed_slack)

        # Default snapping distance (px)
        self.ed_snap = QLineEdit()
        self.ed_snap.setPlaceholderText("15")
        self.ed_snap.setText(self.settings.value("FiberQ/default_snap_distance", "15"))
        layout.addWidget(QLabel("Default Snapping Distance (px):"))
        layout.addWidget(self.ed_snap)

        # Auto labels
        self.chk_labels = QCheckBox("Enable automatic labels")
        self.chk_labels.setChecked(
            self.settings.value("FiberQ/auto_labels", "true") == "true"
        )
        layout.addWidget(self.chk_labels)

        # Default semantic diagram style
        self.cb_semantic = QComboBox()
        self.cb_semantic.addItems(["Classic", "Compact", "Detailed"])
        saved_style = self.settings.value("FiberQ/default_semantic_style", "Classic")
        idx = self.cb_semantic.findText(saved_style)
        if idx >= 0:
            self.cb_semantic.setCurrentIndex(idx)

        layout.addWidget(QLabel("Default Semantic Diagram Style:"))
        layout.addWidget(self.cb_semantic)

        # Save button
        btn_save = QPushButton("Save Settings")
        btn_save.clicked.connect(self.save)
        layout.addWidget(btn_save)

        self.setLayout(layout)

    def save(self):
        """Save settings and close dialog."""
        self.settings.setValue("FiberQ/default_cable_type", self.cb_default_cable.currentText())
        self.settings.setValue("FiberQ/default_slack_length", self.ed_slack.text())
        self.settings.setValue("FiberQ/default_snap_distance", self.ed_snap.text())
        self.settings.setValue("FiberQ/auto_labels", "true" if self.chk_labels.isChecked() else "false")
        self.settings.setValue("FiberQ/default_semantic_style", self.cb_semantic.currentText())
        self.accept()


__all__ = ['FiberQSettingsDialog']
