# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FeatureZSetter
                                 A QGIS plugin
 Sets the Z value of each new/edited feature to a specific value, based on a DEM layer plus an offset. 
 Supports MultiPoint and MultiLineString geometries.
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, Qt
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QToolBar, QMessageBox
from qgis.core import QgsProject, QgsMapLayer, QgsWkbTypes, QgsMessageLog, QgsGeometry, QgsPoint, Qgis
from qgis.utils import iface

import re
from osgeo import gdal
import os.path

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .feature_z_setter_dockwidget import FeatureZSetterDockWidget


class FeatureZSetter:

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            f'i18n/FeatureZSetter_{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr('&Feature Z Setter')

        # Check for CIGeoE toolbar. If exists, add button there; if not exists, create one
        cigeoeToolBarExists = False
        for tb in iface.mainWindow().findChildren(QToolBar):
            if tb.windowTitle() == 'CIGeoE':
                self.toolbar = tb
                cigeoeToolBarExists = True
                break
        if not cigeoeToolBarExists:
            self.toolbar = self.iface.addToolBar('CIGeoE')

        self.toolbar.setObjectName('FeatureZSetter')

        self.pluginIsActive = False
        self.dockwidget = None

        # Instance variables
        self.demLayerName = None
        self.demGDALLayer = None
        self.demLayerGeoTransformed = None
        self.layerBeingEdited = None
        self.noDEMLayerName = "** None **"

    def tr(self, message):
        return QCoreApplication.translate('FeatureZSetter', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        icon_path = ':/plugins/feature_z_setter/icon.png'
        self.add_action(
            icon_path,
            text=self.tr('Feature Z Setter: Set the Z value of features based on a DEM'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.tr('&Feature Z Setter'), action)
            self.iface.removeToolBarIcon(action)
        del self.toolbar

    def run(self):
        if not self.pluginIsActive:
            self.pluginIsActive = True
            if self.dockwidget is None:
                self.dockwidget = FeatureZSetterDockWidget()

                # Connect dockwidget signals to class methods
                self.dockwidget.demLayerCombo.currentIndexChanged.connect(self.newDEMLayerSelected)
                self.dockwidget.refreshButton.clicked.connect(self.refreshLayers)
                self.dockwidget.offsetValue.textChanged.connect(self.refreshLayers)
                self.dockwidget.drapeButton.clicked.connect(self.setAllFeatureZCoordinatesToDEMValue)
                self.dockwidget.flattenButton.clicked.connect(self.setAllFeatureZCoordinatesToItsMaxValue)

            self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

            # Initialize layers and variables
            self.loadLayersIntoComboBox()
            self.setLayerBeingEditedVar()
        else:
            if self.dockwidget is not None:
                self.dockwidget.close()
                self.iface.removeDockWidget(self.dockwidget)
            self.pluginIsActive = False

    def reconnect(self, signal, newhandler=None, oldhandler=None):
        """
        Disconnect oldhandler and connect newhandler to signal.
        """
        while True:
            try:
                if oldhandler is not None:
                    signal.disconnect(oldhandler)
                else:
                    signal.disconnect()
            except TypeError:
                break
        if newhandler is not None:
            signal.connect(newhandler)

    def refreshLayers(self):
        """
        Refresh DEM layers list and active vector layer.
        """
        if not self.pluginIsActive:
            self.run()
            return
        self.demLayerName = None
        self.loadLayersIntoComboBox()
        self.setLayerBeingEditedVar()

    def loadLayersIntoComboBox(self):
        """
        Load raster layers into DEM combo box.
        """
        layers = list(QgsProject.instance().mapLayers().values())
        layers_list = [self.noDEMLayerName]
        for layer in layers:
            if layer.type() == QgsMapLayer.RasterLayer:
                layers_list.append(layer.name())
        self.dockwidget.demLayerCombo.clear()
        self.dockwidget.demLayerCombo.addItems(layers_list)
        self.newDEMLayerSelected()

    def newDEMLayerSelected(self):
        """
        Update selected DEM layer and open raster with GDAL.
        """
        selectedLayerIndex = self.dockwidget.demLayerCombo.currentIndex()
        if selectedLayerIndex < 0:
            return
        self.demLayerName = self.dockwidget.demLayerCombo.currentText()
        layerChecked = False
        layers = list(QgsProject.instance().mapLayers().values())
        for layer in layers:
            if layer.name() == self.demLayerName and layer.type() == QgsMapLayer.RasterLayer:
                layerPath = layer.dataProvider().dataSourceUri()
                try:
                    self.demGDALLayer = gdal.Open(layerPath)
                except Exception:
                    self.demGDALLayer = None
                if self.demGDALLayer is not None:
                    bands = self.demGDALLayer.RasterCount
                    if bands >= 1:
                        self.demLayerGeoTransformed = self.demGDALLayer.GetGeoTransform()
                        layerChecked = True
                        QgsMessageLog.logMessage(f'Layer "{layer.name()}" opened as a DEM, new Z will take it into consideration.', 'FeatureZSetter')
                        break
        if not layerChecked:
            self.demLayerGeoTransformed = None
            self.demGDALLayer = None

    def setLayerBeingEditedVar(self):
        """
        Update the active editable vector layer.
        """
        tempLayer = self.iface.activeLayer()
        if tempLayer is not None and tempLayer.type() == QgsMapLayer.VectorLayer:
            self.layerBeingEdited = tempLayer
            if self.layerBeingEdited.isEditable():
                QgsMessageLog.logMessage(f'{self.layerBeingEdited.name()} is being tracked for changes', 'FeatureZSetter')
        else:
            self.layerBeingEdited = None

    def _set_z_for_geometry(self, geom, get_z_func):
        """
        Return a new QgsGeometry with Z values set using get_z_func(x,y).
        Supports simple and multi geometries.
        """
        wkbType = geom.wkbType()
        geomType = QgsWkbTypes.geometryType(wkbType)
        isMulti = QgsWkbTypes.isMultiType(wkbType)

        if geomType == QgsWkbTypes.PointGeometry:
            if isMulti:
                pts = geom.asMultiPoint()
                new_pts = []
                for p in pts:
                    try:
                        z = get_z_func(p.x(), p.y())
                        p.setZ(z)
                    except Exception:
                        p = QgsPoint(p.x(), p.y(), get_z_func(p.x(), p.y()))
                    new_pts.append(p)
                return QgsGeometry.fromMultiPoint(new_pts)
            else:
                p = geom.asPoint()
                try:
                    z = get_z_func(p.x(), p.y())
                    p.setZ(z)
                except Exception:
                    p = QgsPoint(p.x(), p.y(), get_z_func(p.x(), p.y()))
                return QgsGeometry.fromPoint(p)

        elif geomType == QgsWkbTypes.LineGeometry:
            if isMulti:
                parts = geom.asMultiPolyline()
                new_parts = []
                for line in parts:
                    new_line = []
                    for p in line:
                        try:
                            z = get_z_func(p.x(), p.y())
                            p.setZ(z)
                        except Exception:
                            p = QgsPoint(p.x(), p.y(), get_z_func(p.x(), p.y()))
                        new_line.append(p)
                    new_parts.append(new_line)
                return QgsGeometry.fromMultiPolyline(new_parts)
            else:
                line = geom.asPolyline()
                new_line = []
                for p in line:
                    try:
                        z = get_z_func(p.x(), p.y())
                        p.setZ(z)
                    except Exception:
                        p = QgsPoint(p.x(), p.y(), get_z_func(p.x(), p.y()))
                    new_line.append(p)
                return QgsGeometry.fromPolyline(new_line)

        elif geomType == QgsWkbTypes.PolygonGeometry:
            if isMulti:
                parts = geom.asMultiPolygon()
                new_parts = []
                for poly in parts:
                    new_poly = []
                    for ring in poly:
                        new_ring = []
                        for p in ring:
                            try:
                                z = get_z_func(p.x(), p.y())
                                p.setZ(z)
                            except Exception:
                                p = QgsPoint(p.x(), p.y(), get_z_func(p.x(), p.y()))
                            new_ring.append(p)
                        new_poly.append(new_ring)
                    new_parts.append(new_poly)
                return QgsGeometry.fromMultiPolygon(new_parts)
            else:
                poly = geom.asPolygon()
                new_poly = []
                for ring in poly:
                    new_ring = []
                    for p in ring:
                        try:
                            z = get_z_func(p.x(), p.y())
                            p.setZ(z)
                        except Exception:
                            p = QgsPoint(p.x(), p.y(), get_z_func(p.x(), p.y()))
                        new_ring.append(p)
                    new_poly.append(new_ring)
                return QgsGeometry.fromPolygon(new_poly)

        else:
            return geom

    def is_geometry_with_z(self, geom):
        """
        Check if geometry has Z coordinates.
        """
        try:
            return QgsWkbTypes.hasZ(geom.wkbType())
        except Exception:
            try:
                for v in geom.vertices():
                    try:
                        if v.z() is not None:
                            return True
                    except Exception:
                        continue
            except Exception:
                pass
        return False

    def getZValueFromRaster(self, x, y, layer, layerGeoTransform):
        """
        Get Z value from DEM raster at coordinate (x,y).
        """
        try:
            px = int((float(x) - float(layerGeoTransform[0])) / float(layerGeoTransform[1]))
            py = int((float(y) - float(layerGeoTransform[3])) / float(layerGeoTransform[5]))
            band = layer.GetRasterBand(1)
            data = band.ReadAsArray(px, py, 1, 1)
            try:
                res = data[0][0]
            except Exception:
                res = 0
                QgsMessageLog.logMessage(f"FeatureZSetter: no DEM value available at {x},{y}. Considering offset only.", 'FeatureZSetter')
            return res
        except Exception as e:
            QgsMessageLog.logMessage(f"FeatureZSetter: error reading DEM value: {e}", 'FeatureZSetter')
            return 0

    def featureAddedTriggered(self, fid):
        """
        Update geometry of added feature with Z values based on DEM and offset.
        """
        if self.layerBeingEdited is None:
            return
        feat = None
        for f in self.layerBeingEdited.getFeatures():
            if f.id() == fid:
                feat = f
                break
        if feat is None:
            return

        geom = feat.geometry()
        if geom is None or geom.isEmpty():
            return

        if not self.is_geometry_with_z(geom):
            QgsMessageLog.logMessage("User     tried to add a geometry without Z. No action taken.", 'FeatureZSetter')
            return

        try:
            offset = float(self.dockwidget.offsetValue.text())
        except Exception:
            offset = 0.0
            self.dockwidget.offsetValue.setText("0")
            QgsMessageLog.logMessage("FeatureZSetter: invalid offset value. Set to 0.", 'FeatureZSetter')

        if self.demGDALLayer is not None and self.demLayerGeoTransformed is not None:
            def get_z_func(x, y):
                return self.getZValueFromRaster(x, y, self.demGDALLayer, self.demLayerGeoTransformed) + offset
        else:
            def get_z_func(x, y):
                return offset

        newGeom = self._set_z_for_geometry(geom, get_z_func)
        try:
            feat.setGeometry(newGeom)
            self.layerBeingEdited.updateFeature(feat)
            QgsMessageLog.logMessage(f"FeatureZSetter: geometry updated for added feature id {feat.id()}", 'FeatureZSetter')
        except Exception as e:
            QgsMessageLog.logMessage(f"FeatureZSetter error updating geometry: {e}", 'FeatureZSetter')

    def setAllFeatureZCoordinatesToDEMValue(self):
        """
        Drape selected features' Z coordinates to DEM + offset.
        """
        tempLayer = self.iface.activeLayer()
        if tempLayer is None or not tempLayer.isEditable():
            QMessageBox.information(self.iface.mainWindow(), "Error", "There's no layer selected or selected layer not editable.")
            return

        features = tempLayer.selectedFeatures()
        if not features:
            QMessageBox.information(self.iface.mainWindow(), "Error", "There's no features selected in active layer.")
            return

        try:
            offset = float(self.dockwidget.offsetValue.text())
        except Exception:
            offset = 0.0
            self.dockwidget.offsetValue.setText("0")
            QgsMessageLog.logMessage("FeatureZSetter: invalid offset value. Set to 0.", 'FeatureZSetter')

        changedGeometries = 0
        noZValue = False

        for f in features:
            geom = f.geometry()
            if geom is None or geom.isEmpty():
                continue

            if not self.is_geometry_with_z(geom):
                QgsMessageLog.logMessage(f"User     tried to drape a geometry that does not have a Z coordinate. Skipping feature id {f.id()}.", 'FeatureZSetter')
                noZValue = True
                continue

            if self.demGDALLayer is not None and self.demLayerGeoTransformed is not None:
                def get_z_func(x, y):
                    return self.getZValueFromRaster(x, y, self.demGDALLayer, self.demLayerGeoTransformed) + offset
            else:
                def get_z_func(x, y):
                    return offset

            try:
                newGeom = self._set_z_for_geometry(geom, get_z_func)
                f.setGeometry(newGeom)
                tempLayer.updateFeature(f)
                changedGeometries += 1
            except Exception as e:
                QgsMessageLog.logMessage(f"FeatureZSetter: failed draping feature {f.id()}: {e}", 'FeatureZSetter')
                continue

        if noZValue:
            QMessageBox.information(self.iface.mainWindow(), "Warning", "Some feature(s) did not have Z coordinate; they were skipped.")
        self.iface.messageBar().pushMessage("Feature Z Setter", f"{changedGeometries} geometries draped.", level=Qgis.Success)

        tempLayer.triggerRepaint()
        tempLayer.selectByIds([])
        tempLayer.selectByIds([k.id() for k in features])

    def setAllFeatureZCoordinatesToItsMaxValue(self):
        """
        Flatten selected features' Z coordinates to their maximum Z value.
        """
        tempLayer = self.iface.activeLayer()
        if tempLayer is None or not tempLayer.isEditable():
            QMessageBox.information(self.iface.mainWindow(), "Error", "There's no layer selected or selected layer not editable.")
            return

        features = tempLayer.selectedFeatures()
        if not features:
            QMessageBox.information(self.iface.mainWindow(), "Error", "There's no features selected in active layer.")
            return

        changedGeometries = 0
        noZValue = False

        for f in features:
            geom = f.geometry()
            if geom is None or geom.isEmpty():
                continue

            if not self.is_geometry_with_z(geom):
                QgsMessageLog.logMessage(f"User   tried to flatten a geometry that does not have a Z coordinate. Skipping feature id {f.id()}.", 'FeatureZSetter')
                noZValue = True
                continue

            maxZ = None
            try:
                for v in geom.vertices():
                    try:
                        zval = v.z()
                    except Exception:
                        zval = None
                    if zval is not None:
                        if maxZ is None or float(zval) > float(maxZ):
                            maxZ = zval
            except Exception:
                maxZ = None

            if maxZ is None:
                try:
                    wkt = geom.asWkt()
                    nums = re.findall(r'[-+]?\d*\.\d+|[-+]?\d+', wkt)
                    if len(nums) >= 3:
                        vals = [float(nums[i]) for i in range(2, len(nums), 3)]
                        maxZ = max(vals) if vals else 0.0
                    else:
                        maxZ = 0.0
                except Exception:
                    maxZ = 0.0

            def get_z_func_const(x, y, mz=maxZ):
                return float(mz)

            try:
                newGeom = self._set_z_for_geometry(geom, get_z_func_const)
                f.setGeometry(newGeom)
                tempLayer.updateFeature(f)
                changedGeometries += 1
            except Exception as e:
                QgsMessageLog.logMessage(f"FeatureZSetter: failed flattening feature {f.id()}: {e}", 'FeatureZSetter')