# -*- coding: utf-8 -*-
"""
/***************************************************************************
                        FeaturesBoundingBox for QGIS
                        
                                 A QGIS plugin
                                 
 Show the BBox info for the selected features
                               -------------------
        begin                : 2024-07-01
        copyright            : (C) 2024 by Michel Stuyts
        email                : info@michelstuyts.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QTimer
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMenu, QApplication
from qgis.core import QgsMapLayer, QgsProject, QgsCoordinateReferenceSystem, QgsCoordinateTransform
from qgis.gui import QgsProjectionSelectionDialog

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .fbb_dialog import FBBDialog
import os.path, webbrowser

class FBB:
    def __init__(self, iface):
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FBB_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&FeaturesBoundingBox')
        
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        
    def CopyClipboard(self, text="abc"):
        cb = QApplication.clipboard()
        cb.clear(mode=cb.Clipboard )
        cb.setText(text, mode=cb.Clipboard)
        
    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('FBB', message)


    def add_action(
        self,
        icon_path,
        info_icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=u'Show bounding box info of selected features',
        whats_this=u'Show bounding box info of selected features',
        parent=None):

        icon = QIcon(icon_path)
        infoicon = QIcon(info_icon_path)
        # print(info_icon_path)
                
        actioninfo = QAction(infoicon, self.tr(u'More info'), parent)
        actioninfo.triggered.connect(lambda: webbrowser.open_new_tab("https://gitlab.com/GIS-projects/featuresboundingbox-for-qgis"))
        actioninfo.setEnabled(enabled_flag)
        
        
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(self.tr(status_tip))

        if whats_this is not None:
            action.setWhatsThis(self.tr(whats_this))

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        '''if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)'''
        if add_to_menu:        
            """self.plugin_menu = self.iface.vectorMenu().addMenu(
                QIcon(":/plugins/fbb/icon.svg"), self.tr(u'&FeaturesBoundingBox'),

            )
            self.plugin_menu.addAction(action)
            self.iface.editMenu().addAction(action)"""
            self.fbb_menu = QMenu(self.tr(u'&FeaturesBoundingBox'))
            self.fbb_menu.setIcon(icon)
            self.vector_menu = self.iface.vectorMenu()
            self.vector_menu.addMenu(self.fbb_menu)
            
            
            self.fbb_menu.addAction(action)
            self.fbb_menu.addAction(actioninfo)
        
        self.actions.append(actioninfo)
        self.actions.append(action)
        

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/fbb/icon.png'
        info_icon_path = ':/plugins/fbb/info.png'
        self.add_action(
            icon_path,
            info_icon_path,
            text=self.tr(u'Show bounding box info of selected features'),
            callback=self.run,
            parent=self.iface.mainWindow())

        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&FeaturesBoundingBox'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        if self.first_start == True:
            self.first_start = False
            self.dlg = FBBDialog()

        if self.iface.activeLayer():
            if self.iface.activeLayer().type() == QgsMapLayer.VectorLayer:
                self.dlg.show()
            else:
                pass
        else:
            pass
        
        bbox = self.getBB()
        self.setBB(bbox)
        self.dlg.label_crsbutton_project.setText(self.tr(u"CRS Project"))
        self.dlg.main_title.setText(u"<b>"+self.tr(u"Bounding box info for the selected feature(s) in the current layer")+u"</b>")
        self.dlg.label_layer.setText(self.tr(u"Selected objects from layer:"))
        self.dlg.label_minx.setText(self.tr(u"Min x:"))
        self.dlg.label_maxx.setText(self.tr(u"Max x:"))
        self.dlg.label_miny.setText(self.tr(u"Min y:"))
        self.dlg.label_maxy.setText(self.tr(u"Max y:"))
        self.dlg.label_bbox.setText(self.tr(u"Bounding box:"))
        self.dlg.label_bbox_rounded.setText(self.tr(u"Bounding Box with rounded values:"))
        self.dlg.labelCopied.setText(self.tr(u"BBox Copied to Clipboard"))
        self.dlg.CopyBBOX.setText(self.tr(u'Copy'))
        self.dlg.CopyBBOX_rounded.setText(self.tr(u'Copy'))
        self.dlg.btnCrsProject.clicked.connect(self.SetToProjectCRS)
        self.dlg.btnCrsLayer.clicked.connect(self.SetToLayerCRS)
        
        if self.dlg.fbb_output.toPlainText() != '':
            self.dlg.CopyBBOX.clicked.connect(
                lambda: (
                self.CopyClipboard(self.dlg.fbb_output.toPlainText()),
                self.dlg.labelCopied.setVisible(True),
                QTimer.singleShot(2000, lambda: self.dlg.labelCopied.setVisible(False))
                )
            )
        if self.dlg.fbb_output_rounded.toPlainText() != '':
            self.dlg.CopyBBOX_rounded.clicked.connect(            lambda: (
                self.CopyClipboard(self.dlg.fbb_output_rounded.toPlainText()),
                self.dlg.labelCopied.setVisible(True),
                QTimer.singleShot(2000, lambda: self.dlg.labelCopied.setVisible(False))
                ))
        # Run the dialog event loop
        result = self.dlg.exec()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
      
        
    def SetToProjectCRS(self):
        projectCRS = self.iface.mapCanvas().mapSettings().destinationCrs().authid()
        bbox = self.getBB(projectCRS)
        self.setBB(bbox, projectCRS)

    def SetToLayerCRS(self):
        bbox = self.getBB()
        self.setBB(bbox)

    def setBB(self, bbox, InCRS=None):
        self.dlg.labelCopied.setVisible(False)
        if(bbox == None):
            self.dlg.fbb_output.setText("")  
            self.dlg.fbb_output_rounded.setText("")  

            self.dlg.textMinX.setPlainText("")
            self.dlg.textMinY.setPlainText("")
            self.dlg.textMaxX.setPlainText("")
            self.dlg.textMaxY.setPlainText("")
            self.dlg.btnCrsLayer.setText("")
            self.dlg.btnCrsProject.setText("")
            self.dlg.label_crs.setText("")
            self.dlg.textLayer.setPlainText(self.tr(u'First select a vector layer in the layers panel and select one or more features in that layer. Make sure your layer has a CRS set.'))
            self.dlg.textLayer.setStyleSheet("color: red; font-weight: bold;")
            self.dlg.label_crsbutton_layer.setVisible(False)
            self.dlg.label_crsbutton_project.setVisible(False)
            self.dlg.label_layer.setVisible(False)
            self.dlg.btnCrsLayer.setVisible(False)
            self.dlg.btnCrsProject.setVisible(False)
            self.dlg.label_minx.setVisible(False)
            self.dlg.label_maxx.setVisible(False)
            self.dlg.label_miny.setVisible(False)
            self.dlg.label_maxy.setVisible(False)
            self.dlg.textMinX.setVisible(False)
            self.dlg.textMaxX.setVisible(False)
            self.dlg.textMinY.setVisible(False)
            self.dlg.textMaxY.setVisible(False)
            self.dlg.label_bbox.setVisible(False)
            self.dlg.label_bbox_rounded.setVisible(False)
            self.dlg.CopyBBOX.setVisible(False)
            self.dlg.CopyBBOX_rounded.setVisible(False)
            self.dlg.fbb_output.setVisible(False)
            self.dlg.fbb_output_rounded.setVisible(False)
        else:
            self.dlg.textLayer.setStyleSheet("color: black; font-weight: normal;")
            self.dlg.label_layer.setVisible(True)
            self.dlg.btnCrsLayer.setVisible(True)
            self.dlg.label_minx.setVisible(True)
            self.dlg.label_maxx.setVisible(True)
            self.dlg.label_miny.setVisible(True)
            self.dlg.label_maxy.setVisible(True)
            self.dlg.textMinX.setVisible(True)
            self.dlg.textMaxX.setVisible(True)
            self.dlg.textMinY.setVisible(True)
            self.dlg.textMaxY.setVisible(True)
            self.dlg.label_bbox.setVisible(True)
            self.dlg.label_bbox_rounded.setVisible(True)
            self.dlg.CopyBBOX.setVisible(True)
            self.dlg.CopyBBOX_rounded.setVisible(True)
            self.dlg.fbb_output.setVisible(True)
            self.dlg.fbb_output_rounded.setVisible(True)
            projectCRS = self.iface.mapCanvas().mapSettings().destinationCrs().authid()
            layerCRS = self.iface.activeLayer().crs().authid()
            if(projectCRS == layerCRS):
                self.dlg.btnCrsProject.setVisible(False)
                self.dlg.label_crs.setText(self.tr(u"Bounding Box CRS:"))
                self.dlg.label_crsbutton_layer.setText(self.tr(u"CRS Layer and Project"))
                self.dlg.label_crsbutton_layer.setVisible(True)
                self.dlg.label_crsbutton_project.setVisible(False)
            else:
                self.dlg.btnCrsProject.setVisible(True)
                self.dlg.label_crs.setText(self.tr(u"Choose a Bounding Box CRS:"))
                self.dlg.label_crsbutton_layer.setText(self.tr(u"CRS Layer"))
                self.dlg.label_crsbutton_project.setVisible(True)
                self.dlg.label_crsbutton_layer.setVisible(True)
            self.dlg.fbb_output.setText(str(bbox[0]) + "," + str(bbox[1]) + "," + str(bbox[2]) + "," + str(bbox[3]))
            self.dlg.fbb_output_rounded.setText(str(round(bbox[0])) + "," + str(round(bbox[1])) + "," + str(round(bbox[2])) + "," + str(round(bbox[3])))

            self.dlg.textMinX.setPlainText(str(bbox[0]))
            self.dlg.textMinY.setPlainText(str(bbox[1]))
            self.dlg.textMaxX.setPlainText(str(bbox[2]))
            self.dlg.textMaxY.setPlainText(str(bbox[3]))
            self.dlg.textLayer.setPlainText(str(bbox[4])+" ("+str(bbox[6])+" "+self.tr(u'features selected')+")")
            if bbox[5] == "":
                layercrs = "unknown"
            else:
                layercrs = bbox[5]
            self.dlg.btnCrsLayer.setText(str(layercrs))
            
            if InCRS == layerCRS or InCRS == None:
                self.dlg.btnCrsLayer.setStyleSheet("color: white;background-color: green;  font-weight: bold;")
                self.dlg.btnCrsProject.setStyleSheet("color: black; font-weight: bold;")
            elif InCRS == projectCRS:
                self.dlg.btnCrsLayer.setStyleSheet("color: black; font-weight: bold;")
                self.dlg.btnCrsProject.setStyleSheet("color: white; background-color: green;  font-weight: bold;")
            self.dlg.btnCrsProject.setText(projectCRS)
            
    def getBB(self, crsSetting = None):
        if crsSetting == None and self.iface.activeLayer():
            crsSetting = self.iface.activeLayer().crs().authid()
        # if crsSetting != None:
        #     print(crsSetting)


        if self.iface.activeLayer():
            if self.iface.activeLayer().type() == QgsMapLayer.VectorLayer:
                selectedlayer = self.iface.activeLayer().name()
                layercrs = self.iface.activeLayer().crs().authid()
                xmax = None
                xmin = None
                ymax = None
                ymin = None
                countfeatures = 0
                for feature in self.iface.activeLayer().selectedFeatures():
                    countfeatures += 1
                    SelectedGeometry = feature.geometry()
                    sourceCrs = QgsCoordinateReferenceSystem(layercrs)
                    destCrs = QgsCoordinateReferenceSystem(crsSetting)
                    tr = QgsCoordinateTransform(sourceCrs, destCrs, QgsProject.instance())

                    SelectedGeometry.transform(tr)
                    
                    if xmax:
                        if SelectedGeometry.boundingBox().xMaximum() > xmax:
                            xmax = SelectedGeometry.boundingBox().xMaximum()
                    else:
                        xmax = SelectedGeometry.boundingBox().xMaximum()
                    if ymax:
                        if SelectedGeometry.boundingBox().yMaximum() > ymax:
                            ymax = SelectedGeometry.boundingBox().yMaximum()
                    else:
                        ymax = SelectedGeometry.boundingBox().yMaximum()
                    if xmin:
                        if SelectedGeometry.boundingBox().xMinimum() < xmin:
                            xmin = SelectedGeometry.boundingBox().xMinimum()
                    else:
                        xmin = SelectedGeometry.boundingBox().xMinimum()
                    if ymin:
                        if SelectedGeometry.boundingBox().yMinimum() < ymin:
                            ymin = SelectedGeometry.boundingBox().yMinimum()
                    else:
                        ymin = SelectedGeometry.boundingBox().yMinimum()
                if xmax and ymax and xmin and ymin:                
                    output = xmin,ymin,xmax,ymax, selectedlayer, layercrs, countfeatures
                else:
                    output = None
            else:
                output = None
        else:
            output = None

        return output
    