# -*- coding: utf-8 -*-

"""
/***************************************************************************
 FaunaliaToolkit
                                 A QGIS plugin
 Faunalia Spatial Analysis Toolkit
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-04-30
        copyright            : (C) 2022 by Matteo Ghetta (Faunalia)
        email                : matteo.ghetta@faunalia.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Matteo Ghetta (Faunalia)'
__date__ = '2022-04-30'
__copyright__ = '(C) 2022 by Matteo Ghetta (Faunalia)'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import requests
import json
import plotly.graph_objects as go
from datetime import datetime, timedelta

from qgis.PyQt.QtCore import QCoreApplication, Qt, QVariant

from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterDateTime,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingException,
                       QgsCoordinateReferenceSystem,
                       QgsProcessingParameterFeatureSink,
                       QgsFeature,
                       QgsWkbTypes,
                       QgsCoordinateReferenceSystem,
                       QgsFields,
                       QgsField,
                       QgsGeometry,
                       QgsPointXY,
                       QgsFeatureSink,
                       QgsProcessingParameterPoint,
)

class OpenMeteoWeatherCurrent(QgsProcessingAlgorithm):

    POINT = 'POINT'
    VARIABLES = 'VARIABLES'
    START_DATE = 'START_DATE'
    END_DATE = 'END_DATE'
    TIMEZONE = 'TIMEZONE'
    OUTPUT_PLOT = 'OUTPUT_PLOT'
    OUTPUT_JSON = 'OUTPUT_JSON'
    OUTPUT_LAYER = 'OUTPUT_LAYER'
    OUTPUT_VALUES = 'OUTPUT_VALUES'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.

        self.addParameter(
            QgsProcessingParameterPoint(
                self.POINT,
                self.tr('Point')
            )
        )


        self.meteo_variables = [
            ('Temperature 2 Meters (°C)', 'temperature_2m', go.Scatter(mode='lines', line_color='orange', name='Temperature 2 Meters (°C)')),
            ('Rain (mm)', 'rain', go.Bar(marker_color='blue', name='Rain (mm)')),
            ('Precipitation (mm)', 'precipitation', go.Bar(marker_color='lightblue', name='Precipitation (mm)')),
            ('Cloud cover (%)', 'cloudcover', go.Scatter(mode='lines', name='Cloudcover (%)', line_color='lightblue')),
            ('Wind speed (km/h)', 'windspeed_10m', go.Scatter(mode='lines', name='Wind speed (km/h)', line_color='blue')),
        ]

        self.addParameter(
            QgsProcessingParameterEnum(
                self.VARIABLES,
                self.tr('Meteo Variables'),
                [p[0] for p in self.meteo_variables],
                True
            )
        )

        today = datetime.now()
        minval = today - timedelta(days=60)
        maxval = today + timedelta(days=7)

        self.addParameter(
            QgsProcessingParameterDateTime(
                self.START_DATE,
                self.tr('Start Date'),
                QgsProcessingParameterDateTime.Date,
                defaultValue=None,
                # minValue=minval
            )
        )

        self.addParameter(
            QgsProcessingParameterDateTime(
                self.END_DATE,
                self.tr('End Date'),
                QgsProcessingParameterDateTime.Date,
                defaultValue=None,
                maxValue=maxval
            )
        )

        self.timezones = [
            ('Automatically detect time zone', 'auto'),
            ('America/Anchorage', 'America/Anchorage'),
            ('America/Los_Angeles', 'America/Los_Angeles'),
            ('America/Denver', 'America/Denver'),
            ('America/Chicago', 'America/Chicago'),
            ('America/New_York', 'America/New_York'),
            ('America/Sao_Paulo', 'America/Sao_Paulo'),
            ('GMT+0', 'GMT+0'),
            ('Europe/London', 'Europe/London'),
            ('Europe/Berlin', 'Europe/Berlin'),
            ('Europe/Moscow', 'Europe/Moscow'),
            ('Africa/Cairo', 'Africa/Cairo'),
            ('Asia/Bangkok', 'Asia/Bangkok'),
            ('Asia/Singapore', 'Asia/Singapore'),
            ('Asia/Tokyo', 'Asia/Tokyo'),
            ('Australia/Sydney', 'Australia/Sydney'),
            ('Pacific/Auckland', 'Pacific/Auckland'),
        ]

        timezone_param = QgsProcessingParameterEnum(
            self.TIMEZONE,
            self.tr('Time zone'),
            [p[0] for p in self.timezones],
            defaultValue='Automatically detect time zone'
        )
        timezone_param.setFlags(timezone_param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(timezone_param)

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_PLOT,
                self.tr('Meteo plot'),
                self.tr('HTML files (*.html)')
            )
        )

        self.addParameter((
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_LAYER,
                self.tr('Place')
            )
        ))

        self.addParameter((
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_VALUES,
                self.tr('Values')
            )
        ))

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_JSON,
                self.tr('JSON data'),
                self.tr('JSON files (*.json)'),
                createByDefault=False,
                optional=True
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        point = self.parameterAsPoint(
            parameters,
            self.POINT,
            context,
            QgsCoordinateReferenceSystem("EPSG:4326") # the point will always be converted to 4326
        )

        url =  'https://api.open-meteo.com/v1/forecast?'

        start_date = self.parameterAsDateTime(
            parameters,
            self.START_DATE,
            context
        )
        start_date_str = start_date.date().toString(Qt.ISODate)

        end_date = self.parameterAsDateTime(
            parameters,
            self.END_DATE,
            context
        )
        end_date_str = end_date.date().toString(Qt.ISODate)

        # raise exception if the end date is before the start date
        if end_date < start_date:
            raise QgsProcessingException(self.tr('End Date is smaller than Start Date'))

        timezone_enum = self.parameterAsEnum(
            parameters,
            self.TIMEZONE,
            context
        )
        timezone = self.timezones[timezone_enum][1]


        output_plot = self.parameterAsFileOutput(
            parameters,
            self.OUTPUT_PLOT,
            context
        )

        output_json = self.parameterAsFileOutput(
            parameters,
            self.OUTPUT_JSON,
            context
        )

        # start building the url
        url+=f'latitude={point.y()}&longitude={point.x()}&start_date={start_date_str}&end_date={end_date_str}&timezone={timezone}'

        meteo = self.parameterAsEnums(
            parameters,
            self.VARIABLES,
            context
        )

        # start building the variables
        variables = '&hourly='

        # loop into the chosen variables and update the variable string
        for i in meteo:
            variables+=f'{self.meteo_variables[i][1]},'
            feedback.pushCommandInfo(f'Variables chosen: {self.meteo_variables[i][1]}')
        variables = variables.rstrip(',')

        # update the url with the variables
        url+=variables

        # print the final url
        feedback.pushDebugInfo(f'Build URL: {url}')

        # get the data as dictionary
        data = requests.get(url).json()

        # start empty figure
        fig = go.Figure()

        # fill the empty figure with the plots
        for i in meteo:
            trace = self.meteo_variables[i][2]
            trace.x = data['hourly']['time']
            trace.y = data['hourly'][self.meteo_variables[i][1]]
            fig.add_trace(trace)

        # line of today
        fig.add_vline(datetime.now(), line_color="red", line_dash="dot")

        # update the plot layout
        fig.update_layout(
            title=f'{round(data["latitude"], 2)}°N {round(data["longitude"], 2)}°E {data["elevation"]}m Time in {data["timezone"]}'
        )

        # fig = px.line(data['hourly'], data['hourly']['time'], data['hourly']['temperature_2m'])
        fig.write_html(output_plot)

        results = {}
        results[self.OUTPUT_PLOT] = output_plot

        if output_json:
            with open(output_json, 'w') as f:
                json.dump(data, f, indent=4)
            results[self.OUTPUT_JSON] = output_json

        fields = QgsFields()
        fields.append(QgsField('id', QVariant.Int))
        fields.append(QgsField('longitude', QVariant.Double))
        fields.append(QgsField('latitude', QVariant.Double))

        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT_LAYER,
            context,
            fields,
            QgsWkbTypes.Point,
            QgsCoordinateReferenceSystem('EPSG:4326')
        )

        # create the feature and add it to the sink
        feature = QgsFeature(fields)
        feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(data["longitude"], data["latitude"])))
        feature.setAttributes([1, data["longitude"], data["latitude"]])

        sink.addFeature(feature, QgsFeatureSink.FastInsert)

        # get the dictionary of values (values = {'time': [], 'rain': [], ....})
        values = data['hourly']

        # create the fields
        fields = QgsFields()

        fields.append(QgsField('fk', QVariant.Int))
        # be sure to set the correct type
        for k in values.keys():
            if k == 'time':
                fields.append(QgsField(k, QVariant.DateTime))
            else:
                fields.append(QgsField(k, QVariant.Double))


        # create the second sink (geometryless)
        (sink2, dest_id2) = self.parameterAsSink(
            parameters,
            self.OUTPUT_VALUES,
            context,
            fields,
            QgsWkbTypes.NoGeometry,
            QgsCoordinateReferenceSystem()
        )

        # get the list of single items of each sublist
        l = list(zip(*values.values()))

        # iterate on each sublist and write the values to the feature
        for sublist in l:
            val = [1]
            val.extend(list(sublist))
            feature = QgsFeature()
            feature.setAttributes(val)
            sink2.addFeature(feature, QgsFeatureSink.FastInsert)


        # add the results
        results[self.OUTPUT_LAYER] = dest_id
        results[self.OUTPUT_VALUES] = dest_id2

        return results


    def name(self):
        return 'weather_forecast_hourly'

    def displayName(self):
        return self.tr('Weather Forecast (Hourly)')

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return 'Meteo'

    def tags(self):
        return self.tr('meteo,forecast,open-meteo,data,plot').split(',')

    def shortHelpString(self):
        help_string = '''
        Download hourly weather data from a location (chosen by coordinates). Up to 16 days of forecast are possible.
        The service is providen by <a href="https://open-meteo.com/en">https://open-meteo.com/en</a>.

        The resolution is ~2km.

        Please check the <a href="https://open-meteo.com/en/docs">documentation</a> for variable information or click the Help button.

        No API key are required for <strong>non commercial use</strong>.

        The algorithm outputs the html as the plot of the variables chosen with a red dotted line representing today, a point layer of the clicked location and optionally the whole JSON file with all the fetched values.
        '''

        return help_string

    def helpUrl(self):
        return 'https://faunalia.gitlab.io/faunalia-toolkit/usage/algorithms.html#weather-forecast-hourly'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def tags(self):
        return self.tr('meteo,forecast,download,data,weather,climate').split(',')

    def createInstance(self):
        return OpenMeteoWeatherCurrent()
