# -*- coding: utf-8 -*-

"""
/***************************************************************************
 FaunaliaToolkit
                                 A QGIS plugin
 Faunalia Spatial Analysis Toolkit
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-04-30
        copyright            : (C) 2022 by Matteo Ghetta (Faunalia)
        email                : matteo.ghetta@faunalia.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Matteo Ghetta (Faunalia)'
__date__ = '2022-04-30'
__copyright__ = '(C) 2022 by Matteo Ghetta (Faunalia)'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from faunalia_toolkit.__about__ import DIR_PLUGIN_ROOT

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsProcessingAlgorithm,
                       QgsGeometry,
                       QgsFeature,
                       QgsPointXY,
                       QgsField,
                       QgsFields,
                       QgsWkbTypes,
                       QgsFeatureSink,
                       QgsCoordinateReferenceSystem,
                       QgsProcessingParameterPoint,
                       QgsDistanceArea,
                       QgsProcessingParameterFeatureSink)

class AntipodesFromPoint(QgsProcessingAlgorithm):

    POINT = 'POINT'
    OUTPUT = 'OUTPUT'
    OUTPUT_GEODESIC_LINE = 'OUTPUT_GEODESIC_LINE'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.

        self.addParameter(
            QgsProcessingParameterPoint(
                self.POINT,
                self.tr('Point'),
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Antipodes')
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_GEODESIC_LINE,
                self.tr('Geodesic Line')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        point = self.parameterAsPoint(
            parameters,
            self.POINT,
            context,
            QgsCoordinateReferenceSystem("EPSG:4326") # the point will always be converted to 4326
        )

        # create the fields
        fields = QgsFields()
        fields.append(QgsField('where', QVariant.String))
        fields.append(QgsField('x', QVariant.Double))
        fields.append(QgsField('y', QVariant.Double))

        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            fields,
            QgsWkbTypes.Point,
            QgsCoordinateReferenceSystem('EPSG:4326')
        )

        # empty list that will contain the original and antipodal features
        features = []

        # get the clicked point coordinates
        x = point.x()
        y = point.y()

        # convert the clicked point coordinates as QgsPointXY and to a geometry
        original_point = QgsPointXY(x, y)
        original_geom = QgsGeometry().fromPointXY(original_point)


        # create the original feature with attributes and geometry
        feature = QgsFeature()
        feature.setFields(fields)
        feature.setAttribute('where', 'original')
        feature.setAttribute('x', x)
        feature.setAttribute('y', y)
        feature.setGeometry(original_geom)

        # append the original feature to the list
        features.append(feature)

        # calculate the antipodal coordinates
        if x >=0:
            antipode_x = -180+x
        elif x<0:
            antipode_x = 180+x
        antipode_y = -y

        # create the antipodal coordinates as QgsPointXY and to a QgsGeometry
        antipode_point = QgsPointXY(antipode_x, antipode_y)
        antipode_geom = QgsGeometry().fromPointXY(antipode_point)

        # create the antipodal feature with attributes and geometry
        feature = QgsFeature()
        feature.setFields(fields)
        feature.setAttribute('where', 'antipode')
        feature.setAttribute('x', antipode_x)
        feature.setAttribute('y', antipode_y)
        feature.setGeometry(antipode_geom)

        # append the antipodal feature to the list
        features.append(feature)

        # add the features to the sink
        sink.addFeatures(features, QgsFeatureSink.FastInsert)

        # geodesic line
        fields = QgsFields()
        fields.append(QgsField('id', QVariant.Int))
        fields.append(QgsField('length', QVariant.Double))

        (sink2, dest_id2) = self.parameterAsSink(
            parameters,
            self.OUTPUT_GEODESIC_LINE,
            context,
            fields,
            QgsWkbTypes.MultiLineString,
            QgsCoordinateReferenceSystem('EPSG:4326')
        )

        # create the distance area to calculate the geodesic line
        da = QgsDistanceArea()
        da.setSourceCrs(QgsCoordinateReferenceSystem("EPSG:4326"), context.transformContext())
        da.setEllipsoid(context.ellipsoid())

        geodesic_line = da.geodesicLine(original_point, antipode_point, 40_000_000)
        # geodesic_line = da.geodesicLine(x, y, 40_000_000)

        geometry = QgsGeometry.fromPolylineXY(geodesic_line[0])

        feature = QgsFeature(fields)
        feature.setAttribute('id', 1)
        feature.setAttribute('length', da.measureLength(geometry))
        feature.setGeometry(geometry)

        sink2.addFeature(feature)

        return {self.OUTPUT: dest_id, self.OUTPUT_GEODESIC_LINE: dest_id2}


    def name(self):
        return 'antipodes_from_point'

    def displayName(self):
        return self.tr('Antipodes (from point)')

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return 'Cartography'

    def icon(self):
        return QIcon(str(DIR_PLUGIN_ROOT / "resources/images/antipodes.svg"))

    def tags(self):
        return self.tr('antipodes,flat,earth,cartography').split(',')

    def shortHelpString(self):
        help_string = '''
        Creates the antipode from a selected point clicked on the map.

        Conversion of the coordinates clicked on the map is handled automatically and the final final layer is in EPSG:4326.

        Check <a href="https://www.antipodesmap.com/">https://www.antipodesmap.com/</a> for more curiosities.

        Inspired by Flat Earth Society
        '''

        return help_string

    def helpUrl(self):
        return 'https://faunalia.gitlab.io/faunalia-toolkit/usage/algorithms.html#antipodes-from-point'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def tags(self):
        return self.tr('antipodes,geometry,opposite').split(',')

    def createInstance(self):
        return AntipodesFromPoint()
