# -*- coding: utf-8 -*-

"""
/***************************************************************************
 FaunaliaToolkit
                                 A QGIS plugin
 Faunalia Spatial Analysis Toolkit
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-04-30
        copyright            : (C) 2022 by Matteo Ghetta (Faunalia)
        email                : matteo.ghetta@faunalia.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Matteo Ghetta (Faunalia)'
__date__ = '2022-04-30'
__copyright__ = '(C) 2022 by Matteo Ghetta (Faunalia)'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from faunalia_toolkit.__about__ import DIR_PLUGIN_ROOT

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsFeatureRequest,
                       QgsFields,
                       QgsField,
                       QgsDistanceArea,
                       QgsWkbTypes,
                       QgsGeometry,
                       QgsFeature,
                       QgsPointXY,
                       QgsCoordinateReferenceSystem,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink)

class AntipodesFromPointLayer(QgsProcessingAlgorithm):

    POINT = 'POINT'
    OUTPUT = 'OUTPUT'
    OUTPUT_GEODESIC_LINE = 'OUTPUT_GEODESIC_LINE'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.POINT,
                self.tr('Points'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Antipodes')
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_GEODESIC_LINE,
                self.tr('Geodesic Line'),
                optional=True
            )
        )


    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        points = self.parameterAsSource(parameters, self.POINT, context)

        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            points.fields(),
            points.wkbType(),
            QgsCoordinateReferenceSystem('EPSG:4326')
        )

        # geodesic line
        fields = QgsFields()
        fields.append(QgsField('id', QVariant.Int))
        fields.append(QgsField('length', QVariant.Double))

        (sink2, dest_id2) = self.parameterAsSink(
            parameters,
            self.OUTPUT_GEODESIC_LINE,
            context,
            fields,
            QgsWkbTypes.MultiLineString,
            QgsCoordinateReferenceSystem('EPSG:4326')
        )

        # create the distance area to calculate the geodesic line
        da = QgsDistanceArea()
        da.setSourceCrs(QgsCoordinateReferenceSystem("EPSG:4326"), context.transformContext())
        da.setEllipsoid(context.ellipsoid())

        request = QgsFeatureRequest()
        request.setDestinationCrs(QgsCoordinateReferenceSystem('EPSG:4326'), context.transformContext())

        total = 100 / points.featureCount() if points else 0

        for current, i in enumerate(points.getFeatures(request)):

            if feedback.isCanceled():
                break

            igeom = i.geometry()
            x = igeom.asPoint().x()
            y = igeom.asPoint().y()

            if x >=0:
                antipode_x = -180 + x
            elif x < 0:
                antipode_x = 180 + x
            antipode_y = -y

            antipode_point = QgsPointXY(antipode_x, antipode_y)
            antipode_geom = QgsGeometry().fromPointXY(antipode_point)

            feature = QgsFeature()
            feature.setAttributes(i.attributes())
            feature.setGeometry(antipode_geom)

            sink.addFeature(feature, QgsFeatureSink.FastInsert)

            geodesic_line = da.geodesicLine(i.geometry().asPoint(), antipode_point, 40_000_000)
            geometry = QgsGeometry.fromPolylineXY(geodesic_line[0])

            feature = QgsFeature(fields)
            feature.setAttribute('id', 1)
            feature.setAttribute('length', da.measureLength(geometry))
            feature.setGeometry(geometry)

            if sink2:
                sink2.addFeature(feature)

            feedback.setProgress(int(current * total))

        results = {}
        results[self.OUTPUT] = dest_id

        if sink2:
            results[self.OUTPUT_GEODESIC_LINE] = dest_id2

        return results


    def name(self):
        return 'antipodes_from_layer'

    def displayName(self):
        return self.tr('Antipodes (from layer)')

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return 'Cartography'

    def icon(self):
        return QIcon(str(DIR_PLUGIN_ROOT / "resources/images/antipodes.svg"))

    def tags(self):
        return self.tr('antipodes,flat,earth,cartography').split(',')

    def shortHelpString(self):
        help_string = '''
        Creates the antipode of the chosen point layer.

        Conversion of the coordinates map are handled automatically and the final final layer is in EPSG:4326.

        Check <a href="https://www.antipodesmap.com/">https://www.antipodesmap.com/</a> for more curiosities.

        Inspired by Flat Earth Society
        '''

        return help_string

    def helpUrl(self):
        return 'https://faunalia.gitlab.io/faunalia-toolkit/usage/algorithms.html#antipodes-from-layer'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def tags(self):
        return self.tr('antipodes,geometry,opposite').split(',')

    def createInstance(self):
        return AntipodesFromPointLayer()
