# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Copernicus
                                 A QGIS plugin
 Donwload Copernicus Hourly Data
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-12-27
        copyright            : (C) 2022 by Matteo Ghetta (Faunalia)
        email                : info@faunalia.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Matteo Ghetta (Faunalia)'
__date__ = '2022-12-27'
__copyright__ = '(C) 2022 by Matteo Ghetta (Faunalia)'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import datetime

from qgis.PyQt.QtCore import QCoreApplication, QDate
from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterExtent,
                       QgsCoordinateReferenceSystem,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingException,
                       QgsProcessingUtils
                       )

class CopernicusAlgorithm(QgsProcessingAlgorithm):


    TEMPERATURE = 'TEMPERATURE'
    LAKES = 'LAKES'
    SNOW = 'SNOW'
    YEAR = 'YEAR'
    MONTH = 'MONTH'
    DAY = 'DAY'
    TIME = 'TIME'
    EXTENT = 'EXTENT'

    OUTPUT_FILE= 'OUTPUT_FILE'


    def __init__(self) -> None:
        super().__init__()

        self.valid = None

        try:
            import cdsapi
            self.valid = True
        except ImportError as e:
            self.valid = False

    def initAlgorithm(self, config):


        self.temperature = [
            '2m_dewpoint_temperature',
            '2m_temperature',
            'skin_temperature',
            'soil_temperature_level_1',
            'soil_temperature_level_2',
            'soil_temperature_level_3',
            'soil_temperature_level_4',
        ]

        self.addParameter(
            QgsProcessingParameterEnum(
                self.TEMPERATURE,
                self.tr('Temperature'),
                self.temperature,
                allowMultiple=True,
                optional=True
            )
        )

        self.lakes = [
            'lake_bottom_temperature',
            'lake_ice_depth',
            'lake_ice_temperature',
            'lake_mix_layer_depth',
            'lake_mix_layer_temperature',
            'lake_shape_factor',
            'lake_total_layer_temperature'
        ]

        self.addParameter(
            QgsProcessingParameterEnum(
                self.LAKES,
                self.tr('Lakes'),
                self.lakes,
                allowMultiple=True,
                optional=True
            )
        )

        self.snow = [
            'snow_albedo',
            'snow_cover',
            'snow_density',
            'snow_depth',
            'snow_depth_water_equivalent',
            'snowfall',
            'snowmelt',
            'temperature_of_snow_layer',
        ]

        self.addParameter(
            QgsProcessingParameterEnum(
                self.SNOW,
                self.tr('Snow'),
                self.snow,
                allowMultiple=True,
                optional=True
            )
        )


        self.year = list(map(str, sorted(range(1950, 2024), reverse=True)))

        self.addParameter(
            QgsProcessingParameterEnum(
                self.YEAR,
                self.tr('Year'),
                self.year
            )
        )

        self.month = list(map(str, range(1, 13)))

        self.addParameter(
            QgsProcessingParameterEnum(
                self.MONTH,
                self.tr('Month'),
                self.month
            )
        )

        self.day = list(map(str, range(1, 32)))

        self.addParameter(
            QgsProcessingParameterEnum(
                self.DAY,
                self.tr('Day'),
                self.day,
                allowMultiple=True
            )
        )

        self.time = [f'{datetime.time(i).strftime("%H")}:00' for i in range(24)]

        self.addParameter(
            QgsProcessingParameterEnum(
                self.TIME,
                self.tr('Time'),
                self.time,
                allowMultiple=True,
                defaultValue=[i for i, val in enumerate(self.time)]
            )
        )

        self.addParameter(
            QgsProcessingParameterExtent(
                self.EXTENT,
                self.tr('Sub Region Extraction')
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT_FILE,
                self.tr('Output File'),
                self.tr('GRIB files (*.grib)'),
            )
        )

    def checkDate(self, year, month, days):
        """
        check if the date is valid
        """

        for day in days:
            check_date = QDate(int(year), int(month), int(day))
            if not check_date.isValid():
                raise QgsProcessingException(self.tr(f'The date year:{year}, month:{month}, day:{day} is not valid. Please check again'))


    def processAlgorithm(self, parameters, context, feedback):

        # import the cds library and check theAPI key
        import cdsapi
        c = cdsapi.Client(debug=True)

        if not c.key:
            raise QgsProcessingException(self.tr('You should get an API Key to use the client, please follow the instructions at https://cds.climate.copernicus.eu/api-how-to'))

        variables = []

        temperature = self.parameterAsEnums(
            parameters,
            self.TEMPERATURE,
            context
        )
        temperature = [self.temperature[i] for i in temperature]

        variables.extend(temperature)

        lakes = self.parameterAsEnums(
            parameters,
            self.LAKES,
            context
        )
        lakes = [self.lakes[i] for i in lakes]

        variables.extend(lakes)

        snow = self.parameterAsEnums(
            parameters,
            self.SNOW,
            context
        )
        snow = [self.snow[i] for i in snow]

        variables.extend(snow)

        year = self.parameterAsEnum(
            parameters,
            self.YEAR,
            context
        )
        year = self.year[year]

        month = self.parameterAsEnum(
            parameters,
            self.MONTH,
            context
        )
        month = self.month[month]

        days = self.parameterAsEnums(
            parameters,
            self.DAY,
            context
        )
        days = [self.day[i] for i in days]

        time = self.parameterAsEnums(
            parameters,
            self.TIME,
            context
        )
        time = [self.time[i] for i in time]

        self.checkDate(year, month, days)

        extent = self.parameterAsExtent(
            parameters,
            self.EXTENT,
            context,
            QgsCoordinateReferenceSystem(4326)
        )
        extent = [extent.yMaximum(), extent.xMinimum(), extent.yMinimum(), extent.xMaximum()]

        output_file = self.parameterAsFileOutput(
            parameters,
            self.OUTPUT_FILE,
            context
        )

        r = c.retrieve(
            'reanalysis-era5-land',
            {
                'variable': variables,
                'year': year,
                'month': month,
                'day': days,
                'time': time,
                'area': extent,
                'format': 'grib',
            }
        )

        r.update()
        reply = r.reply

        if reply["state"] in ("failed",):
            feedback.pushDebugInfo(r.error)
            r.error("Message: %s", reply["error"].get("message"))
            r.error("Reason:  %s", reply["error"].get("reason"))
            for n in (
                reply.get("error", {}).get("context", {}).get("traceback", "").split("\n")
            ):
                if n.strip() == "":
                    break
                r.error("  %s", n)
            raise Exception(
                "%s. %s." % (reply["error"].get("message"), reply["error"].get("reason"))
            )

        r.download(output_file)

        context.addLayerToLoadOnCompletion(
            output_file,
            context.LayerDetails(
                name='Mesh',
                project=context.project(),
                outputName='Mesh',
                layerTypeHint=QgsProcessingUtils.LayerHint.Mesh
            )
        )


        return {}

    def name(self):
        return 'download_copernicus_data'

    def displayName(self):
        return self.tr('Download Copernicus Data')

    def group(self):
        return self.tr('Copernicus')

    def groupId(self):
        return 'copernicus'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return CopernicusAlgorithm()

    def tags(self):
        return self.tr('copernicus,grib,meteo,cds,climate').split(',')

    def helpUrl(self):
        return 'https://faunalia.gitlab.io/faunalia-toolkit/usage/algorithms.html#download-copernicus-data'

    def shortHelpString(self):
        description = '''
        <strong>ERA5-Land hourly data from 1950 to present</strong>
        Thanks to the cdsapi of the ERA5 project of Copernicus you can download dataset from 1950 to (almost) today.
        The algorithm is just a simple GUI that uses the cdsapi and fires the request.
        Some checks are performed from the plugin itself (date checking) but the ownership of the whole process is of ERA5 - Copernicus.
        To have a complete documentation of all the variables please visit the <a href="https://cds.climate.copernicus.eu/cdsapp#!/dataset/reanalysis-era5-land?tab=overview">copernicus website</a>.

        Muñoz Sabater, J., (2019): ERA5-Land hourly data from 1981 to present. Copernicus Climate Change Service (C3S) Climate Data Store (CDS). (Accessed on < DD-MMM-YYYY >), 10.24381/cds.e2161bac
        Muñoz Sabater, J., (2021): ERA5-Land hourly data from 1950 to 1980. Copernicus Climate Change Service (C3S) Climate Data Store (CDS). (Accessed on < DD-MMM-YYYY >), 10.24381/cds.e2161bac

        '''
        return self.tr(description)

    def canExecute(self):
        string = '<a href="https://cds.climate.copernicus.eu/cdsapp#!/dataset/reanalysis-era5-land?tab=overview">official documentation</a>'
        if not self.valid:
            return (False, self.tr(f'Please install the cdsapi to run this algorithm. Check the {string}'))
        else:
            return (True, '')
