# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FaultBufferTool
                                 A QGIS plugin
 This plugin will create buffers based on the provided lookup table and attributes in the input shapefile.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-02-01
        git sha              : $Format:%H$
        copyright            : (C) 2025 by ASU
        email                : raswanth@asu.edu
 ***************************************************************************/

 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant, QStandardPaths
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QFileDialog, QPushButton
from qgis import processing

from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsField,
    QgsFeature,
    QgsProcessingContext,
    QgsProcessingFeedback,
    QgsVectorFileWriter,
    QgsSymbol,
    QgsRendererCategory,
    QgsCategorizedSymbolRenderer,
    QgsMessageLog,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsDistanceArea,
    QgsPointXY,
    QgsGeometry
)
from qgis.gui import QgsFileWidget

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .FaultBufferTool_dialog import FaultBufferToolDialog
import os.path
import shutil

class FaultBufferTool:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FaultBufferTool_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&FaultBufferTool')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.dlg = None
        self.buffer_distances = {
            ('P', 1): 200,  # Primary, Quality 1
            ('P', 2): 80,   # Primary, Quality 2
            ('P', 3): 30,   # Primary, Quality 3
            ('P', 4): 10,   # Primary, Quality 4
            ('S', 1): 300,  # Secondary, Quality 1
            ('S', 2): 100,  # Secondary, Quality 2
            ('S', 3): 70,   # Secondary, Quality 3
            ('S', 4): 20    # Secondary, Quality 4
        }

        # Uncertainty table for different criteria - UPDATED based on the provided text
        self.uncertainty_table = {
            # General uncertainty for a predicted rupture
            'general': {
                '50th': 20,
                '84th': 60,
                '97th': 130
            },
            # Use one: Fault confidence
            'confidence': {
                'strong': {'50th': 10, '84th': 25, '97th': 50},
                'distinct': {'50th': 15, '84th': 40, '97th': 60},
                'weak': {'50th': 25, '84th': 55, '97th': 105},
                'uncertain': {'50th': 45, '84th': 120, '97th': 260}
            },
            # Use one: Primary versus Secondary
            'primary_secondary': {
                'primary': {'50th': 20, '84th': 60, '97th': 130},
                'secondary': {'50th': 20, '84th': 60, '97th': 110}
            },
            # Use one: Simple versus complex
            'simple_complex': {
                'simple': {'50th': 20, '84th': 60, '97th': 130},
                'complex': {'50th': 20, '84th': 60, '97th': 100} # Corrected 'Compex' typo
            },
            # Use two: Conf & Prim/Sec
            'conf_prim_sec': {
                'strong_primary': {'50th': 10, '84th': 25, '97th': 35},
                'distinct_primary': {'50th': 15, '84th': 35, '97th': 60},
                'weak_primary': {'50th': 25, '84th': 50, '97th': 105},
                'uncertain_primary': {'50th': 50, '84th': 120, '97th': 228},
                'strong_secondary': {'50th': 15, '84th': 35, '97th': 69},
                'distinct_secondary': {'50th': 20, '84th': 45, '97th': 60},
                'weak_secondary': {'50th': 25, '84th': 70, '97th': 115},
                'uncertain_secondary': {'50th': 25, '84th': 70, '97th': 480}
            },
            # Use two: Conf & Simp/Comp
            'conf_simple_complex': {
                'strong_simple': {'50th': 10, '84th': 25, '97th': 40},
                'distinct_simple': {'50th': 15, '84th': 40, '97th': 60},
                'weak_simple': {'50th': 25, '84th': 50, '97th': 105},
                'uncertain_simple': {'50th': 50, '84th': 125, '97th': 250},
                'strong_complex': {'50th': 10, '84th': 35, '97th': 55},
                'distinct_complex': {'50th': 10, '84th': 35, '97th': 55},
                'weak_complex': {'50th': 25, '84th': 70, '97th': 100},
                'uncertain_complex': {'50th': 35, '84th': 70, '97th': 430}
            },
            # Use two: Prim/Sec & Simp/Comp
            'prim_sec_simple_complex': {
                'primary_simple': {'50th': 20, '84th': 65, '97th': 135},
                'primary_complex': {'50th': 20, '84th': 50, '97th': 65},
                'secondary_simple': {'50th': 20, '84th': 45, '97th': 63},
                'secondary_complex': {'50th': 20, '84th': 65, '97th': 175}
            },
            # Use all three: Conf, Prim/Sec & Simp/Comp
            'all_criteria': {
                'strong_primary_simple': {'50th': 10, '84th': 25, '97th': 40},
                'distinct_primary_simple': {'50th': 15, '84th': 35, '97th': 60},
                'weak_primary_simple': {'50th': 25, '84th': 50, '97th': 105},
                'uncertain_primary_simple': {'50th': 50, '84th': 130, '97th': 255},
                'strong_secondary_simple': {'50th': 10, '84th': 15, '97th': 15},
                'distinct_secondary_simple': {'50th': 30, '84th': 50, '97th': 60},
                'weak_secondary_simple': {'50th': 45, '84th': 100, '97th': 160},
                'uncertain_secondary_simple': {'50th': 45, '84th': 100, '97th': 160},
                'strong_primary_complex': {'50th': 10, '84th': 15, '97th': 20},
                'distinct_primary_complex': {'50th': 10, '84th': 30, '97th': 35},
                'weak_primary_complex': {'50th': 30, '84th': 50, '97th': 90},
                'uncertain_primary_complex': {'50th': 40, '84th': 60, '97th': 90},
                'strong_secondary_complex': {'50th': 20, '84th': 41, '97th': 90},
                'distinct_secondary_complex': {'50th': 20, '84th': 35, '97th': 90},
                'weak_secondary_complex': {'50th': 25, '84th': 70, '97th': 100},
                'uncertain_secondary_complex': {'50th': 25, '84th': 118, '97th': 500}
            },
            # For unpredicted ruptures
            'unpredicted': {
                '50th': 300,
                '84th': 1000,
                '97th': 1700
            }
        }

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('FaultBufferTool', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/FaultBufferTool/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Fault Buffer Tool'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&FaultBufferTool'),
                action)
            self.iface.removeToolBarIcon(action)

    def get_utm_crs(self, longitude, latitude):
        """Calculate the appropriate UTM CRS based on coordinates"""
        # Calculate UTM zone
        # The Earth is divided into 60 UTM zones, each 6 degrees wide
        # We add 180 to shift from (-180,180) range to (0,360) range
        # Then divide by 6 to get the zone number (1-60)
        zone = int((longitude + 180) / 6) + 1
        
        # Determine if Northern or Southern hemisphere
        # - 326xx for Northern hemisphere (latitude > 0)
        # - 327xx for Southern hemisphere (latitude < 0)
        if latitude > 0:
            epsg = f"326{zone:02d}"  # Northern hemisphere
        else:
            epsg = f"327{zone:02d}"  # Southern hemisphere
        
        return QgsCoordinateReferenceSystem(f"EPSG:{epsg}")
    
    def create_asymmetric_buffer(self, geometry, distance, dip_direction, input_crs, segments, buffer_ratio):
        """
        Creates an asymmetric buffer by:
        1. Translating the fault line based on dip direction
        2. Using QGIS's built-in buffer function
        """
        try:
            from qgis.core import QgsFeature, QgsGeometry, QgsVectorLayer, QgsWkbTypes

            # Log initial parameters
            QgsMessageLog.logMessage(f"Starting asymmetric buffer creation: distance={distance}, dip={dip_direction}, ratio={buffer_ratio}", "FaultBufferTool")
            QgsMessageLog.logMessage(f"Input CRS for asymmetric buffer: {input_crs.authid()}", "FaultBufferTool")

            # Calculate translation based on buffer ratio
            translation_dist = distance * ((1-buffer_ratio)/(1+buffer_ratio))
            
            # Calculate diagonal distance for NE, NW, SE, SW directions
            diagonal_dist = translation_dist / (2 ** 0.5)
            
            # Calculate translation offsets based on dip direction
            if dip_direction.upper() == 'N':
                dx, dy = 0, translation_dist
            elif dip_direction.upper() == 'S':
                dx, dy = 0, -translation_dist
            elif dip_direction.upper() == 'E':
                dx, dy = translation_dist, 0
            elif dip_direction.upper() == 'W':
                dx, dy = -translation_dist, 0
            elif dip_direction.upper() == 'NE':
                dx, dy = diagonal_dist, diagonal_dist
            elif dip_direction.upper() == 'NW':
                dx, dy = -diagonal_dist, translation_dist
            elif dip_direction.upper() == 'SE':
                dx, dy = diagonal_dist, -diagonal_dist
            elif dip_direction.upper() == 'SW':
                dx, dy = -diagonal_dist, -diagonal_dist
            else:
                QgsMessageLog.logMessage(f"Invalid dip direction: {dip_direction}", "FaultBufferTool")
                dx, dy = translation_dist, 0  # Default to East if invalid

            QgsMessageLog.logMessage(f"Translation values: dx={dx}, dy={dy}", "FaultBufferTool")

            # Create temporary layer with explicit CRS
            temp_layer = QgsVectorLayer(f"LineString?crs={input_crs.authid()}", "temp", "memory")
            if not temp_layer.isValid():
                QgsMessageLog.logMessage("Failed to create temporary layer", "FaultBufferTool")
                return None

            # Verify the temp layer CRS matches input CRS
            if temp_layer.crs() != input_crs:
                QgsMessageLog.logMessage(f"Warning: Temp layer CRS ({temp_layer.crs().authid()}) doesn't match input CRS ({input_crs.authid()})", "FaultBufferTool")
                temp_layer.setCrs(input_crs)

            # Add feature to temp layer
            temp_provider = temp_layer.dataProvider()
            temp_feat = QgsFeature()
            temp_feat.setGeometry(geometry)
            temp_provider.addFeatures([temp_feat])
            if not temp_provider.addFeatures([temp_feat]):
                QgsMessageLog.logMessage("Failed to add feature to temporary layer", "FaultBufferTool")
                return None
            
            # Run translate algorithm
            translate_params = {
                'INPUT': temp_layer,
                'DELTA_X': dx,
                'DELTA_Y': dy,
                'DELTA_Z': 0,
                'DELTA_M': 0,
                'OUTPUT': 'TEMPORARY_OUTPUT'
            }

            QgsMessageLog.logMessage("Running translate algorithm...", "FaultBufferTool")
            translated_result = processing.run("native:translategeometry", translate_params)
            
            if not translated_result or 'OUTPUT' not in translated_result:
                QgsMessageLog.logMessage("Translation algorithm failed", "FaultBufferTool")
                return None
            
            # Get translated geometry
            translated_layer = translated_result['OUTPUT']
            
            # Verify the translated layer has the correct CRS
            if translated_layer.crs() != input_crs:
                QgsMessageLog.logMessage(f"Warning: Translated layer CRS ({translated_layer.crs().authid()}) doesn't match input CRS ({input_crs.authid()})", "FaultBufferTool")
            
            # Store translated geometry for buffer creation
            translated_geom = None 
            for feat in translated_layer.getFeatures():
                translated_geom = feat.geometry()
                break

            if not translated_geom:
                QgsMessageLog.logMessage("Failed to get translated geometry", "FaultBufferTool")
                return None
                
            QgsMessageLog.logMessage("Creating buffer...", "FaultBufferTool")
            
            # Create buffer
            buffer_geom = translated_geom.buffer(distance, segments)
            
            if not buffer_geom or not buffer_geom.isGeosValid():
                QgsMessageLog.logMessage("Failed to create valid buffer geometry", "FaultBufferTool")
                return None
            
            QgsMessageLog.logMessage(f"Buffer created successfully. Type: {buffer_geom.wkbType()}", "FaultBufferTool")
            return buffer_geom

        except Exception as e:
            QgsMessageLog.logMessage(f"Error in create_asymmetric_buffer: {str(e)}", "FaultBufferTool")
            import traceback
            QgsMessageLog.logMessage(f"Traceback: {traceback.format_exc()}", "FaultBufferTool")
            return None
        
    def get_uncertainty_distance(self, feature):
        """
        Get buffer distance based on uncertainty rankings selected in the UI
        Only uses fields corresponding to checked boxes
        """
        # Get selected confidence interval
        if self.dlg.percentile50RadioButton.isChecked():
            percentile = '50th'
        elif self.dlg.percentile84RadioButton.isChecked():
            percentile = '84th'
        elif self.dlg.percentile97RadioButton.isChecked():
            percentile = '97th'
        
        # General uncertainty (ignore rankings)
        if self.dlg.generalUncertaintyRadioButton.isChecked():
            return self.uncertainty_table['general'][percentile]
        
        available_fields = [f.name() for f in feature.fields()]
        
        # Initialize variables with defaults
        confidence_text = 'uncertain'  # Default
        primary_secondary = 'primary'  # Default
        simple_complex = 'simple'      # Default
        
        # Only use the Quality field if Confidence checkbox is checked
        if self.dlg.confidenceCheckBox.isChecked() and 'Quality' in available_fields:
            confidence = feature['Quality']
            if confidence == 4:
                confidence_text = 'strong'
            elif confidence == 3:
                confidence_text = 'distinct'
            elif confidence == 2:
                confidence_text = 'weak'
            elif confidence == 1:
                confidence_text = 'uncertain'
        
        # Only use the P or S field if Primary/Secondary checkbox is checked
        if self.dlg.primarySecondaryCheckBox.isChecked() and 'PriSec' in available_fields:
            p_or_s = feature["PriSec"].strip().upper()
            primary_secondary = 'primary' if p_or_s == 'P' else 'secondary'
        
        # Only use the SimpComp field if Simple/Complex checkbox is checked
        if self.dlg.simpleComplexCheckBox.isChecked() and 'SimpComp' in available_fields:
            simp_comp = feature['SimpComp'].strip().upper()
            if simp_comp == 'C':
                simple_complex = 'complex'
            elif simp_comp == 'S':
                simple_complex = 'simple'
        
        # Determine which criteria are selected
        conf_selected = self.dlg.confidenceCheckBox.isChecked()
        prim_sec_selected = self.dlg.primarySecondaryCheckBox.isChecked()
        simple_complex_selected = self.dlg.simpleComplexCheckBox.isChecked()
        
        # Now determine which uncertainty table to use based on selected criteria
        
        # Use all three criteria if all are checked
        if conf_selected and prim_sec_selected and simple_complex_selected:
            key = f"{confidence_text}_{primary_secondary}_{simple_complex}"
            return self.uncertainty_table['all_criteria'].get(key, {}).get(percentile, 0)
        
        # Use confidence and primary/secondary if those two are checked
        elif conf_selected and prim_sec_selected:
            key = f"{confidence_text}_{primary_secondary}"
            return self.uncertainty_table['conf_prim_sec'].get(key, {}).get(percentile, 0)
        
        # Use confidence and simple/complex if those two are checked
        elif conf_selected and simple_complex_selected:
            key = f"{confidence_text}_{simple_complex}"
            return self.uncertainty_table['conf_simple_complex'].get(key, {}).get(percentile, 0)
        
        # Use primary/secondary and simple/complex if those two are checked
        elif prim_sec_selected and simple_complex_selected:
            key = f"{primary_secondary}_{simple_complex}"
            return self.uncertainty_table['prim_sec_simple_complex'].get(key, {}).get(percentile, 0)
        
        # Use only confidence if only that is checked
        elif conf_selected:
            return self.uncertainty_table['confidence'].get(confidence_text, {}).get(percentile, 0)
        
        # Use only primary/secondary if only that is checked
        elif prim_sec_selected:
            return self.uncertainty_table['primary_secondary'].get(primary_secondary, {}).get(percentile, 0)
        
        # Use only simple/complex if only that is checked
        elif simple_complex_selected:
            return self.uncertainty_table['simple_complex'].get(simple_complex, {}).get(percentile, 0)
        
        # Fallback to general uncertainty if nothing is selected (shouldn't happen)
        return self.uncertainty_table['general'][percentile]
    
    def validate_required_fields(self, input_layer):
        """
        Validates that the input layer has required fields based on selected options
        Only checks for fields that correspond to checked boxes
        """
        available_fields = [f.name() for f in input_layer.fields()]
        QgsMessageLog.logMessage(f"Available fields: {available_fields}", "FaultBufferTool")
        
        # If using geologic judgment, no field checks needed except Dip_direct for non-strike-slip faults
        if self.dlg.geologicJudgementRadioButton.isChecked():
            if not self.dlg.StrikeslipFaultRadioButton.isChecked():
                if 'Dip_direct' not in available_fields:
                    return False, "Field 'Dip_direct' is required for normal/reverse faults with geologic judgment option"
            return True, ""
        
        # For uncertainty with ranking, check only selected ranking fields
        if self.dlg.uncertaintyWithRankingRadioButton.isChecked():
            # Only check for Quality field if Confidence checkbox is checked
            if self.dlg.confidenceCheckBox.isChecked() and 'Quality' not in available_fields:
                return False, "Required field 'Quality' not found in input layer for Confidence ranking!"
            
            # Only check for P or S field if Primary/Secondary checkbox is checked
            if self.dlg.primarySecondaryCheckBox.isChecked() and 'PriSec' not in available_fields:
                return False, "Required field 'PriSec' not found in input layer for Primary/Secondary ranking!"
            
            # Only check for SimpComp field if Simple/Complex checkbox is checked
            if self.dlg.simpleComplexCheckBox.isChecked() and 'SimpComp' not in available_fields:
                return False, "Required field 'SimpComp' not found in input layer for Simple/Complex ranking!"
            
            # Check for Dip_direct if not using Strike-slip fault type
            if not self.dlg.StrikeslipFaultRadioButton.isChecked() and 'Dip_direct' not in available_fields:
                return False, "Field 'Dip_direct' is required for normal/reverse faults"
        
        return True, ""
    
    # --- UI State Management ---
    def setupDialogConnections(self):
        """Set up signal connections for UI controls"""
        # Connect main uncertainty mode radio buttons
        self.dlg.geologicJudgementRadioButton.toggled.connect(self.update_ui_state)
        self.dlg.generalUncertaintyRadioButton.toggled.connect(self.update_ui_state)
        self.dlg.uncertaintyWithRankingRadioButton.toggled.connect(self.update_ui_state)

        # Connect geologic judgment sub-options
        self.dlg.fromShapefile.toggled.connect(self.update_ui_state) # Use main updater
        self.dlg.inputWidth.toggled.connect(self.update_ui_state) # Use main updater

        # Connect fault type mode radio buttons
        self.dlg.UniformFaultTypeRadioButton.toggled.connect(self.update_ui_state)
        self.dlg.FromShapefileFaultTypeRadioButton.toggled.connect(self.update_ui_state)

    def update_ui_state(self):
        """Updates the enabled/disabled state of UI elements based on selections."""
        if not self.dlg: return

        is_geologic = self.dlg.geologicJudgementRadioButton.isChecked()
        is_general = self.dlg.generalUncertaintyRadioButton.isChecked()
        is_ranking = self.dlg.uncertaintyWithRankingRadioButton.isChecked()
        is_uniform_fault = self.dlg.UniformFaultTypeRadioButton.isChecked()

        # --- Geologic Judgment Section ---
        self.dlg.fromShapefile.setEnabled(is_geologic)
        self.dlg.inputWidth.setEnabled(is_geologic)
        self.dlg.widthinput.setEnabled(is_geologic and self.dlg.inputWidth.isChecked())
        self.dlg.feet.setEnabled(is_geologic)
        self.dlg.meters.setEnabled(is_geologic)
        # Set defaults within geologic mode if needed
        if is_geologic:
            if not (self.dlg.fromShapefile.isChecked() or self.dlg.inputWidth.isChecked()):
                self.dlg.inputWidth.setChecked(True)
            if not (self.dlg.feet.isChecked() or self.dlg.meters.isChecked()):
                self.dlg.meters.setChecked(True)


        # --- Uncertainty Ranking Checkboxes ---
        self.dlg.confidenceCheckBox.setEnabled(is_ranking)
        self.dlg.primarySecondaryCheckBox.setEnabled(is_ranking)
        self.dlg.simpleComplexCheckBox.setEnabled(is_ranking)
        # Uncheck if ranking mode is disabled
        if not is_ranking:
            self.dlg.confidenceCheckBox.setChecked(False)
            self.dlg.primarySecondaryCheckBox.setChecked(False)
            self.dlg.simpleComplexCheckBox.setChecked(False)

        # --- Percentile Radio Buttons ---
        enable_percentiles = is_general or is_ranking
        self.dlg.percentile50RadioButton.setEnabled(enable_percentiles)
        self.dlg.percentile84RadioButton.setEnabled(enable_percentiles)
        self.dlg.percentile97RadioButton.setEnabled(enable_percentiles)
        # Ensure one is checked if enabled, else uncheck
        if enable_percentiles:
            if not (self.dlg.percentile50RadioButton.isChecked() or
                    self.dlg.percentile84RadioButton.isChecked() or
                    self.dlg.percentile97RadioButton.isChecked()):
                self.dlg.percentile50RadioButton.setChecked(True)


        # --- Fault Type Specific Radio Buttons ---
        # Enabled only if UniformFaultTypeRadioButton is checked
        self.dlg.StrikeslipFaultRadioButton.setEnabled(is_uniform_fault)
        self.dlg.NormalFaultRadioButton.setEnabled(is_uniform_fault)
        self.dlg.ReverseFaultRadioButton.setEnabled(is_uniform_fault)
        # Ensure one is checked if Uniform is active
        if is_uniform_fault:
            if not (self.dlg.StrikeslipFaultRadioButton.isChecked() or
                    self.dlg.NormalFaultRadioButton.isChecked() or
                    self.dlg.ReverseFaultRadioButton.isChecked()):
                 self.dlg.StrikeslipFaultRadioButton.setChecked(True) # Default to Strike-slip


    def get_buffer_distance_for_feature(self, feature):
        """Get buffer distance based on geologic judgment settings"""
        
        # Get conversion factor based on units
        conversion_factor = 1.0  # Default for meters
        if self.dlg.feet.isChecked():
            # Convert feet to meters for calculation (if needed)
            conversion_factor = 0.3048
        
        if self.dlg.fromShapefile.isChecked():
            # Use attribute from shapefile
            if 'geo_unc' in [f.name() for f in feature.fields()]:
                buffer_distance = feature['geo_unc']
                # Make sure it's a valid number
                if buffer_distance is None or not isinstance(buffer_distance, (int, float)) or buffer_distance <= 0:
                    QgsMessageLog.logMessage(f"Invalid geo_unc value: {buffer_distance} for feature {feature.id()}", "FaultBufferTool")
                    return 0
                
                # Apply unit conversion
                return buffer_distance * conversion_factor
            else:
                QgsMessageLog.logMessage(f"Feature {feature.id()} does not have geo_unc attribute", "FaultBufferTool")
                return 0
        else:
            # Use user input
            try:
                # User input is already validated in the main run method
                buffer_distance = float(self.dlg.widthinput.text())
                # Apply unit conversion
                return buffer_distance * conversion_factor
            except ValueError:
                return 0
    
    def create_buffer_for_feature(self, feature, buffer_layer, distance, input_layer, transform=None, 
                        utm_crs=None, source_crs=None, buffer_type=""):
        """
        Creates a buffer for a feature with all the proper attributes
        
        Args:
            feature: The input feature to buffer
            buffer_layer: The output buffer layer
            distance: Buffer distance
            input_layer: The original input layer
            transform: Coordinate transform if needed
            utm_crs: UTM CRS if applicable
            source_crs: Source CRS if applicable
            buffer_type: Type of buffer (for geologic judgment)
            
        Returns:
            bool: True if successful, False otherwise
        """
        try:
            fid = feature.id()
            # Get available fields from input feature
            available_fields = [f.name() for f in feature.fields()]
            
            # Get geometry and transform if needed
            geometry = feature.geometry()
            
            # Transform to projected CRS if needed
            if transform and utm_crs:
                QgsMessageLog.logMessage(f"Transforming geometry from {source_crs.authid()} to {utm_crs.authid()}", "FaultBufferTool")
                geometry.transform(transform)
            
            # Set number of segments for buffer
            segments = 5  # Default value
            buffer_ratio = 1.0  # Default value for symmetric buffers
            buffer_geom = None
            dip_direction = None
            is_asymmetric = False
            fault_type_str = "Unknown"
            
            if self.dlg.UniformFaultTypeRadioButton.isChecked():
                # Check which fault type is selected
                if self.dlg.StrikeslipFaultRadioButton.isChecked():
                    # Symmetric buffer for strike-slip faults
                    fault_type_str = "Strike-slip (Uniform)"
                    is_asymmetric = False
                    buffer_ratio = 1.0
                    QgsMessageLog.logMessage(f"Creating symmetric buffer for strike-slip fault, distance={distance}", "FaultBufferTool")
                
                elif self.dlg.NormalFaultRadioButton.isChecked():
                    fault_type_str = "Normal (Uniform)"
                    is_asymmetric = True
                    buffer_ratio = 1/4  # Normal 1:4 FW:HW
                    QgsMessageLog.logMessage(f"Setting up Normal fault buffer with ratio {buffer_ratio}", "FaultBufferTool")
                
                elif self.dlg.ReverseFaultRadioButton.isChecked():
                    fault_type_str = "Reverse (Uniform)"
                    is_asymmetric = True
                    buffer_ratio = 1/2  # Reverse 1:2 FW:HW
                    QgsMessageLog.logMessage(f"Setting up Reverse fault buffer with ratio {buffer_ratio}", "FaultBufferTool")
            
            elif self.dlg.FromShapefileFaultTypeRadioButton.isChecked():
                # Use the buffer type from the shapefile
                if 'Fault_type' in available_fields:
                    fault_code = feature['Fault_type']
                    if isinstance(fault_code, str):
                        fault_code = fault_code.strip().upper()
                        if fault_code == 'S':
                            fault_type_str = "Strike-slip (From Shapefile)"
                            is_asymmetric = False
                            buffer_ratio = 1.0
                            QgsMessageLog.logMessage(f"Creating symmetric buffer for Strike-slip fault from shapefile, distance={distance}", "FaultBufferTool")
                        
                        elif fault_code == 'N':
                            fault_type_str = "Normal (From Shapefile)"
                            is_asymmetric = True
                            buffer_ratio = 1/4
                            QgsMessageLog.logMessage(f"Setting up Normal fault buffer from shapefile with ratio {buffer_ratio}", "FaultBufferTool")
                        
                        elif fault_code == 'R':
                            fault_type_str = "Reverse (From Shapefile)"
                            is_asymmetric = True
                            buffer_ratio = 1/2
                            QgsMessageLog.logMessage(f"Setting up Reverse fault buffer from shapefile with ratio {buffer_ratio}", "FaultBufferTool")
                        
                        else:
                            QgsMessageLog.logMessage(f"Feature {fid}: Invalid Fault_type '{fault_code}'. Defaulting to Strike-slip.", "FaultBufferTool")
                            fault_type_str = f"Strike-slip (Invalid Attribute: {fault_code})"
                            is_asymmetric = False
                            buffer_ratio = 1.0
                    else:
                        QgsMessageLog.logMessage(f"Feature {fid}: Missing or non-string Fault_type attribute '{fault_code}'. Defaulting to Strike-slip.", "FaultBufferTool")
                        fault_type_str = "Strike-slip (Missing/Invalid Attribute)"
                        is_asymmetric = False
                        buffer_ratio = 1.0
                else:
                    QgsMessageLog.logMessage(f"Feature {fid}: Missing 'Fault_type' field. Defaulting to Strike-slip.", "FaultBufferTool")
                    fault_type_str = "Strike-slip (Missing Field)"
                    is_asymmetric = False
                    buffer_ratio = 1.0
            
            # Get dip direction ONLY if needed for asymmetry
            if is_asymmetric:
                if 'Dip_direct' in available_fields:
                    field_value = feature['Dip_direct']
                    if field_value and isinstance(field_value, str) and field_value.strip():
                        dip_dir_candidate = field_value.strip().upper()
                        valid_directions = ['N', 'S', 'E', 'W', 'NE', 'NW', 'SE', 'SW']
                        if dip_dir_candidate in valid_directions:
                            dip_direction = dip_dir_candidate  # Store the valid direction
                        else:
                            QgsMessageLog.logMessage(f"Feature {fid}: Invalid dip direction value '{field_value}'. Cannot create asymmetric buffer. Falling back to symmetric.", "FaultBufferTool")
                            is_asymmetric = False  # Fallback
                            fault_type_str += f" (Symmetric Fallback - Invalid Dip: {field_value})"
                    else:
                        QgsMessageLog.logMessage(f"Feature {fid}: Empty or non-string dip direction value '{field_value}'. Cannot create asymmetric buffer. Falling back to symmetric.", "FaultBufferTool")
                        is_asymmetric = False  # Fallback
                        fault_type_str += " (Symmetric Fallback - Empty/Invalid Dip)"
                else:
                    QgsMessageLog.logMessage(f"Feature {fid}: Missing 'Dip_direct' field required for asymmetric buffer. Cannot create asymmetric buffer. Falling back to symmetric.", "FaultBufferTool")
                    is_asymmetric = False  # Fallback
                    fault_type_str += " (Symmetric Fallback - Missing Dip Field)"
            
            # Create buffer
            if is_asymmetric and dip_direction:
                QgsMessageLog.logMessage(f"Attempting asymmetric buffer for feature {fid}: dist={distance}, ratio={buffer_ratio}, dip={dip_direction}", "FaultBufferTool")
                buffer_geom = self.create_asymmetric_buffer(
                    geometry, distance, dip_direction, utm_crs, segments, buffer_ratio
                )
                if not buffer_geom or buffer_geom.isEmpty():
                    QgsMessageLog.logMessage(f"Feature {fid}: Asymmetric buffer creation failed. Falling back to symmetric buffer.", "FaultBufferTool")
                    # Fallback handled below, buffer_geom is None or empty
                    fault_type_str += " (Symmetric Fallback - Asym Buffer Failed)"
                else:
                    QgsMessageLog.logMessage(f"Feature {fid}: Asymmetric buffer created.", "FaultBufferTool")
            
            # If asymmetric failed or was never attempted/required, create symmetric
            if not buffer_geom or buffer_geom.isEmpty():
                QgsMessageLog.logMessage(f"Creating symmetric buffer for feature {fid}, dist={distance}", "FaultBufferTool")
                buffer_geom = geometry.buffer(distance, segments)
                if not buffer_geom or buffer_geom.isEmpty():
                    QgsMessageLog.logMessage(f"Feature {fid}: Symmetric buffer creation also failed.", "FaultBufferTool")
                    return False  # Cannot proceed if even symmetric fails
            
            # Create buffer feature with appropriate attributes
            if not buffer_geom or buffer_geom.isEmpty() or not buffer_geom.isGeosValid():
                QgsMessageLog.logMessage(f"Feature {fid}: Final buffer geometry is invalid or empty. Skipping.", "FaultBufferTool")
                return False  # Skip this feature
            
            buffer_feature = QgsFeature(buffer_layer.fields())
            buffer_feature.setGeometry(buffer_geom)
            
            # Copy attributes from original feature first
            buffer_field_names = [f.name() for f in buffer_layer.fields()]
            for field in input_layer.fields():  # Iterate through input fields schema
                field_name = field.name()
                if field_name in buffer_field_names:  # Check if field exists in output
                    if field_name in available_fields:  # Check if field exists in *this* input feature
                        buffer_feature.setAttribute(field_name, feature[field_name])
            
            # Set/Overwrite buffer-specific attributes
            if "original_id" in buffer_field_names:
                buffer_feature.setAttribute("original_id", fid)
            if "Buffer_Dist" in buffer_field_names:
                buffer_feature.setAttribute("Buffer_Dist", distance)
            if "Dip_Direction" in buffer_field_names:
                # Only store dip direction if asymmetry was successfully applied
                buffer_feature.setAttribute("Dip_Direction", dip_direction if is_asymmetric and dip_direction else None)
            if "Buffer_Type" in buffer_field_names:
                buffer_feature.setAttribute("Buffer_Type", fault_type_str)
            
            # Add feature to buffer layer
            buffer_provider = buffer_layer.dataProvider()
            if not buffer_provider.addFeature(buffer_feature):
                QgsMessageLog.logMessage(f"Feature {fid}: Failed to add buffered feature to layer.", "FaultBufferTool")
                return False
            
            return True
            
        except Exception as e:
            fid_str = f"feature {feature.id()}" if feature else "unknown feature"
            QgsMessageLog.logMessage(f"Error creating buffer for {fid_str}: {str(e)}", "FaultBufferTool")
            import traceback
            QgsMessageLog.logMessage(f"Traceback: {traceback.format_exc()}", "FaultBufferTool")
            return False
    
    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog
        self.dlg = FaultBufferToolDialog()
        
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = FaultBufferToolDialog()
            
        self.dlg.setupUi(self.dlg)
        
        # --- Setup UI Connections and Initial State ---
        self.setupDialogConnections() # Connect signals to slots
        self.update_ui_state() # Set the initial enabled/disabled states
    
        # Configure the file widget explicitly
        self.dlg.mQgsFileWidget.setStorageMode(QgsFileWidget.SaveFile)
        self.dlg.mQgsFileWidget.setFilter("Shapefiles (*.shp)")
        self.dlg.mQgsFileWidget.setFilePath("")  # Clear any previous path

        # show the dialog
        self.dlg.show()
        
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            
            try:
                # Get the input layer
                input_layer = self.dlg.mMapLayerComboBox.currentLayer()
                if not input_layer:
                    QMessageBox.critical(self.dlg, "Error", "Please select an input layer")
                    return
                
                # Get the output path
                output_path = self.dlg.mQgsFileWidget.filePath()
                if not output_path:
                    QMessageBox.critical(self.dlg, "Error", "Please select an output location")
                    return
                
                if not output_path.endswith('.shp'):
                    output_path += '.shp'
                    
                if output_path and not os.path.dirname(output_path):
                    # Get QGIS project folder as default location
                    project_path = QgsProject.instance().homePath()
                    if project_path:
                        output_path = os.path.join(project_path, output_path)
                    else:
                        # Fallback to user's documents folder
                        output_path = os.path.join(QStandardPaths.writableLocation(QStandardPaths.DocumentsLocation), output_path)
                    
                    QgsMessageLog.logMessage(f"Using default location, full path: {output_path}", "FaultBufferTool")
    
                # Log available fields for debugging
                QgsMessageLog.logMessage(f"Available fields: {[f.name() for f in input_layer.fields()]}", "FaultBufferTool")
                
                # Validate required fields early to fail fast
                is_valid, error_message = self.validate_required_fields(input_layer)
                if not is_valid:
                    QMessageBox.critical(self.dlg, "Error", error_message)
                    return    
                   
                # Get and check the layer's CRS
                source_crs = input_layer.crs()
                               
                # If the CRS is isgeographic, (like EPSG:4326), we'll need to transform to a projected CRS
                if source_crs.isGeographic():
                    # Print some debug information
                    QgsMessageLog.logMessage(f"Source CRS is geographic: {source_crs.description()}", "Buffer Tool")
                    
                    # Get the UTM zome for the layer's extent
                    center_point = input_layer.extent().center()
                    utm_crs = self.get_utm_crs(center_point.x(), center_point.y()) # Get UTM CRS based on center point

                    QgsMessageLog.logMessage(f"Selected UTM CRS: {utm_crs.description()}", "Buffer Tool") 
                    
                    # Create transform context
                    transform = QgsCoordinateTransform(source_crs, utm_crs, QgsProject.instance()) 
                    
                    # Create new layer in UTM projection
                    buffer_layer = QgsVectorLayer(f"Polygon?crs={utm_crs.authid()}", "buffers", "memory")
                else:
                    # Use the same CRS as input if it's already projected
                    QgsMessageLog.logMessage(f"Source CRS is projected: {source_crs.description()}", "Buffer Tool")
                    buffer_layer = QgsVectorLayer(f"Polygon?crs={source_crs.authid()}", "buffers", "memory")
                    transform = None
                    utm_crs = source_crs  # No need to transform if already projected
                
                QgsMessageLog.logMessage(f"Input CRS: {source_crs.authid()}", "FaultBufferTool")
                QgsMessageLog.logMessage(f"Buffer layer CRS: {buffer_layer.crs().authid()}", "FaultBufferTool")
                # QgsMessageLog.logMessage(f"UTM CRS for calculations: {utm_crs.authid()}", "FaultBufferTool")
                
                # Create output layer with appropriate fields
                buffer_provider = buffer_layer.dataProvider()
                
                # First add all the original fields from the input layer
                input_fields = input_layer.fields()
                fields_to_add = []
                for field in input_fields:
                    fields_to_add.append(QgsField(field.name(), field.type(), field.typeName(), 
                                                field.length(), field.precision(), field.comment()))
                
                # Then add the buffer-specific fields
                fields_to_add.extend([
                    QgsField("Buffer_Dist", QVariant.Double, len=20, prec=2),
                    QgsField("Buffer_Type", QVariant.String, len=50)
                ])
                
                # Add all fields to the buffer layer
                buffer_provider.addAttributes(fields_to_add)

                buffer_layer.updateFields()
                QgsMessageLog.logMessage("Buffer layer fields initialized", "FaultBufferTool")
                
                # Process features based on selected mode
                # Inside the run method, replace the existing geologic judgment code:
                if self.dlg.geologicJudgementRadioButton.isChecked():
                    # Handle Geologic judgment option
                    try:
                        # Check if user selected "From shapefile" or "Input width"
                        if self.dlg.fromShapefile.isChecked():
                            # Validate geo_unc field exists
                            if 'geo_unc' not in [f.name() for f in input_layer.fields()]:
                                QMessageBox.critical(self.dlg, "Error", 
                                    "The 'geo_unc' field does not exist in the input layer. Please select a layer with this field or use 'Input width' option.")
                                return
                                
                            # Process each feature with its geo_unc attribute
                            for feature in input_layer.getFeatures():
                                buffer_distance = self.get_buffer_distance_for_feature(feature)
                                if buffer_distance <= 0:
                                    QgsMessageLog.logMessage(f"Skipping feature {feature.id()}: Invalid geo_unc value", "FaultBufferTool")
                                    continue
                                
                                # Create buffer with common method
                                success = self.create_buffer_for_feature(
                                    feature, buffer_layer, buffer_distance, input_layer,
                                    transform, utm_crs, source_crs, "Geologic judgment - From shapefile"
                                )
                                if not success:
                                    QgsMessageLog.logMessage(f"Failed to create buffer for feature {feature.id()}", "FaultBufferTool")
                        
                        else:  # Input width selected
                            # Get the user-specified buffer distance
                            try:
                                user_buffer_distance = float(self.dlg.widthinput.text())
                                if user_buffer_distance <= 0:
                                    QMessageBox.critical(self.dlg, "Error", "Please enter a positive buffer distance value")
                                    return
                            except ValueError:
                                QMessageBox.critical(self.dlg, "Error", "Please enter a valid number for buffer distance")
                                return
                            
                            # Process each feature
                            for feature in input_layer.getFeatures():
                                # Get buffer distance with unit conversion
                                buffer_distance = user_buffer_distance
                                if self.dlg.feet.isChecked():
                                    buffer_distance *= 0.3048  # Convert feet to meters
                                
                                # Create buffer with common method
                                success = self.create_buffer_for_feature(
                                    feature, buffer_layer, buffer_distance, input_layer,
                                    transform, utm_crs, source_crs, "Geologic judgment - Uniform width"
                                )
                                if not success:
                                    QgsMessageLog.logMessage(f"Failed to create buffer for feature {feature.id()}", "FaultBufferTool")
                        
                        QgsMessageLog.logMessage("Geologic judgment buffers created", "FaultBufferTool")
                    
                    except Exception as e:
                        QgsMessageLog.logMessage(f"Error in Geologic judgment buffering: {str(e)}", "FaultBufferTool")
                        QMessageBox.critical(self.dlg, "Error", f"Failed to create Geologic judgment buffers: {str(e)}")
                        return
                
                else:
                    # Process features using uncertainty tables
                    feature_processed = 0
                    for feature in input_layer.getFeatures():
                        try:
                            # Get uncertainty distance
                            distance = self.get_uncertainty_distance(feature)
                            if distance <= 0:
                                QgsMessageLog.logMessage(f"Skipping feature {feature.id()}: invalid distance", "FaultBufferTool")
                                continue
                            
                            # Create buffer with common method
                            success = self.create_buffer_for_feature(
                                feature, buffer_layer, distance, input_layer,
                                transform, utm_crs, source_crs
                            )
                            if success:
                                feature_processed += 1
                            else:
                                QgsMessageLog.logMessage(f"Failed to create buffer for feature {feature.id()}", "FaultBufferTool")
                        except Exception as e:
                            QgsMessageLog.logMessage(f"Error processing feature {feature.id()}: {str(e)}", "FaultBufferTool")
                            continue

                    if feature_processed == 0:
                        QMessageBox.critical(self.dlg, "Error", "No features could be processed. Check log for details.")
                        return
                                    
                QgsMessageLog.logMessage("Buffer created", "FaultBufferTool")
                
                # Save the buffer layer
                options = QgsVectorFileWriter.SaveVectorOptions()
                options.driverName = "ESRI Shapefile"
                options.fileEncoding = "UTF-8"
                    
                # Write the layer to file
                error = QgsVectorFileWriter.writeAsVectorFormatV3(
                    buffer_layer,
                    output_path,
                    QgsProject.instance().transformContext(),
                    options
                )

                if error[0] != QgsVectorFileWriter.NoError:
                    QMessageBox.critical(self.dlg, "Error", f"Failed to save buffer layer: {error}")
                    return
                
                # Add the new layer to the map with proper styling
                output_name = os.path.splitext(os.path.basename(output_path))[0]
                buffer_layer = QgsVectorLayer(output_path, output_name, "ogr")
                if buffer_layer.isValid():
                    # Get plugin directory and locate the QML file
                    plugin_dir = os.path.dirname(__file__)
                    style_path = os.path.join(plugin_dir, "style_buffer.qml")
                    
                    # Create a new path to save a copy of the QML alongside the output shapefile
                    output_dir = os.path.dirname(output_path)
                    output_qml_path = os.path.join(output_dir, f"{output_name}.qml")
                    
                    # Copy the QML file to the output directory if the original exists
                    if os.path.exists(style_path):
                        try:
                            shutil.copy(style_path, output_qml_path)
                            QgsMessageLog.logMessage(f"Style copied to: {output_qml_path}", "FaultBufferTool")
                        except Exception as e:
                            QgsMessageLog.logMessage(f"Failed to copy style: {str(e)}", "FaultBufferTool")
                    else:
                        QgsMessageLog.logMessage(f"Original style file not found at: {style_path}", "FaultBufferTool")
                    
                    # Load the style from the QML file (original path as fallback)
                    if os.path.exists(output_qml_path):
                        buffer_layer.loadNamedStyle(output_qml_path)
                    elif os.path.exists(style_path):
                        buffer_layer.loadNamedStyle(style_path)
                    
                    # Get the layer tree and input layer's node
                    root = QgsProject.instance().layerTreeRoot()
                    input_node = root.findLayer(input_layer.id())
                    
                    # Insert buffer layer below input layer
                    if input_node:
                        QgsProject.instance().addMapLayer(buffer_layer, False)  # False = don't add to legend
                        buffer_node = root.insertLayer(root.children().index(input_node) + 1, buffer_layer)
                    else:
                        # Fallback: just add the layer normally
                        QgsProject.instance().addMapLayer(buffer_layer)

                    buffer_layer.triggerRepaint()
                    
                    # Inform user about the successful operation
                    QMessageBox.information(self.dlg, "Success", f"Buffer created successfully with symbology!")
                else:
                    QMessageBox.critical(self.dlg, "Error", "Failed to load output layer")

            except Exception as e:
                QgsMessageLog.logMessage(f"Unexpected error: {str(e)}", "FaultBufferTool")
                import traceback
                QgsMessageLog.logMessage(f"Traceback: {traceback.format_exc()}", "FaultBufferTool")
                QMessageBox.critical(self.dlg, "Error", f"An unexpected error occurred: {str(e)}")
                return