# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CutterPlugin
                                 A QGIS plugin
 Performs an in-place difference operation on the active layer, using the selected feature to clip all overlapping features.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-11-20
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Emanuele Gigante
        email                : emanuele.gigante01@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.core import QgsProject, QgsGeometry, QgsFeature, Qgis, QgsMessageLog

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .fastcutter_dialog import CutterPluginDialog
import os.path


class CutterPlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CutterPlugin_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&FastCutter')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CutterPlugin', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/fastcutter/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Clip Selection'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&FastCutter'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Esegue il taglio con SUPPORTO MULTI-SELEZIONE e DEBUG"""
        
        # --- DEBUG: Inizio operazione ---
        QgsMessageLog.logMessage("FastCutter: Avvio plugin...", "FastCutter", level=Qgis.Info)

        # 1. Controlli iniziali
        layer = self.iface.activeLayer()
        if not layer:
            self.iface.messageBar().pushMessage("Errore", "Nessun layer selezionato!", level=Qgis.Critical)
            return

        if layer.type() != layer.VectorLayer:
             self.iface.messageBar().pushMessage("Errore", "Il layer attivo non è vettoriale!", level=Qgis.Critical)
             return

        selection = layer.selectedFeatures()
        if not selection:
            self.iface.messageBar().pushMessage("Errore", "Seleziona almeno una feature da usare come coltello!", level=Qgis.Critical)
            return
        
        # --- DEBUG: Info selezione ---
        num_cutters = len(selection)
        QgsMessageLog.logMessage(f"FastCutter: Selezionate {num_cutters} feature come coltello.", "FastCutter", level=Qgis.Info)
        
        # 2. Preparazione Geometria "Coltello Unificato"
        # Se abbiamo più feature, le fondiamo in un'unica geometria
        try:
            # Prende la geometria del primo
            combined_cutter_geom = selection[0].geometry()
            
            # Lista degli ID dei coltelli (per non tagliare se stessi!)
            cutter_ids = [selection[0].id()]

            # Se ce ne sono altri, li unisce (Union)
            if num_cutters > 1:
                for i in range(1, num_cutters):
                    feat = selection[i]
                    combined_cutter_geom = combined_cutter_geom.combine(feat.geometry())
                    cutter_ids.append(feat.id())
            
            QgsMessageLog.logMessage("FastCutter: Geometria coltello combinata creata con successo.", "FastCutter", level=Qgis.Info)

        except Exception as e:
            self.iface.messageBar().pushMessage("Errore", f"Errore nella creazione del coltello: {e}", level=Qgis.Critical)
            QgsMessageLog.logMessage(f"FastCutter ERROR: {e}", "FastCutter", level=Qgis.Critical)
            return

        # 3. Avvia Editing
        if not layer.isEditable():
            layer.startEditing()
            QgsMessageLog.logMessage("FastCutter: Editing avviato automaticamente.", "FastCutter", level=Qgis.Info)

        layer.beginEditCommand("Clip selezione multipla")
        
        count = 0
        # Ottimizzazione: Cerca solo nel rettangolo che contiene tutti i coltelli
        request = combined_cutter_geom.boundingBox()
        
        # 4. Ciclo sulle feature
        processed_count = 0
        for feature in layer.getFeatures(request):
            processed_count += 1
            
            # IMPORTANTE: Salta le feature che sono coltelli!
            if feature.id() in cutter_ids:
                continue
            
            # Verifica intersezione
            if feature.geometry().intersects(combined_cutter_geom):
                try:
                    # Calcola la differenza
                    new_geom = feature.geometry().difference(combined_cutter_geom)
                    
                    # Aggiorna
                    if new_geom and not new_geom.isEmpty():
                        layer.changeGeometry(feature.id(), new_geom)
                        count += 1
                        # Debug dettagliato (solo se necessario)
                        # QgsMessageLog.logMessage(f"Tagliata feature ID: {feature.id()}", "FastCutter", level=Qgis.Info)
                        
                except Exception as e:
                    QgsMessageLog.logMessage(f"FastCutter ERROR su feature {feature.id()}: {e}", "FastCutter", level=Qgis.Warning)

        QgsMessageLog.logMessage(f"FastCutter: Analizzate {processed_count} feature, Tagliate {count}.", "FastCutter", level=Qgis.Info)

        layer.endEditCommand()
        layer.triggerRepaint()
        
        # 5. Risultato finale
        if count > 0:
            msg = f"Operazione completata!\n\n"
            msg += f"Feature selezionate: {num_cutters}\n"
            msg += f"Feature ritagliate: {count}\n\n"
            msg += "Ricorda di salvare se il risultato è corretto."
            QMessageBox.information(self.iface.mainWindow(), "FastCutter - Risultato", msg)
        else:
            QMessageBox.warning(self.iface.mainWindow(), "FastCutter", 
                                "Nessuna feature intersecava l'area selezionata.")