﻿"""
Exportar Layouts a PDF
Autor: David Bosch
Compatibilidad: QGIS 3.34.11 Prizren
Licencia: GPL v3
"""

from qgis.PyQt.QtWidgets import (
    QAction, QDialog, QVBoxLayout, QHBoxLayout, QListWidget, QListWidgetItem,
    QPushButton, QFileDialog, QMessageBox, QLabel, QProgressDialog, QApplication,
    QComboBox, QSlider, QSpinBox, QFormLayout, QCheckBox
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import Qt
from qgis.core import QgsProject, QgsLayoutExporter, Qgis
import os


# -------------------------------
# Diálogo de configuración avanzada
# -------------------------------
class ConfiguracionExportacionDialog(QDialog):
    def __init__(self, settings, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Configuración de exportación PDF")
        self.resize(350, 280)
        self.settings = settings

        layout = QFormLayout()

        self.combo_texto = QComboBox()
        self.combo_texto.addItems([
            "Siempre exportar texto como objetos de texto",
            "Siempre exportar texto como curvas"
        ])
        self.combo_texto.setCurrentIndex(0 if settings.get("texto_editable", True) else 1)
        layout.addRow("Exportación de texto:", self.combo_texto)

        self.combo_compresion = QComboBox()
        self.combo_compresion.addItems(["Lossy (JPEG)", "Lossless (PNG)"])
        self.combo_compresion.setCurrentIndex(0 if settings.get("compresion_lossy", True) else 1)
        layout.addRow("Compresión de imagen:", self.combo_compresion)

        self.slider_calidad = QSlider(Qt.Horizontal)
        self.slider_calidad.setRange(10, 100)
        self.slider_calidad.setValue(settings.get("calidad_jpeg", 75))
        self.lbl_calidad = QLabel(f"{self.slider_calidad.value()}%")
        self.slider_calidad.valueChanged.connect(lambda v: self.lbl_calidad.setText(f"{v}%"))
        calidad_layout = QHBoxLayout()
        calidad_layout.addWidget(self.slider_calidad)
        calidad_layout.addWidget(self.lbl_calidad)
        layout.addRow("Calidad JPEG:", calidad_layout)

        self.chk_vectorial = QCheckBox("Forzar salida vectorial (etiquetas editables - anula transparencias)")
        self.chk_vectorial.setChecked(settings.get("forzar_vectorial", False))
        layout.addRow(self.chk_vectorial)

        self.chk_unico = QCheckBox("Combinar todos los diseños en un único PDF")
        self.chk_unico.setChecked(settings.get("pdf_unico", False))
        layout.addRow(self.chk_unico)

        self.spin_dpi = QSpinBox()
        self.spin_dpi.setRange(72, 600)
        self.spin_dpi.setValue(settings.get("dpi", 150))
        layout.addRow("Resolución (DPI):", self.spin_dpi)

        botones = QHBoxLayout()
        btn_ok = QPushButton("Aceptar")
        btn_ok.clicked.connect(self.accept)
        btn_cancel = QPushButton("Cancelar")
        btn_cancel.clicked.connect(self.reject)
        botones.addWidget(btn_ok)
        botones.addWidget(btn_cancel)
        layout.addRow(botones)
        self.setLayout(layout)

    def get_settings(self):
        return {
            "texto_editable": self.combo_texto.currentIndex() == 0,
            "compresion_lossy": self.combo_compresion.currentIndex() == 0,
            "calidad_jpeg": self.slider_calidad.value(),
            "forzar_vectorial": self.chk_vectorial.isChecked(),
            "pdf_unico": self.chk_unico.isChecked(),
            "dpi": self.spin_dpi.value()
        }


# -------------------------------
# Diálogo principal
# -------------------------------
class ExportarLayoutsDialog(QDialog):
    def __init__(self, layouts, parent=None, settings=None):
        super().__init__(parent)
        self.setWindowTitle("Exportar Layouts a PDF")
        self.resize(420, 500)
        self.settings = settings or {}
        self.layouts = layouts
        self.selection_order = []  # nombres reales en orden de selección

        layout = QVBoxLayout()
        layout.addWidget(QLabel("Selecciona las presentaciones a exportar:"))

        self.chk_todo = QCheckBox("Seleccionar / deseleccionar todas")
        self.chk_todo.stateChanged.connect(self.toggle_all)
        layout.addWidget(self.chk_todo)

        self.listWidget = QListWidget()
        self.listWidget.setSelectionMode(QListWidget.MultiSelection)
        for l in layouts:
            item = QListWidgetItem(l.name())
            item.setData(Qt.UserRole, l.name())
            self.listWidget.addItem(item)
        self.listWidget.selectionModel().selectionChanged.connect(self.on_selection_model_changed)
        layout.addWidget(self.listWidget)

        self.folderButton = QPushButton("📁 Escoge carpeta de salida")
        self.folderButton.clicked.connect(self.choose_folder)
        layout.addWidget(self.folderButton)
        self.output_folder = None

        botones = QHBoxLayout()
        self.configButton = QPushButton("⚙️ Configuración avanzada")
        self.configButton.clicked.connect(self.abrir_configuracion)
        botones.addWidget(self.configButton)
        self.exportButton = QPushButton("Exportar seleccionados")
        self.exportButton.clicked.connect(self.accept)
        botones.addWidget(self.exportButton)
        layout.addLayout(botones)
        self.setLayout(layout)

    def toggle_all(self, state):
        if state == Qt.Checked:
            self.selection_order = []
            for i in range(self.listWidget.count()):
                item = self.listWidget.item(i)
                item.setSelected(True)
                self.selection_order.append(item.data(Qt.UserRole))
        else:
            for i in range(self.listWidget.count()):
                self.listWidget.item(i).setSelected(False)
            self.selection_order = []
        self._refresh_visual_numbers()

    def on_selection_model_changed(self, selected, deselected):
        for index in deselected.indexes():
            item = self.listWidget.item(index.row())
            if item:
                name = item.data(Qt.UserRole)
                if name in self.selection_order:
                    self.selection_order.remove(name)
        for index in selected.indexes():
            item = self.listWidget.item(index.row())
            if item:
                name = item.data(Qt.UserRole)
                if name in self.selection_order:
                    self.selection_order.remove(name)
                self.selection_order.append(name)
        seen, unique = set(), []
        for n in self.selection_order:
            if n not in seen:
                unique.append(n)
                seen.add(n)
        self.selection_order = unique
        self._refresh_visual_numbers()

    def _refresh_visual_numbers(self):
        for i in range(self.listWidget.count()):
            item = self.listWidget.item(i)
            item.setText(item.data(Qt.UserRole))
        for idx, name in enumerate(self.selection_order, start=1):
            for i in range(self.listWidget.count()):
                item = self.listWidget.item(i)
                if item.data(Qt.UserRole) == name:
                    item.setText(f"{idx}. {name}")
                    break

    def choose_folder(self):
        folder = QFileDialog.getExistingDirectory(self, "Selecciona carpeta de salida")
        if folder:
            self.output_folder = folder
            self.folderButton.setText(f"📁 Carpeta: {folder}")

    def selected_layout_names(self):
        if self.selection_order:
            return list(self.selection_order)
        return [self.listWidget.item(i).data(Qt.UserRole)
                for i in range(self.listWidget.count()) if self.listWidget.item(i).isSelected()]

    def abrir_configuracion(self):
        dlg = ConfiguracionExportacionDialog(self.settings, self)
        if dlg.exec_():
            self.settings.update(dlg.get_settings())


# -------------------------------
# Clase principal del plugin
# -------------------------------
class ExportarLayoutsPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.action = None
        self.settings = {
            "texto_editable": True,
            "compresion_lossy": True,
            "calidad_jpeg": 75,
            "forzar_vectorial": False,
            "pdf_unico": True,
            "dpi": 150
        }

    def initGui(self):
        icon_path = os.path.join(os.path.dirname(__file__), "icon.png")
        self.action = QAction(QIcon(icon_path), "Exportar Layouts a PDF", self.iface.mainWindow())
        self.action.triggered.connect(self.mostrar_dialogo)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Exportar Layouts", self.action)

    def unload(self):
        self.iface.removePluginMenu("&Exportar Layouts", self.action)
        self.iface.removeToolBarIcon(self.action)

    def mostrar_dialogo(self):
        project = QgsProject.instance()
        layouts = project.layoutManager().layouts()
        if not layouts:
            QMessageBox.warning(self.iface.mainWindow(), "Sin diseños", "No hay diseños en el proyecto.")
            return

        dlg = ExportarLayoutsDialog(layouts, self.iface.mainWindow(), self.settings)
        while True:
            if not dlg.exec_():
                return
            seleccionados = dlg.selected_layout_names()
            if not seleccionados:
                QMessageBox.warning(self.iface.mainWindow(), "Sin selección", "Selecciona al menos un diseño.")
                continue
            if not dlg.output_folder:
                folder = QFileDialog.getExistingDirectory(self.iface.mainWindow(), "Selecciona carpeta de salida")
                if not folder:
                    if QMessageBox.question(self.iface.mainWindow(), "Carpeta no seleccionada",
                                            "¿Cancelar la exportación?",
                                            QMessageBox.Yes | QMessageBox.No) == QMessageBox.Yes:
                        return
                    else:
                        continue
                dlg.output_folder = folder
            break
        self.exportar_layouts(seleccionados, dlg.output_folder)

    def _detect_image_compression_constants(self):
        if hasattr(QgsLayoutExporter, "ImageCompressionLossy") and hasattr(QgsLayoutExporter, "ImageCompressionLossless"):
            return QgsLayoutExporter.ImageCompressionLossy, QgsLayoutExporter.ImageCompressionLossless
        img_comp = getattr(QgsLayoutExporter, "ImageCompression", None)
        if img_comp is not None and hasattr(img_comp, "Lossy") and hasattr(img_comp, "Lossless"):
            return img_comp.Lossy, img_comp.Lossless
        return 0, 1

    # -------------------------------
    # Exportar layouts (con aviso si sobrescribe)
    # -------------------------------
    def exportar_layouts(self, layout_names, output_folder):
        """
        Exporta los layouts seleccionados respetando el orden de layout_names.
        Detecta sobrescritura y pide confirmación antes de exportar.
        """
        project = QgsProject.instance()
        layout_manager = project.layoutManager()
        exportados, errores = [], []

        # 🔹 Usamos directamente los nombres reales tal como vienen
        clean_names = layout_names

        # --- Detectar sobrescritura ---
        if self.settings.get("pdf_unico", False):
            salida_pdf = os.path.join(
                output_folder,
                f"{os.path.splitext(os.path.basename(project.fileName()))[0] or 'proyecto'}.pdf"
            )
            existing_files = [salida_pdf] if os.path.exists(salida_pdf) else []
        else:
            existing_files = [
                os.path.join(output_folder, f"{name}.pdf")
                for name in clean_names if os.path.exists(os.path.join(output_folder, f"{name}.pdf"))
            ]

        if existing_files:
            msg = "⚠️ Se van a sobrescribir los siguientes archivos:\n\n"
            msg += "\n".join(os.path.basename(f) for f in existing_files)
            msg += "\n\n¿Deseas continuar?"
            respuesta = QMessageBox.question(
                self.iface.mainWindow(),
                "Confirmar sobrescritura",
                msg,
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )
            if respuesta != QMessageBox.Yes:
                QMessageBox.information(self.iface.mainWindow(), "Cancelado", "Exportación cancelada por el usuario.")
                return

        # --- Configuración PDF ---
        lossy_const, lossless_const = self._detect_image_compression_constants()
        pdf_settings = QgsLayoutExporter.PdfExportSettings()
        pdf_settings.dpi = float(self.settings.get("dpi", 150))
        pdf_settings.imageCompression = lossy_const if self.settings.get("compresion_lossy", True) else lossless_const
        try:
            pdf_settings.textRenderFormat = (
                Qgis.TextRenderFormat.AlwaysText
                if self.settings.get("texto_editable", True)
                else Qgis.TextRenderFormat.AlwaysOutlines
            )
        except Exception:
            pass
        try:
            pdf_settings.forceVectorOutput = bool(self.settings.get("forzar_vectorial", False))
        except Exception:
            pass

        # --- Progreso ---
        progress = QProgressDialog("Exportando diseños...", "Cancelar", 0, len(clean_names), self.iface.mainWindow())
        progress.setWindowModality(Qt.WindowModal)
        progress.show()

        # --- Helper: buscar layout exactamente por nombre ---
        def _find_layout_by_name_exact(name):
            for l in layout_manager.layouts():
                if l.name().strip() == name.strip():
                    return l
            return None

        # --- Exportación ---
        if self.settings.get("pdf_unico", False):
            temp_paths = []
            for idx, name in enumerate(clean_names):
                lay = _find_layout_by_name_exact(name)
                if not lay:
                    errores.append(f"No encontrado: {name}")
                    progress.setValue(idx + 1)
                    QApplication.processEvents()
                    continue

                tmp = os.path.join(output_folder, f"__temp_{idx:03d}.pdf")
                result = QgsLayoutExporter(lay).exportToPdf(tmp, pdf_settings)
                if result == QgsLayoutExporter.Success and os.path.exists(tmp):
                    temp_paths.append(tmp)
                else:
                    errores.append(name)
                progress.setValue(idx + 1)
                QApplication.processEvents()
                if progress.wasCanceled():
                    break

            # --- Combinar en orden correcto ---
            if temp_paths:
                try:
                    try:
                        from pypdf import PdfMerger
                    except ImportError:
                        from PyPDF2 import PdfMerger
                    merger = PdfMerger()
                    for p in temp_paths:
                        merger.append(p)
                    with open(salida_pdf, "wb") as f_out:
                        merger.write(f_out)
                    merger.close()
                    for p in temp_paths:
                        try:
                            os.remove(p)
                        except Exception:
                            pass
                    exportados.append(salida_pdf)
                except Exception as e:
                    errores.append(f"Error combinando PDFs: {e}")
            else:
                errores.append("No se generó ningún PDF temporal válido.")

        else:
            for idx, name in enumerate(clean_names):
                lay = _find_layout_by_name_exact(name)
                if not lay:
                    errores.append(f"No encontrado: {name}")
                    progress.setValue(idx + 1)
                    QApplication.processEvents()
                    continue

                out = os.path.join(output_folder, f"{name}.pdf")
                result = QgsLayoutExporter(lay).exportToPdf(out, pdf_settings)
                if result == QgsLayoutExporter.Success and os.path.exists(out):
                    exportados.append(out)
                else:
                    errores.append(name)

                progress.setValue(idx + 1)
                QApplication.processEvents()
                if progress.wasCanceled():
                    break

        progress.close()
        resumen = f"Se exportaron {len(exportados)} archivos correctamente."
        if errores:
            resumen += f"\n⚠️ Errores: {', '.join(errores)}"
        QMessageBox.information(self.iface.mainWindow(), "Resultado", resumen)
    