﻿"""
Exportar Layouts a PDF
Copyright (C) 2025  David Bosch
Licensed under the GNU General Public License v3 (GPLv3)
"""

from qgis.PyQt.QtWidgets import (
    QAction, QDialog, QVBoxLayout, QHBoxLayout, QListWidget, QListWidgetItem,
    QPushButton, QFileDialog, QMessageBox, QLabel, QProgressDialog, QApplication,
    QComboBox, QSlider, QSpinBox, QFormLayout, QCheckBox
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import Qt
from qgis.core import QgsProject, QgsLayoutExporter, Qgis
import os


# -------------------------------
# Diálogo de configuración avanzada
# -------------------------------
class ConfiguracionExportacionDialog(QDialog):
    def __init__(self, settings, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Configuración de exportación PDF")
        self.resize(350, 250)
        self.settings = settings

        layout = QFormLayout()

        # Exportación de texto
        self.combo_texto = QComboBox()
        self.combo_texto.addItems([
            "Siempre exportar texto como objetos de texto",
            "Siempre exportar texto como curvas"
        ])
        self.combo_texto.setCurrentIndex(0 if settings.get("texto_editable", True) else 1)
        layout.addRow("Exportación de texto:", self.combo_texto)

        # Compresión
        self.combo_compresion = QComboBox()
        self.combo_compresion.addItems(["Lossy (JPEG)", "Lossless (PNG)"])
        self.combo_compresion.setCurrentIndex(0 if settings.get("compresion_lossy", True) else 1)
        layout.addRow("Image compression:", self.combo_compresion)

        # Calidad JPEG
        self.slider_calidad = QSlider(Qt.Horizontal)
        self.slider_calidad.setRange(10, 100)
        self.slider_calidad.setValue(settings.get("calidad_jpeg", 75))
        self.lbl_calidad = QLabel(f"{self.slider_calidad.value()}%")
        self.slider_calidad.valueChanged.connect(lambda v: self.lbl_calidad.setText(f"{v}%"))
        calidad_layout = QHBoxLayout()
        calidad_layout.addWidget(self.slider_calidad)
        calidad_layout.addWidget(self.lbl_calidad)
        layout.addRow("Calidad JPEG:", calidad_layout)

        # Forzar vectorial
        self.chk_vectorial = QCheckBox("Forzar salida vectorial (etiquetas editables- anula transparéncias)")
        self.chk_vectorial.setChecked(settings.get("forzar_vectorial", False))
        layout.addRow(self.chk_vectorial)

        # DPI
        self.spin_dpi = QSpinBox()
        self.spin_dpi.setRange(72, 600)
        self.spin_dpi.setValue(settings.get("dpi", 150))
        layout.addRow("Resolución (DPI):", self.spin_dpi)

        # Botones
        botones = QHBoxLayout()
        btn_ok = QPushButton("Aceptar")
        btn_ok.clicked.connect(self.accept)
        btn_cancel = QPushButton("Cancelar")
        btn_cancel.clicked.connect(self.reject)
        botones.addWidget(btn_ok)
        botones.addWidget(btn_cancel)
        layout.addRow(botones)

        self.setLayout(layout)

    def get_settings(self):
        return {
            "texto_editable": self.combo_texto.currentIndex() == 0,
            "compresion_lossy": self.combo_compresion.currentIndex() == 0,
            "calidad_jpeg": self.slider_calidad.value(),
            "forzar_vectorial": self.chk_vectorial.isChecked(),
            "dpi": self.spin_dpi.value()
        }


# -------------------------------
# Diálogo principal del plugin
# -------------------------------
class ExportarLayoutsDialog(QDialog):
    def __init__(self, layouts, parent=None, settings=None):
        super().__init__(parent)
        self.setWindowTitle("Exportar Layouts a PDF")
        self.resize(400, 450)

        self.settings = settings or {}

        layout = QVBoxLayout()
        layout.addWidget(QLabel("Selecciona las presentaciones a exportar:"))

        self.listWidget = QListWidget()
        self.listWidget.setSelectionMode(QListWidget.MultiSelection)
        for l in layouts:
            self.listWidget.addItem(QListWidgetItem(l.name()))
        layout.addWidget(self.listWidget)

        # Carpeta de salida
        self.folderButton = QPushButton("📁 Escoge carpeta de salida")
        self.folderButton.clicked.connect(self.choose_folder)
        layout.addWidget(self.folderButton)

        self.output_folder = None

        # Botones
        botones = QHBoxLayout()
        self.configButton = QPushButton("⚙️ Configuración avanzada")
        self.configButton.clicked.connect(self.abrir_configuracion)
        botones.addWidget(self.configButton)

        self.exportButton = QPushButton("Exportar seleccionados")
        self.exportButton.clicked.connect(self.accept)
        botones.addWidget(self.exportButton)

        layout.addLayout(botones)
        self.setLayout(layout)

    def choose_folder(self):
        folder = QFileDialog.getExistingDirectory(self, "Selecciona carpeta de salida")
        if folder:
            self.output_folder = folder
            self.folderButton.setText(f"📁 Carpeta: {folder}")

    def selected_layout_names(self):
        return [item.text() for item in self.listWidget.selectedItems()]

    def abrir_configuracion(self):
        dlg = ConfiguracionExportacionDialog(self.settings, self)
        if dlg.exec_():
            self.settings.update(dlg.get_settings())


# -------------------------------
# Clase principal del plugin
# -------------------------------
class ExportarLayoutsPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.action = None
        # valores por defecto
        self.settings = {
            "texto_editable": True,
            "compresion_lossy": True,
            "calidad_jpeg": 75,
            "forzar_vectorial": False,
            "dpi": 150
        }

    def initGui(self):
        icon_path = os.path.join(os.path.dirname(__file__), "icon.png")
        self.action = QAction(QIcon(icon_path), "Exportar Layouts a PDF", self.iface.mainWindow())
        self.action.triggered.connect(self.mostrar_dialogo)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Exportar Layouts", self.action)

    def unload(self):
        self.iface.removePluginMenu("&Exportar Layouts", self.action)
        self.iface.removeToolBarIcon(self.action)

    def mostrar_dialogo(self):
        project = QgsProject.instance()
        layouts = project.layoutManager().layouts()

        if not layouts:
            QMessageBox.warning(self.iface.mainWindow(), "Sin diseños", "No hay diseños en el proyecto.")
            return

        dlg = ExportarLayoutsDialog(layouts, self.iface.mainWindow(), self.settings)

        # Permite corregir errores sin cerrar el plugin
        while True:
            # Abrir diálogo principal
            if not dlg.exec_():
                # Usuario canceló -> salir sin exportar
                return

            # Si no se seleccionó ningún diseño
            seleccionados = dlg.selected_layout_names()
            if not seleccionados:
                QMessageBox.warning(
                    self.iface.mainWindow(),
                    "Sin selección",
                    "No seleccionaste ningún diseño. Por favor, selecciona uno o más antes de exportar."
                )
                continue  # volver a mostrar el diálogo

            # Si no se seleccionó carpeta
            if not dlg.output_folder:
                folder = QFileDialog.getExistingDirectory(self.iface.mainWindow(), "Selecciona carpeta de salida")
                if not folder:
                    respuesta = QMessageBox.question(
                        self.iface.mainWindow(),
                        "Carpeta no seleccionada",
                        "No has seleccionado una carpeta de salida.\n¿Quieres cancelar la exportación?",
                        QMessageBox.Yes | QMessageBox.No
                    )
                    if respuesta == QMessageBox.Yes:
                        return
                    else:
                        continue  # volver al diálogo
                else:
                    dlg.output_folder = folder

            # Si llegamos aquí, tenemos todo listo
            break

        # Ejecutar exportación
        self.exportar_layouts(seleccionados, dlg.output_folder)

    def _detect_image_compression_constants(self):
        """
        Devuelve una tupla (lossy_const, lossless_const).
        Intenta diferentes variantes de nombres según la versión/build.
        """
        # Variante 1: atributos directos en QgsLayoutExporter
        if hasattr(QgsLayoutExporter, "ImageCompressionLossy") and hasattr(QgsLayoutExporter, "ImageCompressionLossless"):
            return QgsLayoutExporter.ImageCompressionLossy, QgsLayoutExporter.ImageCompressionLossless

        # Variante 2: enumeración tipo QgsLayoutExporter.ImageCompression.Lossy
        img_comp = getattr(QgsLayoutExporter, "ImageCompression", None)
        if img_comp is not None:
            # puede ser una clase/enum con atributos Lossy/Lossless
            if hasattr(img_comp, "Lossy") and hasattr(img_comp, "Lossless"):
                return img_comp.Lossy, img_comp.Lossless

        # Variante 3: usar enteros por defecto (LOSSY=0, LOSSLESS=1 es una suposición razonable)
        return 0, 1

    def exportar_layouts(self, layout_names, output_folder):
        project = QgsProject.instance()
        layout_manager = project.layoutManager()
        exportados, errores = [], []

        progress = QProgressDialog("Exportando diseños...", "Cancelar", 0, len(layout_names), self.iface.mainWindow())
        progress.setWindowTitle("Exportando")
        progress.setWindowModality(Qt.WindowModal)
        progress.show()

        # detectar constantes de compresión robustamente
        lossy_const, lossless_const = self._detect_image_compression_constants()

        # preparar PdfExportSettings
        pdf_settings = QgsLayoutExporter.PdfExportSettings()
        pdf_settings.dpi = float(self.settings.get("dpi", 150))
        # imageQuality suele estar disponible en 3.34; si no, el atributo se ignorará por Python al asignar
        try:
            pdf_settings.imageQuality = int(self.settings.get("calidad_jpeg", 75))
        except Exception:
            pass

        # textRenderFormat según la GUI
        try:
            pdf_settings.textRenderFormat = (
                Qgis.TextRenderFormat.AlwaysText
                if self.settings.get("texto_editable", True)
                else Qgis.TextRenderFormat.AlwaysOutlines
            )
        except Exception:
            # fallback: no hacer nada si no disponible
            pass

        # imageCompression: usar la constante detectada
        if self.settings.get("compresion_lossy", True):
            pdf_settings.imageCompression = lossy_const
        else:
            pdf_settings.imageCompression = lossless_const

        # Forzar salida vectorial si el usuario lo desea
        try:
            pdf_settings.forceVectorOutput = bool(self.settings.get("forzar_vectorial", True))
        except Exception:
            pass

        # Exportar cada layout seleccionado
        for i, layout in enumerate(layout_manager.layouts()):
            if layout.name() not in layout_names:
                continue

            output_path = os.path.join(output_folder, f"{layout.name()}.pdf")
            exporter = QgsLayoutExporter(layout)
            result = exporter.exportToPdf(output_path, pdf_settings)

            if result == QgsLayoutExporter.Success:
                exportados.append(output_path)
            else:
                errores.append(layout.name())

            progress.setValue(i + 1)
            QApplication.processEvents()
            if progress.wasCanceled():
                break

        progress.close()
        resumen = f"Se exportaron {len(exportados)} diseños correctamente."
        if errores:
            resumen += f"\n⚠️ Error en: {', '.join(errores)}"
        QMessageBox.information(self.iface.mainWindow(), "Resultado", resumen)
