# -*- coding: utf-8 -*-

from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QDialog, QVBoxLayout, QHBoxLayout, QLabel, QSlider, QComboBox, QCheckBox, QDialogButtonBox
from qgis.PyQt.QtCore import Qt

class LayerSettingsDialog(QDialog):
    """Dialog for configuring per-layer export settings."""
    
    def __init__(self, parent=None, settings=None):
        super().__init__(parent)
        self.setWindowTitle("Layer Export Settings")
        self.resize(380, 180) # Compact size
        
        # Default settings
        self.settings = {
            'fill_transparency': 0,
            'stroke_scale': 1.0,
            'export_labels': False,
            'render_as_image': False
        }
        
        if settings:
            self.settings.update(settings)
            
        self.init_gui()
        
    def init_gui(self):
        layout = QVBoxLayout()
        layout.setContentsMargins(15, 15, 15, 10)
        layout.setSpacing(10)
        self.setLayout(layout)
        
        # 1. Fill Transparency
        transp_layout = QHBoxLayout()
        transp_layout.addWidget(QLabel("Fill Transparency:"))
        
        self.slider_transparency = QSlider(Qt.Horizontal)
        self.slider_transparency.setRange(0, 100)
        self.slider_transparency.setValue(self.settings['fill_transparency'])
        self.slider_transparency.valueChanged.connect(self.update_transparency_label)
        transp_layout.addWidget(self.slider_transparency)
        
        self.lbl_transparency_val = QLabel(f"{self.settings['fill_transparency']}%")
        self.lbl_transparency_val.setFixedWidth(35)
        self.lbl_transparency_val.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
        transp_layout.addWidget(self.lbl_transparency_val)
        
        layout.addLayout(transp_layout)
        
        # 2. Stroke Scale
        stroke_layout = QHBoxLayout()
        stroke_layout.addWidget(QLabel("Stroke Scale:"))
        
        self.cmb_stroke = QComboBox()
        self.cmb_stroke.addItems(["0.5x", "1x (Original)", "1.5x", "2x", "3x"])
        
        # Set current index based on value
        scale = self.settings['stroke_scale']
        idx = 1 # Default 1x
        if scale == 0.5: idx = 0
        elif scale == 1.5: idx = 2
        elif scale == 2.0: idx = 3
        elif scale == 3.0: idx = 4
        self.cmb_stroke.setCurrentIndex(idx)
        
        stroke_layout.addWidget(self.cmb_stroke)
        stroke_layout.setStretch(1, 1) # Allow combo to stretch
        layout.addLayout(stroke_layout)
        
        # 3. Checkboxes (Side by Side)
        checks_layout = QHBoxLayout()
        
        self.chk_labels = QCheckBox("Export Labels")
        self.chk_labels.setChecked(self.settings['export_labels'])
        self.chk_labels.setToolTip("Export feature labels to KML")
        checks_layout.addWidget(self.chk_labels)
        
        # Render as Image (Pattern Support)
        self.chk_render_image = QCheckBox("Render as Image")
        self.chk_render_image.setChecked(self.settings['render_as_image'])
        self.chk_render_image.setToolTip("Render as PNG image. Check this for PATTERN fills.")
        checks_layout.addWidget(self.chk_render_image)
        
        layout.addLayout(checks_layout)
        
        # Spacer to push buttons down if resized larger
        layout.addStretch()
        
        # Dialog Buttons
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        
    def update_transparency_label(self, value):
        self.lbl_transparency_val.setText(f"{value}%")
        
    def get_settings(self):
        """Return the current settings from the dialog widgets."""
        scale_text = self.cmb_stroke.currentText()
        scale = 1.0
        if '0.5' in scale_text: scale = 0.5
        elif '1.5' in scale_text: scale = 1.5
        elif '2' in scale_text: scale = 2.0
        elif '3' in scale_text: scale = 3.0
        
        return {
            'fill_transparency': self.slider_transparency.value(),
            'stroke_scale': scale,
            'export_labels': self.chk_labels.isChecked(),
            'render_as_image': self.chk_render_image.isChecked()
        }
