# -*- coding: utf-8 -*-

"""
export2kml.py
QGIS plugin: Export 2KML
"""

import os
import tempfile
import zipfile
try:
    from defusedxml import ElementTree as ET
except ImportError:
    import xml.etree.ElementTree as ET

from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import (
    QAction, QFileDialog, QTableWidgetItem, QCheckBox, QComboBox,
    QMessageBox, QToolBar, QLineEdit, QPushButton, QLabel, QDialog, QMenu
)
from .vertex_limit_dialog import VertexLimitDialog
from .layer_settings_dialog import LayerSettingsDialog
from qgis.PyQt.QtGui import QIcon, QImage, QPainter, QColor, QCursor
from qgis.PyQt.QtCore import Qt, QSize

from qgis.core import (
    QgsProject, QgsVectorLayer, QgsRasterLayer, QgsMapSettings,
    QgsMapRendererParallelJob, QgsMapRendererCustomPainterJob, QgsGeometry, 
    QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsCoordinateTransformContext, 
    QgsFeatureRequest, Qgis, QgsWkbTypes, QgsRectangle, QgsPointXY, QgsMessageLog,
    QgsLayerTree, QgsMapLayerType
)
from qgis.PyQt.QtCore import QObject, QEvent
from qgis.utils import iface as qgis_iface

from osgeo import ogr, gdal
from .vertex_limit_dialog import VertexLimitDialog
from .field_selector_dialog import FieldSelectorDialog
from .symbology_extractor import (
    extract_layer_styles, get_feature_style_id, render_all_marker_styles
)
from .kml_style_generator import add_styles_to_document, get_style_url

gdal.UseExceptions()
ogr.UseExceptions()

KML_PROPS = [
    'Name','description','timestamp','begin','end',
    'altitudeMode','tessellate','extrude','visibility',
    'drawOrder','icon'
]

MAX_VERTICES = 250000
GOOGLE_EARTH_VERTEX_WARNING = 10000  # Google Earth "Map features" limit - show warning
SIMPLIFY_PROMPT_THRESHOLD = 50000   # Ask user if they want to simplify above this

LOG_TAG = "Export2KML"

def log(message: str, level=Qgis.Info):
    """Log message to QGIS log panel."""
    QgsMessageLog.logMessage(message, LOG_TAG, level)

def get_vertex_count(geom: QgsGeometry) -> int:
    """Count vertices in geometry."""
    if not geom or geom.isEmpty():
        return 0
    
    vertex_count = 0
    geom_type = geom.type()
    
    if geom.isMultipart():
        if geom_type == 0:  # MultiPoint
            vertex_count = len(geom.asMultiPoint())
        elif geom_type == 1:  # MultiLineString
            for line in geom.asMultiPolyline():
                vertex_count += len(line)
        elif geom_type == 2:  # MultiPolygon
            for polygon in geom.asMultiPolygon():
                for ring in polygon:
                    vertex_count += len(ring)
    else:
        if geom_type == 0:  # Point
            vertex_count = 1
        elif geom_type == 1:  # LineString
            vertex_count = len(geom.asPolyline())
        elif geom_type == 2:  # Polygon
            polygon = geom.asPolygon()
            for ring in polygon:
                vertex_count += len(ring)
    
    return vertex_count

def validate_coordinates_loosely(geom: QgsGeometry, feature_id) -> tuple:
    """Loosely validate coordinates and return (is_valid, reason, bbox_info)."""
    if not geom or geom.isEmpty():
        return False, "Empty geometry", None
    
    try:
        bbox = geom.boundingBox()
        bbox_info = f"[{bbox.xMinimum():.6f}, {bbox.yMinimum():.6f}, {bbox.xMaximum():.6f}, {bbox.yMaximum():.6f}]"
        
        # Check for obviously invalid coordinates (extremely large values)
        if (abs(bbox.xMinimum()) > 10000000 or abs(bbox.xMaximum()) > 10000000 or
            abs(bbox.yMinimum()) > 10000000 or abs(bbox.yMaximum()) > 10000000):
            return False, "Coordinates extremely large (>10M)", bbox_info
        
        # Check for all coordinates being exactly zero
        if (abs(bbox.xMinimum()) < 0.0000001 and abs(bbox.xMaximum()) < 0.0000001 and
            abs(bbox.yMinimum()) < 0.0000001 and abs(bbox.yMaximum()) < 0.0000001):
            return False, "All coordinates are zero", bbox_info
        
        # More lenient - accept coordinates that might be outside normal WGS84 but not crazy
        return True, "Valid", bbox_info
        
    except Exception as e:
        return False, f"Error getting bbox: {e}", None

def simplify_complex_geometry(geom: QgsGeometry, feature_id, max_vertices=SIMPLIFY_PROMPT_THRESHOLD):
    """Simplify overly complex geometries."""
    if not geom or geom.isEmpty():
        return geom
    
    vertex_count = get_vertex_count(geom)
    if vertex_count <= max_vertices:
        return geom
    
    print(f"Feature {feature_id}: Simplifying complex geometry ({vertex_count:,} vertices)")
    
    try:
        bbox = geom.boundingBox()
        tolerance = max(bbox.width(), bbox.height()) * 0.001
        
        for i in range(5):
            simplified = geom.simplify(tolerance)
            if simplified and not simplified.isEmpty():
                simplified_vertices = get_vertex_count(simplified)
                if simplified_vertices <= max_vertices:
                    print(f"Feature {feature_id}: Simplified to {simplified_vertices:,} vertices")
                    return simplified
            tolerance *= 2
        
        print(f"Feature {feature_id}: Could not simplify sufficiently")
        return geom
        
    except Exception as e:
        print(f"Feature {feature_id}: Error simplifying geometry: {e}")
        return geom

def geometry_to_kml_element(geom: QgsGeometry, feature_id):
    """Convert QGIS geometry to KML geometry element."""
    if not geom or geom.isEmpty():
        return None
    
    try:
        # Try to fix invalid geometries but don't be too strict
        if hasattr(geom, 'isGeosValid') and not geom.isGeosValid():
            print(f"Feature {feature_id}: Attempting to fix invalid geometry...")
            try:
                fixed_geom = geom.makeValid()
                if fixed_geom and not fixed_geom.isEmpty():
                    geom = fixed_geom
                    print(f"Feature {feature_id}: Geometry fixed")
            except:
                print(f"Feature {feature_id}: Could not fix geometry, proceeding anyway")
        
        geom_type_name = QgsWkbTypes.displayString(geom.wkbType())
        vertex_count = get_vertex_count(geom)
        
        # Note: We don't auto-simplify here anymore - user decides via dialog
        # Just log if vertex count is high
        if vertex_count > SIMPLIFY_PROMPT_THRESHOLD:
            print(f"Feature {feature_id}: High vertex count ({vertex_count:,}), user may choose to simplify")
        
        wkt = geom.asWkt()
        
        # Limit WKT size to prevent memory issues
        if len(wkt) > 2000000:
            print(f"Feature {feature_id}: WKT too long, simplifying further")
            geom = simplify_complex_geometry(geom, feature_id, SIMPLIFY_PROMPT_THRESHOLD // 4)
            if not geom or geom.isEmpty():
                return None
            wkt = geom.asWkt()
        
        ogrg = ogr.CreateGeometryFromWkt(wkt)
        if ogrg is None:
            print(f"Feature {feature_id}: Failed to create OGR geometry")
            return None
        
        kml_string = ogrg.ExportToKML()
        if not kml_string:
            print(f"Feature {feature_id}: Failed to export to KML")
            return None
        
        # Parse KML
        try:
            wrapped_kml = f'<root>{kml_string}</root>'
            root = ET.fromstring(wrapped_kml)
            
            geometry_elements = []
            for elem in root:
                if elem.tag in ['Point', 'LineString', 'Polygon', 'MultiGeometry']:
                    geometry_elements.append(elem)
            
            return geometry_elements if geometry_elements else None
            
        except ET.ParseError as e:
            print(f"Feature {feature_id}: XML parse error: {e}")
            return None
            
    except Exception as e:
        print(f"Feature {feature_id}: Geometry conversion error: {e}")
        return None

def make_kml_root():
    kml = ET.Element('kml', xmlns='http://www.opengis.net/kml/2.2')
    return kml, ET.SubElement(kml, 'Document')

def add_vector_layer(doc, layer: QgsVectorLayer, props, total_vertices_so_far, max_vertices, iface_parent, layer_styles=None, style_map_ids=None, progress_callback=None, selected_only=False):
    """Process vector layer with symbology support, progress reporting, and optional selected-only export."""

    
    print(f"\n" + "="*60)
    print(f"PROCESSING LAYER: {layer.name()}")
    print(f"Starting total vertices: {total_vertices_so_far:,}")
    print(f"Feature count: {layer.featureCount()}")
    
    # Get layer CRS information
    layer_crs = layer.crs()
    layer_crs_authid = layer_crs.authid()
    print(f"Layer CRS: {layer_crs_authid} - {layer_crs.description()}")
    
    if not layer.isValid():
        print("❌ Layer is not valid - skipping")
        return total_vertices_so_far, True, None
    
    if layer.featureCount() == 0:
        print("❌ Layer has no features - skipping")
        return total_vertices_so_far, True, None
    
    # Create folder
    folder = ET.SubElement(doc, 'Folder')
    ET.SubElement(folder, 'name').text = props.get('folder_name') or layer.name()
    
    # CRS handling
    wgs84_crs = QgsCoordinateReferenceSystem('EPSG:4326')
    needs_transform = False
    xform = None
    
    if layer_crs_authid == 'EPSG:4326':
        print("✅ Layer is already in WGS84 - NO TRANSFORMATION")
        needs_transform = False
    elif layer_crs.isValid():
        print(f"⚠️ Layer needs transformation: {layer_crs_authid} -> EPSG:4326")
        needs_transform = True
        try:
            xform = QgsCoordinateTransform(layer_crs, wgs84_crs, QgsProject.instance().transformContext())
            print("✅ Coordinate transformation created")
        except Exception as e:
            print(f"❌ Failed to create transformation: {e}")
            needs_transform = False
    else:
        print("⚠️ Invalid CRS - proceeding without transformation")
        needs_transform = False
    
    # Initialize counters and tracking
    fields_avail = [f.name() for f in layer.fields()]
    current_total = total_vertices_so_far
    processed_count = 0
    skipped_count = 0
    user_choice = None
    dialog_shown = False
    
    # Detailed skip tracking
    skip_reasons = {
        'empty_geometry': 0,
        'transform_error': 0,
        'invalid_coords': 0,
        'geometry_conversion_failed': 0,
        'user_skip': 0
    }
    
    geometry_stats = {'Point': 0, 'LineString': 0, 'Polygon': 0, 'MultiGeometry': 0, 'Other': 0}
    
    # Determine feature count based on selection mode
    if selected_only and layer.selectedFeatureCount() > 0:
        total_feature_count = layer.selectedFeatureCount()
        feature_request = QgsFeatureRequest().setFilterFids(layer.selectedFeatureIds())
        print(f"🚀 Starting to process {total_feature_count} SELECTED features...")
    else:
        total_feature_count = layer.featureCount()
        feature_request = QgsFeatureRequest()
        print(f"🚀 Starting to process {total_feature_count} features...")
    
    # Process each feature
    feature_count = 0
    for feat in layer.getFeatures(feature_request):
        feature_count += 1
        
        try:
            # Progress reporting for large datasets
            if feature_count % 100 == 0 or feature_count == total_feature_count:
                progress_pct = int((feature_count / max(total_feature_count, 1)) * 100)
                if progress_callback:
                    progress_callback(progress_pct, f"Processing {layer.name()}: {feature_count}/{total_feature_count}")
                print(f"📊 Progress: {feature_count}/{total_feature_count} features, "
                      f"processed: {processed_count}, skipped: {skipped_count}, "
                      f"total vertices: {current_total:,}")
            
            geom = feat.geometry()
            if not geom or geom.isEmpty():
                skip_reasons['empty_geometry'] += 1
                skipped_count += 1
                continue
            
            original_geom_type = QgsWkbTypes.displayString(geom.wkbType())
            
            # Show details for first few features
            if processed_count + skipped_count < 5:
                print(f"🔍 Feature {feat.id()}: {original_geom_type}")
            
            # Transform if needed
            if needs_transform and xform:
                try:
                    geom_copy = QgsGeometry(geom)
                    geom_copy.transform(xform)
                    geom = geom_copy
                    
                    if processed_count + skipped_count < 3:
                        print(f"🔄 Feature {feat.id()}: Transformed successfully")
                        
                except Exception as e:
                    if processed_count + skipped_count < 10:
                        print(f"❌ Feature {feat.id()}: Transform error: {e}")
                    skip_reasons['transform_error'] += 1
                    skipped_count += 1
                    continue
            
            # Validate coordinates (loosely)
            is_valid, reason, bbox_info = validate_coordinates_loosely(geom, feat.id())
            if not is_valid:
                if processed_count + skipped_count < 10:
                    print(f"❌ Feature {feat.id()}: {reason}, bbox: {bbox_info}")
                skip_reasons['invalid_coords'] += 1
                skipped_count += 1
                continue
            
            # Count vertices BEFORE showing dialog
            vc = get_vertex_count(geom)
            potential_total = current_total + vc
            
            if processed_count + skipped_count < 5:
                print(f"📏 Feature {feat.id()}: {vc:,} vertices, total would be: {potential_total:,}")
            
            # Check vertex thresholds and show appropriate dialog
            ge_warning_shown = getattr(add_vector_layer, 'ge_warning_shown', False)
            simplify_dialog_shown = getattr(add_vector_layer, 'simplify_dialog_shown', False)
            
            # Tier 1: Google Earth warning at 10K (info only)
            if potential_total > GOOGLE_EARTH_VERTEX_WARNING and not ge_warning_shown and user_choice != 'keep':
                print(f"ℹ️ GOOGLE EARTH VERTEX WARNING (10K)")
                print(f"   Current total: {current_total:,}")
                print(f"   Would be: {potential_total:,}")
                
                dlg = VertexLimitDialog(iface_parent, potential_total, GOOGLE_EARTH_VERTEX_WARNING, layer.name(), is_simplify_prompt=False)
                result = dlg.exec_()
                add_vector_layer.ge_warning_shown = True
                
                if result == QDialog.Accepted and dlg.choice:
                    if dlg.choice == 'stop':
                        print("🛑 User chose to stop export")
                        if processed_count == 0:
                            doc.remove(folder)
                        break
                    # 'keep' means continue
                    user_choice = 'keep'
                else:
                    print("❌ Dialog cancelled - stopping")
                    if processed_count == 0:
                        doc.remove(folder)
                    break
            
            # Tier 2: Simplify prompt at 50K
            if potential_total > SIMPLIFY_PROMPT_THRESHOLD and not simplify_dialog_shown and user_choice != 'simplify' and user_choice != 'keep_all':
                print(f"⚠️ SIMPLIFY PROMPT (50K)")
                print(f"   Current total: {current_total:,}")
                print(f"   Would be: {potential_total:,}")
                
                dlg = VertexLimitDialog(iface_parent, potential_total, SIMPLIFY_PROMPT_THRESHOLD, layer.name(), is_simplify_prompt=True)
                result = dlg.exec_()
                add_vector_layer.simplify_dialog_shown = True
                
                if result == QDialog.Accepted and dlg.choice:
                    user_choice = dlg.choice
                    print(f"👤 User chose: {user_choice}")
                    
                    if user_choice == 'stop':
                        print("🛑 User chose to stop export")
                        if processed_count == 0:
                            doc.remove(folder)
                        break
                    elif user_choice == 'simplify':
                        print("🔄 User chose to simplify geometries")
                        # Simplify this geometry
                        geom = simplify_complex_geometry(geom, feat.id(), SIMPLIFY_PROMPT_THRESHOLD // 10)
                        vc = get_vertex_count(geom)
                        potential_total = current_total + vc
                    # 'keep' means keep all vertices
                else:
                    print("❌ Dialog cancelled - stopping")
                    if processed_count == 0:
                        doc.remove(folder)
                    break
            
            # If user chose simplify, simplify subsequent geometries too
            if user_choice == 'simplify':
                geom = simplify_complex_geometry(geom, feat.id(), SIMPLIFY_PROMPT_THRESHOLD // 10)
                vc = get_vertex_count(geom)
            
            # Skip remaining features if user chose skip
            if user_choice == 'skip':
                skip_reasons['user_skip'] += 1
                skipped_count += 1
                continue
            
            # Process this feature
            current_total += vc
            processed_count += 1
            
            # Create placemark
            pm = ET.SubElement(folder, 'Placemark')
            
            # Add style URL if symbology is available
            if layer_styles and style_map_ids:
                feat_style_id = get_feature_style_id(feat, layer_styles, layer)
                style_url = get_style_url(feat_style_id, style_map_ids)
                if style_url:
                    ET.SubElement(pm, 'styleUrl').text = style_url
            
            # Determine name field - auto-select if export_labels is checked and no name set
            name_field = props.get('name')
            if not name_field and props.get('export_labels') and fields_avail:
                # Try to find a 'name' or 'label' field, otherwise first string field, otherwise first field
                for f in fields_avail:
                    if f.lower() in ('name', 'label', 'title', 'id'):
                        name_field = f
                        break
                if not name_field:
                    name_field = fields_avail[0]
            
            # Add name
            if name_field and name_field in fields_avail:
                name_value = feat[name_field]
                ET.SubElement(pm, 'name').text = str(name_value) if name_value is not None else f"Feature {feat.id()}"
            else:
                # Only add default name if export_labels is TRUE or it's a Point layer (names usually shown)
                # Or if we want to ensure every feature has a name for the list view
                ET.SubElement(pm, 'name').text = f"Feature {feat.id()}"
            
            # Add description
            if props.get('description') and props['description'] in fields_avail:
                desc_value = feat[props['description']]
                ET.SubElement(pm, 'description').text = str(desc_value) if desc_value is not None else ""
            
            # Add extended data
            if props.get('fields'):
                ext = ET.SubElement(pm, 'ExtendedData')
                for fld in props['fields']:
                    if fld in fields_avail:
                        dt = ET.SubElement(ext, 'Data', name=fld)
                        field_value = feat[fld]
                        ET.SubElement(dt, 'value').text = str(field_value) if field_value is not None else ""
            
            # Convert geometry to KML
            geometry_elements = geometry_to_kml_element(geom, feat.id())
            if geometry_elements:
                # Add centroid for polygon labels if requested
                if props.get('export_labels') and geom.type() == QgsWkbTypes.PolygonGeometry:
                    try:
                        centroid = geom.centroid()
                        if centroid and not centroid.isEmpty():
                            pt = centroid.asPoint()
                            pt_elem = ET.Element('Point')
                            ET.SubElement(pt_elem, 'coordinates').text = f"{pt.x()},{pt.y()}"
                            geometry_elements.append(pt_elem)
                    except Exception as e:
                        # Don't fail export just because centroid failed
                        print(f"⚠️ Failed to calculate label centroid for feature {feat.id()}: {e}")

                # If multiple elements (e.g. Polygon + Centroid), wrap in MultiGeometry
                target_parent = pm
                if len(geometry_elements) > 1:
                    target_parent = ET.SubElement(pm, 'MultiGeometry')
                
                for geom_elem in geometry_elements:
                    target_parent.append(geom_elem)
                    geom_tag = geom_elem.tag
                    if geom_tag in geometry_stats:
                        geometry_stats[geom_tag] += 1
                    else:
                        geometry_stats['Other'] += 1
            else:
                if processed_count < 10:
                    print(f"❌ Feature {feat.id()}: Failed to convert geometry")
                skip_reasons['geometry_conversion_failed'] += 1
                # Don't increment processed_count for failed conversions
                processed_count -= 1
                current_total -= vc
                skipped_count += 1
                continue
                
        except Exception as e:
            if skipped_count < 10:
                print(f"❌ Feature {feat.id()}: Unexpected error: {e}")
            skipped_count += 1
            continue
    
    # Final summary
    print(f"\n" + "="*60)
    print(f"LAYER {layer.name()} SUMMARY:")
    print(f"📊 Total features in layer: {layer.featureCount()}")
    print(f"✅ Successfully processed: {processed_count:,}")
    print(f"❌ Skipped: {skipped_count:,}")
    print(f"📏 Total vertices added: {current_total - total_vertices_so_far:,}")
    print(f"📏 Running total vertices: {current_total:,}")
    print(f"🎯 Vertex limit: {max_vertices:,}")
    print(f"🔄 CRS transformation: {'Yes' if needs_transform else 'No'}")
    
    if skipped_count > 0:
        print(f"\nSkip reasons breakdown:")
        for reason, count in skip_reasons.items():
            if count > 0:
                print(f"  - {reason}: {count:,}")
    
    if processed_count > 0:
        print(f"\nGeometry types processed:")
        for geom_type, count in geometry_stats.items():
            if count > 0:
                print(f"  - {geom_type}: {count:,}")
    
    print("="*60)
    
    return current_total, True, user_choice

def add_raster_layer(doc, rlayer, props, rasters_dir, max_dimension=2048, progress_callback=None):
    """
    Export a raster layer to KMZ as a PNG ground overlay.
    
    Args:
        doc: KML Document element
        rlayer: QgsRasterLayer to export
        props: Properties dict (folder_name, name, etc.)
        rasters_dir: Directory to save raster PNGs
        max_dimension: Maximum width/height for output PNG (default 2048 for better compatibility)
        progress_callback: Optional callback function(percent, message)
        
    Returns:
        Path to output PNG file
    """
    os.makedirs(rasters_dir, exist_ok=True)
    base = rlayer.name().replace(' ', '_').replace('/', '_')
    
    if progress_callback:
        progress_callback(0, f"Processing raster: {base}")
    
    # Get the source path
    source_path = rlayer.source()
    
    # Check if we need to reproject to EPSG:4326
    layer_crs = rlayer.crs()
    needs_reprojection = layer_crs.authid() != 'EPSG:4326'
    
    try:
        if needs_reprojection:
            # Create a temporary VRT for reprojection
            tmp_vrt = os.path.join(rasters_dir, f'tmp_{base}_4326.vrt')
            
            if progress_callback:
                progress_callback(10, f"Reprojecting {base} to EPSG:4326...")
            
            # Use gdal.Warp with correct keyword arguments (not options list)
            result = gdal.Warp(
                tmp_vrt, 
                source_path,
                format='VRT',
                dstSRS='EPSG:4326'
            )
            
            if result is None:
                raise RuntimeError(f"GDAL Warp failed for {source_path}")
            
            result = None  # Close the dataset
            
            # Load the reprojected layer
            working_layer = QgsRasterLayer(tmp_vrt, base)
        else:
            working_layer = rlayer
            tmp_vrt = source_path
        
        if not working_layer.isValid():
            raise RuntimeError(f'Cannot load raster: {source_path}')
        
        if progress_callback:
            progress_callback(30, f"Rendering {base}...")
        
        # Calculate output dimensions (limit very large rasters for memory efficiency)
        orig_width = working_layer.width()
        orig_height = working_layer.height()
        
        if orig_width > max_dimension or orig_height > max_dimension:
            # Scale down to max_dimension while preserving aspect ratio
            scale = min(max_dimension / orig_width, max_dimension / orig_height)
            out_width = int(orig_width * scale)
            out_height = int(orig_height * scale)
            log(f"Raster {base} reduced from {orig_width}x{orig_height} to {out_width}x{out_height}", Qgis.Info)
        else:
            out_width = orig_width
            out_height = orig_height
        
        # Render raster to image with transparent background (preserves nodata)
        ms = QgsMapSettings()
        ms.setLayers([working_layer])
        ms.setExtent(working_layer.extent())
        ms.setOutputSize(QSize(out_width, out_height))
        ms.setBackgroundColor(QColor(0, 0, 0, 0))  # Transparent background (RGBA with alpha=0)
        
        job = QgsMapRendererParallelJob(ms)
        job.start()
        job.waitForFinished()
        img = job.renderedImage()
        
        if img.isNull():
            raise RuntimeError(f"Failed to render raster: {base}")
        
        if progress_callback:
            progress_callback(70, f"Saving PNG for {base}...")
        
        # Save as PNG with alpha channel (preserves transparency/nodata)
        out_png = os.path.join(rasters_dir, f'{base}.png')
        
        # Convert to ARGB32 format to preserve transparency
        if img.format() != QImage.Format_ARGB32:
            img = img.convertToFormat(QImage.Format_ARGB32)
        
        if not img.save(out_png, 'PNG'):
            raise RuntimeError(f"Failed to save PNG: {out_png}")
        
        if progress_callback:
            progress_callback(85, f"Creating KML overlay for {base}...")
        
        # Get geotransform for LatLonBox
        ds = gdal.Open(tmp_vrt)
        if ds is None:
            raise RuntimeError(f"Cannot open reprojected raster: {tmp_vrt}")
        
        gt = ds.GetGeoTransform()
        raster_width = ds.RasterXSize
        raster_height = ds.RasterYSize
        ds = None  # Close dataset
        
        # Calculate bounds
        north = gt[3]
        south = gt[3] + gt[5] * raster_height
        east = gt[0] + gt[1] * raster_width
        west = gt[0]
        
        # Create KML GroundOverlay element
        go = ET.SubElement(doc, 'GroundOverlay')
        ET.SubElement(go, 'visibility').text = '1'
        ET.SubElement(go, 'name').text = props.get('folder_name') or props.get('name') or base
        
        # Add drawOrder to ensure visibility over terrain
        ET.SubElement(go, 'drawOrder').text = '50'
        ET.SubElement(go, 'altitudeMode').text = 'clampToGround'
        
        # Add description if available
        if props.get('description'):
            ET.SubElement(go, 'description').text = props.get('description')
        
        ico = ET.SubElement(go, 'Icon')
        ET.SubElement(ico, 'href').text = f'rasters/{base}.png'
        
        llb = ET.SubElement(go, 'LatLonBox')
        ET.SubElement(llb, 'north').text = str(north)
        ET.SubElement(llb, 'south').text = str(south)
        ET.SubElement(llb, 'east').text = str(east)
        ET.SubElement(llb, 'west').text = str(west)
        
        if progress_callback:
            progress_callback(100, f"Completed: {base}")
        
        log(f"Exported raster: {base} ({out_width}x{out_height}px)", Qgis.Info)
        return out_png
        
    except Exception as e:
        log(f"Error exporting raster {base}: {e}", Qgis.Warning)
        raise


def add_vector_layer_as_image(doc, vlayer, props, vectors_dir, max_dimension=2048, progress_callback=None, fill_mode='Auto'):
    """
    Export a vector layer to KMZ as a PNG ground overlay.
    This preserves complex symbology like hatched patterns that KML vectors cannot render.
    
    Args:
        doc: KML Document element
        vlayer: QgsVectorLayer to export
        props: Properties dict (folder_name, name, etc.)
        vectors_dir: Directory to save vector PNGs
        max_dimension: Maximum width/height for output PNG
        progress_callback: Optional callback function(percent, message)
        fill_mode: 'Auto', 'Fill', or 'Hollow'
        
    Returns:
        Path to output PNG file
    """
    from qgis.core import (
        QgsCoordinateReferenceSystem, QgsCoordinateTransform,
        QgsProject, QgsMapSettings, QgsMapRendererParallelJob
    )
    from qgis.PyQt.QtCore import QSize
    from qgis.PyQt.QtGui import QImage
    
    os.makedirs(vectors_dir, exist_ok=True)
    base = vlayer.name().replace(' ', '_').replace('/', '_')
    
    if progress_callback:
        progress_callback(0, f"Rendering vector as image: {base}")
    
    try:
        # Get layer extent in EPSG:4326
        layer_crs = vlayer.crs()
        target_crs = QgsCoordinateReferenceSystem('EPSG:4326')
        
        extent = vlayer.extent()
        if layer_crs.authid() != 'EPSG:4326':
            transform = QgsCoordinateTransform(layer_crs, target_crs, QgsProject.instance())
            extent = transform.transformBoundingBox(extent)
        
        if progress_callback:
            progress_callback(20, f"Setting up renderer for {base}...")
        
        # Calculate output dimensions based on extent aspect ratio
        width_deg = extent.width()
        height_deg = extent.height()
        
        if width_deg == 0 or height_deg == 0:
            raise RuntimeError(f"Layer {base} has zero extent")
        
        aspect = width_deg / height_deg
        
        if aspect >= 1:
            out_width = min(max_dimension, 4096)
            out_height = int(out_width / aspect)
        else:
            out_height = min(max_dimension, 4096)
            out_width = int(out_height * aspect)
        
        # Ensure minimum size
        out_width = max(out_width, 256)
        out_height = max(out_height, 256)
        
        if progress_callback:
            progress_callback(40, f"Rendering {base} ({out_width}x{out_height})...")
        
        # Apply hollow mode if requested
        original_renderer = None
        if fill_mode == 'Hollow' and hasattr(vlayer, 'renderer'):
            original_renderer = vlayer.renderer().clone()
            hollow = create_hollow_renderer(vlayer.renderer())
            if hollow:
                vlayer.setRenderer(hollow)
        
        # Setup map settings
        ms = QgsMapSettings()
        ms.setLayers([vlayer])
        ms.setExtent(extent)
        ms.setOutputSize(QSize(out_width, out_height))
        ms.setDestinationCrs(target_crs)
        ms.setBackgroundColor(QColor(0, 0, 0, 0))
        
        # Enable labels if requested
        if props.get('export_labels'):
            ms.setFlag(QgsMapSettings.DrawLabeling, True)
        
        # Render to image
        job = QgsMapRendererParallelJob(ms)
        job.start()
        job.waitForFinished()
        img = job.renderedImage()
        
        # Restore original renderer if modified
        if original_renderer:
            vlayer.setRenderer(original_renderer)
        
        if img.isNull():
            raise RuntimeError(f"Failed to render vector layer: {base}")
        
        if progress_callback:
            progress_callback(70, f"Saving PNG for {base}...")
        
        # Save as PNG with alpha channel
        out_png = os.path.join(vectors_dir, f'{base}.png')
        
        if img.format() != QImage.Format_ARGB32:
            img = img.convertToFormat(QImage.Format_ARGB32)
        
        if not img.save(out_png, 'PNG'):
            raise RuntimeError(f"Failed to save PNG: {out_png}")
        
        if progress_callback:
            progress_callback(85, f"Creating KML overlay for {base}...")
        
        # Get extent bounds for LatLonBox
        north = extent.yMaximum()
        south = extent.yMinimum()
        east = extent.xMaximum()
        west = extent.xMinimum()
        
        # Create KML GroundOverlay
        folder_name = props.get('folder_name', base)
        folder = ET.SubElement(doc, 'Folder')
        ET.SubElement(folder, 'name').text = folder_name
        
        go = ET.SubElement(folder, 'GroundOverlay')
        ET.SubElement(go, 'name').text = f'{base} (Pattern Layer)'
        
        # Add drawOrder to ensure visibility over terrain
        ET.SubElement(go, 'drawOrder').text = '60'
        ET.SubElement(go, 'altitudeMode').text = 'clampToGround'
        
        if props.get('description'):
            ET.SubElement(go, 'description').text = props.get('description')
        
        ico = ET.SubElement(go, 'Icon')
        ET.SubElement(ico, 'href').text = f'vectors/{base}.png'
        
        llb = ET.SubElement(go, 'LatLonBox')
        ET.SubElement(llb, 'north').text = str(north)
        ET.SubElement(llb, 'south').text = str(south)
        ET.SubElement(llb, 'east').text = str(east)
        ET.SubElement(llb, 'west').text = str(west)
        
        if progress_callback:
            progress_callback(100, f"Completed: {base}")
        
        log(f"Exported vector as image: {base} ({out_width}x{out_height}px)", Qgis.Info)
        return out_png
        
    except Exception as e:
        log(f"Error exporting vector as image {base}: {e}", Qgis.Warning)
        raise


def create_hollow_renderer(renderer):
    """
    Create a hollow (outline-only) version of a renderer.
    
    This clones the renderer and modifies all polygon symbols to have no fill,
    only showing the outline/stroke.
    
    Args:
        renderer: The original renderer to modify
        
    Returns:
        Modified renderer with hollow (no fill) polygons
    """
    from qgis.core import (
        QgsSingleSymbolRenderer, QgsCategorizedSymbolRenderer,
        QgsGraduatedSymbolRenderer, QgsSimpleFillSymbolLayer
    )
    from qgis.PyQt.QtCore import Qt
    from qgis.PyQt.QtGui import QColor
    
    if renderer is None:
        return None
    
    # Clone the renderer
    new_renderer = renderer.clone()
    
    def make_symbol_hollow(symbol):
        """Remove fill from a symbol, keeping only the outline"""
        if symbol is None:
            return
        
        for i in range(symbol.symbolLayerCount()):
            sym_layer = symbol.symbolLayer(i)
            
            # Try ALL methods to remove the fill
            # Method 1: Set brush style to NoBrush
            if hasattr(sym_layer, 'setBrushStyle'):
                sym_layer.setBrushStyle(Qt.NoBrush)
                print(f"  🔲 Set NoBrush for symbol layer {i}")
            
            # Method 2: Set fill color to fully transparent (alpha = 0)
            if hasattr(sym_layer, 'setFillColor'):
                transparent = QColor(0, 0, 0, 0)
                sym_layer.setFillColor(transparent)
                print(f"  🔲 Set transparent fill color for symbol layer {i}")
            
            # Method 3: For QgsSimpleFillSymbolLayer, also try setColor
            if isinstance(sym_layer, QgsSimpleFillSymbolLayer):
                # Get current stroke color to preserve it
                stroke_color = sym_layer.strokeColor() if hasattr(sym_layer, 'strokeColor') else QColor(0, 0, 0, 255)
                # Set the main color (fill) to transparent
                sym_layer.setColor(QColor(0, 0, 0, 0))
                # Restore stroke
                if hasattr(sym_layer, 'setStrokeColor'):
                    sym_layer.setStrokeColor(stroke_color)
                print(f"  🔲 Set main color to transparent for symbol layer {i}")
    
    # Handle different renderer types
    if isinstance(new_renderer, QgsSingleSymbolRenderer):
        symbol = new_renderer.symbol()
        make_symbol_hollow(symbol)
        
    elif isinstance(new_renderer, QgsCategorizedSymbolRenderer):
        for cat in new_renderer.categories():
            make_symbol_hollow(cat.symbol())
            
    elif isinstance(new_renderer, QgsGraduatedSymbolRenderer):
        for rng in new_renderer.ranges():
            make_symbol_hollow(rng.symbol())
    else:
        print(f"⚠️ Unknown renderer type for hollow: {type(new_renderer)}")
        # Try to get the symbol directly
        if hasattr(new_renderer, 'symbol'):
            make_symbol_hollow(new_renderer.symbol())
    
    return new_renderer


# def add_vector_layer_as_image(doc, vlayer, props, images_dir, max_dimension=4096, progress_callback=None, fill_mode='Auto'):
#     """
#     Export a vector layer as a PNG ground overlay (preserves complex symbology).
#     
#     This method renders the vector layer exactly as it appears in QGIS,
#     including complex symbology like patterns, gradients, SVG markers, etc.
#     
#     Args:
#         doc: KML Document element
#         vlayer: QgsVectorLayer to export
#         props: Properties dict (folder_name, name, etc.)
#         images_dir: Directory to save the PNG
#         max_dimension: Maximum width/height for output PNG
#         progress_callback: Optional callback function(percent, message)
#         fill_mode: 'Auto', 'Fill', or 'Hollow' - controls polygon fill rendering
#         
#     Returns:
#         Path to output PNG file
#     """
#     os.makedirs(images_dir, exist_ok=True)
#     base = vlayer.name().replace(' ', '_').replace('/', '_').replace('\\', '_')
#     
#     if progress_callback:
#         progress_callback(0, f"Rendering vector as image: {base}")
#     
#     # Store original renderer for restoration
#     original_renderer = None
#     modified_renderer = None
#     
#     try:
#         # Handle fill mode for polygon layers
#         if fill_mode == 'Hollow' and vlayer.geometryType() == 2:  # Polygon
#             print(f"🔲 Applying HOLLOW mode for image rendering: {base}")
#             original_renderer = vlayer.renderer().clone()
#             modified_renderer = create_hollow_renderer(vlayer.renderer())
#             if modified_renderer:
#                 vlayer.setRenderer(modified_renderer)
#                 vlayer.triggerRepaint()
#         
#         # Get layer extent in EPSG:4326 for the GroundOverlay
#         layer_crs = vlayer.crs()
#         extent = vlayer.extent()
#         
#         # Transform extent to EPSG:4326 if needed
#         if layer_crs.authid() != 'EPSG:4326':
#             transform = QgsCoordinateTransform(
#                 layer_crs,
#                 QgsCoordinateReferenceSystem('EPSG:4326'),
#                 QgsCoordinateTransformContext()
#             )
#             extent = transform.transformBoundingBox(extent)
#         
#         if progress_callback:
#             progress_callback(20, f"Rendering {base}...")
#         
#         # Calculate output dimensions based on extent aspect ratio
#         extent_width = extent.width()
#         extent_height = extent.height()
#         
#         if extent_width <= 0 or extent_height <= 0:
#             raise RuntimeError(f"Invalid layer extent for {base}")
#         
#         aspect_ratio = extent_width / extent_height
#         
#         if aspect_ratio >= 1:
#             # Wider than tall
#             out_width = min(max_dimension, 4096)
#             out_height = int(out_width / aspect_ratio)
#         else:
#             # Taller than wide
#             out_height = min(max_dimension, 4096)
#             out_width = int(out_height * aspect_ratio)
#         
#         # Ensure minimum dimensions
#         out_width = max(out_width, 100)
#         out_height = max(out_height, 100)
#         
#         # Render vector layer to image with PROPER transparent background
#         # Create image with transparent background first
#         
#         # Create a transparent image
#         img = QImage(out_width, out_height, QImage.Format_ARGB32_Premultiplied)
#         img.fill(Qt.transparent)  # Explicitly fill with transparent
#         
#         # Setup map settings
#         ms = QgsMapSettings()
#         ms.setLayers([vlayer])
#         ms.setExtent(vlayer.extent())  # Use original extent for proper rendering
#         ms.setOutputSize(QSize(out_width, out_height))
#         ms.setBackgroundColor(QColor(0, 0, 0, 0))  # Transparent background (RGBA with alpha=0)
#         ms.setDestinationCrs(layer_crs)  # Keep original CRS for rendering
#         
#         # Use QPainter to render onto our transparent image
#         painter = QPainter(img)
#         painter.setRenderHint(QPainter.Antialiasing, True)
#         painter.setRenderHint(QPainter.SmoothPixmapTransform, True)
#         
#         # Use custom painter job for better control over transparency
#         job = QgsMapRendererCustomPainterJob(ms, painter)
#         job.start()
#         job.waitForFinished()
#         
#         painter.end()
#         
#         if img.isNull():
#             raise RuntimeError(f"Failed to render vector layer: {base}")
#         
#         if progress_callback:
#             progress_callback(60, f"Saving PNG for {base}...")
#         
#         # Save as PNG with alpha channel (already in correct format)
#         out_png = os.path.join(images_dir, f'{base}.png')
#         
#         if not img.save(out_png, 'PNG'):
#             raise RuntimeError(f"Failed to save PNG: {out_png}")
#         
#         if progress_callback:
#             progress_callback(80, f"Creating KML overlay for {base}...")
#         
#         # Create KML GroundOverlay element
#         go = ET.SubElement(doc, 'GroundOverlay')
#         ET.SubElement(go, 'visibility').text = '1'
#         ET.SubElement(go, 'name').text = props.get('folder_name') or props.get('name') or base
#         
#         if props.get('description'):
#             ET.SubElement(go, 'description').text = props.get('description')
#         
#         ico = ET.SubElement(go, 'Icon')
#         ET.SubElement(ico, 'href').text = f'vectors/{base}.png'
#         
#         llb = ET.SubElement(go, 'LatLonBox')
#         ET.SubElement(llb, 'north').text = str(extent.yMaximum())
#         ET.SubElement(llb, 'south').text = str(extent.yMinimum())
#         ET.SubElement(llb, 'east').text = str(extent.xMaximum())
#         ET.SubElement(llb, 'west').text = str(extent.xMinimum())
#         
#         if progress_callback:
#             progress_callback(100, f"Completed: {base}")
#         
#         log(f"Exported vector as image: {base} ({out_width}x{out_height}px)", Qgis.Info)
#         return out_png
#         
#     except Exception as e:
#         log(f"Error exporting vector as image {base}: {e}", Qgis.Warning)
#         raise
#     finally:
#         # Restore original renderer if it was modified
#         if original_renderer:
#             vlayer.setRenderer(original_renderer)
#             vlayer.triggerRepaint()
#             print(f"✅ Restored original renderer for: {base}")

class LayerTreeContextMenuManager(QObject):
    """Manages context menu additions to layer tree for Export2KML"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.view = qgis_iface.layerTreeView()
        self.menuProvider = self.view.menuProvider()
        self.providers = []
        self.patch()
        self.view.viewport().installEventFilter(self)
    
    def patch(self):
        """Patch the menu provider to add custom actions"""
        if not hasattr(self.menuProvider, "_original_createContextMenu"):
            self.menuProvider._original_createContextMenu = self.menuProvider.createContextMenu
        if not hasattr(self.menuProvider, "providers"):
            self.menuProvider.providers = []
        self.menuProvider.createContextMenu = self.createContextMenu
    
    def createContextMenu(self, event=None):
        """Create context menu with custom actions"""
        menu = self.menuProvider._original_createContextMenu()
        for provider in self.menuProvider.providers:
            try:
                provider(menu, event)
            except TypeError:
                provider(menu)
        return menu
    
    def addProvider(self, provider):
        """Add a context menu provider"""
        if callable(provider) and provider not in self.menuProvider.providers:
            self.providers.append(provider)
            self.menuProvider.providers.append(provider)
    
    def removeProvider(self, provider):
        """Remove a context menu provider"""
        if provider in self.providers:
            self.providers.remove(provider)
        if provider in self.menuProvider.providers:
            self.menuProvider.providers.remove(provider)
    
    def eventFilter(self, obj, event):
        """Filter context menu events"""
        if event.type() == QEvent.ContextMenu:
            menu = self.menuProvider.createContextMenu(event)
            if menu:
                menu.exec_(self.view.mapToGlobal(event.pos()))
                return True
        return False


class Export2KML:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.added_file_layers = []
        self.selected_fields = {}
        self.menu = '&MAS Vector Processing'
        self.actions = []
        self.toolbar = None
        self.context_menu_manager = None
        self.context_menu_provider = None

    def initGui(self):
        ui = os.path.join(self.plugin_dir, 'form.ui')
        self.dlg = uic.loadUi(ui)
        
        # Find or create the shared toolbar
        self.toolbar = self.iface.mainWindow().findChild(QToolBar, 'MASVectorProcessingToolbar')
        if not self.toolbar:
            self.toolbar = self.iface.addToolBar('MAS Vector Processing')
            self.toolbar.setObjectName('MASVectorProcessingToolbar')
        
        # Create main action
        act = QAction(QIcon(os.path.join(self.plugin_dir, 'export2kml.png')),
                     'Export 2KML…', self.iface.mainWindow())
        act.triggered.connect(self.show_dialog)
        self.iface.addPluginToVectorMenu(self.menu, act)
        self.toolbar.addAction(act)
        self.actions.append(act)
        
        # Connect dialog buttons
        self.dlg.btnBrowseFiles.clicked.connect(self.add_layer)
        self.dlg.btnSelectOutput.clicked.connect(self.select_output)
        self.dlg.btnRun.clicked.connect(self.run_export)
        self.dlg.btnCancel.clicked.connect(self.dlg.close)
        self.dlg.tableLayers.cellClicked.connect(self._toggle)
        
        # Connect context menu for layer settings (right-click on table)
        # Explicitly set policy to ensure it works
        self.dlg.tableLayers.setContextMenuPolicy(Qt.CustomContextMenu)
        self.dlg.tableLayers.customContextMenuRequested.connect(self.show_layer_settings_menu)
        
        # Initialize layer settings dictionary (keyed by layer ID)
        self.layer_settings = {}
        
        # Setup context menu for layer tree (right-click)
        self.setup_context_menu()

    def setup_context_menu(self):
        """Setup context menu for layer tree (multiple layer selection)"""
        try:
            self.context_menu_manager = LayerTreeContextMenuManager()
            self.context_menu_provider = self.add_context_menu_action
            self.context_menu_manager.addProvider(self.context_menu_provider)
        except Exception as e:
            log(f"Could not setup context menu: {e}", Qgis.Warning)

    def _is_pattern_layer(self, layer):
        """
        Check if a layer uses pattern fill symbology that KML cannot render as vectors.
        Returns True if the layer should be rasterized for accurate visual output.
        """
        if not isinstance(layer, QgsVectorLayer):
            return False
        
        # Only check polygon layers
        if layer.geometryType() != 2:  # 2 = PolygonGeometry
            return False
        
        try:
            renderer = layer.renderer()
            if not renderer:
                return False
            
            # Get symbols from renderer
            from qgis.core import QgsRenderContext
            ctx = QgsRenderContext()
            symbols = renderer.symbols(ctx)
            
            for symbol in symbols:
                for i in range(symbol.symbolLayerCount()):
                    sym_layer = symbol.symbolLayer(i)
                    layer_type = sym_layer.layerType()
                    
                    # Check for pattern symbol layer types
                    pattern_types = [
                        'LinePatternFill', 'PointPatternFill', 'SVGFill', 
                        'RasterFill', 'CentroidFill', 'RandomMarkerFill'
                    ]
                    if layer_type in pattern_types:
                        print(f"🔍 Detected pattern layer type: {layer_type}")
                        return True
                    
                    # Check brush style for Qt patterns (Dense, Hatch, etc.)
                    if hasattr(sym_layer, 'brushStyle'):
                        brush = sym_layer.brushStyle()
                        # Qt.SolidPattern = 1, Qt.NoBrush = 0
                        # Patterns are 2-14 (Dense1-7, Hor, Ver, Cross, BDiag, FDiag, DiagCross)
                        if brush > 1 and brush <= 14:
                            print(f"🔍 Detected Qt pattern brush style: {brush}")
                            return True
            
            return False
            
        except Exception as e:
            print(f"⚠️ Error detecting pattern: {e}")
            return False

    def add_context_menu_action(self, menu, event=None):
        """Add context menu action for vector layers - supports multiple selection"""
        try:
            # Get selected layers from layer tree
            selected_nodes = self.iface.layerTreeView().selectedNodes()
            vector_layers = []
            
            for node in selected_nodes:
                if QgsLayerTree.isLayer(node):
                    layer = node.layer()
                    if layer and layer.type() == QgsMapLayerType.VectorLayer:
                        vector_layers.append(layer)
            
            if vector_layers:
                # Create export action
                if len(vector_layers) == 1:
                    action_text = f'Export to KML/KMZ'
                else:
                    action_text = f'Export {len(vector_layers)} layers to KML/KMZ'
                
                export_action = QAction(action_text, menu)
                export_action.triggered.connect(lambda: self.run_from_context(vector_layers))
                
                # Find position before Styles
                actions = menu.actions()
                insert_position = len(actions)
                
                for i, action in enumerate(actions):
                    action_text_lower = action.text().lower()
                    if 'style' in action_text_lower:
                        insert_position = i
                        break
                    if action.menu():
                        submenu_title = action.menu().title().lower()
                        if 'style' in submenu_title:
                            insert_position = i
                            break
                
                if insert_position < len(actions):
                    separator = menu.insertSeparator(actions[insert_position])
                    menu.insertAction(separator, export_action)
                else:
                    menu.addSeparator()
                    menu.addAction(export_action)
                    
        except Exception as e:
            log(f"Context menu error: {e}", Qgis.Warning)

    def run_from_context(self, layers):
        """Run export from context menu with pre-selected layers"""
        self._populate()
        
        # Uncheck all layers first
        tbl = self.dlg.tableLayers
        for i in range(tbl.rowCount()):
            cb = tbl.cellWidget(i, 0)
            if cb:
                cb.setChecked(False)
        
        # Check only the context-selected layers
        all_layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
        for layer in layers:
            for i, L in enumerate(all_layers):
                if L == layer and i < tbl.rowCount():
                    cb = tbl.cellWidget(i, 0)
                    if cb:
                        cb.setChecked(True)
                    break
        
        self._update_format()
        self.dlg.setParent(self.iface.mainWindow(), Qt.Tool)
        self.dlg.setWindowFlags(self.dlg.windowFlags() | Qt.Tool)
        self.dlg.show()

    def unload(self):
        # Remove context menu provider
        if self.context_menu_manager and self.context_menu_provider:
            self.context_menu_manager.removeProvider(self.context_menu_provider)
        self.context_menu_manager = None
        self.context_menu_provider = None
        
        # Remove actions from menu and toolbar
        for a in self.actions:
            self.iface.removePluginVectorMenu(self.menu, a)
            if self.toolbar:
                self.toolbar.removeAction(a)
        self.actions.clear()
        
        # Don't delete the toolbar - it may be shared with other plugins
        self.toolbar = None

    def show_dialog(self):
        self._populate()
        self._update_format()
        self.dlg.setParent(self.iface.mainWindow(), Qt.Tool)
        self.dlg.setWindowFlags(self.dlg.windowFlags() | Qt.Tool)
        self.dlg.show()

    def _populate(self):
        layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
        tbl = self.dlg.tableLayers
        
        # Columns: Export?, Fill Mode, Layer Name, Type, CRS, Folder, Fields, + KML_PROPS
        cols = 5 + 2 + len(KML_PROPS)
        tbl.setColumnCount(cols)
        tbl.setHorizontalHeaderLabels(['Export?', 'Fill Mode', 'Layer Name', 'Type', 'CRS', 'Folder', 'Fields'] + KML_PROPS)
        tbl.setRowCount(len(layers))
        
        for i, L in enumerate(layers):
            # Column 0: Export checkbox
            cb = QCheckBox(); cb.setChecked(True); cb.stateChanged.connect(self._update_format)
            cb.setContextMenuPolicy(Qt.CustomContextMenu)
            cb.customContextMenuRequested.connect(lambda pos, r=i, w=cb: self.show_context_menu_for_row(r, w.mapToGlobal(pos)))
            tbl.setCellWidget(i, 0, cb)
            
            # Column 1: Fill Mode dropdown (only for polygon layers)
            fill_combo = QComboBox()
            fill_combo.addItems(['Auto', 'Fill', 'Hollow'])
            if isinstance(L, QgsVectorLayer):
                geom_type = L.geometryType()
                if geom_type == 2:  # Polygon
                    fill_combo.setEnabled(True)
                    fill_combo.setToolTip("Auto: Detect from symbology\nFill: Force filled polygon\nHollow: Force outline only (no fill)")
                else:
                    fill_combo.setEnabled(False)
                    fill_combo.setToolTip("Fill mode only applies to polygon layers")
            else:
                fill_combo.setEnabled(False)
                fill_combo.setToolTip("Fill mode only applies to polygon layers")
            
            fill_combo.setContextMenuPolicy(Qt.CustomContextMenu)
            fill_combo.customContextMenuRequested.connect(lambda pos, r=i, w=fill_combo: self.show_context_menu_for_row(r, w.mapToGlobal(pos)))
            tbl.setCellWidget(i, 1, fill_combo)
            
            # Column 2: Layer Name
            nm = QTableWidgetItem(L.name()); nm.setFlags(nm.flags() & ~Qt.ItemIsEditable)
            tbl.setItem(i, 2, nm)
            
            # Column 3: Type
            tp = 'Raster' if isinstance(L, QgsRasterLayer) else 'Vector'
            it = QTableWidgetItem(tp); it.setFlags(it.flags() & ~Qt.ItemIsEditable)
            tbl.setItem(i, 3, it)
            
            # Column 4: CRS
            if hasattr(L, 'crs') and L.crs().isValid():
                crs_text = L.crs().authid()
                if crs_text == 'EPSG:4326':
                    crs_item = QTableWidgetItem(f"✓ {crs_text}")
                    crs_item.setBackground(Qt.green)
                else:
                    crs_item = QTableWidgetItem(f"⚠ {crs_text}")
                    crs_item.setBackground(Qt.yellow)
            else:
                crs_item = QTableWidgetItem("Unknown")
                crs_item.setBackground(Qt.red)
            
            crs_item.setFlags(crs_item.flags() & ~Qt.ItemIsEditable)
            tbl.setItem(i, 4, crs_item)
            
            # Column 5: Folder
            fld = QLineEdit(L.name())
            fld.setContextMenuPolicy(Qt.CustomContextMenu)
            fld.customContextMenuRequested.connect(lambda pos, r=i, w=fld: self.show_context_menu_for_row(r, w.mapToGlobal(pos)))
            tbl.setCellWidget(i, 5, fld)
            
            # Column 6: Fields button
            if isinstance(L, QgsVectorLayer):
                btn = QPushButton("Select…")
                btn.clicked.connect(lambda _, r=i, LL=L: self._choose_fields(r, LL))
            else:
                btn = QLabel("N/A")
            
            if hasattr(btn, 'setContextMenuPolicy'):
                btn.setContextMenuPolicy(Qt.CustomContextMenu)
                btn.customContextMenuRequested.connect(lambda pos, r=i, w=btn: self.show_context_menu_for_row(r, w.mapToGlobal(pos)))
            tbl.setCellWidget(i, 6, btn)
            
            # Columns 7+: KML properties
            for j in range(len(KML_PROPS)):
                c = QComboBox(); c.addItem('')
                c.setContextMenuPolicy(Qt.CustomContextMenu)
                c.customContextMenuRequested.connect(lambda pos, r=i, w=c: self.show_context_menu_for_row(r, w.mapToGlobal(pos)))
                if isinstance(L, QgsVectorLayer):
                    c.addItems([f.name() for f in L.fields()])
                c.setEditable(True)
                tbl.setCellWidget(i, 7 + j, c)

    def _update_transparency_label(self, value):
        self.dlg.lblTransparencyValue.setText(f"{value}%")

    def _toggle(self, r, c):
        if c == 0:
            w = self.dlg.tableLayers.cellWidget(r, 0)
            w.setChecked(not w.isChecked())

    def show_layer_settings_menu(self, position):
        """Show context menu for layer settings (from table background)."""
        index = self.dlg.tableLayers.indexAt(position)
        if not index.isValid():
            return
        
        # Determine global position from table viewport
        global_pos = self.dlg.tableLayers.viewport().mapToGlobal(position)
        self.show_context_menu_for_row(index.row(), global_pos)

    def show_context_menu_for_row(self, row, global_pos=None):
        """Display context menu for a specific row."""
        # Reconstruct layers list to match table
        layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
        
        if row < 0 or row >= len(layers):
            print(f"❌ Row {row} out of range")
            return
            
        layer = layers[row]
        
        menu = QMenu()
        action_settings = menu.addAction(f"Layer Settings: {layer.name()}")
        action_settings.triggered.connect(lambda: self.open_layer_settings_dialog(row, layer.id()))
        
        if global_pos is None:
            global_pos = QCursor.pos()
            
        menu.exec_(global_pos)

    def open_layer_settings_dialog(self, row, layer_id):
        """Open settings dialog for a specific layer."""
        # Get the layer for auto-detection
        layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
        layer = layers[row] if row < len(layers) else None
        
        # Auto-detect if this is a pattern layer
        is_pattern = self._is_pattern_layer(layer) if layer else False
        
        # Get existing settings or defaults (with smart default for patterns)
        current_settings = self.layer_settings.get(layer_id, {
            'fill_transparency': 0, # Default opaque
            'stroke_scale': 1.0,
            'export_labels': False,
            'render_as_image': is_pattern  # Auto-enable for pattern layers
        })
        
        dlg = LayerSettingsDialog(self.dlg, current_settings)
        if dlg.exec_() == QDialog.Accepted:
            # Save new settings
            new_settings = dlg.get_settings()
            self.layer_settings[layer_id] = new_settings
            print(f"✅ Settings saved for layer {layer_id}: {new_settings}")
            
            # Update visual indicator (maybe add an icon or change text color in table? Optional)
            # For now just logging is enough verification

    def _choose_fields(self, row, lyr):
        src = lyr.source()
        ex = self.selected_fields.get(src, [])
        dlg = FieldSelectorDialog([f.name() for f in lyr.fields()], ex)
        if dlg.exec_():
            ch = dlg.get_selected(); self.selected_fields[src] = ch
            # Column 6 is Fields (was incorrectly 5)
            b = self.dlg.tableLayers.cellWidget(row, 6)
            b.setText(f"{len(ch)} selected" if ch else "Select…")

    def _update_format(self):
        tbl = self.dlg.tableLayers; anyr = False
        layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
        for i, L in enumerate(layers):
            if isinstance(L, QgsRasterLayer) and tbl.cellWidget(i, 0).isChecked():
                anyr = True; break
        self.dlg.radioKML.setEnabled(not anyr)
        if anyr:
            self.dlg.radioKMZ.setChecked(True)

    def add_layer(self):
        p, _ = QFileDialog.getOpenFileName(
            self.dlg, 'Browse Input Files', '',
            'Vector (*.shp *.gpkg *.geojson *.json *.zip *.kml);;'
            'Raster (*.tif *.tiff *.jpg *.jpeg *.png);;All (*.*)'
        )
        
        if not p:
            return
        
        if p.lower().endswith(('.tif', '.tiff', '.jpg', '.jpeg', '.png')):
            L = QgsRasterLayer(p, os.path.basename(p))
        else:
            L = QgsVectorLayer(p, os.path.basename(p), 'ogr')
        
        if not L.isValid():
            QMessageBox.critical(self.iface.mainWindow(), 'Error', 'Cannot load layer')
            return
        
        self.added_file_layers.append(L)
        self._populate(); self._update_format()

    def select_output(self):
        fmt = 'kmz' if self.dlg.radioKMZ.isChecked() else 'kml'
        p, _ = QFileDialog.getSaveFileName(self.dlg, 'Output File', '', f'*.{fmt}')
        if not p:
            return
        if not p.lower().endswith(f'.{fmt}'):
            p += f'.{fmt}'
        self.output_path = p
        self.dlg.editOutput.setText(p)

    def run_export(self):
        """Export with detailed debugging."""
        print("\n" + "🚀"*20)
        print("STARTING DETAILED EXPORT ANALYSIS")
        print("🚀"*20)
        
        try:
            out = getattr(self, 'output_path', None)
            if not out:
                raise ValueError('Please select an output file')
            
            kml, doc = make_kml_root()
            tmp = tempfile.mkdtemp()
            rasdir = os.path.join(tmp, 'rasters')
            os.makedirs(rasdir, exist_ok=True)
            
            layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
            tbl = self.dlg.tableLayers
            
            total_vertices = 0
            should_continue = True
            
            # Get selected layers
            selected_layers = []
            for i, L in enumerate(layers):
                if i < tbl.rowCount() and tbl.cellWidget(i, 0).isChecked():
                    selected_layers.append((i, L))
            
            print(f"📋 Selected layers: {len(selected_layers)}")
            for i, L in selected_layers:
                print(f"   - {L.name()} ({type(L).__name__})")
            
            # Get export options
            selected_only = self.dlg.chkSelectedOnly.isChecked()
            if selected_only:
                print("📌 Mode: Export SELECTED features only")
            
            # Check if ANY layer needs vector-as-image rendering (for checking if we need vecdir)
            # Include auto-detected pattern layers
            any_vector_as_image = False
            vecdir = None
            for i, L in selected_layers:
                lid = L.id()
                settings = self.layer_settings.get(lid, {})
                # Check explicit setting OR auto-detect pattern
                if settings.get('render_as_image', False) or self._is_pattern_layer(L):
                    any_vector_as_image = True
                    break
            
            if any_vector_as_image:
                print("🖼️ Pattern support active: creating vectors directory for image exports")
                vecdir = os.path.join(tmp, 'vectors')
                os.makedirs(vecdir, exist_ok=True)
            
            # Reverse layer order: process bottom-to-top so top layers appear on top in Google Earth
            selected_layers = list(reversed(selected_layers))
            print("📋 Layer order reversed for proper display (bottom-to-top)")
            
            # Setup progress bar
            self.dlg.progressBar.setValue(0)
            self.dlg.progressBar.setFormat("%p% - %v")
            
            def update_progress(pct, message=""):
                self.dlg.progressBar.setValue(pct)
                if message:
                    self.dlg.progressBar.setFormat(f"{message}")
                from qgis.PyQt.QtWidgets import QApplication
                QApplication.processEvents()
            
            # Process each selected layer
            for layer_idx, (i, L) in enumerate(selected_layers):
                if not should_continue:
                    break
                
                # Update layer progress
                layer_progress = int((layer_idx / max(len(selected_layers), 1)) * 100)
                update_progress(layer_progress, f"Processing layer {layer_idx + 1}/{len(selected_layers)}: {L.name()}")
                
                if isinstance(L, QgsRasterLayer):
                    props = {'folder_name': L.name()}
                    try:
                        add_raster_layer(doc, L, props, rasdir, max_dimension=4096, progress_callback=update_progress)
                    except Exception as e:
                        log(f"Failed to export raster layer {L.name()}: {e}", Qgis.Warning)
                        QMessageBox.warning(self.iface.mainWindow(), "Warning", f"Raster export failed: {e}")
                else:
                    # Get properties from UI
                    props = {}
                    
                    # Get fill mode from column 1
                    fill_mode_widget = tbl.cellWidget(i, 1)
                    fill_mode = 'Auto'
                    if fill_mode_widget:
                        fill_mode = fill_mode_widget.currentText()
                    props['fill_mode'] = fill_mode
                    
                    # Get Per-Layer Settings
                    # Auto-detect pattern layers for smart defaults
                    is_pattern = self._is_pattern_layer(L)
                    default_render_as_image = is_pattern  # Auto-enable for patterns
                    
                    settings = self.layer_settings.get(L.id(), {
                        'fill_transparency': 0,
                        'stroke_scale': 1.0,
                        'export_labels': False,
                        'render_as_image': default_render_as_image
                    })
                    
                    # Pass settings to props
                    fill_transparency = settings.get('fill_transparency', 0)
                    props['fill_alpha_scale'] = (100 - fill_transparency) / 100.0
                    props['stroke_scale'] = settings.get('stroke_scale', 1.0)
                    props['export_labels'] = settings.get('export_labels', False)
                    render_as_image = settings.get('render_as_image', default_render_as_image)
                    
                    if is_pattern and render_as_image:
                        print(f"🎨 Layer '{L.name()}' AUTO-DETECTED as pattern layer - will render as image")
                    print(f"🎨 Layer '{L.name()}' Settings: Alpha={props['fill_alpha_scale']:.2f}, Stroke={props['stroke_scale']}x, Labels={props['export_labels']}, Rasterize={render_as_image}")
                    
                    # Get folder name from column 5
                    folder_widget = tbl.cellWidget(i, 5)
                    if folder_widget:
                        fld = folder_widget.text().strip()
                        if fld:
                            props['folder_name'] = fld
                    
                    src = L.source()
                    flds = self.selected_fields.get(src, [])
                    if flds:
                        props['fields'] = flds
                    
                    # KML props start at column 7
                    for j, pn in enumerate(KML_PROPS, start=7):
                        widget = tbl.cellWidget(i, j)
                        if widget:
                            t = widget.currentText().strip()
                            if t: props[pn.lower()] = t
                    
                    # Check render as image (Pattern Support)
                    if render_as_image:
                        try:
                            # Pass fill_mode too, though mostly used for style gen, image render uses it for overriding renderer if needed
                            add_vector_layer_as_image(
                                doc, L, props, vecdir, 
                                max_dimension=4096, 
                                progress_callback=update_progress,
                                fill_mode=fill_mode
                            )
                        except Exception as e:
                            log(f"Failed to export vector as image {L.name()}: {e}", Qgis.Warning)
                            QMessageBox.warning(self.iface.mainWindow(), "Warning", f"Vector image export failed: {e}")
                    else:
                        # Use standard KML placemark method with symbology extraction
                        layer_styles = None
                        style_map_ids = None
                        icon_paths = {}
                        try:
                            layer_styles = extract_layer_styles(L)
                            if layer_styles:
                                print(f"🎨 Extracted symbology: {layer_styles.get('renderer_type', 'unknown')} renderer")
                                print(f"   Styles: {len(layer_styles.get('styles', {}))}")
                                
                                # Apply fill mode based on per-layer setting
                                fill_mode = props.get('fill_mode', 'Auto')
                                if layer_styles.get('geometry_type') == 'Polygon':
                                    for style_id, style_props in layer_styles.get('styles', {}).items():
                                        # Pass fill_mode to style generator for proper outline handling
                                        style_props['fill_mode'] = fill_mode
                                        
                                        # Get scaling values from props
                                        fill_alpha_scale = props.get('fill_alpha_scale', 1.0)
                                        stroke_scale = props.get('stroke_scale', 1.0)
                                        
                                        # Apply fill transparency scaling
                                        if 'fill_color' in style_props:
                                            # KML color is AABBGGRR
                                            abgr = style_props['fill_color']
                                            if len(abgr) == 8:
                                                alpha_hex = abgr[:2]
                                                try:
                                                    alpha = int(alpha_hex, 16)
                                                    new_alpha = int(alpha * fill_alpha_scale)
                                                    new_alpha = max(0, min(255, new_alpha))
                                                    new_abgr = f"{new_alpha:02x}{abgr[2:]}"
                                                    style_props['fill_color'] = new_abgr
                                                except ValueError:
                                                    pass
                                        
                                        # Apply stroke width scaling
                                        if 'stroke_width' in style_props:
                                            try:
                                                width = float(style_props['stroke_width'])
                                                style_props['stroke_width'] = width * stroke_scale
                                            except (ValueError, TypeError):
                                                pass
                                        
                                        if fill_mode == 'Hollow':
                                            style_props['fill'] = False
                                            print(f"🔲 Forced HOLLOW for: {style_id}")
                                        elif fill_mode == 'Fill':
                                            style_props['fill'] = True
                                            print(f"🟢 Forced FILL for: {style_id}")
                                        else:
                                            # Auto - keep detected value
                                            print(f"🔄 AUTO fill={style_props.get('fill', True)} for: {style_id}")
                                
                                # Apply stroke scaling to Line layers too
                                if layer_styles.get('geometry_type') == 'Line':
                                     for style_id, style_props in layer_styles.get('styles', {}).items():
                                        if 'width' in style_props:
                                            try:
                                                width = float(style_props['width'])
                                                style_props['width'] = width * stroke_scale
                                            except (ValueError, TypeError):
                                                pass
                                
                                # Render point icons if this is a point layer (for KMZ)
                                if layer_styles.get('geometry_type') == 'Point':
                                    icons_dir = os.path.join(tmp, 'icons')
                                    icon_paths = render_all_marker_styles(layer_styles, icons_dir, size=32)
                                    if icon_paths:
                                        print(f"🖼️ Rendered {len(icon_paths)} custom point icons")
                                
                                # Add styles to the document with icon paths
                                style_map_ids = add_styles_to_document(doc, layer_styles, icon_paths)
                        except Exception as e:
                            print(f"⚠️ Could not extract symbology: {e}")
                        
                        # Process the layer with symbology, progress, and selection option
                        new_total, should_continue, user_choice = add_vector_layer(
                            doc, L, props, total_vertices, MAX_VERTICES, self.iface.mainWindow(),
                            layer_styles, style_map_ids, update_progress, selected_only
                        )
                        total_vertices = new_total
                        
                        if user_choice == 'stop':
                            QMessageBox.information(
                                self.iface.mainWindow(),
                                "Export Stopped",
                                "Export was stopped. Please unselect some layers to reduce vertex count and try again."
                            )
                            return
            
            print(f"\n" + "🎉"*20)
            print(f"EXPORT COMPLETE")
            print(f"📊 Final total vertices: {total_vertices:,}")
            print(f"🎯 Vertex limit was: {MAX_VERTICES:,}")
            if total_vertices > MAX_VERTICES:
                print(f"⚠️ EXCEEDED LIMIT BY: {total_vertices - MAX_VERTICES:,}")
            print("🎉"*20)
            
            # Write KML file
            kf = os.path.join(tmp, 'doc.kml')
            ET.ElementTree(kml).write(kf, encoding='utf-8', xml_declaration=True)
            
            has_rasters = any(isinstance(L, QgsRasterLayer) for i, L in selected_layers)
            icons_dir = os.path.join(tmp, 'icons')
            has_icons = os.path.exists(icons_dir) and os.listdir(icons_dir)
            vectors_dir = os.path.join(tmp, 'vectors')
            has_vector_images = os.path.exists(vectors_dir) and os.listdir(vectors_dir)
            
            # Use KMZ if there are rasters, custom icons, OR vector images
            if has_rasters or has_icons or has_vector_images:
                # Ensure output is .kmz
                if out.lower().endswith('.kml'):
                    out = out[:-4] + '.kmz'
                    
                with zipfile.ZipFile(out, 'w', zipfile.ZIP_DEFLATED) as zf:
                    zf.write(kf, 'doc.kml')
                    # Include rasters
                    if has_rasters:
                        for fname in os.listdir(rasdir):
                            zf.write(os.path.join(rasdir, fname), f'rasters/{fname}')
                    # Include icons
                    if has_icons:
                        for fname in os.listdir(icons_dir):
                            zf.write(os.path.join(icons_dir, fname), f'icons/{fname}')
                        print(f"📦 Included {len(os.listdir(icons_dir))} icons in KMZ")
                    # Include vector images
                    if has_vector_images:
                        for fname in os.listdir(vectors_dir):
                            zf.write(os.path.join(vectors_dir, fname), f'vectors/{fname}')
                        print(f"📦 Included {len(os.listdir(vectors_dir))} vector images in KMZ")
            else:
                os.replace(kf, out)
            
            self.iface.messageBar().pushMessage(
                'Export 2KML',
                f'Exported to {out} with {total_vertices:,} vertices',
                level=Qgis.Info
            )
            
            self.dlg.close()
            
        except Exception as e:
            print(f"💥 ERROR: {str(e)}")
            import traceback
            print(f"🔍 Traceback: {traceback.format_exc()}")
            QMessageBox.critical(self.iface.mainWindow(), 'Error', str(e))
