# -*- coding: utf-8 -*-

from qgis.PyQt.QtWidgets import QDialog, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, QTextEdit
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QFont

class VertexLimitDialog(QDialog):
    """Dialog shown when vertex count exceeds thresholds."""
    
    def __init__(self, parent, current_vertices, threshold, current_layer_name, is_simplify_prompt=False):
        super().__init__(parent)
        self.choice = None
        self.is_simplify_prompt = is_simplify_prompt
        self.setupUi(current_vertices, threshold, current_layer_name)
        
    def setupUi(self, current_vertices, threshold, current_layer_name):
        if self.is_simplify_prompt:
            self.setWindowTitle("High Vertex Count - Simplify?")
        else:
            self.setWindowTitle("Vertex Count Warning")
        self.setModal(True)
        self.resize(520, 380)
        
        layout = QVBoxLayout(self)
        
        # Warning message
        if self.is_simplify_prompt:
            warning_label = QLabel("⚠️ High Vertex Count Detected")
        else:
            warning_label = QLabel("ℹ️ Google Earth Vertex Limit Info")
        warning_font = QFont()
        warning_font.setBold(True)
        warning_font.setPointSize(14)
        warning_label.setFont(warning_font)
        warning_label.setAlignment(Qt.AlignCenter)
        layout.addWidget(warning_label)
        
        # Details
        details = QTextEdit()
        details.setReadOnly(True)
        details.setMaximumHeight(140)
        
        if self.is_simplify_prompt:
            details.setPlainText(
                f"Current total vertices: {current_vertices:,}\n"
                f"Simplification threshold: {threshold:,}\n"
                f"Current layer: {current_layer_name}\n\n"
                f"The export contains many vertices which may:\n"
                f"• Make the file very large\n"
                f"• Slow down Google Earth\n"
                f"• Fail to import in some cases\n\n"
                f"Would you like to SIMPLIFY or KEEP all vertices?"
            )
        else:
            details.setPlainText(
                f"Current total vertices: {current_vertices:,}\n"
                f"Google Earth limit: {threshold:,}\n"
                f"Current layer: {current_layer_name}\n\n"
                f"Google Earth's 'Map features' mode supports up to {threshold:,} vertices.\n"
                f"You can use the experimental 'Data layer' mode for more.\n\n"
                f"Do you want to continue exporting?"
            )
        layout.addWidget(details)
        
        # Option buttons
        button_layout = QVBoxLayout()
        
        if self.is_simplify_prompt:
            # Simplify button
            simplify_btn = QPushButton("🔄 Simplify Geometry")
            simplify_btn.setStyleSheet("QPushButton { text-align: left; padding: 8px; background-color: #4CAF50; color: white; }")
            simplify_btn.clicked.connect(lambda: self.set_choice("simplify"))
            button_layout.addWidget(simplify_btn)
            
            simplify_desc = QLabel("   Reduce vertex count by simplifying complex geometries. Recommended for large datasets.")
            simplify_desc.setWordWrap(True)
            simplify_desc.setStyleSheet("color: gray; margin-left: 20px; margin-bottom: 10px;")
            button_layout.addWidget(simplify_desc)
            
            # Keep all button
            keep_btn = QPushButton("✅ Keep All Vertices")
            keep_btn.setStyleSheet("QPushButton { text-align: left; padding: 8px; }")
            keep_btn.clicked.connect(lambda: self.set_choice("keep"))
            button_layout.addWidget(keep_btn)
            
            keep_desc = QLabel("   Export all vertices without simplification. File may be large.")
            keep_desc.setWordWrap(True)
            keep_desc.setStyleSheet("color: gray; margin-left: 20px; margin-bottom: 10px;")
            button_layout.addWidget(keep_desc)
            
        else:
            # Continue button
            continue_btn = QPushButton("✅ Continue Export")
            continue_btn.setStyleSheet("QPushButton { text-align: left; padding: 8px; background-color: #2196F3; color: white; }")
            continue_btn.clicked.connect(lambda: self.set_choice("keep"))
            button_layout.addWidget(continue_btn)
            
            continue_desc = QLabel("   Continue exporting. Use Google Earth's 'Data layer' mode if import fails.")
            continue_desc.setWordWrap(True)
            continue_desc.setStyleSheet("color: gray; margin-left: 20px; margin-bottom: 10px;")
            button_layout.addWidget(continue_desc)
        
        # Stop button (always available)
        stop_btn = QPushButton("🛑 Stop Export")
        stop_btn.setStyleSheet("QPushButton { text-align: left; padding: 8px; }")
        stop_btn.clicked.connect(lambda: self.set_choice("stop"))
        button_layout.addWidget(stop_btn)
        
        stop_desc = QLabel("   Cancel the export process.")
        stop_desc.setWordWrap(True)
        stop_desc.setStyleSheet("color: gray; margin-left: 20px; margin-bottom: 10px;")
        button_layout.addWidget(stop_desc)
        
        layout.addLayout(button_layout)
        
    def set_choice(self, choice):
        self.choice = choice
        self.accept()
