# -*- coding: utf-8 -*-
"""
kml_style_generator.py
Generate KML Style elements from extracted QGIS symbology.
"""

import xml.etree.ElementTree as ET


def create_icon_style(color: str = 'ffffffff', scale: float = 1.0, 
                      icon_href: str = None) -> ET.Element:
    """
    Create a KML IconStyle element for point features.
    
    Args:
        color: ABGR hex color string
        scale: Icon scale factor
        icon_href: URL or path to custom icon (None for default pushpin)
        
    Returns:
        IconStyle ET.Element
    """
    icon_style = ET.Element('IconStyle')
    ET.SubElement(icon_style, 'color').text = color
    ET.SubElement(icon_style, 'scale').text = str(scale)
    
    icon = ET.SubElement(icon_style, 'Icon')
    if icon_href:
        ET.SubElement(icon, 'href').text = icon_href
    else:
        # Default Google Earth pushpin
        ET.SubElement(icon, 'href').text = 'http://maps.google.com/mapfiles/kml/pushpin/ylw-pushpin.png'
    
    return icon_style


def create_line_style(color: str = 'ff0000ff', width: float = 2.0) -> ET.Element:
    """
    Create a KML LineStyle element for line features.
    
    Args:
        color: ABGR hex color string
        width: Line width in pixels
        
    Returns:
        LineStyle ET.Element
    """
    line_style = ET.Element('LineStyle')
    ET.SubElement(line_style, 'color').text = color
    # KML width is in pixels, QGIS uses mm. Convert roughly (1mm ≈ 4px at screen resolution)
    kml_width = max(1.0, width * 4)
    ET.SubElement(line_style, 'width').text = str(kml_width)
    
    return line_style


def create_poly_style(fill_color: str = '7f0000ff', fill: bool = True, 
                      outline: bool = True) -> ET.Element:
    """
    Create a KML PolyStyle element for polygon features.
    
    Args:
        fill_color: ABGR hex color string for fill
        fill: Whether to fill the polygon
        outline: Whether to draw outline
        
    Returns:
        PolyStyle ET.Element
    """
    poly_style = ET.Element('PolyStyle')
    
    # When fill is disabled, use fully transparent color (00 alpha in ABGR)
    if fill:
        ET.SubElement(poly_style, 'color').text = fill_color
    else:
        # Use 00 alpha (fully transparent) to ensure no fill is visible
        ET.SubElement(poly_style, 'color').text = '00000000'
    
    ET.SubElement(poly_style, 'fill').text = '1' if fill else '0'
    ET.SubElement(poly_style, 'outline').text = '1' if outline else '0'
    
    return poly_style


def create_label_style(color: str = 'ffffffff', scale: float = 1.0) -> ET.Element:
    """
    Create a KML LabelStyle element.
    
    Args:
        color: ABGR hex color string
        scale: Label scale factor
        
    Returns:
        LabelStyle ET.Element
    """
    label_style = ET.Element('LabelStyle')
    ET.SubElement(label_style, 'color').text = color
    ET.SubElement(label_style, 'scale').text = str(scale)
    
    return label_style


def create_style(style_id: str, style_props: dict, icon_href: str = None) -> ET.Element:
    """
    Create a complete KML Style element based on geometry type and properties.
    
    Args:
        style_id: Unique ID for this style
        style_props: Dictionary of style properties from symbology_extractor
        icon_href: Optional custom icon path for points
        
    Returns:
        Style ET.Element
    """
    style = ET.Element('Style', id=style_id)
    
    geom_type = style_props.get('type', 'Polygon')
    
    if geom_type == 'Point':
        # Point style - IconStyle
        color = style_props.get('color', 'ffffffff')
        size = style_props.get('size', 6)
        # Scale based on size (6mm is default, scale 1.0)
        scale = size / 6.0
        
        icon_style = create_icon_style(color, scale, icon_href)
        style.append(icon_style)
        
        # Add label style
        label_style = create_label_style(color, 0.8)
        style.append(label_style)
        
    elif geom_type == 'Line':
        # Line style - LineStyle
        color = style_props.get('color', 'ff0000ff')
        width = style_props.get('width', 0.5)
        
        line_style = create_line_style(color, width)
        style.append(line_style)
        
        # Add LabelStyle if requested
        if style_props.get('export_labels'):
            label_style = create_label_style(color, 0.8)
            style.append(label_style)
        
    elif geom_type == 'Polygon':
        # Polygon style - PolyStyle + LineStyle for outline
        fill_color = style_props.get('fill_color', style_props.get('color', '7f0000ff'))
        stroke_color = style_props.get('stroke_color', fill_color)
        stroke_width = style_props.get('stroke_width', 0.26)
        fill = style_props.get('fill', True)
        fill_mode = style_props.get('fill_mode', 'Auto')
        
        # Add PolyStyle for fill
        poly_style = create_poly_style(fill_color, fill, outline=True)
        style.append(poly_style)
        
        # Add LineStyle for outline
        # Fill mode: use thin black outline (forced fill)
        # Auto/Hollow mode: use the layer's actual stroke color/width from symbology
        if fill_mode == 'Fill':
            # Forced fill - use thin black outline (1.0 width)
            line_style = create_line_style('ff000000', 1.0)  # Black, 1.0 width
        else:
            # Auto or Hollow - use the detected stroke color from symbology
            line_style = create_line_style(stroke_color, stroke_width)
        style.append(line_style)
        
        # Add LabelStyle if requested
        if style_props.get('export_labels'):
            # Use stroke color for label text usually, or black/white
            label_style = create_label_style('ffffffff', 0.8) # Default white
            style.append(label_style)
    
    return style


def create_style_map(style_id: str, normal_style_id: str, 
                     highlight_style_id: str = None) -> ET.Element:
    """
    Create a KML StyleMap for normal/highlight state transitions.
    
    Args:
        style_id: ID for this StyleMap
        normal_style_id: ID of normal state style
        highlight_style_id: ID of highlight state style (auto-generated if None)
        
    Returns:
        StyleMap ET.Element
    """
    style_map = ET.Element('StyleMap', id=style_id)
    
    # Normal state
    pair_normal = ET.SubElement(style_map, 'Pair')
    ET.SubElement(pair_normal, 'key').text = 'normal'
    ET.SubElement(pair_normal, 'styleUrl').text = f'#{normal_style_id}'
    
    # Highlight state
    pair_highlight = ET.SubElement(style_map, 'Pair')
    ET.SubElement(pair_highlight, 'key').text = 'highlight'
    ET.SubElement(pair_highlight, 'styleUrl').text = f'#{highlight_style_id or normal_style_id}'
    
    return style_map


def create_highlight_style(base_style: ET.Element, style_id: str) -> ET.Element:
    """
    Create a highlight version of a style (slightly larger/brighter).
    
    Args:
        base_style: The base Style element to create highlight from
        style_id: ID for the highlight style
        
    Returns:
        Style ET.Element for highlight state
    """
    import copy
    highlight = copy.deepcopy(base_style)
    highlight.set('id', style_id)
    
    # Increase scale for icons
    icon_style = highlight.find('IconStyle')
    if icon_style is not None:
        scale_elem = icon_style.find('scale')
        if scale_elem is not None:
            try:
                current_scale = float(scale_elem.text)
                scale_elem.text = str(current_scale * 1.2)
            except ValueError:
                pass
    
    # Increase width for lines
    line_style = highlight.find('LineStyle')
    if line_style is not None:
        width_elem = line_style.find('width')
        if width_elem is not None:
            try:
                current_width = float(width_elem.text)
                width_elem.text = str(current_width * 1.5)
            except ValueError:
                pass
    
    return highlight


def add_styles_to_document(doc: ET.Element, layer_styles: dict, 
                           icon_paths: dict = None) -> dict:
    """
    Add all styles from layer_styles to a KML Document element.
    
    Args:
        doc: KML Document ET.Element
        layer_styles: Dictionary from symbology_extractor.extract_layer_styles()
        icon_paths: Dictionary mapping style_ids to relative icon paths (e.g., 'icons/style.png')
        
    Returns:
        Dictionary mapping original style_ids to the StyleMap IDs to use
    """
    if not layer_styles or 'styles' not in layer_styles:
        return {}
    
    if icon_paths is None:
        icon_paths = {}
    
    style_map_ids = {}
    
    for style_id, style_props in layer_styles['styles'].items():
        # Create normal style
        normal_id = f'{style_id}_normal'
        
        # Get custom icon href if available
        icon_href = icon_paths.get(style_id)
        
        normal_style = create_style(normal_id, style_props, icon_href)
        doc.append(normal_style)
        
        # Create highlight style
        highlight_id = f'{style_id}_highlight'
        highlight_style = create_highlight_style(normal_style, highlight_id)
        doc.append(highlight_style)
        
        # Create StyleMap
        map_id = f'{style_id}_map'
        style_map = create_style_map(map_id, normal_id, highlight_id)
        doc.append(style_map)
        
        # Map original ID to the StyleMap ID
        style_map_ids[style_id] = map_id
    
    return style_map_ids


def get_style_url(style_id: str, style_map_ids: dict) -> str:
    """
    Get the styleUrl to use for a placemark.
    
    Args:
        style_id: Original style ID from symbology extractor
        style_map_ids: Dictionary from add_styles_to_document()
        
    Returns:
        Style URL string (e.g., '#layer_style_map')
    """
    if style_map_ids and style_id in style_map_ids:
        return f'#{style_map_ids[style_id]}'
    elif style_id:
        return f'#{style_id}_normal'
    else:
        return None
