# -*- coding: utf-8 -*-
"""
field_selector_dialog.py

Contains the FieldSelectorDialog class, which presents a checked‐list of fields
and allows Shift/Ctrl multi‐selection, plus toggling check state by clicking anywhere.
"""

from qgis.PyQt.QtWidgets import (
    QDialog,
    QDialogButtonBox,
    QListWidget,
    QListWidgetItem,
    QVBoxLayout,
    QLabel
)
from qgis.PyQt.QtCore import Qt


class FieldSelectorDialog(QDialog):
    def __init__(self, layer_fields, selected):
        """
        :param layer_fields: list of field names (strings)
        :param selected: list of field names that should start checked
        """
        super().__init__()
        self.setWindowTitle("Select Fields")
        self.resize(300, 400)

        layout = QVBoxLayout()
        self.list_widget = QListWidget()
        # Enable Shift/Ctrl multi‐selection
        self.list_widget.setSelectionMode(QListWidget.ExtendedSelection)

        for fld in layer_fields:
            item = QListWidgetItem(fld)
            item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
            item.setCheckState(Qt.Checked if fld in selected else Qt.Unchecked)
            self.list_widget.addItem(item)

        # Clicking anywhere on an item toggles its check state
        self.list_widget.itemClicked.connect(self.toggle_item)

        layout.addWidget(QLabel("Choose attributes to export:"))
        layout.addWidget(self.list_widget)

        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)

        self.setLayout(layout)

    def toggle_item(self, item: QListWidgetItem):
        """
        Invert the check state when the item is clicked (anywhere on its row).
        """
        new_state = Qt.Unchecked if item.checkState() == Qt.Checked else Qt.Checked
        item.setCheckState(new_state)

    def get_selected(self):
        """
        Return a list of all field names whose checkState is Qt.Checked.
        """
        return [
            self.list_widget.item(i).text()
            for i in range(self.list_widget.count())
            if self.list_widget.item(i).checkState() == Qt.Checked
        ]
    def get_unselected(self):
        """ Return a list of all field names whose checkState is Qt.Unchecked."""
        return [
            self.list_widget.item(i).text()
            for i in range(self.list_widget.count())
            if self.list_widget.item(i).checkState() == Qt.Unchecked
        ]