# -*- coding: utf-8 -*-

"""
export2kml.py
QGIS plugin: Export 2KML
"""

import os
import tempfile
import zipfile
import xml.etree.ElementTree as ET

from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import (
    QAction, QFileDialog, QTableWidgetItem, QCheckBox, QComboBox,
    QMessageBox, QToolBar, QLineEdit, QPushButton, QLabel, QDialog
)
from qgis.PyQt.QtGui import QIcon, QImage, QPainter
from qgis.PyQt.QtCore import Qt, QSize

from qgis.core import (
    QgsProject, QgsVectorLayer, QgsRasterLayer, QgsMapSettings,
    QgsMapRendererParallelJob, QgsGeometry, QgsCoordinateReferenceSystem,
    QgsCoordinateTransform, QgsCoordinateTransformContext, QgsFeatureRequest,
    Qgis, QgsWkbTypes, QgsRectangle, QgsPointXY
)

from osgeo import ogr, gdal
from .vertex_limit_dialog import VertexLimitDialog
from .field_selector_dialog import FieldSelectorDialog

gdal.UseExceptions()
ogr.UseExceptions()

KML_PROPS = [
    'Name','description','timestamp','begin','end',
    'altitudeMode','tessellate','extrude','visibility',
    'drawOrder','icon'
]

MAX_VERTICES = 250000
MAX_FEATURE_VERTICES = 50000

def get_vertex_count(geom: QgsGeometry) -> int:
    """Count vertices in geometry."""
    if not geom or geom.isEmpty():
        return 0
    
    vertex_count = 0
    geom_type = geom.type()
    
    if geom.isMultipart():
        if geom_type == 0:  # MultiPoint
            vertex_count = len(geom.asMultiPoint())
        elif geom_type == 1:  # MultiLineString
            for line in geom.asMultiPolyline():
                vertex_count += len(line)
        elif geom_type == 2:  # MultiPolygon
            for polygon in geom.asMultiPolygon():
                for ring in polygon:
                    vertex_count += len(ring)
    else:
        if geom_type == 0:  # Point
            vertex_count = 1
        elif geom_type == 1:  # LineString
            vertex_count = len(geom.asPolyline())
        elif geom_type == 2:  # Polygon
            polygon = geom.asPolygon()
            for ring in polygon:
                vertex_count += len(ring)
    
    return vertex_count

def validate_coordinates_loosely(geom: QgsGeometry, feature_id) -> tuple:
    """Loosely validate coordinates and return (is_valid, reason, bbox_info)."""
    if not geom or geom.isEmpty():
        return False, "Empty geometry", None
    
    try:
        bbox = geom.boundingBox()
        bbox_info = f"[{bbox.xMinimum():.6f}, {bbox.yMinimum():.6f}, {bbox.xMaximum():.6f}, {bbox.yMaximum():.6f}]"
        
        # Check for obviously invalid coordinates (extremely large values)
        if (abs(bbox.xMinimum()) > 10000000 or abs(bbox.xMaximum()) > 10000000 or
            abs(bbox.yMinimum()) > 10000000 or abs(bbox.yMaximum()) > 10000000):
            return False, "Coordinates extremely large (>10M)", bbox_info
        
        # Check for all coordinates being exactly zero
        if (abs(bbox.xMinimum()) < 0.0000001 and abs(bbox.xMaximum()) < 0.0000001 and
            abs(bbox.yMinimum()) < 0.0000001 and abs(bbox.yMaximum()) < 0.0000001):
            return False, "All coordinates are zero", bbox_info
        
        # More lenient - accept coordinates that might be outside normal WGS84 but not crazy
        return True, "Valid", bbox_info
        
    except Exception as e:
        return False, f"Error getting bbox: {e}", None

def simplify_complex_geometry(geom: QgsGeometry, feature_id, max_vertices=MAX_FEATURE_VERTICES):
    """Simplify overly complex geometries."""
    if not geom or geom.isEmpty():
        return geom
    
    vertex_count = get_vertex_count(geom)
    if vertex_count <= max_vertices:
        return geom
    
    print(f"Feature {feature_id}: Simplifying complex geometry ({vertex_count:,} vertices)")
    
    try:
        bbox = geom.boundingBox()
        tolerance = max(bbox.width(), bbox.height()) * 0.001
        
        for i in range(5):
            simplified = geom.simplify(tolerance)
            if simplified and not simplified.isEmpty():
                simplified_vertices = get_vertex_count(simplified)
                if simplified_vertices <= max_vertices:
                    print(f"Feature {feature_id}: Simplified to {simplified_vertices:,} vertices")
                    return simplified
            tolerance *= 2
        
        print(f"Feature {feature_id}: Could not simplify sufficiently")
        return geom
        
    except Exception as e:
        print(f"Feature {feature_id}: Error simplifying geometry: {e}")
        return geom

def geometry_to_kml_element(geom: QgsGeometry, feature_id):
    """Convert QGIS geometry to KML geometry element."""
    if not geom or geom.isEmpty():
        return None
    
    try:
        # Try to fix invalid geometries but don't be too strict
        if hasattr(geom, 'isGeosValid') and not geom.isGeosValid():
            print(f"Feature {feature_id}: Attempting to fix invalid geometry...")
            try:
                fixed_geom = geom.makeValid()
                if fixed_geom and not fixed_geom.isEmpty():
                    geom = fixed_geom
                    print(f"Feature {feature_id}: Geometry fixed")
            except:
                print(f"Feature {feature_id}: Could not fix geometry, proceeding anyway")
        
        geom_type_name = QgsWkbTypes.displayString(geom.wkbType())
        vertex_count = get_vertex_count(geom)
        
        # Simplify if too complex
        if vertex_count > MAX_FEATURE_VERTICES:
            geom = simplify_complex_geometry(geom, feature_id)
            if not geom or geom.isEmpty():
                return None
        
        wkt = geom.asWkt()
        
        # Limit WKT size to prevent memory issues
        if len(wkt) > 2000000:
            print(f"Feature {feature_id}: WKT too long, simplifying further")
            geom = simplify_complex_geometry(geom, feature_id, MAX_FEATURE_VERTICES // 4)
            if not geom or geom.isEmpty():
                return None
            wkt = geom.asWkt()
        
        ogrg = ogr.CreateGeometryFromWkt(wkt)
        if ogrg is None:
            print(f"Feature {feature_id}: Failed to create OGR geometry")
            return None
        
        kml_string = ogrg.ExportToKML()
        if not kml_string:
            print(f"Feature {feature_id}: Failed to export to KML")
            return None
        
        # Parse KML
        try:
            wrapped_kml = f'<root>{kml_string}</root>'
            root = ET.fromstring(wrapped_kml)
            
            geometry_elements = []
            for elem in root:
                if elem.tag in ['Point', 'LineString', 'Polygon', 'MultiGeometry']:
                    geometry_elements.append(elem)
            
            return geometry_elements if geometry_elements else None
            
        except ET.ParseError as e:
            print(f"Feature {feature_id}: XML parse error: {e}")
            return None
            
    except Exception as e:
        print(f"Feature {feature_id}: Geometry conversion error: {e}")
        return None

def make_kml_root():
    kml = ET.Element('kml', xmlns='http://www.opengis.net/kml/2.2')
    return kml, ET.SubElement(kml, 'Document')

def add_vector_layer(doc, layer: QgsVectorLayer, props, total_vertices_so_far, max_vertices, iface_parent):
    """Process vector layer with detailed debugging and adjusted validation."""
    
    print(f"\n" + "="*60)
    print(f"PROCESSING LAYER: {layer.name()}")
    print(f"Starting total vertices: {total_vertices_so_far:,}")
    print(f"Feature count: {layer.featureCount()}")
    
    # Get layer CRS information
    layer_crs = layer.crs()
    layer_crs_authid = layer_crs.authid()
    print(f"Layer CRS: {layer_crs_authid} - {layer_crs.description()}")
    
    if not layer.isValid():
        print("❌ Layer is not valid - skipping")
        return total_vertices_so_far, True, None
    
    if layer.featureCount() == 0:
        print("❌ Layer has no features - skipping")
        return total_vertices_so_far, True, None
    
    # Create folder
    folder = ET.SubElement(doc, 'Folder')
    ET.SubElement(folder, 'name').text = props.get('folder_name') or layer.name()
    
    # CRS handling
    wgs84_crs = QgsCoordinateReferenceSystem('EPSG:4326')
    needs_transform = False
    xform = None
    
    if layer_crs_authid == 'EPSG:4326':
        print("✅ Layer is already in WGS84 - NO TRANSFORMATION")
        needs_transform = False
    elif layer_crs.isValid():
        print(f"⚠️ Layer needs transformation: {layer_crs_authid} -> EPSG:4326")
        needs_transform = True
        try:
            xform = QgsCoordinateTransform(layer_crs, wgs84_crs, QgsProject.instance().transformContext())
            print("✅ Coordinate transformation created")
        except Exception as e:
            print(f"❌ Failed to create transformation: {e}")
            needs_transform = False
    else:
        print("⚠️ Invalid CRS - proceeding without transformation")
        needs_transform = False
    
    # Initialize counters and tracking
    fields_avail = [f.name() for f in layer.fields()]
    current_total = total_vertices_so_far
    processed_count = 0
    skipped_count = 0
    user_choice = None
    dialog_shown = False
    
    # Detailed skip tracking
    skip_reasons = {
        'empty_geometry': 0,
        'transform_error': 0,
        'invalid_coords': 0,
        'geometry_conversion_failed': 0,
        'user_skip': 0
    }
    
    geometry_stats = {'Point': 0, 'LineString': 0, 'Polygon': 0, 'MultiGeometry': 0, 'Other': 0}
    
    print(f"🚀 Starting to process {layer.featureCount()} features...")
    
    # Process each feature
    feature_count = 0
    for feat in layer.getFeatures(QgsFeatureRequest()):
        feature_count += 1
        
        try:
            # Progress reporting for large datasets
            if feature_count % 1000 == 0:
                print(f"📊 Progress: {feature_count}/{layer.featureCount()} features, "
                      f"processed: {processed_count}, skipped: {skipped_count}, "
                      f"total vertices: {current_total:,}")
            
            geom = feat.geometry()
            if not geom or geom.isEmpty():
                skip_reasons['empty_geometry'] += 1
                skipped_count += 1
                continue
            
            original_geom_type = QgsWkbTypes.displayString(geom.wkbType())
            
            # Show details for first few features
            if processed_count + skipped_count < 5:
                print(f"🔍 Feature {feat.id()}: {original_geom_type}")
            
            # Transform if needed
            if needs_transform and xform:
                try:
                    geom_copy = QgsGeometry(geom)
                    geom_copy.transform(xform)
                    geom = geom_copy
                    
                    if processed_count + skipped_count < 3:
                        print(f"🔄 Feature {feat.id()}: Transformed successfully")
                        
                except Exception as e:
                    if processed_count + skipped_count < 10:
                        print(f"❌ Feature {feat.id()}: Transform error: {e}")
                    skip_reasons['transform_error'] += 1
                    skipped_count += 1
                    continue
            
            # Validate coordinates (loosely)
            is_valid, reason, bbox_info = validate_coordinates_loosely(geom, feat.id())
            if not is_valid:
                if processed_count + skipped_count < 10:
                    print(f"❌ Feature {feat.id()}: {reason}, bbox: {bbox_info}")
                skip_reasons['invalid_coords'] += 1
                skipped_count += 1
                continue
            
            # Count vertices BEFORE showing dialog
            vc = get_vertex_count(geom)
            potential_total = current_total + vc
            
            if processed_count + skipped_count < 5:
                print(f"📏 Feature {feat.id()}: {vc:,} vertices, total would be: {potential_total:,}")
            
            # Check vertex limit and show dialog
            if potential_total > max_vertices and not dialog_shown and user_choice != 'keep':
                print(f"🚨 VERTEX LIMIT EXCEEDED!")
                print(f"   Current total: {current_total:,}")
                print(f"   This feature: {vc:,}")
                print(f"   Would be: {potential_total:,}")
                print(f"   Limit: {max_vertices:,}")
                
                # Show vertex limit dialog
                dlg = VertexLimitDialog(iface_parent, potential_total, max_vertices, layer.name())
                result = dlg.exec_()
                dialog_shown = True
                
                if result == QDialog.Accepted and dlg.choice:
                    user_choice = dlg.choice
                    print(f"👤 User chose: {user_choice}")
                    
                    if user_choice == 'stop':
                        print("🛑 User chose to stop export")
                        if processed_count == 0:
                            doc.remove(folder)
                        break
                    elif user_choice == 'skip':
                        print("⏭️ User chose to skip remaining features")
                        if processed_count == 0:
                            doc.remove(folder)
                        break
                    elif user_choice == 'keep':
                        print("✅ User chose to keep all features")
                else:
                    print("❌ Dialog cancelled - stopping")
                    if processed_count == 0:
                        doc.remove(folder)
                    break
            
            # Skip remaining features if user chose skip
            if user_choice == 'skip':
                skip_reasons['user_skip'] += 1
                skipped_count += 1
                continue
            
            # Process this feature
            current_total += vc
            processed_count += 1
            
            # Create placemark
            pm = ET.SubElement(folder, 'Placemark')
            
            # Add name
            if props.get('name') and props['name'] in fields_avail:
                name_value = feat[props['name']]
                ET.SubElement(pm, 'name').text = str(name_value) if name_value is not None else f"Feature {feat.id()}"
            else:
                ET.SubElement(pm, 'name').text = f"Feature {feat.id()}"
            
            # Add description
            if props.get('description') and props['description'] in fields_avail:
                desc_value = feat[props['description']]
                ET.SubElement(pm, 'description').text = str(desc_value) if desc_value is not None else ""
            
            # Add extended data
            if props.get('fields'):
                ext = ET.SubElement(pm, 'ExtendedData')
                for fld in props['fields']:
                    if fld in fields_avail:
                        dt = ET.SubElement(ext, 'Data', name=fld)
                        field_value = feat[fld]
                        ET.SubElement(dt, 'value').text = str(field_value) if field_value is not None else ""
            
            # Convert geometry to KML
            geometry_elements = geometry_to_kml_element(geom, feat.id())
            if geometry_elements:
                for geom_elem in geometry_elements:
                    pm.append(geom_elem)
                    geom_tag = geom_elem.tag
                    if geom_tag in geometry_stats:
                        geometry_stats[geom_tag] += 1
                    else:
                        geometry_stats['Other'] += 1
            else:
                if processed_count < 10:
                    print(f"❌ Feature {feat.id()}: Failed to convert geometry")
                skip_reasons['geometry_conversion_failed'] += 1
                # Don't increment processed_count for failed conversions
                processed_count -= 1
                current_total -= vc
                skipped_count += 1
                continue
                
        except Exception as e:
            if skipped_count < 10:
                print(f"❌ Feature {feat.id()}: Unexpected error: {e}")
            skipped_count += 1
            continue
    
    # Final summary
    print(f"\n" + "="*60)
    print(f"LAYER {layer.name()} SUMMARY:")
    print(f"📊 Total features in layer: {layer.featureCount()}")
    print(f"✅ Successfully processed: {processed_count:,}")
    print(f"❌ Skipped: {skipped_count:,}")
    print(f"📏 Total vertices added: {current_total - total_vertices_so_far:,}")
    print(f"📏 Running total vertices: {current_total:,}")
    print(f"🎯 Vertex limit: {max_vertices:,}")
    print(f"🔄 CRS transformation: {'Yes' if needs_transform else 'No'}")
    
    if skipped_count > 0:
        print(f"\nSkip reasons breakdown:")
        for reason, count in skip_reasons.items():
            if count > 0:
                print(f"  - {reason}: {count:,}")
    
    if processed_count > 0:
        print(f"\nGeometry types processed:")
        for geom_type, count in geometry_stats.items():
            if count > 0:
                print(f"  - {geom_type}: {count:,}")
    
    print("="*60)
    
    return current_total, True, user_choice

def add_raster_layer(doc, path, props, rasters_dir):
    # Keep existing raster logic
    os.makedirs(rasters_dir, exist_ok=True)
    base = os.path.splitext(os.path.basename(path))[0]
    tmp = os.path.join(rasters_dir, f'tmp_{base}_4326.vrt')
    
    gdal.Warp(
        tmp, path,
        format='VRT',
        dstSRS='EPSG:4326',
        options=['SRC_METHOD=NO_GEOTRANSFORM']
    )
    
    rlayer = QgsRasterLayer(tmp, base)
    if not rlayer.isValid():
        raise RuntimeError(f'Cannot load {path}')
    
    ms = QgsMapSettings()
    ms.setLayers([rlayer])
    ms.setExtent(rlayer.extent())
    ms.setOutputSize(QSize(rlayer.width(), rlayer.height()))
    job = QgsMapRendererParallelJob(ms)
    job.start(); job.waitForFinished()
    img = job.renderedImage()
    
    out_png = os.path.join(rasters_dir, f'{base}.png')
    rgb = QImage(img.size(), QImage.Format_RGB32)
    rgb.fill(Qt.white)
    p = QPainter(rgb); p.drawImage(0, 0, img); p.end()
    rgb.save(out_png, 'PNG')
    
    ds = gdal.Open(tmp)
    gt = ds.GetGeoTransform()
    ds = None
    
    go = ET.SubElement(doc, 'GroundOverlay')
    ET.SubElement(go, 'visibility').text = '1'
    ET.SubElement(go, 'name').text = props.get('folder_name') or props.get('name') or base
    ico = ET.SubElement(go, 'Icon')
    ET.SubElement(ico, 'href').text = f'rasters/{base}.png'
    llb = ET.SubElement(go, 'LatLonBox')
    ET.SubElement(llb, 'north').text = str(gt[3])
    ET.SubElement(llb, 'south').text = str(gt[3] + gt[5] * rlayer.height())
    ET.SubElement(llb, 'east').text = str(gt[0] + gt[1] * rlayer.width())
    ET.SubElement(llb, 'west').text = str(gt[0])
    
    return out_png

class Export2KML:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.added_file_layers = []
        self.selected_fields = {}
        self.menu = '&MAS Vector Processing'
        self.actions = []

    def initGui(self):
        ui = os.path.join(self.plugin_dir, 'form.ui')
        self.dlg = uic.loadUi(ui)
        
        tb = self.iface.mainWindow().findChild(QToolBar, 'MASVectorProcessingToolbar')
        if not tb:
            tb = self.iface.addToolBar('MAS Vector Processing')
            tb.setObjectName('MASVectorProcessingToolbar')
        
        act = QAction(QIcon(os.path.join(self.plugin_dir, 'export2kml.png')),
                     'Export 2KML…', self.iface.mainWindow())
        act.triggered.connect(self.show_dialog)
        self.iface.addPluginToVectorMenu(self.menu, act)
        tb.addAction(act)
        self.actions.append(act)
        
        self.dlg.btnBrowseFiles.clicked.connect(self.add_layer)
        self.dlg.btnSelectOutput.clicked.connect(self.select_output)
        self.dlg.btnRun.clicked.connect(self.run_export)
        self.dlg.btnCancel.clicked.connect(self.dlg.close)
        self.dlg.tableLayers.cellClicked.connect(self._toggle)

    def unload(self):
        for a in self.actions:
            self.iface.removePluginMenu(self.menu, a)
            self.iface.removeToolBarIcon(a)
        self.actions.clear()

    def show_dialog(self):
        self._populate()
        self._update_format()
        self.dlg.setParent(self.iface.mainWindow(), Qt.Tool)
        self.dlg.setWindowFlags(self.dlg.windowFlags() | Qt.Tool)
        self.dlg.show()

    def _populate(self):
        layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
        tbl = self.dlg.tableLayers
        
        cols = 4 + 2 + len(KML_PROPS)
        tbl.setColumnCount(cols)
        tbl.setHorizontalHeaderLabels(['Export?','Layer Name','Type','CRS','Folder','Fields'] + KML_PROPS)
        tbl.setRowCount(len(layers))
        
        for i, L in enumerate(layers):
            cb = QCheckBox(); cb.setChecked(True); cb.stateChanged.connect(self._update_format)
            tbl.setCellWidget(i, 0, cb)
            
            nm = QTableWidgetItem(L.name()); nm.setFlags(nm.flags() & ~Qt.ItemIsEditable)
            tbl.setItem(i, 1, nm)
            
            tp = 'Raster' if isinstance(L, QgsRasterLayer) else 'Vector'
            it = QTableWidgetItem(tp); it.setFlags(it.flags() & ~Qt.ItemIsEditable)
            tbl.setItem(i, 2, it)
            
            if hasattr(L, 'crs') and L.crs().isValid():
                crs_text = L.crs().authid()
                if crs_text == 'EPSG:4326':
                    crs_item = QTableWidgetItem(f"✓ {crs_text}")
                    crs_item.setBackground(Qt.green)
                else:
                    crs_item = QTableWidgetItem(f"⚠ {crs_text}")
                    crs_item.setBackground(Qt.yellow)
            else:
                crs_item = QTableWidgetItem("Unknown")
                crs_item.setBackground(Qt.red)
            
            crs_item.setFlags(crs_item.flags() & ~Qt.ItemIsEditable)
            tbl.setItem(i, 3, crs_item)
            
            fld = QLineEdit(L.name()); tbl.setCellWidget(i, 4, fld)
            
            if isinstance(L, QgsVectorLayer):
                btn = QPushButton("Select…")
                btn.clicked.connect(lambda _, r=i, LL=L: self._choose_fields(r, LL))
            else:
                btn = QLabel("N/A")
            tbl.setCellWidget(i, 5, btn)
            
            for j in range(len(KML_PROPS)):
                c = QComboBox(); c.addItem('')
                if isinstance(L, QgsVectorLayer):
                    c.addItems([f.name() for f in L.fields()])
                c.setEditable(True)
                tbl.setCellWidget(i, 6 + j, c)

    def _toggle(self, r, c):
        if c == 0:
            w = self.dlg.tableLayers.cellWidget(r, 0)
            w.setChecked(not w.isChecked())

    def _choose_fields(self, row, lyr):
        src = lyr.source()
        ex = self.selected_fields.get(src, [])
        dlg = FieldSelectorDialog([f.name() for f in lyr.fields()], ex)
        if dlg.exec_():
            ch = dlg.get_selected(); self.selected_fields[src] = ch
            b = self.dlg.tableLayers.cellWidget(row, 5)
            b.setText(f"{len(ch)} selected" if ch else "Select…")

    def _update_format(self):
        tbl = self.dlg.tableLayers; anyr = False
        layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
        for i, L in enumerate(layers):
            if isinstance(L, QgsRasterLayer) and tbl.cellWidget(i, 0).isChecked():
                anyr = True; break
        self.dlg.radioKML.setEnabled(not anyr)
        if anyr:
            self.dlg.radioKMZ.setChecked(True)

    def add_layer(self):
        p, _ = QFileDialog.getOpenFileName(
            self.dlg, 'Browse Input Files', '',
            'Vector (*.shp *.gpkg *.geojson *.json *.zip *.kml);;'
            'Raster (*.tif *.tiff *.jpg *.jpeg *.png);;All (*.*)'
        )
        
        if not p:
            return
        
        if p.lower().endswith(('.tif', '.tiff', '.jpg', '.jpeg', '.png')):
            L = QgsRasterLayer(p, os.path.basename(p))
        else:
            L = QgsVectorLayer(p, os.path.basename(p), 'ogr')
        
        if not L.isValid():
            QMessageBox.critical(self.iface.mainWindow(), 'Error', 'Cannot load layer')
            return
        
        self.added_file_layers.append(L)
        self._populate(); self._update_format()

    def select_output(self):
        fmt = 'kmz' if self.dlg.radioKMZ.isChecked() else 'kml'
        p, _ = QFileDialog.getSaveFileName(self.dlg, 'Output File', '', f'*.{fmt}')
        if not p:
            return
        if not p.lower().endswith(f'.{fmt}'):
            p += f'.{fmt}'
        self.output_path = p
        self.dlg.editOutput.setText(p)

    def run_export(self):
        """Export with detailed debugging."""
        print("\n" + "🚀"*20)
        print("STARTING DETAILED EXPORT ANALYSIS")
        print("🚀"*20)
        
        try:
            out = getattr(self, 'output_path', None)
            if not out:
                raise ValueError('Please select an output file')
            
            kml, doc = make_kml_root()
            tmp = tempfile.mkdtemp()
            rasdir = os.path.join(tmp, 'rasters')
            os.makedirs(rasdir, exist_ok=True)
            
            layers = list(QgsProject.instance().mapLayers().values()) + self.added_file_layers
            tbl = self.dlg.tableLayers
            
            total_vertices = 0
            should_continue = True
            
            # Get selected layers
            selected_layers = []
            for i, L in enumerate(layers):
                if i < tbl.rowCount() and tbl.cellWidget(i, 0).isChecked():
                    selected_layers.append((i, L))
            
            print(f"📋 Selected layers: {len(selected_layers)}")
            for i, L in selected_layers:
                print(f"   - {L.name()} ({type(L).__name__})")
            
            # Process each selected layer
            for i, L in selected_layers:
                if not should_continue:
                    break
                
                if isinstance(L, QgsRasterLayer):
                    props = {'folder_name': L.name()}
                    add_raster_layer(doc, L.source(), props, rasdir)
                else:
                    # Get properties from UI
                    props = {}
                    
                    folder_widget = tbl.cellWidget(i, 4)
                    if folder_widget:
                        fld = folder_widget.text().strip()
                        if fld:
                            props['folder_name'] = fld
                    
                    src = L.source()
                    flds = self.selected_fields.get(src, [])
                    if flds:
                        props['fields'] = flds
                    
                    for j, pn in enumerate(KML_PROPS, start=6):
                        widget = tbl.cellWidget(i, j)
                        if widget:
                            t = widget.currentText().strip()
                            if t and t != '':
                                props[pn.lower()] = t
                    
                    # Process the layer
                    new_total, should_continue, user_choice = add_vector_layer(
                        doc, L, props, total_vertices, MAX_VERTICES, self.iface.mainWindow()
                    )
                    total_vertices = new_total
                    
                    if user_choice == 'stop':
                        QMessageBox.information(
                            self.iface.mainWindow(),
                            "Export Stopped",
                            "Export was stopped. Please unselect some layers to reduce vertex count and try again."
                        )
                        return
            
            print(f"\n" + "🎉"*20)
            print(f"EXPORT COMPLETE")
            print(f"📊 Final total vertices: {total_vertices:,}")
            print(f"🎯 Vertex limit was: {MAX_VERTICES:,}")
            if total_vertices > MAX_VERTICES:
                print(f"⚠️ EXCEEDED LIMIT BY: {total_vertices - MAX_VERTICES:,}")
            print("🎉"*20)
            
            # Write KML file
            kf = os.path.join(tmp, 'doc.kml')
            ET.ElementTree(kml).write(kf, encoding='utf-8', xml_declaration=True)
            
            has_rasters = any(isinstance(L, QgsRasterLayer) for i, L in selected_layers)
            
            if has_rasters:
                with zipfile.ZipFile(out, 'w', zipfile.ZIP_DEFLATED) as zf:
                    zf.write(kf, 'doc.kml')
                    for fname in os.listdir(rasdir):
                        zf.write(os.path.join(rasdir, fname), f'rasters/{fname}')
            else:
                os.replace(kf, out)
            
            self.iface.messageBar().pushMessage(
                'Export 2KML',
                f'Exported to {out} with {total_vertices:,} vertices',
                level=Qgis.Info
            )
            
            self.dlg.close()
            
        except Exception as e:
            print(f"💥 ERROR: {str(e)}")
            import traceback
            print(f"🔍 Traceback: {traceback.format_exc()}")
            QMessageBox.critical(self.iface.mainWindow(), 'Error', str(e))
