# exporter_dialog.py
# -*- coding: utf-8 -*-

"""
Exporter Dialog
User interface for configuring Excel Auto Exporter
"""

import os
from qgis.PyQt.QtCore import pyqtSignal, Qt, QSize
from qgis.PyQt.QtWidgets import (
    QDialog,
    QVBoxLayout,
    QHBoxLayout,
    QPushButton,
    QListWidget,
    QLabel,
    QCheckBox,
    QLineEdit,
    QFileDialog,
    QListWidgetItem,
    QMessageBox,
    QGroupBox,
    QSplitter,
    QWidget,
    QComboBox,
)
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsProject, QgsVectorLayer, QgsWkbTypes, QgsSettings, QgsApplication


class ExporterDialog(QDialog):
    """Dialog for configuring Excel Auto Exporter."""

    layers_config_changed = pyqtSignal()
    export_requested = pyqtSignal(str)
    export_all_requested = pyqtSignal()

    def __init__(self, iface, config_manager, excel_engine, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.config_manager = config_manager
        self.excel_engine = excel_engine
        self.plugin_dir = os.path.dirname(__file__)
        self.all_layers = []

        self.setupUi()
        self.load_configuration()

    def _theme_icon(self, icon_name):
        icon = QgsApplication.getThemeIcon(icon_name)
        return icon if not icon.isNull() else QIcon()

    def get_geometry_icon(self, geometry_type):
        icon_path = None
        if geometry_type == QgsWkbTypes.PointGeometry:
            icon_path = os.path.join(self.plugin_dir, "point.png")
        elif geometry_type == QgsWkbTypes.LineGeometry:
            icon_path = os.path.join(self.plugin_dir, "line.png")
        elif geometry_type == QgsWkbTypes.PolygonGeometry:
            icon_path = os.path.join(self.plugin_dir, "polygon.png")

        if icon_path and os.path.exists(icon_path):
            return QIcon(icon_path)
        return QIcon()

    def get_geometry_type_name(self, geometry_type):
        if geometry_type == QgsWkbTypes.PointGeometry:
            return "Point"
        if geometry_type == QgsWkbTypes.LineGeometry:
            return "Line"
        if geometry_type == QgsWkbTypes.PolygonGeometry:
            return "Polygon"
        return "Unknown"

    def get_layer_group_path(self, layer):
        project = QgsProject.instance()
        root = project.layerTreeRoot()
        layer_id = layer.id()

        for tree_layer in root.findLayers():
            if tree_layer.layerId() == layer_id:
                path_parts = []
                parent = tree_layer.parent()
                while parent and parent != root:
                    path_parts.insert(0, parent.name())
                    parent = parent.parent()
                return " / ".join(path_parts)
        return ""

    def setupUi(self):
        self.setWindowTitle("Excel Auto Exporter - Configuration")
        self.setWindowFlags(
            Qt.Window
            | Qt.WindowMinimizeButtonHint
            | Qt.WindowMaximizeButtonHint
            | Qt.WindowCloseButtonHint
        )
        self.resize(800, 650)

        self.main_layout = QVBoxLayout()
        self.main_layout.setContentsMargins(6, 6, 6, 6)
        self.main_layout.setSpacing(6)

        self._setup_instructions_ui()
        self._setup_available_layers_ui()
        self._setup_monitored_layers_ui()
        self._setup_settings_ui()
        self._setup_actions_ui()

        self.setLayout(self.main_layout)

        self.refresh_layer_list()
        self.refresh_monitored_list()

    def _setup_instructions_ui(self):
        instructions = QLabel(
            "Select layers to monitor and export to Excel.\n"
            "Use search to quickly find layers and manage monitoring.\n"
            "Changes in monitored layers can trigger automatic export."
        )
        instructions.setWordWrap(True)
        instructions.setStyleSheet(
            "QLabel { background-color: #f3f6f8; padding: 8px; border-radius: 4px; }"
        )
        self.main_layout.addWidget(instructions)

    def _setup_available_layers_ui(self):
        self.main_splitter = QSplitter(Qt.Vertical)
        self.main_splitter.setChildrenCollapsible(False)

        available_widget = QWidget()
        available_widget.setMinimumHeight(150)
        available_layout = QVBoxLayout(available_widget)
        available_layout.setContentsMargins(2, 2, 2, 2)

        group = QGroupBox("Available Layers")
        group_layout = QVBoxLayout()

        search_layout = QHBoxLayout()
        search_layout.addWidget(QLabel("Search:"))

        self.search_box = QLineEdit()
        self.search_box.setPlaceholderText("Filter by layer name or group path")
        self.search_box.textChanged.connect(self.filter_layers)
        search_layout.addWidget(self.search_box)

        clear_search_btn = QPushButton("Clear")
        clear_search_btn.setMaximumWidth(56)
        clear_search_btn.clicked.connect(self.search_box.clear)
        search_layout.addWidget(clear_search_btn)

        group_layout.addLayout(search_layout)

        self.layer_list = QListWidget()
        self.layer_list.setSelectionMode(QListWidget.MultiSelection)
        self.layer_list.setIconSize(QSize(20, 20))
        group_layout.addWidget(self.layer_list)

        self.layer_count_label = QLabel("Layers: 0")
        self.layer_count_label.setStyleSheet("QLabel { color: #666; font-size: 10px; }")
        group_layout.addWidget(self.layer_count_label)

        buttons_row = QHBoxLayout()

        self.refresh_btn = QPushButton("Refresh")
        self.refresh_btn.setIcon(self._theme_icon("/mActionRefresh.svg"))
        self.refresh_btn.setToolTip("Reload all layers from project")
        self.refresh_btn.clicked.connect(self.refresh_layer_list)
        buttons_row.addWidget(self.refresh_btn)

        buttons_row.addStretch()

        self.add_layer_btn = QPushButton("Add to Monitoring")
        self.add_layer_btn.setIcon(self._theme_icon("/symbologyAdd.svg"))
        self.add_layer_btn.clicked.connect(self.add_layers_to_monitoring)
        self.add_layer_btn.setStyleSheet(
            "QPushButton { font-weight: bold; background-color: #2196F3; color: white; padding: 6px; }"
        )
        buttons_row.addWidget(self.add_layer_btn)

        group_layout.addLayout(buttons_row)
        group.setLayout(group_layout)
        available_layout.addWidget(group)
        self.main_splitter.addWidget(available_widget)

    def _setup_monitored_layers_ui(self):
        monitored_widget = QWidget()
        monitored_widget.setMinimumHeight(120)
        monitored_layout = QVBoxLayout(monitored_widget)
        monitored_layout.setContentsMargins(2, 2, 2, 2)

        monitored_group = QGroupBox("Monitored Layers")
        monitored_group_layout = QVBoxLayout()

        self.monitored_list = QListWidget()
        self.monitored_list.setIconSize(QSize(20, 20))
        monitored_group_layout.addWidget(self.monitored_list)

        monitored_btn_layout = QHBoxLayout()

        self.export_one_btn = QPushButton("Export Selected")
        self.export_one_btn.setIcon(self._theme_icon("/mActionSaveAs.svg"))
        self.export_one_btn.clicked.connect(self.export_selected_layer)
        monitored_btn_layout.addWidget(self.export_one_btn)

        self.remove_layer_btn = QPushButton("Remove")
        self.remove_layer_btn.setIcon(self._theme_icon("/symbologyRemove.svg"))
        self.remove_layer_btn.clicked.connect(self.remove_layer_from_monitoring)
        monitored_btn_layout.addWidget(self.remove_layer_btn)

        monitored_group_layout.addLayout(monitored_btn_layout)
        monitored_group.setLayout(monitored_group_layout)
        monitored_layout.addWidget(monitored_group)

        self.main_splitter.addWidget(monitored_widget)
        self.main_splitter.setSizes([400, 250])
        self.main_splitter.setStretchFactor(0, 3)
        self.main_splitter.setStretchFactor(1, 2)

        self.main_layout.addWidget(self.main_splitter)

    def _setup_settings_ui(self):
        settings_group = QGroupBox("Settings")
        settings_layout = QVBoxLayout()

        self.auto_export_checkbox = QCheckBox("Enable automatic export on layer changes")
        self.auto_export_checkbox.setChecked(self.config_manager.is_auto_export_enabled())
        self.auto_export_checkbox.stateChanged.connect(self.on_auto_export_changed)
        settings_layout.addWidget(self.auto_export_checkbox)

        engine_layout = QHBoxLayout()
        engine_label = QLabel("Export engine:")
        engine_layout.addWidget(engine_label)

        self.engine_combo = QComboBox()
        self.engine_combo.addItem("Auto (detect safe option)", "auto")
        self.engine_combo.addItem("openpyxl (default)", "openpyxl")
        self.engine_combo.addItem("xlsxwriter (safer on Windows)", "xlsxwriter")
        self.engine_combo.addItem("CSV (universal fallback)", "csv")

        settings = QgsSettings()
        current_engine = settings.value("excel_auto_exporter/export_engine", "auto")
        idx = self.engine_combo.findData(current_engine)
        if idx >= 0:
            self.engine_combo.setCurrentIndex(idx)

        self.engine_combo.currentIndexChanged.connect(self.on_engine_changed)
        engine_layout.addWidget(self.engine_combo)
        settings_layout.addLayout(engine_layout)

        info = QLabel("'Auto' uses xlsxwriter on Windows/QGIS 3.40+ to avoid crashes")
        info.setWordWrap(True)
        info.setStyleSheet("QLabel { color: #666; font-size: 10px; font-style: italic; }")
        settings_layout.addWidget(info)

        folder_layout = QHBoxLayout()
        folder_layout.addWidget(QLabel("Default output folder:"))

        self.output_folder_edit = QLineEdit()
        self.output_folder_edit.setText(self.config_manager.get_default_output_folder())
        folder_layout.addWidget(self.output_folder_edit)

        browse_btn = QPushButton("Browse...")
        browse_btn.clicked.connect(self.browse_output_folder)
        folder_layout.addWidget(browse_btn)

        settings_layout.addLayout(folder_layout)
        settings_group.setLayout(settings_layout)
        self.main_layout.addWidget(settings_group)

    def _setup_actions_ui(self):
        button_layout = QHBoxLayout()

        self.export_all_btn = QPushButton("Export All Monitored Layers")
        self.export_all_btn.setIcon(self._theme_icon("/mActionSaveAs.svg"))
        self.export_all_btn.clicked.connect(self.export_all_layers)
        self.export_all_btn.setStyleSheet(
            "QPushButton { background-color: #4CAF50; color: white; font-weight: bold; padding: 8px; }"
        )
        button_layout.addWidget(self.export_all_btn)

        button_layout.addStretch()

        close_btn = QPushButton("Close")
        close_btn.clicked.connect(self.close)
        close_btn.setMinimumWidth(100)
        button_layout.addWidget(close_btn)

        self.main_layout.addLayout(button_layout)

    def refresh_layer_list(self):
        self.layer_list.clear()
        self.all_layers = []

        for layer in QgsProject.instance().mapLayers().values():
            if not isinstance(layer, QgsVectorLayer):
                continue

            self.all_layers.append(layer.name())

            geom_type = layer.geometryType()
            icon = self.get_geometry_icon(geom_type)
            geom_name = self.get_geometry_type_name(geom_type)
            group_path = self.get_layer_group_path(layer)

            display_text = (
                f"{layer.name()} ({geom_name}) — [{group_path}]"
                if group_path
                else f"{layer.name()} ({geom_name})"
            )

            item = QListWidgetItem(icon, display_text)
            item.setData(Qt.UserRole, layer.name())
            item.setData(Qt.UserRole + 1, group_path)

            tooltip_parts = [
                f"Type: {geom_name}",
                f"Fields: {len(layer.fields())}",
                f"Features: {layer.featureCount()}",
            ]
            if group_path:
                tooltip_parts.append(f"Group path: {group_path}")

            item.setToolTip("\n".join(tooltip_parts))
            self.layer_list.addItem(item)

        self.update_layer_count()
        if self.search_box.text():
            self.filter_layers(self.search_box.text())

    def filter_layers(self, search_text):
        search_text = (search_text or "").lower()
        visible_count = 0

        for i in range(self.layer_list.count()):
            item = self.layer_list.item(i)
            layer_name = (item.data(Qt.UserRole) or "").lower()
            group_path = (item.data(Qt.UserRole + 1) or "").lower()
            full_text = f"{layer_name} {group_path}"

            matches = search_text in full_text
            item.setHidden(not matches)
            if matches:
                visible_count += 1

        total = len(self.all_layers)
        if search_text:
            self.layer_count_label.setText(f"Showing: {visible_count} of {total} layers")
            self.layer_count_label.setStyleSheet("QLabel { color: #2196F3; font-weight: bold; }")
        else:
            self.layer_count_label.setText(f"Total layers: {total}")
            self.layer_count_label.setStyleSheet("QLabel { color: #666; }")

    def update_layer_count(self):
        total = len(self.all_layers)
        self.layer_count_label.setText(f"Total layers: {total}")
        self.layer_count_label.setStyleSheet("QLabel { color: #666; }")

    def refresh_monitored_list(self):
        self.monitored_list.clear()
        monitored = self.config_manager.get_monitored_layers()

        for layer_name, cfg in monitored.items():
            output_path = cfg.get("output_path", "Not configured")
            fields_count = len(cfg.get("fields", []))

            layer = self.get_layer_by_name(layer_name)
            if layer:
                icon = self.get_geometry_icon(layer.geometryType())
                group_path = self.get_layer_group_path(layer)
            else:
                icon = QIcon()
                group_path = ""

            layer_part = f"{layer_name} [{group_path}]" if group_path else layer_name
            display_text = f"{layer_part} → {os.path.basename(output_path)} ({fields_count} fields)"

            item = QListWidgetItem(icon, display_text)
            item.setData(Qt.UserRole, layer_name)
            if group_path:
                item.setData(Qt.UserRole + 1, group_path)

            tooltip = [
                f"Layer: {layer_name}",
                f"Output: {output_path}",
                f"Fields exported: {fields_count}",
            ]
            if group_path:
                tooltip.append(f"Group path: {group_path}")
            item.setToolTip("\n".join(tooltip))

            self.monitored_list.addItem(item)

    def add_layers_to_monitoring(self):
        selected_items = self.layer_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "No Selection", "Please select at least one layer to add.")
            return

        added_count = 0
        skipped_count = 0

        for item in selected_items:
            layer_name = item.data(Qt.UserRole)
            monitored = self.config_manager.get_monitored_layers()
            if layer_name in monitored:
                skipped_count += 1
                continue

            layer = self.get_layer_by_name(layer_name)
            if layer and layer.isValid():
                fields = [f.name() for f in layer.fields()]
                default_folder = self.output_folder_edit.text() or self.config_manager.get_default_output_folder()
                output_path = os.path.join(default_folder, f"{layer_name}.xlsx")
                self.config_manager.add_monitored_layer(layer_name, fields, output_path)
                added_count += 1

        self.refresh_monitored_list()
        self.layers_config_changed.emit()

        msg = f"{added_count} layer(s) added to monitoring."
        if skipped_count > 0:
            msg += f"\n{skipped_count} layer(s) skipped (already monitored)."
        QMessageBox.information(self, "Layers Added", msg)

    def remove_layer_from_monitoring(self):
        selected_items = self.monitored_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "No Selection", "Please select a monitored layer to remove.")
            return

        for item in selected_items:
            layer_name = item.data(Qt.UserRole)
            self.config_manager.remove_monitored_layer(layer_name)

        self.refresh_monitored_list()
        self.layers_config_changed.emit()

        QMessageBox.information(
            self,
            "Layer Removed",
            f"{len(selected_items)} layer(s) removed from monitoring.",
        )

    def export_selected_layer(self):
        selected_items = self.monitored_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "No Selection", "Please select a layer to export.")
            return

        for item in selected_items:
            self.export_requested.emit(item.data(Qt.UserRole))

    def export_all_layers(self):
        self.export_all_requested.emit()

    def browse_output_folder(self):
        folder = QFileDialog.getExistingDirectory(
            self,
            "Select Default Output Folder",
            self.output_folder_edit.text(),
        )
        if folder:
            self.output_folder_edit.setText(folder)
            self.config_manager.set_default_output_folder(folder)

    def on_auto_export_changed(self, state):
        enabled = state == Qt.Checked
        self.config_manager.set_auto_export(enabled)
        self.layers_config_changed.emit()

    def on_engine_changed(self, _index):
        engine = self.engine_combo.currentData()
        settings = QgsSettings()
        settings.setValue("excel_auto_exporter/export_engine", engine)

        if engine == "csv":
            QMessageBox.information(
                self,
                "Export Engine Changed",
                "CSV export selected.\n\nAll exports will now use CSV format instead of Excel.",
            )
        elif engine == "xlsxwriter":
            try:
                import xlsxwriter  # noqa: F401
                QMessageBox.information(
                    self,
                    "Export Engine Changed",
                    "xlsxwriter engine selected (recommended for Windows/QGIS 3.40+).",
                )
            except ImportError:
                QMessageBox.warning(
                    self,
                    "xlsxwriter Not Available",
                    "xlsxwriter is not installed.\n\n"
                    "Install it with:\npip install xlsxwriter\n\n"
                    "Falling back to CSV until installed.",
                )
        elif engine == "openpyxl":
            import platform
            if platform.system() == "Windows":
                QMessageBox.warning(
                    self,
                    "Export Engine Changed",
                    "openpyxl selected on Windows.\n\n"
                    "Note: openpyxl may cause crashes on Windows/QGIS 3.40+.\n"
                    "Consider using 'Auto' or 'xlsxwriter' instead.\n\n"
                    "Reference: https://github.com/qgis/QGIS/issues/58205",
                )

        self.layers_config_changed.emit()

    def load_configuration(self):
        self.auto_export_checkbox.setChecked(self.config_manager.is_auto_export_enabled())
        self.output_folder_edit.setText(self.config_manager.get_default_output_folder())

    def get_layer_by_name(self, layer_name):
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsVectorLayer) and layer.name() == layer_name:
                return layer
        return None
