# excel_engine.py
# -*- coding: utf-8 -*-

"""
Excel Export Engine
Handles export of layers to Excel/CSV and background task execution.
"""

import os
import platform
from qgis.core import QgsMessageLog, Qgis, QgsTask, QgsVectorLayer


class ExportTask(QgsTask):
    """Background export task using QGIS Task Manager."""

    def __init__(
        self,
        engine,
        layer_name,
        fields_to_export,
        output_path,
        export_engine,
        layer_source=None,
        provider_type=None,
        preloaded_rows=None,
        on_finished_callback=None,
    ):
        super().__init__(f"Export {layer_name}", QgsTask.CanCancel)
        self.engine = engine
        self.layer_name = layer_name
        self.fields_to_export = fields_to_export
        self.output_path = output_path
        self.export_engine = export_engine
        self.layer_source = layer_source
        self.provider_type = provider_type
        self.preloaded_rows = preloaded_rows
        self.on_finished_callback = on_finished_callback
        self.success = False
        self.message = ""

    def run(self):
        try:
            rows = self.preloaded_rows

            # Preferred path in real QGIS: clone layer in worker thread
            if rows is None and self.layer_source and self.provider_type:
                cloned_layer = QgsVectorLayer(self.layer_source, self.layer_name, self.provider_type)
                if not cloned_layer.isValid():
                    raise ValueError(f"Unable to clone layer for export: {self.layer_name}")
                rows = self.engine.prepare_rows(cloned_layer, self.fields_to_export)

            if rows is None:
                raise ValueError(f"No row source available for layer: {self.layer_name}")

            self.success, self.message = self.engine.export_rows(
                rows=rows,
                output_path=self.output_path,
                export_engine=self.export_engine,
            )
            return self.success

        except Exception as e:
            self.success = False
            self.message = f"Error exporting {self.layer_name}: {e}"
            QgsMessageLog.logMessage(self.message, "Excel Auto Exporter", Qgis.Critical)
            return False

    def finished(self, result):
        if self.on_finished_callback:
            self.on_finished_callback(self.layer_name, bool(result and self.success), self.message)


class ExcelEngine:
    """Handles Excel export operations."""

    def __init__(self):
        self.pandas_available = self.check_dependencies()
        self.xlsxwriter_available = self.check_xlsxwriter()

    def check_dependencies(self):
        try:
            import pandas  # noqa: F401
            import openpyxl  # noqa: F401
            return True
        except ImportError:
            QgsMessageLog.logMessage(
                "Missing dependencies: pandas and/or openpyxl not installed.\n"
                "Please install with: pip install pandas openpyxl",
                "Excel Auto Exporter",
                Qgis.Critical,
            )
            return False

    def check_xlsxwriter(self):
        try:
            import importlib.util
            spec = importlib.util.find_spec("xlsxwriter")
            return spec is not None
        except (ImportError, AttributeError, ValueError) as e:
            QgsMessageLog.logMessage(
                f"Error checking xlsxwriter availability: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )
            return False

    def is_problematic_environment(self):
        if platform.system() != "Windows":
            return False

        try:
            from qgis.core import Qgis as QgisCore
            version = QgisCore.QGIS_VERSION
            parts = version.split(".")
            if len(parts) >= 2:
                major = int(parts[0])
                minor = int(parts[1])
                return major == 3 and minor >= 34
        except (ImportError, AttributeError, ValueError) as e:
            QgsMessageLog.logMessage(
                f"Error detecting QGIS environment: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )
        return False

    def get_recommended_engine(self, user_preference="openpyxl"):
        is_problematic = self.is_problematic_environment()

        if user_preference == "openpyxl" and is_problematic:
            if self.xlsxwriter_available:
                QgsMessageLog.logMessage(
                    "Windows + QGIS 3.34+ detected. Using xlsxwriter instead of openpyxl.",
                    "Excel Auto Exporter",
                    Qgis.Warning,
                )
                return "xlsxwriter", ".xlsx", "Using xlsxwriter (safer on Windows/QGIS 3.40+)"

            QgsMessageLog.logMessage(
                "Windows + QGIS 3.34+ detected. openpyxl may crash. Falling back to CSV.",
                "Excel Auto Exporter",
                Qgis.Warning,
            )
            return "csv", ".csv", "Using CSV fallback (openpyxl crash risk detected)"

        if user_preference == "xlsxwriter":
            if self.xlsxwriter_available:
                return "xlsxwriter", ".xlsx", None
            return "csv", ".csv", "xlsxwriter not available, using CSV"

        if user_preference == "csv":
            return "csv", ".csv", None

        return "openpyxl", ".xlsx", None

    @staticmethod
    def _sanitize_filename(name):
        """Return filesystem-safe filename for Windows/Linux."""
        forbidden_chars = '/\\:*?"<>|'
        sanitized = "".join("_" if ch in forbidden_chars else ch for ch in (name or ""))
        sanitized = sanitized.strip().strip(".")
        return sanitized or "export"

    def _normalize_output_path(self, output_path, ext):
        output_dir = os.path.dirname(output_path)
        base_name = os.path.basename(output_path)
        name_wo_ext, _ = os.path.splitext(base_name)
        safe_name = self._sanitize_filename(name_wo_ext)
        return os.path.join(output_dir, f"{safe_name}{ext}")

    def _ensure_output_dir_writable(self, output_path):
        output_dir = os.path.dirname(output_path)
        if output_dir:
            if not os.path.exists(output_dir):
                os.makedirs(output_dir, exist_ok=True)
            if not os.access(output_dir, os.W_OK):
                raise PermissionError(f"Output directory is not writable: {output_dir}")
        else:
            cwd = os.getcwd()
            if not os.access(cwd, os.W_OK):
                raise PermissionError(f"Current directory is not writable: {cwd}")

    def prepare_rows(self, layer, fields_to_export):
        if not layer or not layer.isValid():
            raise ValueError("Layer is not valid")

        rows = []
        for feature in layer.getFeatures():
            row = {}
            for field_name in fields_to_export:
                is_area_field = any(
                    kw in field_name.lower()
                    for kw in [
                        "area", "qm", "ha", "superficie", "flache", "fläche",
                        "sqm", "m2", "m²", "hectare", "acre",
                    ]
                )

                if is_area_field:
                    geom = feature.geometry()
                    row[field_name] = round(geom.area(), 2) if geom and not geom.isEmpty() else 0
                else:
                    field_idx = layer.fields().indexOf(field_name)
                    if field_idx >= 0:
                        val = feature.attribute(field_name)
                        row[field_name] = val if val is not None else ""
                    else:
                        row[field_name] = ""
            rows.append(row)

        return rows

    def export_rows(self, rows, output_path, export_engine="openpyxl"):
        if not self.pandas_available:
            return False, "Dependencies not available"
        if not rows:
            return False, "No data to export"

        try:
            import pandas as pd

            engine, ext, warning = self.get_recommended_engine(export_engine)
            output_path = self._normalize_output_path(output_path, ext)
            self._ensure_output_dir_writable(output_path)

            df = pd.DataFrame(rows)

            if engine == "csv":
                df.to_csv(output_path, index=False, encoding="utf-8-sig")
                success_msg = (
                    f"Successfully exported {len(rows)} features to "
                    f"{os.path.basename(output_path)} (CSV)"
                )
            elif engine == "xlsxwriter":
                writer = pd.ExcelWriter(output_path, engine="xlsxwriter")
                df.to_excel(writer, sheet_name="Sheet1", index=False)
                writer.close()
                success_msg = (
                    f"Successfully exported {len(rows)} features to "
                    f"{os.path.basename(output_path)} (xlsxwriter)"
                )
            else:
                df.to_excel(output_path, index=False, engine="openpyxl")
                success_msg = (
                    f"Successfully exported {len(rows)} features to "
                    f"{os.path.basename(output_path)}"
                )

            if warning:
                success_msg += f" [{warning}]"

            return True, success_msg

        except Exception as e:
            error_msg = f"Error exporting layer: {e}"
            QgsMessageLog.logMessage(error_msg, "Excel Auto Exporter", Qgis.Critical)
            return False, error_msg

    def create_export_task(
        self,
        layer_name,
        layer,
        fields_to_export,
        output_path,
        export_engine,
        on_finished_callback,
    ):
        layer_source = layer.source() if hasattr(layer, "source") and callable(layer.source) else None
        provider_type = (
            layer.providerType()
            if hasattr(layer, "providerType") and callable(layer.providerType)
            else None
        )

        preloaded_rows = None
        if not layer_source or not provider_type:
            # Fallback for non-cloneable layers / tests
            preloaded_rows = self.prepare_rows(layer, fields_to_export)

        return ExportTask(
            engine=self,
            layer_name=layer_name,
            fields_to_export=fields_to_export,
            output_path=output_path,
            export_engine=export_engine,
            layer_source=layer_source,
            provider_type=provider_type,
            preloaded_rows=preloaded_rows,
            on_finished_callback=on_finished_callback,
        )

    def export_layer(self, layer, fields_to_export, output_path, export_engine="openpyxl"):
        """Synchronous path kept for manual export compatibility and tests."""
        try:
            rows = self.prepare_rows(layer, fields_to_export)
            return self.export_rows(rows, output_path, export_engine)
        except Exception as e:
            error_msg = f"Error exporting layer: {e}"
            QgsMessageLog.logMessage(error_msg, "Excel Auto Exporter", Qgis.Critical)
            return False, error_msg

    def get_layer_fields(self, layer):
        if not layer or not layer.isValid():
            return []
        return [field.name() for field in layer.fields()]
