# excel_auto_exporter.py
# -*- coding: utf-8 -*-

"""
Excel Auto Exporter Plugin for QGIS
Main plugin class
"""

import os
from qgis.PyQt.QtCore import QObject, QTimer
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtGui import QIcon
from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsMessageLog,
    Qgis,
    QgsSettings,
    QgsApplication,
)

from .config_manager import ConfigManager
from .excel_engine import ExcelEngine
from .exporter_dialog import ExporterDialog


class ExcelAutoExporter(QObject):
    """Main plugin class for Excel Auto Exporter."""

    def __init__(self, iface):
        super().__init__()
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        self.config_manager = ConfigManager()
        self.excel_engine = ExcelEngine()

        self.monitored_layers = {}
        self.dialog = None
        self.action = None

        self.export_engine = self.get_export_engine_preference()

        # Debounce queue
        self.pending_exports = set()
        self.debounce_interval_ms = 1000
        self.debounce_timer = QTimer()
        self.debounce_timer.setSingleShot(True)
        self.debounce_timer.setInterval(self.debounce_interval_ms)
        self.debounce_timer.timeout.connect(self._perform_export)

        # Avoid overlapping async tasks per layer
        self.active_tasks = {}

    def get_export_engine_preference(self):
        settings = QgsSettings()
        return settings.value("excel_auto_exporter/export_engine", "auto")

    def set_export_engine_preference(self, engine):
        settings = QgsSettings()
        settings.setValue("excel_auto_exporter/export_engine", engine)
        self.export_engine = engine

    def initGui(self):
        try:
            icon_path = os.path.join(self.plugin_dir, "icon.png")
            self.action = QAction(QIcon(icon_path), "Excel Auto Exporter", self.iface.mainWindow())
            self.action.triggered.connect(self.run)
            self.action.setStatusTip("Configure Excel Auto Exporter")
            self.action.setWhatsThis("Configure layers to export automatically")

            self.iface.addToolBarIcon(self.action)
            self.iface.addPluginToMenu("&Excel Auto Exporter", self.action)

            QgsProject.instance().readProject.connect(self.on_project_loaded)
            QgsProject.instance().layersAdded.connect(self.on_layers_added)
            QgsProject.instance().layersRemoved.connect(self.on_layers_removed)

            if not self.excel_engine.pandas_available:
                self.show_dependency_warning()

            QTimer.singleShot(1000, self.restore_monitoring_from_config)
            self.show_message("Excel Auto Exporter loaded successfully", Qgis.Info)
        except Exception as e:
            self.show_message(f"Error initializing plugin: {e}", Qgis.Critical)

    def unload(self):
        try:
            QgsProject.instance().readProject.disconnect(self.on_project_loaded)
            QgsProject.instance().layersAdded.disconnect(self.on_layers_added)
            QgsProject.instance().layersRemoved.disconnect(self.on_layers_removed)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error disconnecting project signals: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )

        self.clear_all_monitoring()

        if self.action:
            self.iface.removePluginMenu("&Excel Auto Exporter", self.action)
            self.iface.removeToolBarIcon(self.action)

        if self.dialog:
            self.dialog.close()

    def run(self):
        if not self.excel_engine.pandas_available:
            self.show_dependency_warning()
            return

        if not self.dialog:
            self.dialog = ExporterDialog(
                self.iface,
                self.config_manager,
                self.excel_engine,
                parent=self.iface.mainWindow(),
            )
            self.dialog.layers_config_changed.connect(self.update_monitoring)
            self.dialog.export_requested.connect(self.export_layer_by_name)
            self.dialog.export_all_requested.connect(self.export_all_monitored)

        self.dialog.show()
        self.dialog.raise_()
        self.dialog.activateWindow()

    def restore_monitoring_from_config(self):
        try:
            monitored = self.config_manager.get_monitored_layers()
            if not monitored:
                return

            for layer_name in monitored.keys():
                layer = self.get_layer_by_name(layer_name)
                if layer and layer.isValid():
                    self.monitor_layer(layer_name)

            if self.monitored_layers:
                self.show_message(
                    f"Restored monitoring for {len(self.monitored_layers)} layer(s)",
                    Qgis.Success,
                )
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error restoring configuration: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )

    def update_monitoring(self):
        try:
            self.export_engine = self.get_export_engine_preference()
            self.clear_all_monitoring()

            monitored = self.config_manager.get_monitored_layers()
            for layer_name in monitored.keys():
                self.monitor_layer(layer_name)
        except Exception as e:
            self.show_message(f"Error updating monitoring: {e}", Qgis.Warning)

    def monitor_layer(self, layer_name):
        try:
            layer = self.get_layer_by_name(layer_name)
            if not layer or not layer.isValid():
                self.show_message(f"Layer '{layer_name}' not found or invalid", Qgis.Warning)
                return False

            if layer_name in self.monitored_layers:
                old_layer = self.monitored_layers[layer_name]
                try:
                    old_layer.attributeValueChanged.disconnect()
                    old_layer.geometryChanged.disconnect()
                    old_layer.featureAdded.disconnect()
                    old_layer.featureDeleted.disconnect()
                except Exception as e:
                    QgsMessageLog.logMessage(
                        f"Error disconnecting old signals for {layer_name}: {e}",
                        "Excel Auto Exporter",
                        Qgis.Warning,
                    )

            if self.config_manager.is_auto_export_enabled():
                layer.attributeValueChanged.connect(
                    lambda fid, field, value, ln=layer_name: self.on_layer_changed(ln)
                )
                layer.geometryChanged.connect(
                    lambda fid, geom, ln=layer_name: self.on_layer_changed(ln)
                )
                layer.featureAdded.connect(
                    lambda fid, ln=layer_name: self.on_layer_changed(ln)
                )
                layer.featureDeleted.connect(
                    lambda fid, ln=layer_name: self.on_layer_changed(ln)
                )

            self.monitored_layers[layer_name] = layer
            self.show_message(f"Monitoring activated for: {layer_name}", Qgis.Success)

            # Initial export request (debounced path)
            self.on_layer_changed(layer_name)
            return True
        except Exception as e:
            self.show_message(f"Error monitoring layer {layer_name}: {e}", Qgis.Critical)
            return False

    def clear_all_monitoring(self):
        for layer_name, layer in self.monitored_layers.items():
            try:
                if layer and layer.isValid():
                    layer.attributeValueChanged.disconnect()
                    layer.geometryChanged.disconnect()
                    layer.featureAdded.disconnect()
                    layer.featureDeleted.disconnect()
            except Exception as e:
                QgsMessageLog.logMessage(
                    f"Error clearing signals for {layer_name}: {e}",
                    "Excel Auto Exporter",
                    Qgis.Warning,
                )

        self.monitored_layers.clear()

    def on_layer_changed(self, layer_name):
        if not self.config_manager.is_auto_export_enabled():
            return

        self.pending_exports.add(layer_name)
        self.debounce_timer.start()

    def _perform_export(self):
        layers_to_export = list(self.pending_exports)
        self.pending_exports.clear()

        for layer_name in layers_to_export:
            if layer_name in self.active_tasks:
                QgsMessageLog.logMessage(
                    f"Task already running for layer '{layer_name}', skipping duplicate queue.",
                    "Excel Auto Exporter",
                    Qgis.Info,
                )
                continue

            layer = self.monitored_layers.get(layer_name) or self.get_layer_by_name(layer_name)
            if not layer or not layer.isValid():
                self.show_message(f"Layer '{layer_name}' not found", Qgis.Warning)
                continue

            config = self.config_manager.get_monitored_layers()
            layer_config = config.get(layer_name)
            if not layer_config:
                self.show_message(f"No configuration found for '{layer_name}'", Qgis.Warning)
                continue

            fields = layer_config.get("fields", [])
            output_path = layer_config.get("output_path", "")
            if not fields:
                self.show_message(f"No fields configured for '{layer_name}'", Qgis.Warning)
                continue
            if not output_path:
                self.show_message(f"No output path configured for '{layer_name}'", Qgis.Warning)
                continue

            chosen_engine = self.export_engine if self.export_engine != "auto" else "openpyxl"

            try:
                task = self.excel_engine.create_export_task(
                    layer_name=layer_name,
                    layer=layer,
                    fields_to_export=fields,
                    output_path=output_path,
                    export_engine=chosen_engine,
                    on_finished_callback=self._on_task_finished,
                )
                self.active_tasks[layer_name] = task
                QgsApplication.taskManager().addTask(task)
            except Exception as e:
                QgsMessageLog.logMessage(
                    f"Error creating export task for {layer_name}: {e}",
                    "Excel Auto Exporter",
                    Qgis.Critical,
                )

    def _on_task_finished(self, layer_name, success, message):
        self.active_tasks.pop(layer_name, None)
        if success:
            self.show_message(f"✅ {message}", Qgis.Success)
        else:
            self.show_message(f"❌ {message}", Qgis.Critical)

    def export_layer_by_name(self, layer_name):
        """Manual/explicit export (synchronous compatibility path)."""
        try:
            layer = self.monitored_layers.get(layer_name) or self.get_layer_by_name(layer_name)
            if not layer or not layer.isValid():
                self.show_message(f"Layer '{layer_name}' not found", Qgis.Warning)
                return False

            config = self.config_manager.get_monitored_layers()
            layer_config = config.get(layer_name)
            if not layer_config:
                self.show_message(f"No configuration found for '{layer_name}'", Qgis.Warning)
                return False

            fields = layer_config.get("fields", [])
            output_path = layer_config.get("output_path", "")
            if not fields:
                self.show_message(f"No fields configured for '{layer_name}'", Qgis.Warning)
                return False
            if not output_path:
                self.show_message(f"No output path configured for '{layer_name}'", Qgis.Warning)
                return False

            chosen_engine = self.export_engine if self.export_engine != "auto" else "openpyxl"
            ok, msg = self.excel_engine.export_layer(layer, fields, output_path, chosen_engine)
            self.show_message(f"✅ {msg}" if ok else f"❌ {msg}", Qgis.Success if ok else Qgis.Critical)
            return ok
        except Exception as e:
            error_msg = f"Error exporting {layer_name}: {e}"
            self.show_message(error_msg, Qgis.Critical)
            QgsMessageLog.logMessage(error_msg, "Excel Auto Exporter", Qgis.Critical)
            return False

    def export_all_monitored(self):
        try:
            monitored = self.config_manager.get_monitored_layers()
            if not monitored:
                self.show_message("No layers configured for monitoring", Qgis.Warning)
                return

            success_count = 0
            fail_count = 0
            for layer_name in monitored.keys():
                if self.export_layer_by_name(layer_name):
                    success_count += 1
                else:
                    fail_count += 1

            self.show_message(
                f"Export complete: {success_count} successful, {fail_count} failed",
                Qgis.Success if fail_count == 0 else Qgis.Warning,
            )
        except Exception as e:
            self.show_message(f"Error exporting all layers: {e}", Qgis.Critical)

    def on_project_loaded(self):
        try:
            QTimer.singleShot(500, self.restore_monitoring_from_config)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error in on_project_loaded: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )

    def on_layers_added(self, layers):
        try:
            monitored = self.config_manager.get_monitored_layers()
            for layer in layers:
                if isinstance(layer, QgsVectorLayer) and layer.name() in monitored:
                    self.monitor_layer(layer.name())
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error in on_layers_added: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )

    def on_layers_removed(self, layer_ids):
        try:
            to_remove = []
            for layer_name, layer in self.monitored_layers.items():
                if layer.id() in layer_ids:
                    to_remove.append(layer_name)
            for layer_name in to_remove:
                del self.monitored_layers[layer_name]
                self.active_tasks.pop(layer_name, None)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error in on_layers_removed: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )

    def get_layer_by_name(self, layer_name):
        try:
            for layer in QgsProject.instance().mapLayers().values():
                if isinstance(layer, QgsVectorLayer) and layer.name() == layer_name:
                    return layer
            return None
        except Exception:
            return None

    def show_message(self, message, level=Qgis.Info):
        try:
            if self.iface:
                self.iface.messageBar().pushMessage(
                    "Excel Auto Exporter", message, level, duration=3
                )
            QgsMessageLog.logMessage(message, "Excel Auto Exporter", level)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error showing message: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )

    def show_dependency_warning(self):
        QMessageBox.warning(
            self.iface.mainWindow(),
            "Excel Auto Exporter - Missing Dependencies",
            "Required Python libraries are not installed:\n\n"
            "• pandas\n"
            "• openpyxl\n\n"
            "Please install them using:\n"
            "pip install pandas openpyxl\n\n"
            "Or from OSGeo4W Shell:\n"
            "python -m pip install pandas openpyxl",
        )
