# config_manager.py
# -*- coding: utf-8 -*-
"""
Configuration Manager
Handles saving and loading plugin configuration
"""

import json
import os
from qgis.core import QgsMessageLog, Qgis
from qgis.PyQt.QtCore import QSettings


class ConfigManager:
    """Manages plugin configuration persistence."""

    def __init__(self):
        self.settings = QSettings()
        self.config_key = "excel_auto_exporter"

    def save_config(self, config):
        try:
            config_json = json.dumps(config)
            self.settings.setValue(f"{self.config_key}/config", config_json)
            QgsMessageLog.logMessage(
                "Configuration saved successfully",
                "Excel Auto Exporter",
                Qgis.Info,
            )
            return True
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error saving configuration: {e}",
                "Excel Auto Exporter",
                Qgis.Critical,
            )
            return False

    def load_config(self):
        try:
            config_json = self.settings.value(f"{self.config_key}/config", "{}")
            config = json.loads(config_json)
            if not config:
                config = self.get_default_config()
            return config
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error loading configuration: {e}",
                "Excel Auto Exporter",
                Qgis.Warning,
            )
            return self.get_default_config()

    def get_default_config(self):
        return {
            "monitored_layers": {},
            "auto_export_enabled": True,
            "default_output_folder": os.path.expanduser("~/Documents/QGIS_Excel_Exports"),
        }

    def add_monitored_layer(self, layer_name, fields, output_path):
        config = self.load_config()
        config["monitored_layers"][layer_name] = {
            "fields": fields,
            "output_path": output_path,
        }
        self.save_config(config)

    def remove_monitored_layer(self, layer_name):
        config = self.load_config()
        if layer_name in config["monitored_layers"]:
            del config["monitored_layers"][layer_name]
            self.save_config(config)

    def get_monitored_layers(self):
        config = self.load_config()
        return config.get("monitored_layers", {})

    def is_auto_export_enabled(self):
        config = self.load_config()
        return config.get("auto_export_enabled", True)

    def set_auto_export(self, enabled):
        config = self.load_config()
        config["auto_export_enabled"] = enabled
        self.save_config(config)

    def get_default_output_folder(self):
        config = self.load_config()
        return config.get(
            "default_output_folder",
            os.path.expanduser("~/Documents/QGIS_Excel_Exports"),
        )

    def set_default_output_folder(self, folder_path):
        config = self.load_config()
        config["default_output_folder"] = folder_path
        self.save_config(config)
