# -*- coding: utf-8 -*-

"""
Excel Export Engine
Handles the actual export of layers to Excel format
"""

import os
import sys
import platform
from qgis.core import QgsMessageLog, Qgis, QgsVectorLayer, Qgis as QgisCore

class ExcelEngine:
    """Handles Excel export operations"""
    
    def __init__(self):
        self.pandas_available = self.check_dependencies()
        self.xlsxwriter_available = self.check_xlsxwriter()
        
    def check_dependencies(self):
        """
        Check if required Python libraries are available
        :return: True if dependencies are available
        :rtype: bool
        """
        try:
            import pandas
            import openpyxl
            return True
        except ImportError:
            QgsMessageLog.logMessage(
                "Missing dependencies: pandas and/or openpyxl not installed.\n"
                "Please install with: pip install pandas openpyxl",
                "Excel Auto Exporter",
                Qgis.Critical
            )
            return False
    
    def check_xlsxwriter(self):
        """
        Check if xlsxwriter is available (safer alternative to openpyxl)
        :return: True if xlsxwriter is available
        :rtype: bool
        """
        try:
            import importlib.util
            spec = importlib.util.find_spec("xlsxwriter")
            return spec is not None
        except:
            return False
    
    def is_problematic_environment(self):
        """
        Detect if running in known problematic environment (Windows + QGIS 3.34+)
        :return: True if environment is known to have openpyxl crashes
        :rtype: bool
        """
        if platform.system() != "Windows":
            return False
        
        try:
            from qgis.core import Qgis as QgisCore
            version = QgisCore.QGIS_VERSION
            # Extract major.minor version (e.g., "3.40.4" -> 3.40)
            parts = version.split('.')
            if len(parts) >= 2:
                major = int(parts[0])
                minor = int(parts[1])
                # QGIS 3.34+ on Windows has reported openpyxl crashes
                if major == 3 and minor >= 34:
                    return True
        except:
            pass
        
        return False
    
    def get_recommended_engine(self, user_preference='openpyxl'):
        """
        Determine best export engine based on environment and availability
        :param user_preference: User's preferred engine ('openpyxl', 'xlsxwriter', 'csv')
        :return: Tuple (engine, format_extension, warning_message)
        :rtype: tuple
        """
        is_problematic = self.is_problematic_environment()
        
        # If user wants openpyxl but we're in problematic environment
        if user_preference == 'openpyxl' and is_problematic:
            if self.xlsxwriter_available:
                QgsMessageLog.logMessage(
                    "⚠️ Windows + QGIS 3.34+ detected. Using xlsxwriter instead of openpyxl to avoid crashes.\n"
                    "Reference: https://github.com/qgis/QGIS/issues/58205",
                    "Excel Auto Exporter",
                    Qgis.Warning
                )
                return ('xlsxwriter', '.xlsx', 'Using xlsxwriter (safer on Windows/QGIS 3.40+)')
            else:
                QgsMessageLog.logMessage(
                    "⚠️ Windows + QGIS 3.34+ detected. openpyxl may cause crashes.\n"
                    "Falling back to CSV export. Install xlsxwriter for Excel support:\n"
                    "pip install xlsxwriter",
                    "Excel Auto Exporter",
                    Qgis.Warning
                )
                return ('csv', '.csv', 'Using CSV fallback (openpyxl crash risk detected)')
        
        # User explicitly wants xlsxwriter
        if user_preference == 'xlsxwriter':
            if self.xlsxwriter_available:
                return ('xlsxwriter', '.xlsx', None)
            else:
                return ('csv', '.csv', 'xlsxwriter not available, using CSV')
        
        # User explicitly wants CSV
        if user_preference == 'csv':
            return ('csv', '.csv', None)
        
        # Default: openpyxl (if not problematic)
        return ('openpyxl', '.xlsx', None)
    
    def export_layer(self, layer, fields_to_export, output_path, export_engine='openpyxl'):
        """
        Export a QGIS layer to Excel
        :param layer: QGIS vector layer to export
        :type layer: QgsVectorLayer
        :param fields_to_export: List of field names to include
        :type fields_to_export: list
        :param output_path: Path where Excel file will be saved
        :type output_path: str
        :param export_engine: Export engine to use ('openpyxl', 'xlsxwriter', 'csv')
        :type export_engine: str
        :return: Tuple (success, message)
        :rtype: tuple
        """
        if not self.pandas_available:
            return False, "Dependencies not available"
        
        if not layer or not layer.isValid():
            return False, f"Layer is not valid"
        
        try:
            import pandas as pd
            
            # Determine best engine and adjust output path if needed
            engine, ext, warning = self.get_recommended_engine(export_engine)
            
            # Adjust output path extension if needed
            base_path, current_ext = os.path.splitext(output_path)
            if current_ext != ext:
                output_path = base_path + ext
            
            # Collect data
            data = []
            for feature in layer.getFeatures():
                row = {}
                for field_name in fields_to_export:
                    # Auto-detect area fields and calculate from geometry
                    # Detects: area, qm, ha, superficie, fläche, flaeche, sqm, m2, etc.
                    is_area_field = any(keyword in field_name.lower() for keyword in [
                        'area', 'qm', 'ha', 'superficie', 'flache', 'fläche',
                        'sqm', 'm2', 'm²', 'hectare', 'acre'
                    ])
                    
                    if is_area_field:
                        # Always calculate area from geometry for area-related fields
                        if feature.geometry() and not feature.geometry().isEmpty():
                            area = feature.geometry().area()
                            row[field_name] = round(area, 2)
                        else:
                            row[field_name] = 0
                    else:
                        # Standard field reading for non-area fields
                        field_index = layer.fields().indexOf(field_name)
                        if field_index >= 0:
                            value = feature.attribute(field_name)
                            # Convert NULL to empty string for better Excel compatibility
                            row[field_name] = value if value is not None else ""
                        else:
                            row[field_name] = ""
                
                data.append(row)
            
            if not data:
                return False, "No data to export"
            
            # Create DataFrame
            df = pd.DataFrame(data)
            
            # Ensure output directory exists
            output_dir = os.path.dirname(output_path)
            if output_dir and not os.path.exists(output_dir):
                os.makedirs(output_dir, exist_ok=True)
            
            # Export using appropriate engine
            if engine == 'csv':
                df.to_csv(output_path, index=False, encoding='utf-8-sig')
                success_msg = f"Successfully exported {len(data)} features to {os.path.basename(output_path)} (CSV)"
            elif engine == 'xlsxwriter':
                writer = pd.ExcelWriter(output_path, engine='xlsxwriter')
                df.to_excel(writer, sheet_name='Sheet1', index=False)
                writer.close()
                success_msg = f"Successfully exported {len(data)} features to {os.path.basename(output_path)} (xlsxwriter)"
            else:  # openpyxl
                df.to_excel(output_path, index=False, engine='openpyxl')
                success_msg = f"Successfully exported {len(data)} features to {os.path.basename(output_path)}"
            
            # Add warning to message if fallback was used
            if warning:
                success_msg += f" [{warning}]"
            
            return True, success_msg
        
        except Exception as e:
            error_msg = f"Error exporting layer: {str(e)}"
            QgsMessageLog.logMessage(error_msg, "Excel Auto Exporter", Qgis.Critical)
            return False, error_msg
    
    def get_layer_fields(self, layer):
        """
        Get list of field names from a layer
        :param layer: QGIS vector layer
        :type layer: QgsVectorLayer
        :return: List of field names
        :rtype: list
        """
        if not layer or not layer.isValid():
            return []
        
        return [field.name() for field in layer.fields()]
