# -*- coding: utf-8 -*-

"""
Excel Auto Exporter Plugin for QGIS
Main plugin class - Universal version for public use
"""

import os
from qgis.PyQt.QtCore import QObject, QTimer, Qt
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtGui import QIcon
from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsMessageLog,
    Qgis,
    QgsSettings
)

from .config_manager import ConfigManager
from .excel_engine import ExcelEngine
from .exporter_dialog import ExporterDialog


class ExcelAutoExporter(QObject):
    """Main plugin class for Excel Auto Exporter"""
    
    def __init__(self, iface):
        """
        Constructor
        :param iface: An interface instance that will be passed to this class
        :type iface: QgsInterface
        """
        super().__init__()
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
        # Initialize managers
        self.config_manager = ConfigManager()
        self.excel_engine = ExcelEngine()
        
        # Plugin state
        self.monitored_layers = {}
        self.dialog = None
        self.toolbar = None
        self.action = None
        
        # Load export engine preference
        self.export_engine = self.get_export_engine_preference()
    
    def get_export_engine_preference(self):
        """
        Get user's preferred export engine from settings
        :return: Export engine preference ('openpyxl', 'xlsxwriter', 'csv', or 'auto')
        :rtype: str
        """
        settings = QgsSettings()
        engine = settings.value("excel_auto_exporter/export_engine", "auto")
        return engine
    
    def set_export_engine_preference(self, engine):
        """
        Save user's preferred export engine to settings
        :param engine: Export engine ('openpyxl', 'xlsxwriter', 'csv', or 'auto')
        :type engine: str
        """
        settings = QgsSettings()
        settings.setValue("excel_auto_exporter/export_engine", engine)
        self.export_engine = engine
    
    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI"""
        try:
            # Create action that will start plugin configuration
            icon_path = os.path.join(self.plugin_dir, 'icon.png')
            self.action = QAction(
                QIcon(icon_path),
                "Excel Auto Exporter",
                self.iface.mainWindow()
            )
            self.action.triggered.connect(self.run)
            self.action.setStatusTip("Configure Excel Auto Exporter")
            self.action.setWhatsThis("Configure layers to export to Excel automatically")
            
            # Add toolbar button and menu item
            self.iface.addToolBarIcon(self.action)
            self.iface.addPluginToMenu("&Excel Auto Exporter", self.action)
            
            # Connect to project signals
            QgsProject.instance().readProject.connect(self.on_project_loaded)
            QgsProject.instance().layersAdded.connect(self.on_layers_added)
            QgsProject.instance().layersRemoved.connect(self.on_layers_removed)
            
            # Check dependencies
            if not self.excel_engine.pandas_available:
                self.show_dependency_warning()
            
            # Load saved configuration and setup monitoring
            QTimer.singleShot(1000, self.restore_monitoring_from_config)
            
            self.show_message("Excel Auto Exporter loaded successfully", Qgis.Info)
            
        except Exception as e:
            self.show_message(f"Error initializing plugin: {str(e)}", Qgis.Critical)
    
    def unload(self):
        """Remove the plugin menu item and icon from QGIS GUI"""
        try:
            # Disconnect signals
            QgsProject.instance().readProject.disconnect(self.on_project_loaded)
            QgsProject.instance().layersAdded.disconnect(self.on_layers_added)
            QgsProject.instance().layersRemoved.disconnect(self.on_layers_removed)
        except:
            pass
        
        # Disconnect layer signals
        self.clear_all_monitoring()
        
        # Remove menu and toolbar
        self.iface.removePluginMenu("&Excel Auto Exporter", self.action)
        self.iface.removeToolBarIcon(self.action)
        
        # Clean up dialog
        if self.dialog:
            self.dialog.close()
    
    def run(self):
        """Run method that performs all the real work"""
        if not self.excel_engine.pandas_available:
            self.show_dependency_warning()
            return
        
        # Create dialog if not exists
        if not self.dialog:
            self.dialog = ExporterDialog(
                self.iface,
                self.config_manager,
                self.excel_engine,
                parent=self.iface.mainWindow()
            )
            
            # Connect dialog signals
            self.dialog.layers_config_changed.connect(self.update_monitoring)
            self.dialog.export_requested.connect(self.export_layer_by_name)
            self.dialog.export_all_requested.connect(self.export_all_monitored)
        
        # Show the dialog
        self.dialog.show()
        self.dialog.raise_()
        self.dialog.activateWindow()
    
    def restore_monitoring_from_config(self):
        """Restore layer monitoring from saved configuration"""
        try:
            monitored = self.config_manager.get_monitored_layers()
            
            if not monitored:
                return
            
            # Try to find and monitor configured layers
            for layer_name in monitored.keys():
                layer = self.get_layer_by_name(layer_name)
                if layer and layer.isValid():
                    self.monitor_layer(layer_name)
            
            if self.monitored_layers:
                count = len(self.monitored_layers)
                self.show_message(
                    f"Restored monitoring for {count} layer(s) from previous session",
                    Qgis.Success
                )
        
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error restoring configuration: {str(e)}",
                "Excel Auto Exporter",
                Qgis.Warning
            )
    
    def update_monitoring(self):
        """Update layer monitoring based on current configuration"""
        try:
            # Reload export engine preference (in case it was changed in dialog)
            self.export_engine = self.get_export_engine_preference()
            
            # Clear current monitoring
            self.clear_all_monitoring()
            
            # Get configured layers
            monitored = self.config_manager.get_monitored_layers()
            
            # Setup monitoring for each configured layer
            for layer_name in monitored.keys():
                self.monitor_layer(layer_name)
        
        except Exception as e:
            self.show_message(f"Error updating monitoring: {str(e)}", Qgis.Warning)
    
    def monitor_layer(self, layer_name):
        """
        Start monitoring a specific layer
        :param layer_name: Name of the layer to monitor
        :type layer_name: str
        """
        try:
            layer = self.get_layer_by_name(layer_name)
            
            if not layer or not layer.isValid():
                self.show_message(f"Layer '{layer_name}' not found or invalid", Qgis.Warning)
                return False
            
            # Disconnect previous connections if any
            if layer_name in self.monitored_layers:
                old_layer = self.monitored_layers[layer_name]
                try:
                    old_layer.attributeValueChanged.disconnect()
                    old_layer.geometryChanged.disconnect()
                    old_layer.featureAdded.disconnect()
                    old_layer.featureDeleted.disconnect()
                except:
                    pass
            
            # Only connect signals if auto-export is enabled
            if self.config_manager.is_auto_export_enabled():
                # Connect change signals
                layer.attributeValueChanged.connect(
                    lambda fid, field, value, ln=layer_name: self.on_layer_changed(ln)
                )
                layer.geometryChanged.connect(
                    lambda fid, geom, ln=layer_name: self.on_layer_changed(ln)
                )
                layer.featureAdded.connect(
                    lambda fid, ln=layer_name: self.on_layer_changed(ln)
                )
                layer.featureDeleted.connect(
                    lambda fid, ln=layer_name: self.on_layer_changed(ln)
                )
            
            # Store monitored layer
            self.monitored_layers[layer_name] = layer
            
            self.show_message(f"Monitoring activated for: {layer_name}", Qgis.Success)
            
            # Export immediately
            self.export_layer_by_name(layer_name)
            
            return True
        
        except Exception as e:
            self.show_message(f"Error monitoring layer {layer_name}: {str(e)}", Qgis.Critical)
            return False
    
    def clear_all_monitoring(self):
        """Stop monitoring all layers"""
        for layer_name, layer in self.monitored_layers.items():
            try:
                if layer and layer.isValid():
                    layer.attributeValueChanged.disconnect()
                    layer.geometryChanged.disconnect()
                    layer.featureAdded.disconnect()
                    layer.featureDeleted.disconnect()
            except:
                pass
        
        self.monitored_layers.clear()
    
    def on_layer_changed(self, layer_name):
        """
        Called when a monitored layer changes
        :param layer_name: Name of the changed layer
        :type layer_name: str
        """
        if self.config_manager.is_auto_export_enabled():
            self.export_layer_by_name(layer_name)
    
    def export_layer_by_name(self, layer_name):
        """
        Export a specific layer to Excel
        :param layer_name: Name of the layer to export
        :type layer_name: str
        """
        try:
            # Get layer
            layer = self.monitored_layers.get(layer_name) or self.get_layer_by_name(layer_name)
            
            if not layer or not layer.isValid():
                self.show_message(f"Layer '{layer_name}' not found", Qgis.Warning)
                return False
            
            # Get configuration
            config = self.config_manager.get_monitored_layers()
            layer_config = config.get(layer_name)
            
            if not layer_config:
                self.show_message(f"No configuration found for layer '{layer_name}'", Qgis.Warning)
                return False
            
            fields = layer_config.get('fields', [])
            output_path = layer_config.get('output_path', '')
            
            if not fields:
                self.show_message(f"No fields configured for layer '{layer_name}'", Qgis.Warning)
                return False
            
            if not output_path:
                self.show_message(f"No output path configured for layer '{layer_name}'", Qgis.Warning)
                return False
            
            # Export using excel engine WITH CONFIGURED ENGINE
            export_engine = self.export_engine if self.export_engine != 'auto' else 'openpyxl'
            success, message = self.excel_engine.export_layer(layer, fields, output_path, export_engine)
            
            if success:
                self.show_message(f"✅ {message}", Qgis.Success)
            else:
                self.show_message(f"❌ {message}", Qgis.Critical)
            
            return success
        
        except Exception as e:
            error_msg = f"Error exporting {layer_name}: {str(e)}"
            self.show_message(error_msg, Qgis.Critical)
            QgsMessageLog.logMessage(error_msg, "Excel Auto Exporter", Qgis.Critical)
            return False
    
    def export_all_monitored(self):
        """Export all monitored layers"""
        try:
            monitored = self.config_manager.get_monitored_layers()
            
            if not monitored:
                self.show_message("No layers configured for monitoring", Qgis.Warning)
                return
            
            success_count = 0
            fail_count = 0
            
            for layer_name in monitored.keys():
                if self.export_layer_by_name(layer_name):
                    success_count += 1
                else:
                    fail_count += 1
            
            self.show_message(
                f"Export complete: {success_count} successful, {fail_count} failed",
                Qgis.Success if fail_count == 0 else Qgis.Warning
            )
        
        except Exception as e:
            self.show_message(f"Error exporting all layers: {str(e)}", Qgis.Critical)
    
    def on_project_loaded(self):
        """Called when a project is loaded"""
        try:
            # Restore monitoring for configured layers
            QTimer.singleShot(500, self.restore_monitoring_from_config)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error in on_project_loaded: {str(e)}",
                "Excel Auto Exporter",
                Qgis.Warning
            )
    
    def on_layers_added(self, layers):
        """
        Called when layers are added to the project
        :param layers: List of added layers
        :type layers: list
        """
        try:
            monitored = self.config_manager.get_monitored_layers()
            
            for layer in layers:
                if isinstance(layer, QgsVectorLayer):
                    if layer.name() in monitored:
                        # Layer was configured but wasn't in project before
                        self.monitor_layer(layer.name())
        
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error in on_layers_added: {str(e)}",
                "Excel Auto Exporter",
                Qgis.Warning
            )
    
    def on_layers_removed(self, layer_ids):
        """
        Called when layers are removed from the project
        :param layer_ids: List of removed layer IDs
        :type layer_ids: list
        """
        try:
            # Remove from monitored layers if they were being monitored
            layers_to_remove = []
            for layer_name, layer in self.monitored_layers.items():
                if layer.id() in layer_ids:
                    layers_to_remove.append(layer_name)
            
            for layer_name in layers_to_remove:
                del self.monitored_layers[layer_name]
        
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error in on_layers_removed: {str(e)}",
                "Excel Auto Exporter",
                Qgis.Warning
            )
    
    def get_layer_by_name(self, layer_name):
        """
        Get a layer by its name
        :param layer_name: Name of the layer
        :type layer_name: str
        :return: QgsVectorLayer or None
        :rtype: QgsVectorLayer
        """
        try:
            for layer in QgsProject.instance().mapLayers().values():
                if isinstance(layer, QgsVectorLayer) and layer.name() == layer_name:
                    return layer
            return None
        except Exception:
            return None
    
    def show_message(self, message, level=Qgis.Info):
        """
        Show message in QGIS message bar and log
        :param message: Message to display
        :type message: str
        :param level: Message level
        :type level: Qgis.MessageLevel
        """
        try:
            if self.iface:
                self.iface.messageBar().pushMessage(
                    "Excel Auto Exporter",
                    message,
                    level,
                    duration=3
                )
            QgsMessageLog.logMessage(message, "Excel Auto Exporter", level)
        except Exception:
            pass
    
    def show_dependency_warning(self):
        """Show warning about missing dependencies"""
        QMessageBox.warning(
            self.iface.mainWindow(),
            "Excel Auto Exporter - Missing Dependencies",
            "Required Python libraries are not installed:\n\n"
            "• pandas\n"
            "• openpyxl\n\n"
            "Please install them using:\n"
            "pip install pandas openpyxl\n\n"
            "Or from OSGeo4W Shell:\n"
            "python -m pip install pandas openpyxl"
        )
