# -*- coding: utf-8 -*-
"""
Excel Export Engine
Handles the actual export of layers to Excel format
"""

import os
from qgis.core import QgsMessageLog, Qgis, QgsVectorLayer

class ExcelEngine:
    """Handles Excel export operations"""
    
    def __init__(self):
        self.pandas_available = self.check_dependencies()
    
    def check_dependencies(self):
        """
        Check if required Python libraries are available
        
        :return: True if dependencies are available
        :rtype: bool
        """
        try:
            import pandas
            import openpyxl
            return True
        except ImportError:
            QgsMessageLog.logMessage(
                "Missing dependencies: pandas and/or openpyxl not installed.\n"
                "Please install with: pip install pandas openpyxl",
                "Excel Auto Exporter",
                Qgis.Critical
            )
            return False
    
    def export_layer(self, layer, fields_to_export, output_path):
        """
        Export a QGIS layer to Excel
        
        :param layer: QGIS vector layer to export
        :type layer: QgsVectorLayer
        :param fields_to_export: List of field names to include
        :type fields_to_export: list
        :param output_path: Path where Excel file will be saved
        :type output_path: str
        :return: Tuple (success, message)
        :rtype: tuple
        """
        if not self.pandas_available:
            return False, "Dependencies not available"
        
        if not layer or not layer.isValid():
            return False, f"Layer is not valid"
        
        try:
            import pandas as pd
            
            # Collect data
            data = []
            for feature in layer.getFeatures():
                row = {}
                for field_name in fields_to_export:
                    # Auto-detect area fields and calculate from geometry
                    # Detects: area, qm, ha, superficie, fläche, flaeche, sqm, m2, etc.
                    is_area_field = any(keyword in field_name.lower() for keyword in [
                        'area', 'qm', 'ha', 'superficie', 'flache', 'fläche', 
                        'sqm', 'm2', 'm²', 'hectare', 'acre'
                    ])
                    
                    if is_area_field:
                        # Always calculate area from geometry for area-related fields
                        if feature.geometry() and not feature.geometry().isEmpty():
                            area = feature.geometry().area()
                            row[field_name] = round(area, 2)
                        else:
                            row[field_name] = 0
                    else:
                        # Standard field reading for non-area fields
                        field_index = layer.fields().indexOf(field_name)
                        if field_index >= 0:
                            value = feature.attribute(field_name)
                            # Convert NULL to empty string for better Excel compatibility
                            row[field_name] = value if value is not None else ""
                        else:
                            row[field_name] = ""
                
                data.append(row)
            
            if not data:
                return False, "No data to export"
            
            # Create DataFrame
            df = pd.DataFrame(data)
            
            # Ensure output directory exists
            output_dir = os.path.dirname(output_path)
            if output_dir and not os.path.exists(output_dir):
                os.makedirs(output_dir, exist_ok=True)
            
            # Export to Excel
            df.to_excel(output_path, index=False, engine='openpyxl')
            
            return True, f"Successfully exported {len(data)} features to {os.path.basename(output_path)}"
            
        except Exception as e:
            error_msg = f"Error exporting layer: {str(e)}"
            QgsMessageLog.logMessage(error_msg, "Excel Auto Exporter", Qgis.Critical)
            return False, error_msg
    
    def get_layer_fields(self, layer):
        """
        Get list of field names from a layer
        
        :param layer: QGIS vector layer
        :type layer: QgsVectorLayer
        :return: List of field names
        :rtype: list
        """
        if not layer or not layer.isValid():
            return []
        
        return [field.name() for field in layer.fields()]
